/**
 * @file PairingGenivi.h
 *
 * @par SW-Component
 * State machine for pairing
 *
 * @brief Implementation of Genivi pairing state machine.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Header file for implementation of Genivi pairing state machine.
 */

#ifndef _PAIRING_GENIVI_H_
#define _PAIRING_GENIVI_H_

#include "IPairingRequest.h"
#include "IPairingGenivi.h"
#include "IDeviceObserver.h"
#include "PairingGeniviData.h"
#include "EvolutionGeniviStackTypes.h"

namespace btstackif {
namespace genivi {

// class forward declarations
class IObjectPathManagerGenivi;

/**
 * Pairing class.
 */
class PairingGenivi : public IPairingRequest, public IPairingGenivi, public IDeviceObserver
{
public:
   /**
    * Default constructor.
    */
   PairingGenivi();

   /**
    * Destructor.
    */
   virtual ~PairingGenivi();

   virtual void reset(void);

   virtual void setCallback(IN IPairingCallback* callback);

   virtual void setControlIf(IN IBasicControl* control);

   virtual void startPairing(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address);

   virtual void cancelPairing(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address);

   virtual void setPinCode(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address, IN const BTSPinCode& pinCode);

   virtual void confirmSecureSimplePairing(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address, IN const BTSSspMode mode, IN const BTSNumericValue& numValue, IN const BTSConfirmationMode confirm);

   virtual void setObjectPathManagerIf(IN IObjectPathManagerGenivi* objectPathManager);

   virtual void handlePairingRequest(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSObjectPath& device, IN const BTSSspMode sspMode, IN const BTSNumericValue& numValue, IN const act_t token);

   virtual bool handleCancelRequest(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const bool sent, IN const act_t token);

   virtual bool handleCancelRequestRequest(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSObjectPath& device, IN const bool sent, IN const act_t token);

   virtual void handlePairResult(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSObjectPath& device, IN const BTSIpcCommonErrorCode errorCode);

   virtual void handleCancelPairingResult(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSObjectPath& device, IN const BTSIpcCommonErrorCode errorCode);

   virtual void handleReportLinkKey(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSObjectPath& device, IN const BTSLinkKey& linkKey);

   virtual void handleIncomingPairingComp(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSObjectPath& device, IN const BTSRequestResult result);

   virtual void setTriggerSimulateStackReset(IN const bool enable);

   virtual void deviceAdded(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address);

   virtual void deviceRemoved(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address);

   virtual void deviceAvailable(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address);

   virtual void deviceUnavailable(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address);

   virtual void deviceConnectionStatus(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const BTSStatusTransition aclTransition, IN const bool aclConnected, IN const BTSStatusTransition anyProfileTransition, IN const bool anyProfileConnected);

   virtual void deviceCreationFinished(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const BTSRequestResult result);

   virtual void deviceRemovalFinished(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const BTSRequestResult result);

private:
   IPairingCallback* _callback; /**< callback interface */
   IBasicControl* _controlIf; /**< basic control interface */
   IObjectPathManagerGenivi* _objectPathManagerIf; /**< object path manager interface */
   ::std::map< BTSObjectPath, act_t > _tokenList; /**< list with DBUS tokens */
   ::std::vector< PairingGeniviData > _pendingList; /**< list with pending devices */

   bool _testTriggerSimulateStackReset; /**< test flag */

   void createPairMsg(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, IN const BTSBDAddress& address, IN const BTSObjectPath& device) const;

   void createCancelPairingMsg(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, IN const BTSBDAddress& address, IN const BTSObjectPath& device) const;

   void createRequestPinCodeRes(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, IN const BTSBDAddress& address, IN const BTSPinCode& pinCode, IN const act_t token, IN const bool responseFlag = false) const;

   void createRequestConfirmationRes(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, IN const BTSBDAddress& address, IN const BTSConfirmationMode confirm, IN const act_t token, IN const bool responseFlag = false) const;

   void createRequestAuthorizationRes(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, IN const BTSBDAddress& address, IN const BTSConfirmationMode confirm, IN const act_t token, IN const bool responseFlag = false) const;

   void createRequestPasskeyRes(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, IN const BTSBDAddress& address, IN const BTSNumericValue& numValue, IN const act_t token, IN const bool responseFlag = false) const;

   void createDisplayPinCodeRes(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, IN const BTSBDAddress& address, IN const act_t token, IN const bool responseFlag = false) const;

   void createCancelRes(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, IN const act_t token, IN const bool responseFlag = false) const;

   void createCancelRequestRes(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, IN const act_t token, IN const bool responseFlag = false) const;

   void forwardRequest(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const BTSSspMode sspMode, IN const BTSNumericValue& numValue);

   void checkForDisplayPinCodeRequest(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, IN const BTSObjectPath& device, IN const BTSBDAddress& address, IN const BTSSspMode sspMode) const;

   void addDevice(IN const BTSObjectPath& device, IN const BTSSspMode sspMode, IN const BTSNumericValue& numValue);

   bool isDevicePending(IN const BTSObjectPath& device) const;

   bool isDevicePending(OUT BTSSspMode& sspMode, OUT BTSNumericValue& numValue, IN const BTSObjectPath& device) const;

   void removeDevice(IN const BTSObjectPath& device);
};

} //genivi
} //btstackif

#endif //_PAIRING_GENIVI_H_
