/**
 * @file ObjectPathManagerGenivi.h
 *
 * @par SW-Component
 * Control
 *
 * @brief StateMachine for object path handling.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details StateMachine for object path handling.
 */

#ifndef _OBJECT_PATH_MANAGER_GENIVI_H_
#define _OBJECT_PATH_MANAGER_GENIVI_H_

#include "IObjectPathManagerGenivi.h"

namespace btstackif {
namespace genivi {

/**
 *
 */
class ObjectPathManager : public IObjectPathManagerGenivi
{
public:
   /**
    * Get instance.
    *
    * @return = pointer to own instance
    */
   static IObjectPathManagerGenivi& getInstance(void);

   virtual void reset(void);

   //------------------------------------------------------------------------------

   virtual void setAddress2ObjectPath(IN const BTSBDAddress& address, IN const BTSObjectPath& objPath);

   virtual void removeAddress2ObjectPath(IN const BTSObjectPath& objPath);

   virtual bool getObjectPath4Address(OUT BTSObjectPath& objPath, IN const BTSBDAddress& address) const;

   virtual bool getAddress4ObjectPath(OUT BTSBDAddress& address, IN const BTSObjectPath& objPath) const;

   virtual void getAllObjectPaths4Address(OUT ::std::vector< BTSObjectPath >& objPathList) const;

   virtual void removeAllAddress2ObjectPathMappings(void);

   //------------------------------------------------------------------------------

   virtual void setProtocol2ObjectPath(IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSObjectPath& objPath);

   virtual void setProtocol2ObjectPath(IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSMasInstanceId masInstance, IN const BTSObjectPath& objPath);

   virtual void setProtocol2ObjectPath(IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSUuid& uuid, IN const BTSObjectPath& objPath);

   virtual void removeProtocol2ObjectPath(IN const BTSObjectPath& objPath);

   virtual void removeProtocol2ObjectPath(IN const BTSBDAddress& address, IN const BTSProtocolId protocol);

   virtual void removeProtocol2ObjectPath(IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSMasInstanceId masInstance);

   virtual void removeProtocol2ObjectPath(IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSUuid& uuid);

   virtual bool getObjectPath4Protocol(OUT BTSObjectPath& objPath, IN const BTSBDAddress& address, IN const BTSProtocolId protocol) const;

   virtual bool getObjectPath4Protocol(OUT BTSObjectPath& objPath, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSMasInstanceId masInstance) const;

   virtual bool getObjectPath4Protocol(OUT BTSObjectPath& objPath, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSUuid& uuid) const;

   virtual bool getProtocol4ObjectPath(OUT BTSBDAddress& address, OUT BTSProtocolId& protocol, IN const BTSObjectPath& objPath) const;

   virtual bool getProtocol4ObjectPath(OUT BTSBDAddress& address, OUT BTSProtocolId& protocol, OUT BTSMasInstanceId& masInstance, IN const BTSObjectPath& objPath) const;

   virtual bool getProtocol4ObjectPath(OUT BTSBDAddress& address, OUT BTSProtocolId& protocol, OUT BTSUuid& uuid, IN const BTSObjectPath& objPath) const;

   virtual void removeAllProtocol2ObjectPathMappings(void);

   virtual void printObjectPath2ProtocolMapping(IN const BTSObjectPath& objPath) const;

   //------------------------------------------------------------------------------

   virtual void setModem2ObjectPath(IN const BTSBDAddress& address, IN const BTSObjectPath& objPath);

   virtual void removeModem2ObjectPath(IN const BTSObjectPath& objPath);

   virtual bool getObjectPath4Modem(OUT BTSObjectPath& objPath, IN const BTSBDAddress& address) const;

   virtual bool getModem4ObjectPath(OUT BTSBDAddress& address, IN const BTSObjectPath& objPath) const;

   virtual void removeAllModem2ObjectPathMappings(void);

   //------------------------------------------------------------------------------

private:
   /**
    * Default constructor.
    */
   ObjectPathManager();

   /**
    * Destructor.
    */
   ~ObjectPathManager();

   /**
    * We need following:
    * # device:
    *    - for interfaces: org.bluez.Device1, org.bluez.evo.AVP, org.bluez.Network1
    *    - address <-> object path
    * # service/protocol:
    *    - for interfaces: org.bluez.Service, org.bluez.obex.Session + org.bluez.obex.PhonebookAccess, net.connman.Service
    *    - for protocols: HFP, AVP, PBAP, PAN(via ConnMan)
    *    - address + protocol <-> object path
    * # service/protocol:
    *    - for interfaces: org.bluez.obex.Session + org.bluez.obex.MessageAccess
    *    - for protocols: MAP
    *    - address + protocol + MAS instance <-> object path
    * # modem:
    *    - for interfaces: org.ofono.Handsfree, org.ofono.Modem, org.ofono.VoiceCallManager, org.ofono.NetworkRegistration, org.ofono.CallVolume, org.ofono.evo.Siri
    *    - address <-> object path
    * # call:
    *    - for interfaces: org.ofono.VoiceCall
    *    - address + call count <-> object path
    * # network operator:
    *    - for interfaces: org.ofono.NetworkOperator
    *    - address + operator count <-> object path
    * # transfer:
    *    - for interfaces: org.bluez.obex.Transfer
    *    - for protocols: MAP + PBAP
    *    - address + protocol + MAS instance (MAP) + transfer count <-> object path
    * # message:
    *    - for interfaces: org.bluez.obex.Message
    *    - for protocols: MAP
    *    - address + protocol + MAS instance (MAP) + message count <-> object path
    */

   /**
    * Service/Protocol entry.
    */
   class ProtocolEntry
   {
   public:
      ProtocolEntry();
      ProtocolEntry(IN const ProtocolEntry& ref);
      ProtocolEntry& operator=(IN const ProtocolEntry& ref);
      bool operator==(IN const ProtocolEntry& ref) const;
      bool operator!=(IN const ProtocolEntry& ref) const;
      bool operator<(const ProtocolEntry& ref) const;
      bool operator>(const ProtocolEntry& ref) const;
      virtual ~ProtocolEntry();
      // member with public access
      BTSBDAddress deviceAddress; /**< device address */
      BTSProtocolId protocolId; /**< protocol id */
      BTSUuid sppUuid; /**< SPP UUID */
      BTSMasInstanceId masInstanceId; /**< MAS instance id */

   protected:
      virtual int compare(const ProtocolEntry& ref) const;
   };

#if 0
   /**
    * Simple count based entry.
    */
   class SimpleCountEntry
   {
   public:
      SimpleCountEntry();
      SimpleCountEntry(IN const SimpleCountEntry& ref);
      SimpleCountEntry& operator=(IN const SimpleCountEntry& ref);
      bool operator==(IN const SimpleCountEntry& ref) const;
      bool operator!=(IN const SimpleCountEntry& ref) const;
      bool operator<(const SimpleCountEntry& ref) const;
      bool operator>(const SimpleCountEntry& ref) const;
      virtual ~SimpleCountEntry();
      // member with public access
      BTSBDAddress deviceAddress; /**< device address */
      unsigned int count; /**< count */

   protected:
      virtual int compare(const SimpleCountEntry& ref) const;
   };

   /**
    * Extended count based entry.
    */
   class ExtendedCountEntry : public ProtocolEntry
   {
   public:
      ExtendedCountEntry();
      ExtendedCountEntry(IN const ExtendedCountEntry& ref);
      ExtendedCountEntry& operator=(IN const ExtendedCountEntry& ref);
      bool operator==(IN const ExtendedCountEntry& ref) const;
      bool operator!=(IN const ExtendedCountEntry& ref) const;
      bool operator<(const ExtendedCountEntry& ref) const;
      bool operator>(const ExtendedCountEntry& ref) const;
      virtual ~ExtendedCountEntry();
      // member with public access
      unsigned int count; /**< count */

   protected:
      virtual int compare(const ExtendedCountEntry& ref) const;
   };
#endif

   /**
    * # device:
    */
   ::std::map<BTSBDAddress, BTSObjectPath> _device2ObjectPath; /**< address -> object path */
   ::std::map<BTSObjectPath, BTSBDAddress> _objectPath2Device; /**< object path -> address */

   /**
    * # service/protocol:
    */
   ::std::map<ProtocolEntry, BTSObjectPath> _protocol2ObjectPath; /**< protocol -> object path */
   ::std::map<BTSObjectPath, ProtocolEntry> _objectPath2Protocol; /**< object path -> protocol */

   /**
    * # modem:
    */
   ::std::map<BTSBDAddress, BTSObjectPath> _modem2ObjectPath; /**< address -> object path */
   ::std::map<BTSObjectPath, BTSBDAddress> _objectPath2Modem; /**< object path -> address */

#if 0
   /**
    * # call:
    */
   ::std::map<SimpleCountEntry, BTSObjectPath> _call2ObjectPath; /**< call -> object path */
   ::std::map<BTSObjectPath, SimpleCountEntry> _objectPath2Call; /**< object path -> call */

   /**
    * # network operator:
    */
   ::std::map<SimpleCountEntry, BTSObjectPath> _operator2ObjectPath; /**< operator -> object path */
   ::std::map<BTSObjectPath, SimpleCountEntry> _objectPath2Operator; /**< object path -> operator */

   /**
    * # transfer:
    */
   ::std::map<ExtendedCountEntry, BTSObjectPath> _transfer2ObjectPath; /**< transfer -> object path */
   ::std::map<BTSObjectPath, ExtendedCountEntry> _objectPath2Transfer; /**< object path -> transfer */

   /**
    * # message:
    */
   ::std::map<ExtendedCountEntry, BTSObjectPath> _message2ObjectPath; /**< message -> object path */
   ::std::map<BTSObjectPath, ExtendedCountEntry> _objectPath2Message; /**< object path -> message */
#endif

   //------------------------------------------------------------------------------

   void intSetProtocol2ObjectPath(IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSObjectPath& objPath);

   void intRemoveProtocol2ObjectPath(IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance);

   bool intGetObjectPath4Protocol(OUT BTSObjectPath& objPath, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance) const;

   bool intGetProtocol4ObjectPath(OUT BTSBDAddress& address, OUT BTSProtocolId& protocol, OUT BTSUuid& uuid, OUT BTSMasInstanceId& masInstance, IN const BTSObjectPath& objPath) const;
};

} //genivi
} //btstackif

#endif //_OBJECT_PATH_MANAGER_GENIVI_H_
