/**
 * @file LocalAdapterModesGenivi.h
 *
 * @par SW-Component
 * State machine for local adapter modes
 *
 * @brief Implementation of Genivi local adapter modes state machine.
 *
 * @copyright (C) 2017 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Header file for implementation of Genivi local adapter modes state machine.
 */

#ifndef _LOCAL_ADAPTER_MODES_GENIVI_H_
#define _LOCAL_ADAPTER_MODES_GENIVI_H_

#include "ILocalAdapterModesRequest.h"
#include "ILocalAdapterModesGenivi.h"
#include "EvolutionGeniviStackTypes.h"

namespace btstackif {
namespace genivi {

/**
 * Local adapter modes class.
 */
class LocalAdapterModesGenivi : public ILocalAdapterModesRequest, public ILocalAdapterModesGenivi
{
public:
   /**
    * Default constructor.
    */
   LocalAdapterModesGenivi();

   /**
    * Destructor.
    */
   virtual ~LocalAdapterModesGenivi();

   virtual void reset(void);

   virtual void setCallback(IN ILocalAdapterModesCallback* callback);

   virtual void setControlIf(IN IBasicControl* control);

   virtual void setDiscoverable(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const bool enabled);

   virtual void setPairable(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const bool enabled);

   virtual void setConnectable(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const bool enabled);

   virtual void setDiscoverableTimeout(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSDiscoverableTimeout timeout);

   virtual void setPairableTimout(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSDiscoverableTimeout timeout);

   virtual void getDiscoverable(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList);

   virtual void getPairable(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList);

   virtual void getConnectable(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList);

   virtual void sendVirtualDiscoverableUpdate(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const bool enabled, IN const BTSIpcCommonErrorCode errorCode);

   virtual void sendVirtualPairableUpdate(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const bool enabled, IN const BTSIpcCommonErrorCode errorCode);

   virtual void sendVirtualConnectableUpdate(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const bool enabled, IN const BTSIpcCommonErrorCode errorCode);

   virtual void handleDiscoverableUpdate(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const bool enabled, IN const BTSIpcCommonErrorCode errorCode, IN const bool response, IN const bool force = false);

   virtual void handlePairableUpdate(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const bool enabled, IN const BTSIpcCommonErrorCode errorCode, IN const bool response, IN const bool force = false);

   virtual void handleConnectableUpdate(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const bool enabled, IN const BTSIpcCommonErrorCode errorCode, IN const bool response, IN const bool force = false);

   virtual void handleDiscoverableTimeoutUpdate(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSDiscoverableTimeout timeout, IN const BTSIpcCommonErrorCode errorCode, IN const bool response, IN const bool force = false);

   virtual void handlePairableTimeoutUpdate(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSDiscoverableTimeout timeout, IN const BTSIpcCommonErrorCode errorCode, IN const bool response, IN const bool force = false);

   virtual void handleAdapterProperties(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSDbusPropertyList& properties, IN const BTSIpcCommonErrorCode errorCode);

private:
   ILocalAdapterModesCallback* _callback; /**< callback interface */
   IBasicControl* _controlIf; /**< control interface */
   bool _adapterDataReceived; /**< flag for data received */

   void setInitialDiscoverableMode(IN const bool enabled);

   void setInitialPairableMode(IN const bool enabled);

   void setInitialConnectableMode(IN const bool enabled);

   void createSetAdapterDiscoverableMsg(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, IN const bool enabled) const;

   void createSetAdapterPairableMsg(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, IN const bool enabled) const;

   void createSetAdapterConnectableMsg(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, IN const bool enabled) const;

   void createSetAdapterDiscoverabletimeoutMsg(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, IN const BTSDiscoverableTimeout timeout) const;

   void createSetAdapterPairabletimeoutMsg(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, IN const BTSDiscoverableTimeout timeout) const;

   void createGetAdapterDiscoverableMsg(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList) const;

   void createGetAdapterPairableMsg(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList) const;

   void createGetAdapterConnectableMsg(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList) const;
};

} //genivi
} //btstackif

#endif //_LOCAL_ADAPTER_MODES_GENIVI_H_
