/**
 * @file LinkKeyGenivi.h
 *
 * @par SW-Component
 * State machine for link key
 *
 * @brief Implementation of Genivi link key state machine.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Header file for implementation of Genivi link key state machine.
 */

#ifndef _LINK_KEY_GENIVI_H_
#define _LINK_KEY_GENIVI_H_

#include "ILinkKeyRequest.h"
#include "ILinkKeyGenivi.h"
#include "IDeviceObserver.h"
#include "EvolutionGeniviStackTypes.h"

namespace btstackif {
namespace genivi {

// class forward declarations
class IObjectPathManagerGenivi;

/**
 * Link key class.
 */
class LinkKeyGenivi : public ILinkKeyRequest, public ILinkKeyGenivi, public IDeviceObserver
{
public:
   /**
    * Default constructor.
    */
   LinkKeyGenivi();

   /**
    * Destructor.
    */
   virtual ~LinkKeyGenivi();

   virtual void reset(void);

   virtual void setCallback(IN ILinkKeyCallback* callback);

   virtual void setLinkKey(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address, IN const BTSLinkKeyType linkKeyType, IN const BTSLinkKey& linkKey, IN const BTSDLinkKey& dLinkKey);

   virtual void setObjectPathManagerIf(IN IObjectPathManagerGenivi* objectPathManager);

   virtual void handleRequestLinkkey(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSObjectPath& device, IN const act_t token);

   virtual bool handleCancelRequest(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const bool sent, IN const act_t token);

   virtual bool handleCancelRequestRequest(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSObjectPath& device, IN const bool sent, IN const act_t token);

   virtual void deviceAdded(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address);

   virtual void deviceRemoved(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address);

   virtual void deviceAvailable(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address);

   virtual void deviceUnavailable(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address);

   virtual void deviceConnectionStatus(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const BTSStatusTransition aclTransition, IN const bool aclConnected, IN const BTSStatusTransition anyProfileTransition, IN const bool anyProfileConnected);

   virtual void deviceCreationFinished(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const BTSRequestResult result);

   virtual void deviceRemovalFinished(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const BTSRequestResult result);

private:
   ILinkKeyCallback* _callback; /**< callback interface */
   IObjectPathManagerGenivi* _objectPathManagerIf; /**< object path manager interface */
   ::std::map< BTSObjectPath, act_t > _tokenList; /**< list with DBUS tokens */
   ::std::vector< BTSObjectPath > _pendingList; /**< list with pending devices */

   void createRequestLinkkeyResMsg(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, IN const BTSBDAddress& address, IN const BTSLinkKey& linkKey, IN const act_t token) const;

   void createCancelRes(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, IN const act_t token, IN const bool responseFlag = false) const;

   void createCancelRequestRes(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, IN const act_t token, IN const bool responseFlag = false) const;

   bool isDevicePending(IN const BTSObjectPath& device) const;

   void removeDevice(IN const BTSObjectPath& device);
};

} //genivi
} //btstackif

#endif //_LINK_KEY_GENIVI_H_
