/**
 * @file ISwitchBluetoothGenivi.h
 *
 * @par SW-Component
 * State machine for switch Bluetooth on/off
 *
 * @brief Interface definition for Genivi switch Bluetooth on/off state machine.
 *
 * @copyright (C) 2018 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details This file defines the Genivi interface for switch Bluetooth on/off state machine.
 */

#ifndef _I_SWITCH_BLUETOOTH_GENIVI_H_
#define _I_SWITCH_BLUETOOTH_GENIVI_H_

#include "EvolutionGeniviStackTypes.h"

namespace btstackif {

// class forward declarations
class ISwitchBluetoothRequest;
class Bts2Ipc_BaseMessage;
class Bts2App_BaseMessage;

namespace genivi {

/**
 * Interface definition.
 */
class ISwitchBluetoothGenivi
{
public:
   /**
    * Destructor.
    */
   virtual ~ISwitchBluetoothGenivi() {}

   /**
    * Get request interface.
    *
    * @return = request interface
    */
   virtual ISwitchBluetoothRequest* getRequestIf(void) = 0;

   /**
    * Handle received adapter properties.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] properties: properties
    * @param[in] errorCode: error code
    */
   virtual void handleAdapterProperties(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSDbusPropertyList& properties, IN const BTSIpcCommonErrorCode errorCode) = 0;

   /**
    * Handle data returned for HW version request.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] status: status flag
    * @param[in] btVersion: BT version
    * @param[in] manufacturerName: manufacturer name
    * @param[in] subVersion: sub version
    * @param[in] extInfo: extended information
    */
   virtual void handleGetHwVersionConfirmation(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSStatusCode status, IN const BTSBtCoreVersion btVersion, IN const BTSManufacturerNameController manufacturerName, IN const BTSSubVersionController subVersion, IN const BTSExtensionInfo& extInfo) = 0;

   /**
    * Handle HCI mode update.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] hciMode: HCI mode flag
    * @param[in] errorCode: error code
    */
   virtual void handleHciModeUpdate(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const bool hciMode, IN const BTSIpcCommonErrorCode errorCode) = 0;

   /**
    * Handle adapter powered update message.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] powered: powered flag
    * @param[in] errorCode: error code
    * @param[in] response: response flag (update or return)
    * @param[in] force: force flag
    */
   virtual void handlePoweredUpdate(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const bool powered, IN const BTSIpcCommonErrorCode errorCode, IN const bool response, IN const bool force = false) = 0;
};

} //genivi
} //btstackif

#endif //_I_SWITCH_BLUETOOTH_GENIVI_H_
