/**
 * @file IStartupGenivi.h
 *
 * @par SW-Component
 * State machine for startup
 *
 * @brief Interface definition for Genivi startup state machine.
 *
 * @copyright (C) 2018 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details This file defines the Genivi interface for startup state machine.
 */

#ifndef _I_STARTUP_GENIVI_H_
#define _I_STARTUP_GENIVI_H_

#include "EvolutionGeniviStackTypes.h"

namespace btstackif {

// class forward declarations
class IStartupRequest;
class DbusServiceStatus;
class Bts2Ipc_BaseMessage;
class Bts2App_BaseMessage;

namespace genivi {

// class forward declarations
class IHfpDeviceCapabilitiesGenivi;

/**
 * Interface definition.
 */
class IStartupGenivi
{
public:
   /**
    * Destructor.
    */
   virtual ~IStartupGenivi() {}

   /**
    * Set HFP device capabilities interface.
    *
    * @param[in] hfpDeviceCapabilitiesGeniviIf: HFP device capabilities interface
    */
   virtual void setHfpDeviceCapabilitiesGeniviIf(IN IHfpDeviceCapabilitiesGenivi* hfpDeviceCapabilitiesGeniviIf) = 0;

   /**
    * Get request interface.
    *
    * @return = request interface
    */
   virtual IStartupRequest* getRequestIf(void) = 0;

   /**
    * Add to DBus service group.
    *
    * @param[in] userMode: user mode
    * @param[in] status: DBus service status entry
    * @param[in] enabled: enabled flag for DBus service status entry
    */
   virtual void addToServiceGroup(IN const BTSUserMode userMode, IN DbusServiceStatus& status, IN const bool enabled) = 0;

   /**
    * Add DBus interface(s).
    *
    * @param[out] dbusInterfaces: DBus interfaces
    * @param[in] userMode: user mode
    */
   virtual void addInterfaces(OUT ::std::vector< BTSDbusInterfaceItem >& dbusInterfaces, IN const BTSUserMode userMode) = 0;

   /**
    * Update service availability.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] interface: interface
    * @param[in] availability: DBUS service available flag
    * @param[in] busType: bus type
    * @param[in] busName: bus name
    * @param[in] objPath: object path
    *
    * @return = true: group availability shall be checked,
    * @return = false: check of group availability not necessary
    */
   virtual bool updateDbusServiceAvailability(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem,
            IN const BTSGenDbusServiceInterface interface, IN const BTSDbusServiceAvailability availability, IN const BTSCommonEnumClass busType, IN const BTSBusName& busName, IN const BTSObjectPath& objPath) = 0;

   /**
    * Check new group service availability.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] checkGroupAvailability: check or check not
    */
   virtual void checkNewGroupAvailability(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const bool checkGroupAvailability) = 0;

   /**
    * Handle received adapter properties.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] properties: properties
    * @param[in] errorCode: error code
    */
   virtual void handleAdapterProperties(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSDbusPropertyList& properties, IN const BTSIpcCommonErrorCode errorCode) = 0;

   /**
    * Handle adapter initialization.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    */
   virtual void handleAdapterInitialization(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem) = 0;

   /**
    * Handle register agent result.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] errorCode: error code
    */
   virtual void handleRegisterAgentResult(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSIpcCommonErrorCode errorCode) = 0;
};

} //genivi
} //btstackif

#endif //_I_STARTUP_GENIVI_H_
