/**
 * @file IPairingGenivi.h
 *
 * @par SW-Component
 * State machine for pairing
 *
 * @brief Interface definition for Genivi pairing state machine.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details This file defines the Genivi interface for pairing state machine.
 */

#ifndef _I_PAIRING_GENIVI_H_
#define _I_PAIRING_GENIVI_H_

#include "BtStackInternalTypes.h"

namespace btstackif {

// class forward declarations
class Bts2Ipc_BaseMessage;
class Bts2App_BaseMessage;

namespace genivi {

// class forward declarations
class IObjectPathManagerGenivi;

/**
 * Interface definition.
 */
class IPairingGenivi
{
public:
   /**
    * Destructor.
    */
   virtual ~IPairingGenivi() {}

   /**
    * Set object path manager interface.
    *
    * @param[in] objectPathManager: object path manager interface
    */
   virtual void setObjectPathManagerIf(IN IObjectPathManagerGenivi* objectPathManager) = 0;

   /**
    * Handle pairing request from remote device.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] device: device object path
    * @param[in] sspMode: pairing mode
    * @param[in] numValue: numeric value / passkey
    * @param[in] token: DBUS token
    */
   virtual void handlePairingRequest(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSObjectPath& device, IN const BTSSspMode sspMode, IN const BTSNumericValue& numValue, IN const act_t token) = 0;

   /**
    * Handle cancel request from stack.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] sent: flag indicating that response message was already sent
    * @param[in] token: DBUS token
    *
    * @return = true: request was answered,
    * @return = false: request was not answered
    */
   virtual bool handleCancelRequest(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const bool sent, IN const act_t token) = 0;

   /**
    * Handle cancel request from stack.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] device: object path of device
    * @param[in] sent: flag indicating that response message was already sent
    * @param[in] token: DBUS token
    *
    * @return = true: request was answered,
    * @return = false: request was not answered
    */
   virtual bool handleCancelRequestRequest(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSObjectPath& device, IN const bool sent, IN const act_t token) = 0;

   /**
    * Handle pair result.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] device: device object path
    * @param[in] errorCode: error code
    */
   virtual void handlePairResult(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSObjectPath& device, IN const BTSIpcCommonErrorCode errorCode) = 0;

   /**
    * Handle cancel pairing result.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] device: device object path
    * @param[in] errorCode: error code
    */
   virtual void handleCancelPairingResult(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSObjectPath& device, IN const BTSIpcCommonErrorCode errorCode) = 0;

   /**
    * Handle report link key indication.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] device: device object path
    * @param[in] linkKey: link key
    */
   virtual void handleReportLinkKey(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSObjectPath& device, IN const BTSLinkKey& linkKey) = 0;

   /**
    * Handle incoming pairing complete indication.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] device: device object path
    * @param[in] result: result
    */
   virtual void handleIncomingPairingComp(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSObjectPath& device, IN const BTSRequestResult result) = 0;

   /**
    * Set test flag.
    *
    * @param[in] enable: test flag
    */
   virtual void setTriggerSimulateStackReset(IN const bool enable) = 0;
};

} //genivi
} //btstackif

#endif //_I_PAIRING_GENIVI_H_
