/**
 * @file IObjectPathManagerGenivi.h
 *
 * @par SW-Component
 * Control
 *
 * @brief StateMachine for object path handling (interface).
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details StateMachine for object path handling (interface).
 */

#ifndef _I_OBJECT_PATH_MANAGER_GENIVI_H_
#define _I_OBJECT_PATH_MANAGER_GENIVI_H_

#include "EvolutionGeniviStackTypes.h"

namespace btstackif {
namespace genivi {

/**
 *
 */
class IObjectPathManagerGenivi
{
public:
   /**
    * Destructor.
    */
   virtual ~IObjectPathManagerGenivi() {}

   /**
    * Reset all internal data.
    */
   virtual void reset(void) = 0;

   //------------------------------------------------------------------------------

   /**
    * Set address vs. object path mapping.
    *
    * @param[in] address: device address
    * @param[in] objPath: object path
    */
   virtual void setAddress2ObjectPath(IN const BTSBDAddress& address, IN const BTSObjectPath& objPath) = 0;

   /**
    * Remove address vs. object path mapping.
    *
    * @param[in] objPath: object path
    */
   virtual void removeAddress2ObjectPath(IN const BTSObjectPath& objPath) = 0;

   /**
    * Get object path for address.
    *
    * @param[out] objPath: object path
    * @param[in] address: device address
    *
    * @return = true: mapping found,
    * @return = false: mapping not found
    */
   virtual bool getObjectPath4Address(OUT BTSObjectPath& objPath, IN const BTSBDAddress& address) const = 0;

   /**
    * Get address for object path.
    *
    * @param[out] address: device address
    * @param[in] objPath: object path
    *
    * @return = true: mapping found,
    * @return = false: mapping not found
    */
   virtual bool getAddress4ObjectPath(OUT BTSBDAddress& address, IN const BTSObjectPath& objPath) const = 0;

   /**
    * Get all object paths for stored addresses.
    *
    * @param[out] objPathList: object path list
    */
   virtual void getAllObjectPaths4Address(OUT ::std::vector< BTSObjectPath >& objPathList) const = 0;

   /**
    * Remove all address vs. object path mappings.
    */
   virtual void removeAllAddress2ObjectPathMappings(void) = 0;

   //------------------------------------------------------------------------------

   /**
    * Set service/protocol vs. object path mapping.
    * Valid for: HFP, AVP, PBAP, PAN(via ConnMan).
    *
    * @param[in] address: device address
    * @param[in] protocol: service/protocol
    * @param[in] objPath: object path
    */
   virtual void setProtocol2ObjectPath(IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSObjectPath& objPath) = 0;

   /**
    * Set service/protocol vs. object path mapping.
    * Valid for: MAP.
    *
    * @param[in] address: device address
    * @param[in] protocol: service/protocol
    * @param[in] masInstance: MAS instance
    * @param[in] objPath: object path
    */
   virtual void setProtocol2ObjectPath(IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSMasInstanceId masInstance, IN const BTSObjectPath& objPath) = 0;

   /**
    * Set service/protocol vs. object path mapping.
    * Valid for: SPP.
    *
    * @param[in] address: device address
    * @param[in] protocol: service/protocol
    * @param[in] uuid: SPP UUID
    * @param[in] objPath: object path
    */
   virtual void setProtocol2ObjectPath(IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSUuid& uuid, IN const BTSObjectPath& objPath) = 0;

   /**
    * Remove service/protocol vs. object path mapping.
    * Valid for: HFP, AVP, PBAP, PAN(via ConnMan), MAP.
    *
    * @param[in] objPath: object path
    */
   virtual void removeProtocol2ObjectPath(IN const BTSObjectPath& objPath) = 0;

   /**
    * Remove service/protocol vs. object path mapping.
    * Valid for: HFP, AVP, PBAP, PAN(via ConnMan).
    *
    * @param[in] address: device address
    * @param[in] protocol: service/protocol
    */
   virtual void removeProtocol2ObjectPath(IN const BTSBDAddress& address, IN const BTSProtocolId protocol) = 0;

   /**
    * Remove service/protocol vs. object path mapping.
    * Valid for: MAP.
    *
    * @param[in] address: device address
    * @param[in] protocol: service/protocol
    * @param[in] masInstance: MAS instance
    */
   virtual void removeProtocol2ObjectPath(IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSMasInstanceId masInstance) = 0;

   /**
    * Remove service/protocol vs. object path mapping.
    * Valid for: SPP.
    *
    * @param[in] address: device address
    * @param[in] protocol: service/protocol
    * @param[in] uuid: SPP UUID
    */
   virtual void removeProtocol2ObjectPath(IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSUuid& uuid) = 0;

   /**
    * Get object path for service/protocol.
    * Valid for: HFP, AVP, PBAP, PAN(via ConnMan).
    *
    * @param[out] objPath: object path
    * @param[in] address: device address
    * @param[in] protocol: service/protocol
    *
    * @return = true: mapping found,
    * @return = false: mapping not found
    */
   virtual bool getObjectPath4Protocol(OUT BTSObjectPath& objPath, IN const BTSBDAddress& address, IN const BTSProtocolId protocol) const = 0;

   /**
    * Get object path for service/protocol.
    * Valid for: MAP.
    *
    * @param[out] objPath: object path
    * @param[in] address: device address
    * @param[in] protocol: service/protocol
    * @param[in] masInstance: MAS instance
    *
    * @return = true: mapping found,
    * @return = false: mapping not found
    */
   virtual bool getObjectPath4Protocol(OUT BTSObjectPath& objPath, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSMasInstanceId masInstance) const = 0;

   /**
    * Get object path for service/protocol.
    * Valid for: SPP.
    *
    * @param[out] objPath: object path
    * @param[in] address: device address
    * @param[in] protocol: service/protocol
    * @param[in] uuid: SPP UUID
    *
    * @return = true: mapping found,
    * @return = false: mapping not found
    */
   virtual bool getObjectPath4Protocol(OUT BTSObjectPath& objPath, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSUuid& uuid) const = 0;

   /**
    * Get service/protocol for object path.
    * Valid for: HFP, AVP, PBAP, PAN(via ConnMan).
    *
    * @param[out] address: device address
    * @param[out] protocol: service/protocol
    * @param[in] objPath: object path
    *
    * @return = true: mapping found,
    * @return = false: mapping not found
    */
   virtual bool getProtocol4ObjectPath(OUT BTSBDAddress& address, OUT BTSProtocolId& protocol, IN const BTSObjectPath& objPath) const = 0;

   /**
    * Get service/protocol for object path.
    * Valid for: MAP.
    *
    * @param[out] address: device address
    * @param[out] protocol: service/protocol
    * @param[out] masInstance: MAS instance
    * @param[in] objPath: object path
    *
    * @return = true: mapping found,
    * @return = false: mapping not found
    */
   virtual bool getProtocol4ObjectPath(OUT BTSBDAddress& address, OUT BTSProtocolId& protocol, OUT BTSMasInstanceId& masInstance, IN const BTSObjectPath& objPath) const = 0;

   /**
    * Get service/protocol for object path.
    * Valid for: SPP.
    *
    * @param[out] address: device address
    * @param[out] protocol: service/protocol
    * @param[out] uuid: SPP UUID
    * @param[in] objPath: object path
    *
    * @return = true: mapping found,
    * @return = false: mapping not found
    */
   virtual bool getProtocol4ObjectPath(OUT BTSBDAddress& address, OUT BTSProtocolId& protocol, OUT BTSUuid& uuid, IN const BTSObjectPath& objPath) const = 0;

   /**
    * Remove all protocol vs. object path mappings.
    */
   virtual void removeAllProtocol2ObjectPathMappings(void) = 0;

   /**
    * Print object path to service/protocol mapping.
    *
    * @param[in] objPath: object path
    */
   virtual void printObjectPath2ProtocolMapping(IN const BTSObjectPath& objPath) const = 0;

   //------------------------------------------------------------------------------

   /**
    * Set modem vs. object path mapping.
    *
    * @param[in] address: modem address
    * @param[in] objPath: object path
    */
   virtual void setModem2ObjectPath(IN const BTSBDAddress& address, IN const BTSObjectPath& objPath) = 0;

   /**
    * Remove modem vs. object path mapping.
    *
    * @param[in] objPath: object path
    */
   virtual void removeModem2ObjectPath(IN const BTSObjectPath& objPath) = 0;

   /**
    * Get object path for modem.
    *
    * @param[out] objPath: object path
    * @param[in] address: modem address
    *
    * @return = true: mapping found,
    * @return = false: mapping not found
    */
   virtual bool getObjectPath4Modem(OUT BTSObjectPath& objPath, IN const BTSBDAddress& address) const = 0;

   /**
    * Get modem for object path.
    *
    * @param[out] address: modem address
    * @param[in] objPath: object path
    *
    * @return = true: mapping found,
    * @return = false: mapping not found
    */
   virtual bool getModem4ObjectPath(OUT BTSBDAddress& address, IN const BTSObjectPath& objPath) const = 0;

   /**
    * Remove all modem vs. object path mappings.
    */
   virtual void removeAllModem2ObjectPathMappings(void) = 0;

   //------------------------------------------------------------------------------
};

} //genivi
} //btstackif

#endif //_I_OBJECT_PATH_MANAGER_GENIVI_H_
