/**
 * @file IDeviceManagerGenivi.h
 *
 * @par SW-Component
 * State machine for device manager
 *
 * @brief Interface definition for Genivi device manager state machine.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details This file defines the Genivi interface for device manager state machine.
 */

#ifndef _I_DEVICE_MANAGER_GENIVI_H_
#define _I_DEVICE_MANAGER_GENIVI_H_

#include "BtStackInternalTypes.h"

namespace btstackif {

// class forward declarations
class Bts2Ipc_BaseMessage;
class Bts2App_BaseMessage;
class App2Bts_BaseMessage;

namespace genivi {

// class forward declarations
class IObjectPathManagerGenivi;

/**
 * Interface definition.
 */
class IDeviceManagerGenivi
{
public:
   /**
    * Destructor.
    */
   virtual ~IDeviceManagerGenivi() {}

   /**
    * Set object path manager interface.
    *
    * @param[in] objectPathManager: object path manager interface
    */
   virtual void setObjectPathManagerIf(IN IObjectPathManagerGenivi* objectPathManager) = 0;

   /**
    * Signal for device added.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] address: device address
    * @param[in] device: device object path
    */
   virtual void deviceAdded(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const BTSObjectPath& device) = 0;

   /**
    * Signal for device added.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] properties: device properties
    */
   virtual void deviceAdded(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSDbusPropertyList& properties) = 0;

   /**
    * Signal for device added.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] address: device address
    * @param[in] name: name
    * @param[in] cod: cod
    * @param[in] connected: connected
    */
   virtual void deviceAdded(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const BTSDeviceName& name, IN const BTSCod cod, IN const bool connected) = 0;

   /**
    * Signal for device removed.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] device: device object path
    */
   virtual void deviceRemoved(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSObjectPath& device) = 0;

   /**
    * Update ACL status of device connection.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] device: device object path
    * @param[in] connected: connected flag
    * @param[in] errorCode: error code
    * @param[in] response: response flag (update or return)
    */
   virtual void updateAclStatus(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSObjectPath& device, IN const bool connected, IN const BTSIpcCommonErrorCode errorCode, IN const bool response) = 0;

   /**
    * Update device name.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] device: device object path
    * @param[in] name: name
    * @param[in] errorCode: error code
    * @param[in] response: response flag (update or return)
    */
   virtual void updateName(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSObjectPath& device, IN const BTSDeviceName& name, IN const BTSIpcCommonErrorCode errorCode, IN const bool response) = 0;

   /**
    * Update role.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] device: device object path
    * @param[in] master: master/slave flag
    * @param[in] errorCode: error code
    * @param[in] response: response flag (update or return)
    */
   virtual void updateRole(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSObjectPath& device, IN const bool master, IN const BTSIpcCommonErrorCode errorCode, IN const bool response) = 0;

   /**
    * Update control pause.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] device: device object path
    * @param[in] enableAvpPause: enable avp pause flag
    * @param[in] errorCode: error code
    * @param[in] response: response flag (update or return)
    */
   virtual void updateEnableAvpPause(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSObjectPath& device, IN const bool enableAvpPause, IN const BTSIpcCommonErrorCode errorCode, IN const bool response) = 0;

   /**
    * Update power mode.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] device: device object path
    * @param[in] mode: power mode
    * @param[in] errorCode: error code
    * @param[in] response: response flag (update or return)
    */
   virtual void updatePowerMode(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSObjectPath& device, IN const BTSBtPowerMode mode, IN const BTSIpcCommonErrorCode errorCode, IN const bool response) = 0;

   /**
    * Update service availability.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] device: device object path
    * @param[in] availability: DBUS service available flag
    */
   virtual void updateDbusServiceAvailability(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSObjectPath& device, IN const BTSDbusServiceAvailability availability) = 0;

   /**
    * Handle result for device disconnect.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] device: device object path
    * @param[in] errorCode: error code (result)
    */
   virtual void handleDeviceDisconnectResult(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSObjectPath& device, IN const BTSIpcCommonErrorCode errorCode) = 0;

   /**
    * Handle result for create device.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] address: device address (always set)
    * @param[in] device: device object path (only set in success case)
    * @param[in] errorCode: error code (result)
    */
   virtual void handleCreateDeviceResult(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const BTSObjectPath& device, IN const BTSIpcCommonErrorCode errorCode) = 0;

   /**
    * Handle result for remove device.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] errorCode: error code (result)
    */
   virtual void handleRemoveDeviceResult(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSIpcCommonErrorCode errorCode) = 0;

   /**
    * Get App2Bts message. TODO: remove after connection management is completely reworked
    *
    * @param[in] address: device address
    *
    * @return = App2Bts message if found else NULL
    */
   virtual App2Bts_BaseMessage* getApp2BtsWorkingMessage(IN const BTSBDAddress& address) = 0;
};

} //genivi
} //btstackif

#endif //_I_DEVICE_MANAGER_GENIVI_H_
