/**
 * @file DeviceManagerGenivi.h
 *
 * @par SW-Component
 * State machine for device manager
 *
 * @brief Implementation of Genivi device manager state machine.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Header file for implementation of Genivi device manager state machine.
 */

#ifndef _DEVICE_MANAGER_GENIVI_H_
#define _DEVICE_MANAGER_GENIVI_H_

#include "IDeviceManagerRequest.h"
#include "IDeviceManagerGenivi.h"
#include "EvolutionGeniviStackTypes.h"
#include "DbusServiceAvailability.h"

namespace btstackif {
namespace genivi {

// class forward declarations
class IObjectPathManagerGenivi;

/**
 * Device manager class.
 */
class DeviceManagerGenivi : public IDeviceManagerRequest, public IDeviceManagerGenivi
{
public:
   /**
    * Default constructor.
    */
   DeviceManagerGenivi();

   /**
    * Destructor.
    */
   virtual ~DeviceManagerGenivi();

   virtual void reset(void);

   virtual void setCallback(IN IDeviceManagerCallback* callback);

   virtual void setControlIf(IN IBasicControl* control);

   virtual void disconnect(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address, IN const bool pauseBtStreaming);

   virtual void create(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address);

   virtual void remove(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address);

   virtual void sendVirtualDeviceConnectedUpdate(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address, IN const bool connected, IN const BTSIpcCommonErrorCode errorCode);

   virtual void setObjectPathManagerIf(IN IObjectPathManagerGenivi* objectPathManager);

   virtual void deviceAdded(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const BTSObjectPath& device);

   virtual void deviceAdded(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSDbusPropertyList& properties);

   virtual void deviceAdded(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const BTSDeviceName& name, IN const BTSCod cod, IN const bool connected);

   virtual void deviceRemoved(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSObjectPath& device);

   virtual void updateAclStatus(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSObjectPath& device, IN const bool connected, IN const BTSIpcCommonErrorCode errorCode, IN const bool response);

   virtual void updateName(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSObjectPath& device, IN const BTSDeviceName& name, IN const BTSIpcCommonErrorCode errorCode, IN const bool response);

   virtual void updateRole(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSObjectPath& device, IN const bool master, IN const BTSIpcCommonErrorCode errorCode, IN const bool response);

   virtual void updateEnableAvpPause(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSObjectPath& device, IN const bool enableAvpPause, IN const BTSIpcCommonErrorCode errorCode, IN const bool response);

   virtual void updatePowerMode(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSObjectPath& device, IN const BTSBtPowerMode mode, IN const BTSIpcCommonErrorCode errorCode, IN const bool response);

   virtual void updateDbusServiceAvailability(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSObjectPath& device, IN const BTSDbusServiceAvailability availability);

   virtual void handleDeviceDisconnectResult(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSObjectPath& device, IN const BTSIpcCommonErrorCode errorCode);

   virtual void handleCreateDeviceResult(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const BTSObjectPath& device, IN const BTSIpcCommonErrorCode errorCode);

   virtual void handleRemoveDeviceResult(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSIpcCommonErrorCode errorCode);

   virtual App2Bts_BaseMessage* getApp2BtsWorkingMessage(IN const BTSBDAddress& address);

private:
   IDeviceManagerCallback* _callback; /**< callback interface */
   IBasicControl* _controlIf; /**< basic control interface */
   IObjectPathManagerGenivi* _objectPathManagerIf; /**< object path manager interface */
   DbusServiceStatus _serviceStatusDevice; /**< device service status */

   void createDeviceDisconnectMsg(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, IN const BTSBDAddress& address, IN const BTSObjectPath& device) const;

   void createDeviceCreateMsg(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, IN const BTSBDAddress& address) const;

   void createDeviceRemoveMsg(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, IN const BTSBDAddress& address, IN const BTSObjectPath& device) const;

   void createSetEnableAvpPauseMsg(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, IN const BTSBDAddress& address, IN const BTSObjectPath& device, IN const bool enableAvpPause) const;

   void createDeviceGetDataMsg(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, IN const BTSBDAddress& address, IN const BTSObjectPath& device) const;

   void setDbusServiceAvailability(OUT unsigned int& secondaryData, IN const bool enable) const;

   bool getDbusServiceAvailability(IN const unsigned int secondaryData) const;

   void checkForDataComplete(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN IDeviceManagerCallback& callback, IN const BTSBDAddress& address) const;
};

} //genivi
} //btstackif

#endif //_DEVICE_MANAGER_GENIVI_H_
