/**
 * @file IntrospectableDbusCallbackIf.cpp
 *
 * @par SW-Component
 * CcDbusIf
 *
 * @brief Introspectable DBUS Callback implementation.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Introspectable DBUS callback implementation.
 */

#include "IntrospectableDbusCallbackIf.h"
#include "Ipc2Bts_MessageWrapper_GEN.h"
#include "BtsUtils.h"

#include "cc_dbus_if/EvolutionGeniviUtility.h"
#include "cc_dbus_if/EvolutionGeniviDbusTypes.h"
#include "cc_dbus_if/EvolutionGeniviDbusParser.h"

#include "TraceClasses.h"
#include "FwAssert.h"
#include "FwTrace.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_BTS_DBUS
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/IntrospectableDbusCallbackIf.cpp.trc.h"
#endif
#endif

using namespace ::introspect::org::freedesktop::DBus::Introspectable;

using namespace ::asf::core;

namespace btstackif {
namespace genivi {

IntrospectableDbusCallbackIf::IntrospectableDbusCallbackIf()
{
}

IntrospectableDbusCallbackIf::IntrospectableDbusCallbackIf(IDbusRecHandler* recHandler) : DbusCallbackIf(recHandler)
{
}

IntrospectableDbusCallbackIf::~IntrospectableDbusCallbackIf()
{
}

// "ServiceAvailableIF" implementation --- start
void IntrospectableDbusCallbackIf::onIntrospectableProxyAvailable(const ::std::string& busName, const ::std::string& objPath, const ::ccdbusif::DbusBusType busType, void* user, const ServiceState previousState, const ServiceState currentState)
{
   (void)(user);
   (void)(previousState);

   BTSDbusServiceAvailability availabilityEvent = BTS_DBUS_SERVICE_NOT_AVAILABLE;

   if(ServiceState__Available == currentState)
   {
      availabilityEvent = BTS_DBUS_SERVICE_AVAILABLE;
   }

   Ipc2Bts_ServiceAvailabilityConnection* ptrMsg = ptrNew_Ipc2Bts_ServiceAvailabilityConnection();

   if(NULL != ptrMsg)
   {
      ptrMsg->setInterface(BTS_GEN_DBUS_SERVICE_INTROSPECTABLE);
      ptrMsg->setAvailabilityEvent(availabilityEvent);
      ptrMsg->setBusType(busType);
      ptrMsg->setBusName(busName);
      ptrMsg->setObjPath(objPath);
   }

   onSignal(ptrMsg, ::ccdbusif::DEFAULT_ACT, true);
}

void IntrospectableDbusCallbackIf::onIntrospectableProxyUnavailable(const ::std::string& busName, const ::std::string& objPath, const ::ccdbusif::DbusBusType busType, void* user, const ServiceState previousState, const ServiceState currentState)
{
   (void)(user);
   (void)(previousState);

   BTSDbusServiceAvailability availabilityEvent = BTS_DBUS_SERVICE_NOT_AVAILABLE;

   if(ServiceState__Available == currentState)
   {
      availabilityEvent = BTS_DBUS_SERVICE_AVAILABLE;
   }

   Ipc2Bts_ServiceAvailabilityConnection* ptrMsg = ptrNew_Ipc2Bts_ServiceAvailabilityConnection();

   if(NULL != ptrMsg)
   {
      ptrMsg->setInterface(BTS_GEN_DBUS_SERVICE_INTROSPECTABLE);
      ptrMsg->setAvailabilityEvent(availabilityEvent);
      ptrMsg->setBusType(busType);
      ptrMsg->setBusName(busName);
      ptrMsg->setObjPath(objPath);
   }

   onSignal(ptrMsg, ::ccdbusif::DEFAULT_ACT, true);
}
// "ServiceAvailableIF" implementation --- end

// TODO: complete rework is needed; object path has to be set for all messages

void IntrospectableDbusCallbackIf::onIntrospectError(const ::boost::shared_ptr< IntrospectableProxy >& proxy, const ::boost::shared_ptr< IntrospectError >& error)
{
   (void)(proxy);
   (void)(error);
}

void IntrospectableDbusCallbackIf::onIntrospectResponse(const ::boost::shared_ptr< IntrospectableProxy >& proxy, const ::boost::shared_ptr< IntrospectResponse >& response)
{
   // HINT: proxy object delivers following data:
   //    const std::string& getDBusBusName() const;
   //    const std::string& getDBusObjectPath() const;
   //    const std::string& getInterfaceName() const;
   //    TODO: ASF 1.25.0-rc1 --- ::DBusBusType getBusType() const;

   ::ccdbusif::evolution::EvolutionGeniviDbusParser parser;

   parser.setTraces(true);
   parser.parseIntrospection(response->getXml(), ::ccdbusif::BUS_TYPE_SYSTEM /*TODO: ASF 1.25.0-rc1 --- proxy->getBusType()*/, proxy->getDBusBusName(), proxy->getDBusObjectPath(), proxy->getInterfaceName());

   // TODO
}

} //genivi
} //btstackif
