/**
 * @file EvolutionGeniviDbusServiceCallbackIf.cpp
 *
 * @par SW-Component
 * BtStackIf
 *
 * @brief EvolutionGenivi DBUS Callback Interface for Service.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details EvolutionGenivi DBUS Callback Interface for Service.
 */

#include "EvolutionGeniviDbusServiceCallbackIf.h"
#include "Ipc2Bts_MessageWrapper_GEN.h"
#include "BtsUtils.h"
#include "EvolutionGeniviUtils.h"
#include "cc_dbus_if/EvolutionGeniviDbusParser.h"

using namespace ::org::bluez::Service;
using namespace ::asf::core;

namespace btstackif {
namespace genivi {

EvolutionGeniviDbusServiceCallbackIf::EvolutionGeniviDbusServiceCallbackIf()
{
}

EvolutionGeniviDbusServiceCallbackIf::EvolutionGeniviDbusServiceCallbackIf(IDbusRecHandler* recHandler) : DbusCallbackIf(recHandler)
{
}

EvolutionGeniviDbusServiceCallbackIf::~EvolutionGeniviDbusServiceCallbackIf()
{
}

// "ServiceAvailableIF" implementation --- start
void EvolutionGeniviDbusServiceCallbackIf::onProxyAvailable(const ServiceState previousState, const ServiceState currentState, const ::std::string& objPath)
{
   (void)(previousState);

   BTSDbusServiceAvailability availabilityEvent = BTS_DBUS_SERVICE_NOT_AVAILABLE;

   if(ServiceState__Available == currentState)
   {
      availabilityEvent = BTS_DBUS_SERVICE_AVAILABLE;
   }

   Ipc2Bts_ServiceAvailabilityConnection* ptrMsg = ptrNew_Ipc2Bts_ServiceAvailabilityConnection();

   if(NULL != ptrMsg)
   {
      ptrMsg->setInterface(BTS_GEN_DBUS_SERVICE_SERVICE);
      ptrMsg->setObjPath(objPath);
      ptrMsg->setAvailabilityEvent(availabilityEvent);
   }

   onSignal(ptrMsg, ::ccdbusif::DEFAULT_ACT, true);
}

void EvolutionGeniviDbusServiceCallbackIf::onProxyUnavailable(const ServiceState previousState, const ServiceState currentState, const ::std::string& objPath)
{
   (void)(previousState);

   BTSDbusServiceAvailability availabilityEvent = BTS_DBUS_SERVICE_NOT_AVAILABLE;

   if(ServiceState__Available == currentState)
   {
      availabilityEvent = BTS_DBUS_SERVICE_AVAILABLE;
   }

   Ipc2Bts_ServiceAvailabilityConnection* ptrMsg = ptrNew_Ipc2Bts_ServiceAvailabilityConnection();

   if(NULL != ptrMsg)
   {
      ptrMsg->setInterface(BTS_GEN_DBUS_SERVICE_SERVICE);
      ptrMsg->setObjPath(objPath);
      ptrMsg->setAvailabilityEvent(availabilityEvent);
   }

   onSignal(ptrMsg, ::ccdbusif::DEFAULT_ACT, true);
}
// "ServiceAvailableIF" implementation --- end

// org/bluez/ServiceProxy implementation --- start
void EvolutionGeniviDbusServiceCallbackIf::onConnectError(const ::boost::shared_ptr< ServiceProxy >& proxy, const ::boost::shared_ptr< ConnectError >& error)
{
   Ipc2Bts_ConnectResult* ptrResultMsg = ptrNew_Ipc2Bts_ConnectResult();
   if(NULL != ptrResultMsg)
   {
      ptrResultMsg->setService(proxy->getDBusObjectPath());
   }
   onError(ptrResultMsg, error, convertErrorCode2InternalValue(error->getName(), error->getMessage()));
}

void EvolutionGeniviDbusServiceCallbackIf::onConnectResponse(const ::boost::shared_ptr< ServiceProxy >& proxy, const ::boost::shared_ptr< ConnectResponse >& response)
{
   Ipc2Bts_ConnectResult* ptrResultMsg = ptrNew_Ipc2Bts_ConnectResult();
   if(NULL != ptrResultMsg)
   {
      ptrResultMsg->setService(proxy->getDBusObjectPath());
   }
   onResponse(ptrResultMsg, response);
}

void EvolutionGeniviDbusServiceCallbackIf::onDisconnectError(const ::boost::shared_ptr< ServiceProxy >& proxy, const ::boost::shared_ptr< DisconnectError >& error)
{
   Ipc2Bts_DisconnectResult* ptrResultMsg = ptrNew_Ipc2Bts_DisconnectResult();
   if(NULL != ptrResultMsg)
   {
      ptrResultMsg->setService(proxy->getDBusObjectPath());
   }
   onError(ptrResultMsg, error, convertErrorCode2InternalValue(error->getName(), error->getMessage()));
}

void EvolutionGeniviDbusServiceCallbackIf::onDisconnectResponse(const ::boost::shared_ptr< ServiceProxy >& proxy, const ::boost::shared_ptr< DisconnectResponse >& response)
{
   Ipc2Bts_DisconnectResult* ptrResultMsg = ptrNew_Ipc2Bts_DisconnectResult();
   if(NULL != ptrResultMsg)
   {
      ptrResultMsg->setService(proxy->getDBusObjectPath());
   }
   onResponse(ptrResultMsg, response);
}

void EvolutionGeniviDbusServiceCallbackIf::onRemoteUUIDError(const ::boost::shared_ptr< ServiceProxy >& proxy, const ::boost::shared_ptr< RemoteUUIDError >& error)
{
   Ipc2Bts_ServiceRemoteUuidUpdate* ptrResultMsg = ptrNew_Ipc2Bts_ServiceRemoteUuidUpdate();
   if(NULL != ptrResultMsg)
   {
      ptrResultMsg->setService(proxy->getDBusObjectPath());
   }
   onError(ptrResultMsg, error, convertErrorCode2InternalValue(error->getName(), error->getMessage()));
}

void EvolutionGeniviDbusServiceCallbackIf::onRemoteUUIDUpdate(const ::boost::shared_ptr< ServiceProxy >& proxy, const ::boost::shared_ptr< RemoteUUIDUpdate >& update)
{
   Ipc2Bts_ServiceRemoteUuidUpdate* ptrResultMsg = ptrNew_Ipc2Bts_ServiceRemoteUuidUpdate();
   if(NULL != ptrResultMsg)
   {
      ptrResultMsg->setService(proxy->getDBusObjectPath());
      ::ccdbusif::evolution::EvolutionGeniviDbusParser parser;
      ptrResultMsg->setProtocol(convertUuid2Protocol(parser.getUuid2Enum(update->getRemoteUUID())));
   }
   onUpdate(ptrResultMsg, update->getAct());
}

void EvolutionGeniviDbusServiceCallbackIf::onStateError(const ::boost::shared_ptr< ServiceProxy >& proxy, const ::boost::shared_ptr< StateError >& error)
{
   Ipc2Bts_ServiceStateUpdate* ptrResultMsg = ptrNew_Ipc2Bts_ServiceStateUpdate();
   if(NULL != ptrResultMsg)
   {
      ptrResultMsg->setService(proxy->getDBusObjectPath());
   }
   onError(ptrResultMsg, error, convertErrorCode2InternalValue(error->getName(), error->getMessage()));
}

void EvolutionGeniviDbusServiceCallbackIf::onStateUpdate(const ::boost::shared_ptr< ServiceProxy >& proxy, const ::boost::shared_ptr< StateUpdate >& update)
{
   Ipc2Bts_ServiceStateUpdate* ptrResultMsg = ptrNew_Ipc2Bts_ServiceStateUpdate();
   if(NULL != ptrResultMsg)
   {
      ptrResultMsg->setService(proxy->getDBusObjectPath());
      ::ccdbusif::evolution::EvolutionGeniviDbusParser parser;
      ptrResultMsg->setServiceState(convertServiceState2InternalValue(parser.convertString2ServiceState(update->getState())));
   }
   onUpdate(ptrResultMsg, update->getAct());
}

void EvolutionGeniviDbusServiceCallbackIf::onDeviceError(const ::boost::shared_ptr< ServiceProxy >& proxy, const ::boost::shared_ptr< DeviceError >& error)
{
   Ipc2Bts_ServiceDeviceUpdate* ptrResultMsg = ptrNew_Ipc2Bts_ServiceDeviceUpdate();
   if(NULL != ptrResultMsg)
   {
      ptrResultMsg->setService(proxy->getDBusObjectPath());
   }
   onError(ptrResultMsg, error, convertErrorCode2InternalValue(error->getName(), error->getMessage()));
}

void EvolutionGeniviDbusServiceCallbackIf::onDeviceUpdate(const ::boost::shared_ptr< ServiceProxy >& proxy, const ::boost::shared_ptr< DeviceUpdate >& update)
{
   Ipc2Bts_ServiceDeviceUpdate* ptrResultMsg = ptrNew_Ipc2Bts_ServiceDeviceUpdate();
   if(NULL != ptrResultMsg)
   {
      ptrResultMsg->setService(proxy->getDBusObjectPath());
      ptrResultMsg->setDevice(update->getDevice());
   }
   onUpdate(ptrResultMsg, update->getAct());
}

void EvolutionGeniviDbusServiceCallbackIf::onVersionError(const ::boost::shared_ptr< ServiceProxy >& proxy, const ::boost::shared_ptr< VersionError >& error)
{
   Ipc2Bts_ServiceVersionUpdate* ptrResultMsg = ptrNew_Ipc2Bts_ServiceVersionUpdate();
   if(NULL != ptrResultMsg)
   {
      ptrResultMsg->setService(proxy->getDBusObjectPath());
   }
   onError(ptrResultMsg, error, convertErrorCode2InternalValue(error->getName(), error->getMessage()));
}

void EvolutionGeniviDbusServiceCallbackIf::onVersionUpdate(const ::boost::shared_ptr< ServiceProxy >& proxy, const ::boost::shared_ptr< VersionUpdate >& update)
{
   Ipc2Bts_ServiceVersionUpdate* ptrResultMsg = ptrNew_Ipc2Bts_ServiceVersionUpdate();
   if(NULL != ptrResultMsg)
   {
      ptrResultMsg->setService(proxy->getDBusObjectPath());
      ptrResultMsg->setVersion(update->getVersion());
   }
   onUpdate(ptrResultMsg, update->getAct());
}
// org/bluez/ServiceProxy implementation --- end

} //genivi
} //btstackif
