/**
 * @file ConnManDbusServiceCallbackIf.cpp
 *
 * @par SW-Component
 * BtStackIf
 *
 * @brief ConnMan DBUS Callback Interface for Service.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details ConnMan DBUS Callback Interface for Service.
 */

#include "ConnManDbusServiceCallbackIf.h"
#include "Ipc2Bts_MessageWrapper_GEN.h"
// #include "BtsUtils.h"
#include "EvolutionGeniviUtils.h"
#include "cc_dbus_if/ConnManDbusParser.h"
#include "cc_dbus_if/ConnManUtility.h"
#include "FwStringUtils.h"
#include "FwAssert.h"

using namespace ::net::connman::Service;
using namespace ::asf::core;

namespace btstackif {
namespace genivi {

ConnManDbusServiceCallbackIf::ConnManDbusServiceCallbackIf()
{
}

ConnManDbusServiceCallbackIf::ConnManDbusServiceCallbackIf(IDbusRecHandler* recHandler) : DbusCallbackIf(recHandler)
{
}

ConnManDbusServiceCallbackIf::~ConnManDbusServiceCallbackIf()
{
}

// "ServiceAvailableIF" implementation --- start
void ConnManDbusServiceCallbackIf::onProxyAvailable(const ServiceState previousState, const ServiceState currentState, const ::std::string& objPath)
{
   (void)(previousState);

   BTSDbusServiceAvailability availabilityEvent = BTS_DBUS_SERVICE_NOT_AVAILABLE;

   if(ServiceState__Available == currentState)
   {
      availabilityEvent = BTS_DBUS_SERVICE_AVAILABLE;
   }

   Ipc2Bts_ServiceAvailabilityConnection* ptrMsg = ptrNew_Ipc2Bts_ServiceAvailabilityConnection();

   if(NULL != ptrMsg)
   {
      ptrMsg->setInterface(BTS_GEN_DBUS_SERVICE_CONNMAN_SERVICE);
      ptrMsg->setObjPath(objPath);
      ptrMsg->setAvailabilityEvent(availabilityEvent);
   }

   onSignal(ptrMsg, ::ccdbusif::DEFAULT_ACT, true);
}

void ConnManDbusServiceCallbackIf::onProxyUnavailable(const ServiceState previousState, const ServiceState currentState, const ::std::string& objPath)
{
   (void)(previousState);

   BTSDbusServiceAvailability availabilityEvent = BTS_DBUS_SERVICE_NOT_AVAILABLE;

   if(ServiceState__Available == currentState)
   {
      availabilityEvent = BTS_DBUS_SERVICE_AVAILABLE;
   }

   Ipc2Bts_ServiceAvailabilityConnection* ptrMsg = ptrNew_Ipc2Bts_ServiceAvailabilityConnection();

   if(NULL != ptrMsg)
   {
      ptrMsg->setInterface(BTS_GEN_DBUS_SERVICE_CONNMAN_SERVICE);
      ptrMsg->setObjPath(objPath);
      ptrMsg->setAvailabilityEvent(availabilityEvent);
   }

   onSignal(ptrMsg, ::ccdbusif::DEFAULT_ACT, true);
}
// "ServiceAvailableIF" implementation --- end

// net/connman/ServiceProxy implementation --- start
void ConnManDbusServiceCallbackIf::onGetPropertiesError(const ::boost::shared_ptr< ServiceProxy >& proxy, const ::boost::shared_ptr< GetPropertiesError >& error)
{
   (void)(proxy);
   (void)(error);
}

void ConnManDbusServiceCallbackIf::onGetPropertiesResponse(const ::boost::shared_ptr< ServiceProxy >& proxy, const ::boost::shared_ptr< GetPropertiesResponse >& response)
{
   (void)(proxy);
   (void)(response);
}

void ConnManDbusServiceCallbackIf::onConnectError(const ::boost::shared_ptr< ServiceProxy >& proxy, const ::boost::shared_ptr< ConnectError >& error)
{
   Ipc2Bts_ConnManServiceConnectResult* ptrResultMsg = ptrNew_Ipc2Bts_ConnManServiceConnectResult();
   if(NULL != ptrResultMsg)
   {
      ptrResultMsg->setDevice(proxy->getDBusObjectPath());
   }
   onError(ptrResultMsg, error, convertErrorCode2InternalValue(error->getName(), error->getMessage()));
}

void ConnManDbusServiceCallbackIf::onConnectResponse(const ::boost::shared_ptr< ServiceProxy >& proxy, const ::boost::shared_ptr< ConnectResponse >& response)
{
   Ipc2Bts_ConnManServiceConnectResult* ptrResultMsg = ptrNew_Ipc2Bts_ConnManServiceConnectResult();
   if(NULL != ptrResultMsg)
   {
      ptrResultMsg->setDevice(proxy->getDBusObjectPath());
   }
   onResponse(ptrResultMsg, response->getAct());
}

void ConnManDbusServiceCallbackIf::onDisconnectError(const ::boost::shared_ptr< ServiceProxy >& proxy, const ::boost::shared_ptr< DisconnectError >& error)
{
   Ipc2Bts_ConnManServiceDisconnectResult* ptrResultMsg = ptrNew_Ipc2Bts_ConnManServiceDisconnectResult();
   if(NULL != ptrResultMsg)
   {
      ptrResultMsg->setDevice(proxy->getDBusObjectPath());
   }
   onError(ptrResultMsg, error, convertErrorCode2InternalValue(error->getName(), error->getMessage()));
}

void ConnManDbusServiceCallbackIf::onDisconnectResponse(const ::boost::shared_ptr< ServiceProxy >& proxy, const ::boost::shared_ptr< DisconnectResponse >& response)
{
   Ipc2Bts_ConnManServiceDisconnectResult* ptrResultMsg = ptrNew_Ipc2Bts_ConnManServiceDisconnectResult();
   if(NULL != ptrResultMsg)
   {
      ptrResultMsg->setDevice(proxy->getDBusObjectPath());
   }
   onResponse(ptrResultMsg, response->getAct());
}

void ConnManDbusServiceCallbackIf::onPropertyChangedError(const ::boost::shared_ptr< ServiceProxy >& proxy, const ::boost::shared_ptr< PropertyChangedError >& error)
{
   (void)(proxy);
   (void)(error);
   // is never triggered
   FW_NORMAL_ASSERT_ALWAYS();
}

void ConnManDbusServiceCallbackIf::onPropertyChangedSignal(const ::boost::shared_ptr< ServiceProxy >& proxy, const ::boost::shared_ptr< PropertyChangedSignal >& signal)
{
   const ::std::string& objPath = proxy->getDBusObjectPath();

   // check for bluetooth as part of object path to avoid internal message traffic (no engineering solution but it helps TODO: maybe store Bluetooth related object paths here and check if match)
   ::std::string copyObjPath(objPath);
   ::fw::convertString2LowerCase(copyObjPath);
   if(false == ::ccdbusif::connman_utility::isBluetoothObjectPath(copyObjPath))
   {
      // ignore
      return;
   }

   /*
    * we are interested in following properties:
    * - State: ready/online => connected
    * - Ethernet: dict entry Interface
    */
   ::ccdbusif::connman::ConnManDbusParser parser;
   parser.setTraces(true);
   ::ccdbusif::connman::ServiceProperty propName = parser.getServiceProperty2Enum(signal->getName());

   if(::ccdbusif::connman::SERVICE_STATE == propName)
   {
      BTSDbusPropertyList outPropertyList;

      parser.parseProperty(outPropertyList, signal->getValueMutable(), signal->getName(), (int)::ccdbusif::connman::IF_SERVICE);

      if(1 == outPropertyList.size())
      {
         const ::ccdbusif::DbusVariantProperty& outProperty = outPropertyList[0];

         Ipc2Bts_ConnManServiceStateUpdate* ptrResultMsg = ptrNew_Ipc2Bts_ConnManServiceStateUpdate();
         if(NULL != ptrResultMsg)
         {
            ptrResultMsg->setDevice(objPath);
            ptrResultMsg->setValue(parser.getServiceState2Enum(outProperty.propData.getString()));
         }
         onUpdate(ptrResultMsg, signal->getAct());
      }
      else
      {
         FW_NORMAL_ASSERT_ALWAYS();
      }
   }
   else if(::ccdbusif::connman::SERVICE_ETHERNET == propName)
   {
#if 0
      // following was observed: DBUS log:
      [ 1629.717837] signal sender=:1.18 -> dest=(null destination) serial=786 path=/net/connman/service/bluetooth_E0750AFFC70F_AC37438E2A6F; interface=net.connman.Service; member=PropertyChanged
         string "Ethernet"
         variant       array [
               dict entry(
                  string "Method"
                  variant                string "auto"
               )
               dict entry(
                  string "Interface"
                  variant                string "bnep0"
               )
               dict entry(
                  string "Address"
                  variant                string "E0:75:0A:FF:C7:0F"
               )
               dict entry(
                  string "MTU"
                  variant                uint16 1500
               )
            ]
      [ 1629.737490] signal sender=:1.18 -> dest=(null destination) serial=787 path=/net/connman/service/bluetooth_E0750AFFC70F_AC37438E2A6F; interface=net.connman.Service; member=PropertyChanged
         string "Ethernet"
         variant       array [
               dict entry(
                  string "Method"
                  variant                string "auto"
               )
               dict entry(
                  string "Address"
                  variant                string "E0:75:0A:FF:C7:0F"
               )
               dict entry(
                  string "MTU"
                  variant                uint16 1500
               )
            ]
      [ 1629.739318] signal sender=:1.21 -> dest=(null destination) serial=2238 path=/org/alps/evolution/hli; interface=org.alps.evo.DM; member=BtApplDeviceDisconnectInd
         byte 0
         array of bytes [
            ac 37 43 8e 2a 6f
         ]
         byte 2
         uint64 524288
         byte 0
#endif

      BTSDbusPropertyList outPropertyList;

      parser.parseProperty(outPropertyList, signal->getValueMutable(), signal->getName(), (int)::ccdbusif::connman::IF_SERVICE);

      Ipc2Bts_ConnManServiceInterfaceUpdate* ptrResultMsg = ptrNew_Ipc2Bts_ConnManServiceInterfaceUpdate();
      if(NULL != ptrResultMsg)
      {
         ptrResultMsg->setDevice(objPath);

         for(size_t i = 0; i < outPropertyList.size(); i++)
         {
            const ::ccdbusif::DbusVariantProperty& outProperty = outPropertyList[i];

            if((::ccdbusif::connman::ServiceProperty)outProperty.propEnum == ::ccdbusif::connman::SERVICE_ETHERNET_INTERFACE)
            {
               ptrResultMsg->setInterface(outProperty.propData.getString());
               break;
            }
         }

         // note: it can happen that Interface parameter is not part of property Ethernet; continue with empty Interface name
      }
      onUpdate(ptrResultMsg, signal->getAct());
   }
}
// net/connman/ServiceProxy implementation --- end

} //genivi
} //btstackif
