/**
 * @file ConnManDbusManagerCallbackIf.cpp
 *
 * @par SW-Component
 * BtStackIf
 *
 * @brief ConnMan DBUS Callback Interface for Manager.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details ConnMan DBUS Callback Interface for Manager.
 */

#include "ConnManDbusManagerCallbackIf.h"
#include "Ipc2Bts_MessageWrapper_GEN.h"
// #include "BtsUtils.h"
// #include "EvolutionGeniviUtils.h"
#include "cc_dbus_if/ConnManDbusParser.h"
#include "cc_dbus_if/ConnManUtility.h"
#include "FwStringUtils.h"
#include "FwAssert.h"

using namespace ::net::connman::Manager;
using namespace ::asf::core;

namespace btstackif {
namespace genivi {

ConnManDbusManagerCallbackIf::ConnManDbusManagerCallbackIf()
{
   _serviceIf = NULL;
}

ConnManDbusManagerCallbackIf::ConnManDbusManagerCallbackIf(IDbusRecHandler* recHandler) : DbusCallbackIf(recHandler)
{
   _serviceIf = NULL;
}

ConnManDbusManagerCallbackIf::~ConnManDbusManagerCallbackIf()
{
   _serviceIf = NULL;
}

// "ServiceAvailableIF" implementation --- start
void ConnManDbusManagerCallbackIf::onProxyAvailable(const ServiceState previousState, const ServiceState currentState, const ::std::string& objPath)
{
   (void)(previousState);
   (void)(objPath);

   BTSDbusServiceAvailability availabilityEvent = BTS_DBUS_SERVICE_NOT_AVAILABLE;

   if(ServiceState__Available == currentState)
   {
      availabilityEvent = BTS_DBUS_SERVICE_AVAILABLE;
   }

   Ipc2Bts_ServiceAvailabilityConnection* ptrMsg = ptrNew_Ipc2Bts_ServiceAvailabilityConnection();

   if(NULL != ptrMsg)
   {
      ptrMsg->setInterface(BTS_GEN_DBUS_SERVICE_CONNMAN_MANAGER);
      ptrMsg->setAvailabilityEvent(availabilityEvent);
   }

   onSignal(ptrMsg, ::ccdbusif::DEFAULT_ACT, true);
}

void ConnManDbusManagerCallbackIf::onProxyUnavailable(const ServiceState previousState, const ServiceState currentState, const ::std::string& objPath)
{
   (void)(previousState);
   (void)(objPath);

   BTSDbusServiceAvailability availabilityEvent = BTS_DBUS_SERVICE_NOT_AVAILABLE;

   if(ServiceState__Available == currentState)
   {
      availabilityEvent = BTS_DBUS_SERVICE_AVAILABLE;
   }

   Ipc2Bts_ServiceAvailabilityConnection* ptrMsg = ptrNew_Ipc2Bts_ServiceAvailabilityConnection();

   if(NULL != ptrMsg)
   {
      ptrMsg->setInterface(BTS_GEN_DBUS_SERVICE_CONNMAN_MANAGER);
      ptrMsg->setAvailabilityEvent(availabilityEvent);
   }

   onSignal(ptrMsg, ::ccdbusif::DEFAULT_ACT, true);
}
// "ServiceAvailableIF" implementation --- end

// net/connman/ManagerProxy implementation --- start
void ConnManDbusManagerCallbackIf::onGetPropertiesError(const ::boost::shared_ptr< ManagerProxy >& proxy, const ::boost::shared_ptr< GetPropertiesError >& error)
{
   (void)(proxy);
   (void)(error);
}

void ConnManDbusManagerCallbackIf::onGetPropertiesResponse(const ::boost::shared_ptr< ManagerProxy >& proxy, const ::boost::shared_ptr< GetPropertiesResponse >& response)
{
   (void)(proxy);
   (void)(response);
}

void ConnManDbusManagerCallbackIf::onGetTechnologiesError(const ::boost::shared_ptr< ManagerProxy >& proxy, const ::boost::shared_ptr< GetTechnologiesError >& error)
{
   (void)(proxy);
   (void)(error);
}

void ConnManDbusManagerCallbackIf::onGetTechnologiesResponse(const ::boost::shared_ptr< ManagerProxy >& proxy, const ::boost::shared_ptr< GetTechnologiesResponse >& response)
{
   (void)(proxy);
   (void)(response);
}

void ConnManDbusManagerCallbackIf::onGetServicesError(const ::boost::shared_ptr< ManagerProxy >& proxy, const ::boost::shared_ptr< GetServicesError >& error)
{
   (void)(proxy);
   onError(ptrNew_Ipc2Bts_GetConnManServicesResult(), error);
}

void ConnManDbusManagerCallbackIf::onGetServicesResponse(const ::boost::shared_ptr< ManagerProxy >& proxy, const ::boost::shared_ptr< GetServicesResponse >& response)
{
   (void)(proxy);
   ::std::vector< GetServicesResponseServicesStruct >& serviceList = response->getServicesMutable();

   /*
    * GetServicesResponseServicesStruct <=> ServicesChangedSignalChangedStruct
    * received data matches ServicesChangedSignal
    * GetServicesResponseServicesStruct contains:
    * - const ::std::string& getElem1() const: object path
    * - ::std::map< ::std::string, ::asf::dbus::DBusVariant >& getElem2Mutable(): dictionary
    *
    * GetServicesResponse delivers full information
    * we are interested in:
    * - "Type" => check for "bluetooth"
    * - device address => to be derived from object path because there is no parameter indicating device address; device name is available but this might be ambiguous
    */

   // go through the list
   for(::std::vector< GetServicesResponseServicesStruct >::iterator it = serviceList.begin(); it != serviceList.end(); ++it)
   {
      checkServiceEntry(it->getElem1(), it->getElem2Mutable(), response->getAct());
   }

   // create result message
   Ipc2Bts_GetConnManServicesResult* ptrResultMsg = ptrNew_Ipc2Bts_GetConnManServicesResult();
   if(NULL != ptrResultMsg)
   {
      // information not needed --- ptrResultMsg->setServices(rxedServiceList);
   }
   onResponse(ptrResultMsg, response->getAct());
}

void ConnManDbusManagerCallbackIf::onPropertyChangedError(const ::boost::shared_ptr< ManagerProxy >& proxy, const ::boost::shared_ptr< PropertyChangedError >& error)
{
   (void)(proxy);
   (void)(error);
   // is never triggered
   FW_NORMAL_ASSERT_ALWAYS();
}

void ConnManDbusManagerCallbackIf::onPropertyChangedSignal(const ::boost::shared_ptr< ManagerProxy >& proxy, const ::boost::shared_ptr< PropertyChangedSignal >& signal)
{
   (void)(proxy);
   (void)(signal);
}

void ConnManDbusManagerCallbackIf::onTechnologyAddedError(const ::boost::shared_ptr< ManagerProxy >& proxy, const ::boost::shared_ptr< TechnologyAddedError >& error)
{
   (void)(proxy);
   (void)(error);
   // is never triggered
   FW_NORMAL_ASSERT_ALWAYS();
}

void ConnManDbusManagerCallbackIf::onTechnologyAddedSignal(const ::boost::shared_ptr< ManagerProxy >& proxy, const ::boost::shared_ptr< TechnologyAddedSignal >& signal)
{
   (void)(proxy);
   (void)(signal);
}

void ConnManDbusManagerCallbackIf::onTechnologyRemovedError(const ::boost::shared_ptr< ManagerProxy >& proxy, const ::boost::shared_ptr< TechnologyRemovedError >& error)
{
   (void)(proxy);
   (void)(error);
   // is never triggered
   FW_NORMAL_ASSERT_ALWAYS();
}

void ConnManDbusManagerCallbackIf::onTechnologyRemovedSignal(const ::boost::shared_ptr< ManagerProxy >& proxy, const ::boost::shared_ptr< TechnologyRemovedSignal >& signal)
{
   (void)(proxy);
   (void)(signal);
}

void ConnManDbusManagerCallbackIf::onServicesChangedError(const ::boost::shared_ptr< ManagerProxy >& proxy, const ::boost::shared_ptr< ServicesChangedError >& error)
{
   (void)(proxy);
   (void)(error);
   // is never triggered
   FW_NORMAL_ASSERT_ALWAYS();
}

void ConnManDbusManagerCallbackIf::onServicesChangedSignal(const ::boost::shared_ptr< ManagerProxy >& proxy, const ::boost::shared_ptr< ServicesChangedSignal >& signal)
{
   (void)(proxy);
   ::std::vector< ServicesChangedSignalChangedStruct >& serviceList = signal->getChangedMutable();
   const ::std::vector< ::std::string >& removedList = signal->getRemoved();

   /*
    * GetServicesResponseServicesStruct <=> ServicesChangedSignalChangedStruct
    * received data matches GetServicesResponse
    * ServicesChangedSignalChangedStruct contains:
    * - const ::std::string& getElem1() const: object path
    * - ::std::map< ::std::string, ::asf::dbus::DBusVariant >& getElem2Mutable(): dictionary
    *
    * ServicesChangedSignal only delivers full information in case of new service was added else only object path is delivered
    * we are interested in:
    * - "Type" => check for "bluetooth"
    * - device address => to be derived from object path because there is no parameter indicating device address; device name is available but this might be ambiguous
    * we are interested in:
    * - removed object path
    */

   // go through the list
   for(::std::vector< ServicesChangedSignalChangedStruct >::iterator it = serviceList.begin(); it != serviceList.end(); ++it)
   {
      checkServiceEntry(it->getElem1(), it->getElem2Mutable(), signal->getAct());
   }

   // go through the list
   for(::std::vector< ::std::string >::const_iterator it = removedList.begin(); it != removedList.end(); ++it)
   {
      checkRemovedEntry(*it, signal->getAct());
   }
}
// net/connman/ManagerProxy implementation --- end

void ConnManDbusManagerCallbackIf::checkServiceEntry(const ::std::string& objPath, ::std::map< ::std::string, ::asf::dbus::DBusVariant >& paramDict, const act_t token)
{
   // check for non-empty object path and non-empty dictionary
   if((true == objPath.empty()) || (0 == paramDict.size()))
   {
      //  no information to be processed => return
      return;
   }

   // existing entry in case of GetServicesResponse or new entry in case of ServicesChangedSignal
   // do same handling for both: find entry with type Bluetooth
   ::ccdbusif::connman::ConnManDbusParser parser;
   parser.setTraces(true);
   BTSDbusPropertyList outPropertyList;

   if(true == parser.findAndParseProperty(outPropertyList, paramDict, parser.getServiceProperty2String(::ccdbusif::connman::SERVICE_TYPE), (int)::ccdbusif::connman::IF_SERVICE))
   {
      if(1 == outPropertyList.size())
      {
         const ::ccdbusif::DbusVariantProperty& outProperty = outPropertyList[0];

         if((false == outProperty.propData.getString().empty()) && (parser.getServiceType2String(::ccdbusif::connman::SERVICE_TYPE_BLUETOOTH) == outProperty.propData.getString()))
         {
            // the found entry is related to Bluetooth
            // derive device address from object path
            BTSBDAddress address;
            ::ccdbusif::connman_utility::convertObjectPath2BDAddress(address, objPath);
            ::fw::convertString2LowerCase(address);

            // forward object path mapping
            Ipc2Bts_AddProtocolObjectPathMapping* ptrMsg = ptrNew_Ipc2Bts_AddProtocolObjectPathMapping();
            if(0 != ptrMsg)
            {
               ptrMsg->setProtocol(BTS_PROTO_PAN);
               ptrMsg->setBDAddress(address);
               ptrMsg->setObjPath(objPath);
            }
            onSignal(ptrMsg, token, true);

            // create proxy
            if(0 != _serviceIf)
            {
               _serviceIf->serviceCreated(objPath);
            }

            // properties not needed because connection is only possible after interaction with Bluetooth device manager application (at least link key is needed)
         }
         else
         {
            // other types e.g. WIFI
         }
      }
      else
      {
         FW_NORMAL_ASSERT_ALWAYS();
      }
   }
   else
   {
      FW_NORMAL_ASSERT_ALWAYS();
   }
}

void ConnManDbusManagerCallbackIf::checkRemovedEntry(const ::std::string& objPath, const act_t token)
{
   // check for non-empty object path
   if(true == objPath.empty())
   {
      // no information to be processed => return
      return;
   }

   // entry was removed
   // check for bluetooth as part of object path to avoid internal message traffic (no engineering solution but it helps)
   ::std::string copyObjPath(objPath);
   ::fw::convertString2LowerCase(copyObjPath);
   if(false == ::ccdbusif::connman_utility::isBluetoothObjectPath(copyObjPath))
   {
      // ignore
      return;
   }

   // destroy proxy
   if(0 != _serviceIf)
   {
      _serviceIf->serviceRemoved(objPath, true);
   }

   // send removed mapping
   Ipc2Bts_DelProtocolObjectPathMapping* ptrMsg = ptrNew_Ipc2Bts_DelProtocolObjectPathMapping();
   if(NULL != ptrMsg)
   {
      ptrMsg->setObjPath(objPath);
      ptrMsg->setObjectId(objPath);
   }
   onSignal(ptrMsg, token);
}

} //genivi
} //btstackif
