/**
 * @file EvolutionGeniviSerialCallbackIf.cpp
 *
 * @par SW-Component
 * BtStackIf
 *
 * @brief EvolutionGenivi DBUS Callback for Serial.
 *
 * @copyright (C) 2016 - 2018 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details EvolutionGenivi DBUS Callback implementation for Serial.
 */

#include "EvolutionGeniviSerialCallbackIf.h"
#include "Ipc2Bts_MessageWrapper_GEN.h"
#include "TraceClasses.h"
#include "FwAssert.h"
#include "FwTrace.h"
#include "EvolutionGeniviUtils.h"
#include "cc_dbus_if/EvolutionGeniviDbusParser.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_BTS_DBUS
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/EvolutionGeniviSerialCallbackIf.cpp.trc.h"
#endif
#endif

using namespace ::org::bluez::Serial;

namespace btstackif {
namespace genivi {

EvolutionGeniviSerialCallbackIf::EvolutionGeniviSerialCallbackIf(IDbusRecHandler* recHandler) :
DbusCallbackIf(recHandler),
_callbackId(0)
{
}

EvolutionGeniviSerialCallbackIf::~EvolutionGeniviSerialCallbackIf()
{
}

// "ServiceAvailableIF" implementation --- start
void EvolutionGeniviSerialCallbackIf::onAvailableCb(const bool available, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const ::ccdbusif::DbusBusType busType /*= ::ccdbusif::BUS_TYPE_SYSTEM*/)
{
   (void)(busName);
   (void)(busType);

   BTSDbusServiceAvailability availabilityEvent(BTS_DBUS_SERVICE_NOT_AVAILABLE);

   if(true == available)
   {
      availabilityEvent = BTS_DBUS_SERVICE_AVAILABLE;
   }


   Ipc2Bts_ServiceAvailabilityConnection* msg = ptrNew_Ipc2Bts_ServiceAvailabilityConnection();

   if(0 != msg)
   {
      msg->setInterface(BTS_GEN_DBUS_SERVICE_SERIAL);
      msg->setObjPath(objPath);
      msg->setAvailabilityEvent(availabilityEvent);
   }

   onSignal(msg, ::ccdbusif::DEFAULT_ACT, true);
}

void EvolutionGeniviSerialCallbackIf::onUnavailableCb(const bool available, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const ::ccdbusif::DbusBusType busType /*= ::ccdbusif::BUS_TYPE_SYSTEM*/)
{
   (void)(objPath);
   (void)(busName);
   (void)(busType);

   BTSDbusServiceAvailability availabilityEvent(BTS_DBUS_SERVICE_NOT_AVAILABLE);

   if(true == available)
   {
      availabilityEvent = BTS_DBUS_SERVICE_AVAILABLE;
   }


   Ipc2Bts_ServiceAvailabilityConnection* msg = ptrNew_Ipc2Bts_ServiceAvailabilityConnection();

   if(0 != msg)
   {
      msg->setInterface(BTS_GEN_DBUS_SERVICE_SERIAL);
      msg->setObjPath(objPath);
      msg->setAvailabilityEvent(availabilityEvent);
   }

   onSignal(msg, ::ccdbusif::DEFAULT_ACT, true);
}
// "ServiceAvailableIF" implementation --- end

// /org/bluez/SerialProxy implementation --- start
void EvolutionGeniviSerialCallbackIf::onSendDataErrorCb(const ::boost::shared_ptr< SendDataError >& error, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const ::ccdbusif::DbusBusType busType /*= ::ccdbusif::BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   // ignore
   (void)(error);
   (void)(objPath);
   (void)(busName);
   (void)(busType);
   (void)(interfaceName);
}

void EvolutionGeniviSerialCallbackIf::onSendDataResponseCb(const ::boost::shared_ptr< SendDataResponse >& response, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const ::ccdbusif::DbusBusType busType /*= ::ccdbusif::BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   // ignore
   (void)(response);
   (void)(objPath);
   (void)(busName);
   (void)(busType);
   (void)(interfaceName);
}

void EvolutionGeniviSerialCallbackIf::onSppDataIndErrorCb(const ::boost::shared_ptr< SppDataIndError >& error, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const ::ccdbusif::DbusBusType busType /*= ::ccdbusif::BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   // is never triggered
   (void)(error);
   (void)(objPath);
   (void)(busName);
   (void)(busType);
   (void)(interfaceName);
}

void EvolutionGeniviSerialCallbackIf::onSppDataIndSignalCb(const ::boost::shared_ptr< SppDataIndSignal >& signal, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const ::ccdbusif::DbusBusType busType /*= ::ccdbusif::BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   // ignore
   (void)(signal);
   (void)(objPath);
   (void)(busName);
   (void)(busType);
   (void)(interfaceName);
}

void EvolutionGeniviSerialCallbackIf::onFlowCtrlSendErrorCb(const ::boost::shared_ptr< FlowCtrlSendError >& error, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const ::ccdbusif::DbusBusType busType /*= ::ccdbusif::BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   // is never triggered
   (void)(error);
   (void)(objPath);
   (void)(busName);
   (void)(busType);
   (void)(interfaceName);
}

void EvolutionGeniviSerialCallbackIf::onFlowCtrlSendSignalCb(const ::boost::shared_ptr< FlowCtrlSendSignal >& signal, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const ::ccdbusif::DbusBusType busType /*= ::ccdbusif::BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   // ignore
   (void)(signal);
   (void)(objPath);
   (void)(busName);
   (void)(busType);
   (void)(interfaceName);
}

void EvolutionGeniviSerialCallbackIf::onDeviceFilePathErrorCb(const ::boost::shared_ptr< DeviceFilePathError >& error, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const ::ccdbusif::DbusBusType busType /*= ::ccdbusif::BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   // is never triggered
   (void)(error);
   (void)(objPath);
   (void)(busName);
   (void)(busType);
   (void)(interfaceName);
}

void EvolutionGeniviSerialCallbackIf::onDeviceFilePathSignalCb(const ::boost::shared_ptr< DeviceFilePathSignal >& signal, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const ::ccdbusif::DbusBusType busType /*= ::ccdbusif::BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   (void)(busName);
   (void)(busType);
   (void)(interfaceName);

   Ipc2Bts_SerialDeviceFilePath* msg = ptrNew_Ipc2Bts_SerialDeviceFilePath();
   if(0 != msg)
   {
      msg->setService(objPath);
      convertStringBinary2String(msg->getDeviceNameMutable(), signal->getPath());
   }
   onSignal(msg, signal->getAct());
}

void EvolutionGeniviSerialCallbackIf::onFlowCtrlRecvErrorCb(const ::boost::shared_ptr< FlowCtrlRecvError >& error, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const ::ccdbusif::DbusBusType busType /*= ::ccdbusif::BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   // ignore
   (void)(error);
   (void)(objPath);
   (void)(busName);
   (void)(busType);
   (void)(interfaceName);
}

void EvolutionGeniviSerialCallbackIf::onFlowCtrlRecvUpdateCb(const ::boost::shared_ptr< FlowCtrlRecvUpdate >& update, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const ::ccdbusif::DbusBusType busType /*= ::ccdbusif::BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   // ignore
   (void)(update);
   (void)(objPath);
   (void)(busName);
   (void)(busType);
   (void)(interfaceName);
}
// /org/bluez/SerialProxy implementation --- end

} //genivi
} //btstackif
