/**
 * @file TelephonyControl_GEN.cpp
 *
 * @par SW-Component
 * Main
 *
 * @brief Telephony Control.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Implementation of telephony control functionality.
 */

#include "TelephonyControl_GEN.h"
#include "App2Bts_MessageWrapper.h"
#include "Bts2App_MessageWrapper.h"
#include "Bts2Ipc_MessageWrapper_GEN.h"
#include "Ipc2Bts_MessageWrapper_GEN.h"
#include "EvolutionGeniviStackTypes.h"
#include "EvolutionGeniviUtils.h"
#include "BtsUtils.h"
#include "TraceClasses.h"
#include "FwAssert.h"
#include "FwTrace.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_BTS_CONTROL
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/TelephonyControl_GEN.cpp.trc.h"
#endif
#endif

namespace btstackif {
namespace genivi {

TelephonyControl::TelephonyControl()
: BasicControl(BTS_FB_TELEPHONY)
{
   /*********** start here with specific class members ***************************/
   /*********** end here with specific class members *****************************/
}

TelephonyControl::~TelephonyControl()
{
   /*********** start here with specific class members ***************************/
   /*********** end here with specific class members *****************************/
}

void TelephonyControl::pushApp2BtsMessage(IN App2Bts_BaseMessage* ptrMessage)
{
   if(NULL == ptrMessage)
   {
      // should never happen else you have programmed something wrong
      // #error_indication
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   if(_functionBlock != ptrMessage->getFunctionBlock())
   {
      // should never happen else you have programmed something wrong
      // #error_indication
      FW_NORMAL_ASSERT_ALWAYS();
      delete ptrMessage;
      return;
   }

   // HINT: if any error happens now the error has to be handled within this function or within any called sub-function

   bool similarOpCodeInWorkingQueue = false;

   // check for skipping similar opcode check
   if(false == skipSimilarOpCodeCheck(ptrMessage))
   {
      // get similar opcodes
      ::std::vector<BTSApp2BtsMessageMasterCompareItem> itemList;
      getSimilarOpCodes(itemList, ptrMessage);

      // check if opcode is in working queue
      similarOpCodeInWorkingQueue = isSimilarOpCodeInWorkingQueue(itemList);
   }

   // check if opcode is in working queue
   if(true == similarOpCodeInWorkingQueue)
   {
      // push to waiting queue
      pushApp2BtsMsgToWaitingQueue(ptrMessage, false); // single worker thread
      ETG_TRACE_USR3((" pushApp2BtsMessage(): App2Bts 0x%04X pushed to waiting queue", ptrMessage->getTraceOpCode()));
   }
   else
   {
      BTSApp2BtsOpcode opcode = ptrMessage->getOpCode();
      ::std::vector<Bts2Ipc_BaseMessage*> sendBts2IpcMsgList;
      sendBts2IpcMsgList.reserve(10);
      ::std::vector<Bts2App_BaseMessage*> sendBts2AppMsgList;
      sendBts2AppMsgList.reserve(10);
      bool deleteApp2BtsMsg = false;
      BTSApp2BtsMessageCompareItem compareItem;
      ptrMessage->getCompareItem(compareItem);

      ETG_TRACE_USR3((" pushApp2BtsMessage(): App2Bts 0x%04X to be processed", ptrMessage->getTraceOpCode()));

      // handle message depending on opcode
      // - set marker to delete message in sub-handler function if necessary
      // - push message to working queue in sub-handler function if necessary
      // - create direct answer message (Bts2App) in sub-handler function if necessary
      // - create Bts2Ipc message in sub-handler function if necessary
      // - handle any error in sub-handler function because there is the best place to handle
      switch(opcode)
      {
         case App2BtsOC_SetTelephonyDummy:
            handleApp2BtsSetTelephonyDummy(sendBts2IpcMsgList, sendBts2AppMsgList, deleteApp2BtsMsg, static_cast<App2Bts_SetTelephonyDummy*>(ptrMessage)); /*lint !e1774: Could use dynamic_cast to downcast polymorphic type*/ // no RTTI, use static_cast instead of dynamic_cast, we know the type of the message
            break;
         // all other
         default:
            FW_NORMAL_ASSERT_ALWAYS();
            deleteApp2BtsMsg = true;
            break;
      }

      if(0 < sendBts2IpcMsgList.size())
      {
         _bts2IpcMessageWasSent = true; // this works only if single thread handling TODO: recheck again
      }

      sendBts2IpcMessageList(sendBts2IpcMsgList, compareItem);

      sendBts2AppMessageList(sendBts2AppMsgList);

      if(true == deleteApp2BtsMsg)
      {
         delete ptrMessage;
      }
   }
}

void TelephonyControl::pushIpc2BtsMessage(IN Ipc2Bts_BaseMessage* ptrMessage)
{
   if(NULL == ptrMessage)
   {
      // should never happen else you have programmed something wrong
      // #error_indication
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   if(_functionBlock != ptrMessage->getBtsDestinationFunctionBlock())
   {
      // should never happen else you have programmed something wrong
      // #error_indication
      FW_NORMAL_ASSERT_ALWAYS();
      delete ptrMessage;
      return;
   }

   // HINT: if any error happens now the error has to be handled within this function or within any called sub-function

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)ptrMessage->getOpCode();
   ::std::vector<Bts2Ipc_BaseMessage*> sendBts2IpcMsgList;
   sendBts2IpcMsgList.reserve(10);
   ::std::vector<Bts2App_BaseMessage*> sendBts2AppMsgList;
   sendBts2AppMsgList.reserve(10);
   App2Bts_BaseMessage* ptrApp2BtsQueueMessage;
   bool deleteApp2BtsMsg = false;

   // find related APP2BTS message
   // for req -> cfm sequence there must be a message in working queue
   // for ind sequence there can be a message in working queue
   ptrApp2BtsQueueMessage = findApp2BtsWorkingMessageWrapper(ptrMessage->getApp2BtsCompareItem());

   // special handling is needed for e.g. connect/disconnect
   BTSApp2BtsMessageCompareItem specialCompareItem;
   specialCompareItem.opCode = App2BtsOC_None;

   // handle message depending on opcode
   // - do not delete Ipc2Bts message in sub-handler function because this done at the end of this function
   // - set marker to delete related App2Bts message in sub-handler function if necessary
   // - create Bts2App message in sub-handler function if necessary
   // - create Bts2Ipc message in sub-handler function if necessary
   // - handle any error in sub-handler function because there is the best place to handle
   switch(opcode)
   {
      case Ipc2BtsOC_ServiceAvailabilityTelephony:
         handleIpc2BtsServiceAvailabilityTelephony(sendBts2IpcMsgList, sendBts2AppMsgList, specialCompareItem, deleteApp2BtsMsg, &ptrApp2BtsQueueMessage, static_cast<Ipc2Bts_ServiceAvailabilityTelephony*>(ptrMessage)); /*lint !e1774: Could use dynamic_cast to downcast polymorphic type*/ // no RTTI, use static_cast instead of dynamic_cast, we know the type of the message
         break;
      case Ipc2BtsOC_AddHfpInterfaceObjectPathMapping:
         handleIpc2BtsAddHfpInterfaceObjectPathMapping(sendBts2IpcMsgList, sendBts2AppMsgList, specialCompareItem, deleteApp2BtsMsg, &ptrApp2BtsQueueMessage, static_cast<Ipc2Bts_AddHfpInterfaceObjectPathMapping*>(ptrMessage)); /*lint !e1774: Could use dynamic_cast to downcast polymorphic type*/ // no RTTI, use static_cast instead of dynamic_cast, we know the type of the message
         break;
      case Ipc2BtsOC_DelHfpInterfaceObjectPathMapping:
         handleIpc2BtsDelHfpInterfaceObjectPathMapping(sendBts2IpcMsgList, sendBts2AppMsgList, specialCompareItem, deleteApp2BtsMsg, &ptrApp2BtsQueueMessage, static_cast<Ipc2Bts_DelHfpInterfaceObjectPathMapping*>(ptrMessage)); /*lint !e1774: Could use dynamic_cast to downcast polymorphic type*/ // no RTTI, use static_cast instead of dynamic_cast, we know the type of the message
         break;
      case Ipc2BtsOC_GetHandsfreeProperties:
         handleIpc2BtsGetHandsfreeProperties(sendBts2IpcMsgList, sendBts2AppMsgList, specialCompareItem, deleteApp2BtsMsg, &ptrApp2BtsQueueMessage, static_cast<Ipc2Bts_GetHandsfreeProperties*>(ptrMessage)); /*lint !e1774: Could use dynamic_cast to downcast polymorphic type*/ // no RTTI, use static_cast instead of dynamic_cast, we know the type of the message
         break;
      case Ipc2BtsOC_GetHandsfreePropertiesExt:
         handleIpc2BtsGetHandsfreePropertiesExt(sendBts2IpcMsgList, sendBts2AppMsgList, specialCompareItem, deleteApp2BtsMsg, &ptrApp2BtsQueueMessage, static_cast<Ipc2Bts_GetHandsfreePropertiesExt*>(ptrMessage)); /*lint !e1774: Could use dynamic_cast to downcast polymorphic type*/ // no RTTI, use static_cast instead of dynamic_cast, we know the type of the message
         break;
      case Ipc2BtsOC_HandsfreeFeaturesUpdate:
         handleIpc2BtsHandsfreeFeaturesUpdate(sendBts2IpcMsgList, sendBts2AppMsgList, specialCompareItem, deleteApp2BtsMsg, &ptrApp2BtsQueueMessage, static_cast<Ipc2Bts_HandsfreeFeaturesUpdate*>(ptrMessage)); /*lint !e1774: Could use dynamic_cast to downcast polymorphic type*/ // no RTTI, use static_cast instead of dynamic_cast, we know the type of the message
         break;
      case Ipc2BtsOC_HandsfreeInbandRingingUpdate:
         handleIpc2BtsHandsfreeInbandRingingUpdate(sendBts2IpcMsgList, sendBts2AppMsgList, specialCompareItem, deleteApp2BtsMsg, &ptrApp2BtsQueueMessage, static_cast<Ipc2Bts_HandsfreeInbandRingingUpdate*>(ptrMessage)); /*lint !e1774: Could use dynamic_cast to downcast polymorphic type*/ // no RTTI, use static_cast instead of dynamic_cast, we know the type of the message
         break;
      case Ipc2BtsOC_GetModems:
         handleIpc2BtsGetModems(sendBts2IpcMsgList, sendBts2AppMsgList, specialCompareItem, deleteApp2BtsMsg, &ptrApp2BtsQueueMessage, static_cast<Ipc2Bts_GetModems*>(ptrMessage)); /*lint !e1774: Could use dynamic_cast to downcast polymorphic type*/ // no RTTI, use static_cast instead of dynamic_cast, we know the type of the message
         break;
      case Ipc2BtsOC_GetModemsExt:
         handleIpc2BtsGetModemsExt(sendBts2IpcMsgList, sendBts2AppMsgList, specialCompareItem, deleteApp2BtsMsg, &ptrApp2BtsQueueMessage, static_cast<Ipc2Bts_GetModemsExt*>(ptrMessage)); /*lint !e1774: Could use dynamic_cast to downcast polymorphic type*/ // no RTTI, use static_cast instead of dynamic_cast, we know the type of the message
         break;
      case Ipc2BtsOC_ModemAdded:
         handleIpc2BtsModemAdded(sendBts2IpcMsgList, sendBts2AppMsgList, specialCompareItem, deleteApp2BtsMsg, &ptrApp2BtsQueueMessage, static_cast<Ipc2Bts_ModemAdded*>(ptrMessage)); /*lint !e1774: Could use dynamic_cast to downcast polymorphic type*/ // no RTTI, use static_cast instead of dynamic_cast, we know the type of the message
         break;
      case Ipc2BtsOC_ModemAddedExt:
         handleIpc2BtsModemAddedExt(sendBts2IpcMsgList, sendBts2AppMsgList, specialCompareItem, deleteApp2BtsMsg, &ptrApp2BtsQueueMessage, static_cast<Ipc2Bts_ModemAddedExt*>(ptrMessage)); /*lint !e1774: Could use dynamic_cast to downcast polymorphic type*/ // no RTTI, use static_cast instead of dynamic_cast, we know the type of the message
         break;
      case Ipc2BtsOC_ModemRemoved:
         handleIpc2BtsModemRemoved(sendBts2IpcMsgList, sendBts2AppMsgList, specialCompareItem, deleteApp2BtsMsg, &ptrApp2BtsQueueMessage, static_cast<Ipc2Bts_ModemRemoved*>(ptrMessage)); /*lint !e1774: Could use dynamic_cast to downcast polymorphic type*/ // no RTTI, use static_cast instead of dynamic_cast, we know the type of the message
         break;
      case Ipc2BtsOC_GetModemProperties:
         handleIpc2BtsGetModemProperties(sendBts2IpcMsgList, sendBts2AppMsgList, specialCompareItem, deleteApp2BtsMsg, &ptrApp2BtsQueueMessage, static_cast<Ipc2Bts_GetModemProperties*>(ptrMessage)); /*lint !e1774: Could use dynamic_cast to downcast polymorphic type*/ // no RTTI, use static_cast instead of dynamic_cast, we know the type of the message
         break;
      case Ipc2BtsOC_GetModemPropertiesExt:
         handleIpc2BtsGetModemPropertiesExt(sendBts2IpcMsgList, sendBts2AppMsgList, specialCompareItem, deleteApp2BtsMsg, &ptrApp2BtsQueueMessage, static_cast<Ipc2Bts_GetModemPropertiesExt*>(ptrMessage)); /*lint !e1774: Could use dynamic_cast to downcast polymorphic type*/ // no RTTI, use static_cast instead of dynamic_cast, we know the type of the message
         break;
      case Ipc2BtsOC_ModemInterfacesUpdate:
         handleIpc2BtsModemInterfacesUpdate(sendBts2IpcMsgList, sendBts2AppMsgList, specialCompareItem, deleteApp2BtsMsg, &ptrApp2BtsQueueMessage, static_cast<Ipc2Bts_ModemInterfacesUpdate*>(ptrMessage)); /*lint !e1774: Could use dynamic_cast to downcast polymorphic type*/ // no RTTI, use static_cast instead of dynamic_cast, we know the type of the message
         break;
      // all other
      default:
         FW_NORMAL_ASSERT_ALWAYS();
         break;
   }

   if(0 < sendBts2IpcMsgList.size())
   {
      if(NULL != ptrApp2BtsQueueMessage)
      {
         BTSApp2BtsMessageCompareItem compareItem;
         ptrApp2BtsQueueMessage->getCompareItem(compareItem);
         sendBts2IpcMessageList(sendBts2IpcMsgList, compareItem);
      }
      else
      {
         ::std::vector<Bts2Ipc_BaseMessage*> respSendBts2IpcMsgList;
         respSendBts2IpcMsgList.reserve(sendBts2IpcMsgList.size());

         for(size_t i = 0; i < sendBts2IpcMsgList.size(); i++)
         {
            if(NULL != sendBts2IpcMsgList[i])
            {
               if(true == sendBts2IpcMsgList[i]->getResponseMessageFlag())
               {
                  // response message: it could be possible that there is no related application message
                  respSendBts2IpcMsgList.push_back(sendBts2IpcMsgList[i]);
               }
               else
               {
                  // should never happen else you have programmed something wrong
                  // #error_indication
                  FW_NORMAL_ASSERT_ALWAYS();

                  delete sendBts2IpcMsgList[i];
               }
            }
         }

         sendBts2IpcMsgList.clear();

         if(0 < respSendBts2IpcMsgList.size())
         {
            BTSApp2BtsMessageCompareItem compareItem;
            sendBts2IpcMessageList(respSendBts2IpcMsgList, compareItem);
         }
      }
   }

   sendBts2AppMessageList(sendBts2AppMsgList);

   // delete received message
   delete ptrMessage;

   // remove related APP2BTS message and check waiting queue
   if((true == deleteApp2BtsMsg) && (NULL != ptrApp2BtsQueueMessage))
   {
      removeApp2BtsWorkingMessage(ptrApp2BtsQueueMessage);

      if(App2BtsOC_None == specialCompareItem.opCode)
      {
         checkWaitingQueue(ptrApp2BtsQueueMessage);
      }

      delete ptrApp2BtsQueueMessage;
   }

   // check waiting queue (special handling for connect/disconnect)
   if((App2BtsOC_None != specialCompareItem.opCode) && (App2BtsOC_Ignore != specialCompareItem.opCode))
   {
      checkWaitingQueueExtended(specialCompareItem);
   }
}

void TelephonyControl::setStackConfiguration(IN const BTSFunctionBlock component, IN const BTSInterfaceType stackInterface, IN const BTSFunctionBlock subComponent, IN const BTSUserMode userMode,
         OUT ::std::vector<BTSDbusInterfaceItem>& dbusInterfaces, IN const BTSLocalConfigurationContainer& configuration)
{
   (void)(component);
   (void)(stackInterface);
   (void)(subComponent);
   (void)(userMode);
   (void)(dbusInterfaces);
   (void)(configuration);
}

void TelephonyControl::triggerInitializedCallback(void)
{
}

void TelephonyControl::createDbusServiceAvailabilityMessage(IN const BTSCommonEnumClass interface, IN const BTSDbusServiceAvailability availabilityEvent)
{
   (void)(interface);
   (void)(availabilityEvent);
}

void TelephonyControl::createDbusServiceAvailabilityMessage(IN const BTSCommonEnumClass interface, IN const BTSDbusServiceAvailability availabilityEvent, IN const BTSBusName& busName, IN const BTSObjectPath& objPath, IN const BTSCommonEnumClass busType)
{
   (void)(interface);
   (void)(availabilityEvent);
   (void)(busName);
   (void)(objPath);
   (void)(busType);
}

void TelephonyControl::setSubControlTestCommand(IN const char* testCommand, IN const unsigned int testData)
{
   if(NULL == testCommand)
   {
      return;
   }

   (void)(testData);
}

void TelephonyControl::setSubControlTestCommand(IN const char* testCommand, IN const unsigned char* testData)
{
   if(NULL == testCommand)
   {
      return;
   }

   if(NULL == testData)
   {
      return;
   }
}

void TelephonyControl::sendDirectAnswerForApp2BtsMessages(IN const ::std::vector<App2Bts_BaseMessage*>& msgList, IN const BTSCommonEnumClass resultCode, IN const BTSCommonEnumClass statusCode)
{
   (void)(resultCode);
   (void)(statusCode);

   // TODO: recheck what to do with doubled requests - answer or ignore? what to do with status?

   if(0 == msgList.size())
   {
      // should never happen else you have programmed something wrong
      // #error_indication
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   App2Bts_BaseMessage* ptrMessage;

   for(size_t i = 0; i < msgList.size(); i++)
   {
      ptrMessage = msgList[i];

      if(NULL == ptrMessage)
      {
         // should never happen else you have programmed something wrong
         // #error_indication
         FW_NORMAL_ASSERT_ALWAYS();
         continue;
      }

      if(_functionBlock != ptrMessage->getFunctionBlock())
      {
         // should never happen else you have programmed something wrong
         // #error_indication
         FW_NORMAL_ASSERT_ALWAYS();
         continue;
      }

      // if(true == _handleDoubledRequests) TODO: check after merging
      {
         BTSApp2BtsOpcode opcode = ptrMessage->getOpCode();
         ::std::vector<Bts2App_BaseMessage*> sendBts2AppMsgList;
         sendBts2AppMsgList.reserve(10);

         ETG_TRACE_USR3((" sendDirectAnswerForApp2BtsMessages(): App2Bts 0x%04X to be handled", ptrMessage->getTraceOpCode()));

         // send direct answer depending on opcode
         // - create direct answer message (Bts2App) in sub-handler function
         // - handle any error in sub-handler function because there is the best place to handle
         // - do not delete message; this is done outside of this function
         switch(opcode)
         {
            case App2BtsOC_SetTelephonyDummy:
               handleDoubledApp2BtsSetTelephonyDummy(sendBts2AppMsgList, static_cast<App2Bts_SetTelephonyDummy*>(ptrMessage), resultCode, statusCode); /*lint !e1774: Could use dynamic_cast to downcast polymorphic type*/ // no RTTI, use static_cast instead of dynamic_cast, we know the type of the message
               break;
            // all other
            default:
               FW_NORMAL_ASSERT_ALWAYS();
               break;
         }

         sendBts2AppMessageList(sendBts2AppMsgList);
      }
   }
}

bool TelephonyControl::doApp2BtsMsgPrecheck(OUT bool& rejectRequest, OUT BTSCommonEnumClass& resultCode, OUT BTSCommonEnumClass& statusCode, OUT bool& skipOpCodeCheck, IN App2Bts_BaseMessage* ptrMessage)
{
   if(NULL == ptrMessage)
   {
      // should never happen else you have programmed something wrong
      // #error_indication
      FW_NORMAL_ASSERT_ALWAYS();
      return false;
   }

   (void)(rejectRequest);
   (void)(resultCode);
   (void)(statusCode);
   (void)(skipOpCodeCheck);

   return true;
}

void TelephonyControl::getSimilarOpCodes(OUT ::std::vector<BTSApp2BtsMessageMasterCompareItem>& itemList, IN const App2Bts_BaseMessage* ptrMessage)
{
   if(NULL == ptrMessage)
   {
      // should never happen else you have programmed something wrong
      // #error_indication
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   BTSApp2BtsOpcode opCode = ptrMessage->getOpCode();
   BTSApp2BtsMessageMasterCompareItem item;

   if((App2BtsOC_TelephonyBlockStart < opCode) && (opCode < App2BtsOC_TelephonyBlockEnd))
   {
      size_t reserveSize = 0;

      // TODO: complete switch
      switch(opCode)
      {
         case App2BtsOC_SetTelephonyDummy:
            reserveSize += 1;
            itemList.reserve(reserveSize);
            item.opCode = opCode;
            itemList.push_back(item);
            break;
         // add all other here if necessary
         default:
            FW_NORMAL_ASSERT_ALWAYS();
            break;
      }
   }
   else
   {
      // opcode in wrong range
      FW_NORMAL_ASSERT_ALWAYS();

      // add at least input opcode
      item.opCode = opCode;
      itemList.push_back(item);
   }
}

void TelephonyControl::getMatchingOpCodes(OUT ::std::vector<BTSApp2BtsMessageMasterCompareItem>& itemList, OUT ::std::vector<BTSApp2BtsMessageMasterCompareItem>& highPrioItemList, IN const App2Bts_BaseMessage* ptrMessage)
{
   if(NULL == ptrMessage)
   {
      // should never happen else you have programmed something wrong
      // #error_indication
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   (void)(highPrioItemList);
   BTSApp2BtsOpcode opCode = ptrMessage->getOpCode();
   BTSApp2BtsMessageMasterCompareItem item;
   ptrMessage->getCompareItem(item);

   if((App2BtsOC_TelephonyBlockStart < opCode) && (opCode < App2BtsOC_TelephonyBlockEnd))
   {
      size_t reserveSize;

      // TODO: complete switch
      switch(opCode)
      {
         case App2BtsOC_SetTelephonyDummy:
            reserveSize = 1;
            itemList.reserve(reserveSize);
            item.opCode = opCode;
            itemList.push_back(item);
            break;
         // add all other here if necessary
         default:
            FW_NORMAL_ASSERT_ALWAYS();
            break;
      }
   }
   else
   {
      // opcode in wrong range
      FW_NORMAL_ASSERT_ALWAYS();

      // add at least input opcode
      item.opCode = opCode;
      itemList.push_back(item);
   }
}

bool TelephonyControl::skipSimilarOpCodeCheck(IN const App2Bts_BaseMessage* ptrMessage)
{
   (void)(ptrMessage);
   // implement if necessary

   return false;
}

void TelephonyControl::checkWaitingQueueExtended(IN const BTSApp2BtsMessageCompareItem& compareItem)
{
   (void)(compareItem);
   // implement if necessary
}

void TelephonyControl::handleDoubledApp2BtsMessages(IN const ::std::vector<App2Bts_BaseMessage*>& msgList)
{
   // TODO: recheck what to do with doubled requests - answer or ignore? what to do with status?

   if(0 == msgList.size())
   {
      // should never happen else you have programmed something wrong
      // #error_indication
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   App2Bts_BaseMessage* ptrMessage;

   for(size_t i = 0; i < msgList.size(); i++)
   {
      ptrMessage = msgList[i];

      if(NULL == ptrMessage)
      {
         // should never happen else you have programmed something wrong
         // #error_indication
         FW_NORMAL_ASSERT_ALWAYS();
         continue;
      }

      if(_functionBlock != ptrMessage->getFunctionBlock())
      {
         // should never happen else you have programmed something wrong
         // #error_indication
         FW_NORMAL_ASSERT_ALWAYS();
         continue;
      }

      if(true == _handleDoubledRequests)
      {
         BTSApp2BtsOpcode opcode = ptrMessage->getOpCode();
         ::std::vector<Bts2App_BaseMessage*> sendBts2AppMsgList;
         sendBts2AppMsgList.reserve(10);

         ETG_TRACE_USR3((" handleDoubledApp2BtsMessages(): App2Bts 0x%04X to be handled", ptrMessage->getTraceOpCode()));

         // handle doubled messages depending on opcode
         // - create direct answer message (Bts2App) in sub-handler function
         // - handle any error in sub-handler function because there is the best place to handle
         // - do not delete message; this is done at the end of this function
         switch(opcode)
         {
            case App2BtsOC_SetTelephonyDummy:
               handleDoubledApp2BtsSetTelephonyDummy(sendBts2AppMsgList, static_cast<App2Bts_SetTelephonyDummy*>(ptrMessage), (BTSCommonEnumClass)BTS_REQ_SUCCESS, BTS_COMMON_ENUM_CLASS_DEFAULT_VALUE); /*lint !e1774: Could use dynamic_cast to downcast polymorphic type*/ // no RTTI, use static_cast instead of dynamic_cast, we know the type of the message
               break;
            // all other
            default:
               FW_NORMAL_ASSERT_ALWAYS();
               break;
         }

         sendBts2AppMessageList(sendBts2AppMsgList);
      }
   }
}

/*********** start here with specific class members + methods *****************/
/*********** end here with specific class members + methods *******************/

void TelephonyControl::handleDoubledApp2BtsSetTelephonyDummy(OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, IN App2Bts_SetTelephonyDummy* ptrMessage, IN const BTSCommonEnumClass resultCode, IN const BTSCommonEnumClass statusCode)
{
   (void)(bts2AppMsgList);
   (void)(resultCode);
   (void)(statusCode);

   if(NULL == ptrMessage)
   {
      // should never happen else you have programmed something wrong
      // #error_indication
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   // send answer directly because doubled request - set all to success
   // TODO
}

void TelephonyControl::handleApp2BtsSetTelephonyDummy(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT bool& deleteApp2BtsMessage, IN App2Bts_SetTelephonyDummy* ptrMessage)
{
   (void)(bts2AppMsgList);
   (void)(deleteApp2BtsMessage);

   if(NULL == ptrMessage)
   {
      // should never happen else you have programmed something wrong
      // #error_indication
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   // TODO: implement
   _app2BtsWorkingQueue.push(ptrMessage, false);

   bts2IpcMsgList.reserve(1);

   // TODO: implement
}

void TelephonyControl::handleIpc2BtsServiceAvailabilityTelephony(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSApp2BtsMessageCompareItem& compareItem, OUT bool& deleteApp2BtsMessage, INOUT App2Bts_BaseMessage** app2BtsMessage, IN Ipc2Bts_ServiceAvailabilityTelephony* ptrMessage)
{
   if(NULL == ptrMessage)
   {
      // should never happen else you have programmed something wrong
      // #error_indication
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   if(NULL == app2BtsMessage)
   {
      // should never happen else you have programmed something wrong
      // #error_indication
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   // *app2BtsMessage is allowed to be NULL e.g. in case of indication/signal message

   bts2AppMsgList.reserve(1);

   // TODO: implement
   (void)(bts2IpcMsgList);
   (void)(compareItem);
   (void)(deleteApp2BtsMessage);

   // do not delete App2BtsMessage because this is a status message
}

void TelephonyControl::handleIpc2BtsAddHfpInterfaceObjectPathMapping(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSApp2BtsMessageCompareItem& compareItem, OUT bool& deleteApp2BtsMessage, INOUT App2Bts_BaseMessage** app2BtsMessage, IN Ipc2Bts_AddHfpInterfaceObjectPathMapping* ptrMessage)
{
   if(NULL == ptrMessage)
   {
      // should never happen else you have programmed something wrong
      // #error_indication
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   if(NULL == app2BtsMessage)
   {
      // should never happen else you have programmed something wrong
      // #error_indication
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   // *app2BtsMessage is allowed to be NULL e.g. in case of indication/signal message

   bts2AppMsgList.reserve(1);

   // TODO: implement
   (void)(bts2IpcMsgList);
   (void)(compareItem);
   (void)(deleteApp2BtsMessage);

   // do not delete App2BtsMessage because this is a status message
}

void TelephonyControl::handleIpc2BtsDelHfpInterfaceObjectPathMapping(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSApp2BtsMessageCompareItem& compareItem, OUT bool& deleteApp2BtsMessage, INOUT App2Bts_BaseMessage** app2BtsMessage, IN Ipc2Bts_DelHfpInterfaceObjectPathMapping* ptrMessage)
{
   if(NULL == ptrMessage)
   {
      // should never happen else you have programmed something wrong
      // #error_indication
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   if(NULL == app2BtsMessage)
   {
      // should never happen else you have programmed something wrong
      // #error_indication
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   // *app2BtsMessage is allowed to be NULL e.g. in case of indication/signal message

   bts2AppMsgList.reserve(1);

   // TODO: implement
   (void)(bts2IpcMsgList);
   (void)(compareItem);
   (void)(deleteApp2BtsMessage);

   // do not delete App2BtsMessage because this is a status message
}

void TelephonyControl::handleIpc2BtsGetHandsfreeProperties(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSApp2BtsMessageCompareItem& compareItem, OUT bool& deleteApp2BtsMessage, INOUT App2Bts_BaseMessage** app2BtsMessage, IN Ipc2Bts_GetHandsfreeProperties* ptrMessage)
{
   if(NULL == ptrMessage)
   {
      // should never happen else you have programmed something wrong
      // #error_indication
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   if(NULL == app2BtsMessage)
   {
      // should never happen else you have programmed something wrong
      // #error_indication
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   // *app2BtsMessage is allowed to be NULL e.g. in case of indication/signal message

   bts2AppMsgList.reserve(1);

   // TODO: implement
   (void)(bts2IpcMsgList);
   (void)(compareItem);
   (void)(deleteApp2BtsMessage);

   // do not delete App2BtsMessage because this is a status message
}

void TelephonyControl::handleIpc2BtsGetHandsfreePropertiesExt(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSApp2BtsMessageCompareItem& compareItem, OUT bool& deleteApp2BtsMessage, INOUT App2Bts_BaseMessage** app2BtsMessage, IN Ipc2Bts_GetHandsfreePropertiesExt* ptrMessage)
{
   if(NULL == ptrMessage)
   {
      // should never happen else you have programmed something wrong
      // #error_indication
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   if(NULL == app2BtsMessage)
   {
      // should never happen else you have programmed something wrong
      // #error_indication
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   // *app2BtsMessage is allowed to be NULL e.g. in case of indication/signal message

   bts2AppMsgList.reserve(1);

   // TODO: implement
   (void)(bts2IpcMsgList);
   (void)(compareItem);
   (void)(deleteApp2BtsMessage);

   // do not delete App2BtsMessage because this is a status message
}

void TelephonyControl::handleIpc2BtsHandsfreeFeaturesUpdate(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSApp2BtsMessageCompareItem& compareItem, OUT bool& deleteApp2BtsMessage, INOUT App2Bts_BaseMessage** app2BtsMessage, IN Ipc2Bts_HandsfreeFeaturesUpdate* ptrMessage)
{
   if(NULL == ptrMessage)
   {
      // should never happen else you have programmed something wrong
      // #error_indication
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   if(NULL == app2BtsMessage)
   {
      // should never happen else you have programmed something wrong
      // #error_indication
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   // *app2BtsMessage is allowed to be NULL e.g. in case of indication/signal message

   bts2AppMsgList.reserve(1);

   // TODO: implement
   (void)(bts2IpcMsgList);
   (void)(compareItem);
   (void)(deleteApp2BtsMessage);

   // do not delete App2BtsMessage because this is a status message
}

void TelephonyControl::handleIpc2BtsHandsfreeInbandRingingUpdate(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSApp2BtsMessageCompareItem& compareItem, OUT bool& deleteApp2BtsMessage, INOUT App2Bts_BaseMessage** app2BtsMessage, IN Ipc2Bts_HandsfreeInbandRingingUpdate* ptrMessage)
{
   if(NULL == ptrMessage)
   {
      // should never happen else you have programmed something wrong
      // #error_indication
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   if(NULL == app2BtsMessage)
   {
      // should never happen else you have programmed something wrong
      // #error_indication
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   // *app2BtsMessage is allowed to be NULL e.g. in case of indication/signal message

   bts2AppMsgList.reserve(1);

   // TODO: implement
   (void)(bts2IpcMsgList);
   (void)(compareItem);
   (void)(deleteApp2BtsMessage);

   // do not delete App2BtsMessage because this is a status message
}

void TelephonyControl::handleIpc2BtsGetModems(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSApp2BtsMessageCompareItem& compareItem, OUT bool& deleteApp2BtsMessage, INOUT App2Bts_BaseMessage** app2BtsMessage, IN Ipc2Bts_GetModems* ptrMessage)
{
   if(NULL == ptrMessage)
   {
      // should never happen else you have programmed something wrong
      // #error_indication
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   if(NULL == app2BtsMessage)
   {
      // should never happen else you have programmed something wrong
      // #error_indication
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   // *app2BtsMessage is allowed to be NULL e.g. in case of indication/signal message

   bts2AppMsgList.reserve(1);

   // TODO: implement
   (void)(bts2IpcMsgList);
   (void)(compareItem);
   (void)(deleteApp2BtsMessage);

   // do not delete App2BtsMessage because this is a status message
}

void TelephonyControl::handleIpc2BtsGetModemsExt(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSApp2BtsMessageCompareItem& compareItem, OUT bool& deleteApp2BtsMessage, INOUT App2Bts_BaseMessage** app2BtsMessage, IN Ipc2Bts_GetModemsExt* ptrMessage)
{
   if(NULL == ptrMessage)
   {
      // should never happen else you have programmed something wrong
      // #error_indication
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   if(NULL == app2BtsMessage)
   {
      // should never happen else you have programmed something wrong
      // #error_indication
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   // *app2BtsMessage is allowed to be NULL e.g. in case of indication/signal message

   bts2AppMsgList.reserve(1);

   // TODO: implement
   (void)(bts2IpcMsgList);
   (void)(compareItem);
   (void)(deleteApp2BtsMessage);

   // do not delete App2BtsMessage because this is a status message
}

void TelephonyControl::handleIpc2BtsModemAdded(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSApp2BtsMessageCompareItem& compareItem, OUT bool& deleteApp2BtsMessage, INOUT App2Bts_BaseMessage** app2BtsMessage, IN Ipc2Bts_ModemAdded* ptrMessage)
{
   if(NULL == ptrMessage)
   {
      // should never happen else you have programmed something wrong
      // #error_indication
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   if(NULL == app2BtsMessage)
   {
      // should never happen else you have programmed something wrong
      // #error_indication
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   // *app2BtsMessage is allowed to be NULL e.g. in case of indication/signal message

   bts2AppMsgList.reserve(1);

   // TODO: implement
   (void)(bts2IpcMsgList);
   (void)(compareItem);
   (void)(deleteApp2BtsMessage);

   // do not delete App2BtsMessage because this is a status message
}

void TelephonyControl::handleIpc2BtsModemAddedExt(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSApp2BtsMessageCompareItem& compareItem, OUT bool& deleteApp2BtsMessage, INOUT App2Bts_BaseMessage** app2BtsMessage, IN Ipc2Bts_ModemAddedExt* ptrMessage)
{
   if(NULL == ptrMessage)
   {
      // should never happen else you have programmed something wrong
      // #error_indication
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   if(NULL == app2BtsMessage)
   {
      // should never happen else you have programmed something wrong
      // #error_indication
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   // *app2BtsMessage is allowed to be NULL e.g. in case of indication/signal message

   bts2AppMsgList.reserve(1);

   // TODO: implement
   (void)(bts2IpcMsgList);
   (void)(compareItem);
   (void)(deleteApp2BtsMessage);

   // do not delete App2BtsMessage because this is a status message
}

void TelephonyControl::handleIpc2BtsModemRemoved(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSApp2BtsMessageCompareItem& compareItem, OUT bool& deleteApp2BtsMessage, INOUT App2Bts_BaseMessage** app2BtsMessage, IN Ipc2Bts_ModemRemoved* ptrMessage)
{
   if(NULL == ptrMessage)
   {
      // should never happen else you have programmed something wrong
      // #error_indication
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   if(NULL == app2BtsMessage)
   {
      // should never happen else you have programmed something wrong
      // #error_indication
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   // *app2BtsMessage is allowed to be NULL e.g. in case of indication/signal message

   bts2AppMsgList.reserve(1);

   // TODO: implement
   (void)(bts2IpcMsgList);
   (void)(compareItem);
   (void)(deleteApp2BtsMessage);

   // do not delete App2BtsMessage because this is a status message
}

void TelephonyControl::handleIpc2BtsGetModemProperties(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSApp2BtsMessageCompareItem& compareItem, OUT bool& deleteApp2BtsMessage, INOUT App2Bts_BaseMessage** app2BtsMessage, IN Ipc2Bts_GetModemProperties* ptrMessage)
{
   if(NULL == ptrMessage)
   {
      // should never happen else you have programmed something wrong
      // #error_indication
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   if(NULL == app2BtsMessage)
   {
      // should never happen else you have programmed something wrong
      // #error_indication
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   // *app2BtsMessage is allowed to be NULL e.g. in case of indication/signal message

   bts2AppMsgList.reserve(1);

   // TODO: implement
   (void)(bts2IpcMsgList);
   (void)(compareItem);
   (void)(deleteApp2BtsMessage);

   // do not delete App2BtsMessage because this is a status message
}

void TelephonyControl::handleIpc2BtsGetModemPropertiesExt(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSApp2BtsMessageCompareItem& compareItem, OUT bool& deleteApp2BtsMessage, INOUT App2Bts_BaseMessage** app2BtsMessage, IN Ipc2Bts_GetModemPropertiesExt* ptrMessage)
{
   if(NULL == ptrMessage)
   {
      // should never happen else you have programmed something wrong
      // #error_indication
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   if(NULL == app2BtsMessage)
   {
      // should never happen else you have programmed something wrong
      // #error_indication
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   // *app2BtsMessage is allowed to be NULL e.g. in case of indication/signal message

   bts2AppMsgList.reserve(1);

   // TODO: implement
   (void)(bts2IpcMsgList);
   (void)(compareItem);
   (void)(deleteApp2BtsMessage);

   // do not delete App2BtsMessage because this is a status message
}

void TelephonyControl::handleIpc2BtsModemInterfacesUpdate(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSApp2BtsMessageCompareItem& compareItem, OUT bool& deleteApp2BtsMessage, INOUT App2Bts_BaseMessage** app2BtsMessage, IN Ipc2Bts_ModemInterfacesUpdate* ptrMessage)
{
   if(NULL == ptrMessage)
   {
      // should never happen else you have programmed something wrong
      // #error_indication
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   if(NULL == app2BtsMessage)
   {
      // should never happen else you have programmed something wrong
      // #error_indication
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   // *app2BtsMessage is allowed to be NULL e.g. in case of indication/signal message

   bts2AppMsgList.reserve(1);

   // TODO: implement
   (void)(bts2IpcMsgList);
   (void)(compareItem);
   (void)(deleteApp2BtsMessage);

   // do not delete App2BtsMessage because this is a status message
}

} //genivi
} //btstackif
