/**
 * @file PhonebookControl_GEN.cpp
 *
 * @par SW-Component
 * Main
 *
 * @brief Phonebook Control.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Implementation of phonebook control functionality.
 */

#include "PhonebookControl_GEN.h"
#include "App2Bts_MessageWrapper.h"
#include "Bts2App_MessageWrapper.h"
#include "Bts2Ipc_MessageWrapper_GEN.h"
#include "Ipc2Bts_MessageWrapper_GEN.h"
#include "EvolutionGeniviStackTypes.h"
#include "EvolutionGeniviUtils.h"
#include "BtsUtils.h"
#include "TraceClasses.h"
#include "FwAssert.h"
#include "FwTrace.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_BTS_CONTROL
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/PhonebookControl_GEN.cpp.trc.h"
#endif
#endif

namespace btstackif {
namespace genivi {

PhonebookControl::PhonebookControl()
: BasicControl(BTS_FB_PHONEBOOK)
{
   /*********** start here with specific class members ***************************/
   /*********** end here with specific class members *****************************/
}

PhonebookControl::~PhonebookControl()
{
   /*********** start here with specific class members ***************************/
   /*********** end here with specific class members *****************************/
}

void PhonebookControl::pushApp2BtsMessage(IN App2Bts_BaseMessage* ptrMessage)
{
   if(NULL == ptrMessage)
   {
      // should never happen else you have programmed something wrong
      // #error_indication
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   if(_functionBlock != ptrMessage->getFunctionBlock())
   {
      // should never happen else you have programmed something wrong
      // #error_indication
      FW_NORMAL_ASSERT_ALWAYS();
      delete ptrMessage;
      return;
   }

   // HINT: if any error happens now the error has to be handled within this function or within any called sub-function

   bool similarOpCodeInWorkingQueue = false;

   // check for skipping similar opcode check
   if(false == skipSimilarOpCodeCheck(ptrMessage))
   {
      // get similar opcodes
      ::std::vector<BTSApp2BtsMessageMasterCompareItem> itemList;
      getSimilarOpCodes(itemList, ptrMessage);

      // check if opcode is in working queue
      similarOpCodeInWorkingQueue = isSimilarOpCodeInWorkingQueue(itemList);
   }

   // check if opcode is in working queue
   if(true == similarOpCodeInWorkingQueue)
   {
      // push to waiting queue
      pushApp2BtsMsgToWaitingQueue(ptrMessage, false); // single worker thread
      ETG_TRACE_USR3((" pushApp2BtsMessage(): App2Bts 0x%04X pushed to waiting queue", ptrMessage->getTraceOpCode()));
   }
   else
   {
      BTSApp2BtsOpcode opcode = ptrMessage->getOpCode();
      ::std::vector<Bts2Ipc_BaseMessage*> sendBts2IpcMsgList;
      sendBts2IpcMsgList.reserve(10);
      ::std::vector<Bts2App_BaseMessage*> sendBts2AppMsgList;
      sendBts2AppMsgList.reserve(10);
      bool deleteApp2BtsMsg = false;
      BTSApp2BtsMessageCompareItem compareItem;
      ptrMessage->getCompareItem(compareItem);

      ETG_TRACE_USR3((" pushApp2BtsMessage(): App2Bts 0x%04X to be processed", ptrMessage->getTraceOpCode()));

      // handle message depending on opcode
      // - set marker to delete message in sub-handler function if necessary
      // - push message to working queue in sub-handler function if necessary
      // - create direct answer message (Bts2App) in sub-handler function if necessary
      // - create Bts2Ipc message in sub-handler function if necessary
      // - handle any error in sub-handler function because there is the best place to handle
      switch(opcode)
      {
         case App2BtsOC_SetPhonebookDummy:
            handleApp2BtsSetPhonebookDummy(sendBts2IpcMsgList, sendBts2AppMsgList, deleteApp2BtsMsg, static_cast<App2Bts_SetPhonebookDummy*>(ptrMessage)); /*lint !e1774: Could use dynamic_cast to downcast polymorphic type*/ // no RTTI, use static_cast instead of dynamic_cast, we know the type of the message
            break;
         // all other
         default:
            FW_NORMAL_ASSERT_ALWAYS();
            deleteApp2BtsMsg = true;
            break;
      }

      if(0 < sendBts2IpcMsgList.size())
      {
         _bts2IpcMessageWasSent = true; // this works only if single thread handling TODO: recheck again
      }

      sendBts2IpcMessageList(sendBts2IpcMsgList, compareItem);

      sendBts2AppMessageList(sendBts2AppMsgList);

      if(true == deleteApp2BtsMsg)
      {
         delete ptrMessage;
      }
   }
}

void PhonebookControl::pushIpc2BtsMessage(IN Ipc2Bts_BaseMessage* ptrMessage)
{
   if(NULL == ptrMessage)
   {
      // should never happen else you have programmed something wrong
      // #error_indication
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   if(_functionBlock != ptrMessage->getBtsDestinationFunctionBlock())
   {
      // should never happen else you have programmed something wrong
      // #error_indication
      FW_NORMAL_ASSERT_ALWAYS();
      delete ptrMessage;
      return;
   }

   // HINT: if any error happens now the error has to be handled within this function or within any called sub-function

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)ptrMessage->getOpCode();
   ::std::vector<Bts2Ipc_BaseMessage*> sendBts2IpcMsgList;
   sendBts2IpcMsgList.reserve(10);
   ::std::vector<Bts2App_BaseMessage*> sendBts2AppMsgList;
   sendBts2AppMsgList.reserve(10);
   App2Bts_BaseMessage* ptrApp2BtsQueueMessage;
   bool deleteApp2BtsMsg = false;

   // find related APP2BTS message
   // for req -> cfm sequence there must be a message in working queue
   // for ind sequence there can be a message in working queue
   ptrApp2BtsQueueMessage = findApp2BtsWorkingMessageWrapper(ptrMessage->getApp2BtsCompareItem());

   // special handling is needed for e.g. connect/disconnect
   BTSApp2BtsMessageCompareItem specialCompareItem;
   specialCompareItem.opCode = App2BtsOC_None;

   // handle message depending on opcode
   // - do not delete Ipc2Bts message in sub-handler function because this done at the end of this function
   // - set marker to delete related App2Bts message in sub-handler function if necessary
   // - create Bts2App message in sub-handler function if necessary
   // - create Bts2Ipc message in sub-handler function if necessary
   // - handle any error in sub-handler function because there is the best place to handle
   switch(opcode)
   {
      case Ipc2BtsOC_ServiceAvailabilityPhonebook:
         handleIpc2BtsServiceAvailabilityPhonebook(sendBts2IpcMsgList, sendBts2AppMsgList, specialCompareItem, deleteApp2BtsMsg, &ptrApp2BtsQueueMessage, static_cast<Ipc2Bts_ServiceAvailabilityPhonebook*>(ptrMessage)); /*lint !e1774: Could use dynamic_cast to downcast polymorphic type*/ // no RTTI, use static_cast instead of dynamic_cast, we know the type of the message
         break;
      case Ipc2BtsOC_PimSessionSourceUpdate:
         handleIpc2BtsPimSessionSourceUpdate(sendBts2IpcMsgList, sendBts2AppMsgList, specialCompareItem, deleteApp2BtsMsg, &ptrApp2BtsQueueMessage, static_cast<Ipc2Bts_PimSessionSourceUpdate*>(ptrMessage)); /*lint !e1774: Could use dynamic_cast to downcast polymorphic type*/ // no RTTI, use static_cast instead of dynamic_cast, we know the type of the message
         break;
      case Ipc2BtsOC_PimSessionDestinationUpdate:
         handleIpc2BtsPimSessionDestinationUpdate(sendBts2IpcMsgList, sendBts2AppMsgList, specialCompareItem, deleteApp2BtsMsg, &ptrApp2BtsQueueMessage, static_cast<Ipc2Bts_PimSessionDestinationUpdate*>(ptrMessage)); /*lint !e1774: Could use dynamic_cast to downcast polymorphic type*/ // no RTTI, use static_cast instead of dynamic_cast, we know the type of the message
         break;
      case Ipc2BtsOC_PimSessionChannelUpdate:
         handleIpc2BtsPimSessionChannelUpdate(sendBts2IpcMsgList, sendBts2AppMsgList, specialCompareItem, deleteApp2BtsMsg, &ptrApp2BtsQueueMessage, static_cast<Ipc2Bts_PimSessionChannelUpdate*>(ptrMessage)); /*lint !e1774: Could use dynamic_cast to downcast polymorphic type*/ // no RTTI, use static_cast instead of dynamic_cast, we know the type of the message
         break;
      case Ipc2BtsOC_PimSessionTargetUpdate:
         handleIpc2BtsPimSessionTargetUpdate(sendBts2IpcMsgList, sendBts2AppMsgList, specialCompareItem, deleteApp2BtsMsg, &ptrApp2BtsQueueMessage, static_cast<Ipc2Bts_PimSessionTargetUpdate*>(ptrMessage)); /*lint !e1774: Could use dynamic_cast to downcast polymorphic type*/ // no RTTI, use static_cast instead of dynamic_cast, we know the type of the message
         break;
      case Ipc2BtsOC_PimTransferCancelResult:
         handleIpc2BtsPimTransferCancelResult(sendBts2IpcMsgList, sendBts2AppMsgList, specialCompareItem, deleteApp2BtsMsg, &ptrApp2BtsQueueMessage, static_cast<Ipc2Bts_PimTransferCancelResult*>(ptrMessage)); /*lint !e1774: Could use dynamic_cast to downcast polymorphic type*/ // no RTTI, use static_cast instead of dynamic_cast, we know the type of the message
         break;
      case Ipc2BtsOC_PimTransferStatusUpdate:
         handleIpc2BtsPimTransferStatusUpdate(sendBts2IpcMsgList, sendBts2AppMsgList, specialCompareItem, deleteApp2BtsMsg, &ptrApp2BtsQueueMessage, static_cast<Ipc2Bts_PimTransferStatusUpdate*>(ptrMessage)); /*lint !e1774: Could use dynamic_cast to downcast polymorphic type*/ // no RTTI, use static_cast instead of dynamic_cast, we know the type of the message
         break;
      case Ipc2BtsOC_PimTransferSessionUpdate:
         handleIpc2BtsPimTransferSessionUpdate(sendBts2IpcMsgList, sendBts2AppMsgList, specialCompareItem, deleteApp2BtsMsg, &ptrApp2BtsQueueMessage, static_cast<Ipc2Bts_PimTransferSessionUpdate*>(ptrMessage)); /*lint !e1774: Could use dynamic_cast to downcast polymorphic type*/ // no RTTI, use static_cast instead of dynamic_cast, we know the type of the message
         break;
      case Ipc2BtsOC_PimTransferNameUpdate:
         handleIpc2BtsPimTransferNameUpdate(sendBts2IpcMsgList, sendBts2AppMsgList, specialCompareItem, deleteApp2BtsMsg, &ptrApp2BtsQueueMessage, static_cast<Ipc2Bts_PimTransferNameUpdate*>(ptrMessage)); /*lint !e1774: Could use dynamic_cast to downcast polymorphic type*/ // no RTTI, use static_cast instead of dynamic_cast, we know the type of the message
         break;
      case Ipc2BtsOC_PimTransferTypeUpdate:
         handleIpc2BtsPimTransferTypeUpdate(sendBts2IpcMsgList, sendBts2AppMsgList, specialCompareItem, deleteApp2BtsMsg, &ptrApp2BtsQueueMessage, static_cast<Ipc2Bts_PimTransferTypeUpdate*>(ptrMessage)); /*lint !e1774: Could use dynamic_cast to downcast polymorphic type*/ // no RTTI, use static_cast instead of dynamic_cast, we know the type of the message
         break;
      case Ipc2BtsOC_PimTransferTimeUpdate:
         handleIpc2BtsPimTransferTimeUpdate(sendBts2IpcMsgList, sendBts2AppMsgList, specialCompareItem, deleteApp2BtsMsg, &ptrApp2BtsQueueMessage, static_cast<Ipc2Bts_PimTransferTimeUpdate*>(ptrMessage)); /*lint !e1774: Could use dynamic_cast to downcast polymorphic type*/ // no RTTI, use static_cast instead of dynamic_cast, we know the type of the message
         break;
      case Ipc2BtsOC_PimTransferSizeUpdate:
         handleIpc2BtsPimTransferSizeUpdate(sendBts2IpcMsgList, sendBts2AppMsgList, specialCompareItem, deleteApp2BtsMsg, &ptrApp2BtsQueueMessage, static_cast<Ipc2Bts_PimTransferSizeUpdate*>(ptrMessage)); /*lint !e1774: Could use dynamic_cast to downcast polymorphic type*/ // no RTTI, use static_cast instead of dynamic_cast, we know the type of the message
         break;
      case Ipc2BtsOC_PimTransferTransferredUpdate:
         handleIpc2BtsPimTransferTransferredUpdate(sendBts2IpcMsgList, sendBts2AppMsgList, specialCompareItem, deleteApp2BtsMsg, &ptrApp2BtsQueueMessage, static_cast<Ipc2Bts_PimTransferTransferredUpdate*>(ptrMessage)); /*lint !e1774: Could use dynamic_cast to downcast polymorphic type*/ // no RTTI, use static_cast instead of dynamic_cast, we know the type of the message
         break;
      case Ipc2BtsOC_PimTransferFilenameUpdate:
         handleIpc2BtsPimTransferFilenameUpdate(sendBts2IpcMsgList, sendBts2AppMsgList, specialCompareItem, deleteApp2BtsMsg, &ptrApp2BtsQueueMessage, static_cast<Ipc2Bts_PimTransferFilenameUpdate*>(ptrMessage)); /*lint !e1774: Could use dynamic_cast to downcast polymorphic type*/ // no RTTI, use static_cast instead of dynamic_cast, we know the type of the message
         break;
      // all other
      default:
         FW_NORMAL_ASSERT_ALWAYS();
         break;
   }

   if(0 < sendBts2IpcMsgList.size())
   {
      if(NULL != ptrApp2BtsQueueMessage)
      {
         BTSApp2BtsMessageCompareItem compareItem;
         ptrApp2BtsQueueMessage->getCompareItem(compareItem);
         sendBts2IpcMessageList(sendBts2IpcMsgList, compareItem);
      }
      else
      {
         ::std::vector<Bts2Ipc_BaseMessage*> respSendBts2IpcMsgList;
         respSendBts2IpcMsgList.reserve(sendBts2IpcMsgList.size());

         for(size_t i = 0; i < sendBts2IpcMsgList.size(); i++)
         {
            if(NULL != sendBts2IpcMsgList[i])
            {
               if(true == sendBts2IpcMsgList[i]->getResponseMessageFlag())
               {
                  // response message: it could be possible that there is no related application message
                  respSendBts2IpcMsgList.push_back(sendBts2IpcMsgList[i]);
               }
               else
               {
                  // should never happen else you have programmed something wrong
                  // #error_indication
                  FW_NORMAL_ASSERT_ALWAYS();

                  delete sendBts2IpcMsgList[i];
               }
            }
         }

         sendBts2IpcMsgList.clear();

         if(0 < respSendBts2IpcMsgList.size())
         {
            BTSApp2BtsMessageCompareItem compareItem;
            sendBts2IpcMessageList(respSendBts2IpcMsgList, compareItem);
         }
      }
   }

   sendBts2AppMessageList(sendBts2AppMsgList);

   // delete received message
   delete ptrMessage;

   // remove related APP2BTS message and check waiting queue
   if((true == deleteApp2BtsMsg) && (NULL != ptrApp2BtsQueueMessage))
   {
      removeApp2BtsWorkingMessage(ptrApp2BtsQueueMessage);

      if(App2BtsOC_None == specialCompareItem.opCode)
      {
         checkWaitingQueue(ptrApp2BtsQueueMessage);
      }

      delete ptrApp2BtsQueueMessage;
   }

   // check waiting queue (special handling for connect/disconnect)
   if((App2BtsOC_None != specialCompareItem.opCode) && (App2BtsOC_Ignore != specialCompareItem.opCode))
   {
      checkWaitingQueueExtended(specialCompareItem);
   }
}

void PhonebookControl::setStackConfiguration(IN const BTSFunctionBlock component, IN const BTSInterfaceType stackInterface, IN const BTSFunctionBlock subComponent, IN const BTSUserMode userMode,
         OUT ::std::vector<BTSDbusInterfaceItem>& dbusInterfaces, IN const BTSLocalConfigurationContainer& configuration)
{
   (void)(component);
   (void)(stackInterface);
   (void)(subComponent);
   (void)(userMode);
   (void)(dbusInterfaces);
   (void)(configuration);
}

void PhonebookControl::triggerInitializedCallback(void)
{
}

void PhonebookControl::createDbusServiceAvailabilityMessage(IN const BTSCommonEnumClass interface, IN const BTSDbusServiceAvailability availabilityEvent)
{
   (void)(interface);
   (void)(availabilityEvent);
}

void PhonebookControl::createDbusServiceAvailabilityMessage(IN const BTSCommonEnumClass interface, IN const BTSDbusServiceAvailability availabilityEvent, IN const BTSBusName& busName, IN const BTSObjectPath& objPath, IN const BTSCommonEnumClass busType)
{
   (void)(interface);
   (void)(availabilityEvent);
   (void)(busName);
   (void)(objPath);
   (void)(busType);
}

void PhonebookControl::setSubControlTestCommand(IN const char* testCommand, IN const unsigned int testData)
{
   if(NULL == testCommand)
   {
      return;
   }

   (void)(testData);
}

void PhonebookControl::setSubControlTestCommand(IN const char* testCommand, IN const unsigned char* testData)
{
   if(NULL == testCommand)
   {
      return;
   }

   if(NULL == testData)
   {
      return;
   }
}

void PhonebookControl::sendDirectAnswerForApp2BtsMessages(IN const ::std::vector<App2Bts_BaseMessage*>& msgList, IN const BTSCommonEnumClass resultCode, IN const BTSCommonEnumClass statusCode)
{
   (void)(resultCode);
   (void)(statusCode);

   // TODO: recheck what to do with doubled requests - answer or ignore? what to do with status?

   if(0 == msgList.size())
   {
      // should never happen else you have programmed something wrong
      // #error_indication
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   App2Bts_BaseMessage* ptrMessage;

   for(size_t i = 0; i < msgList.size(); i++)
   {
      ptrMessage = msgList[i];

      if(NULL == ptrMessage)
      {
         // should never happen else you have programmed something wrong
         // #error_indication
         FW_NORMAL_ASSERT_ALWAYS();
         continue;
      }

      if(_functionBlock != ptrMessage->getFunctionBlock())
      {
         // should never happen else you have programmed something wrong
         // #error_indication
         FW_NORMAL_ASSERT_ALWAYS();
         continue;
      }

      // if(true == _handleDoubledRequests) TODO: check after merging
      {
         BTSApp2BtsOpcode opcode = ptrMessage->getOpCode();
         ::std::vector<Bts2App_BaseMessage*> sendBts2AppMsgList;
         sendBts2AppMsgList.reserve(10);

         ETG_TRACE_USR3((" sendDirectAnswerForApp2BtsMessages(): App2Bts 0x%04X to be handled", ptrMessage->getTraceOpCode()));

         // send direct answer depending on opcode
         // - create direct answer message (Bts2App) in sub-handler function
         // - handle any error in sub-handler function because there is the best place to handle
         // - do not delete message; this is done outside of this function
         switch(opcode)
         {
            case App2BtsOC_SetPhonebookDummy:
               handleDoubledApp2BtsSetPhonebookDummy(sendBts2AppMsgList, static_cast<App2Bts_SetPhonebookDummy*>(ptrMessage), resultCode, statusCode); /*lint !e1774: Could use dynamic_cast to downcast polymorphic type*/ // no RTTI, use static_cast instead of dynamic_cast, we know the type of the message
               break;
            // all other
            default:
               FW_NORMAL_ASSERT_ALWAYS();
               break;
         }

         sendBts2AppMessageList(sendBts2AppMsgList);
      }
   }
}

bool PhonebookControl::doApp2BtsMsgPrecheck(OUT bool& rejectRequest, OUT BTSCommonEnumClass& resultCode, OUT BTSCommonEnumClass& statusCode, OUT bool& skipOpCodeCheck, IN App2Bts_BaseMessage* ptrMessage)
{
   if(NULL == ptrMessage)
   {
      // should never happen else you have programmed something wrong
      // #error_indication
      FW_NORMAL_ASSERT_ALWAYS();
      return false;
   }

   (void)(rejectRequest);
   (void)(resultCode);
   (void)(statusCode);
   (void)(skipOpCodeCheck);

   return true;
}

void PhonebookControl::getSimilarOpCodes(OUT ::std::vector<BTSApp2BtsMessageMasterCompareItem>& itemList, IN const App2Bts_BaseMessage* ptrMessage)
{
   if(NULL == ptrMessage)
   {
      // should never happen else you have programmed something wrong
      // #error_indication
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   BTSApp2BtsOpcode opCode = ptrMessage->getOpCode();
   BTSApp2BtsMessageMasterCompareItem item;

   if((App2BtsOC_PhonebookBlockStart < opCode) && (opCode < App2BtsOC_PhonebookBlockEnd))
   {
      size_t reserveSize = 0;

      // TODO: complete switch
      switch(opCode)
      {
         case App2BtsOC_SetPhonebookDummy:
            reserveSize += 1;
            itemList.reserve(reserveSize);
            item.opCode = opCode;
            itemList.push_back(item);
            break;
         // add all other here if necessary
         default:
            FW_NORMAL_ASSERT_ALWAYS();
            break;
      }
   }
   else
   {
      // opcode in wrong range
      FW_NORMAL_ASSERT_ALWAYS();

      // add at least input opcode
      item.opCode = opCode;
      itemList.push_back(item);
   }
}

void PhonebookControl::getMatchingOpCodes(OUT ::std::vector<BTSApp2BtsMessageMasterCompareItem>& itemList, OUT ::std::vector<BTSApp2BtsMessageMasterCompareItem>& highPrioItemList, IN const App2Bts_BaseMessage* ptrMessage)
{
   if(NULL == ptrMessage)
   {
      // should never happen else you have programmed something wrong
      // #error_indication
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   (void)(highPrioItemList);
   BTSApp2BtsOpcode opCode = ptrMessage->getOpCode();
   BTSApp2BtsMessageMasterCompareItem item;
   ptrMessage->getCompareItem(item);

   if((App2BtsOC_PhonebookBlockStart < opCode) && (opCode < App2BtsOC_PhonebookBlockEnd))
   {
      size_t reserveSize;

      // TODO: complete switch
      switch(opCode)
      {
         case App2BtsOC_SetPhonebookDummy:
            reserveSize = 1;
            itemList.reserve(reserveSize);
            item.opCode = opCode;
            itemList.push_back(item);
            break;
         // add all other here if necessary
         default:
            FW_NORMAL_ASSERT_ALWAYS();
            break;
      }
   }
   else
   {
      // opcode in wrong range
      FW_NORMAL_ASSERT_ALWAYS();

      // add at least input opcode
      item.opCode = opCode;
      itemList.push_back(item);
   }
}

bool PhonebookControl::skipSimilarOpCodeCheck(IN const App2Bts_BaseMessage* ptrMessage)
{
   (void)(ptrMessage);
   // implement if necessary

   return false;
}

void PhonebookControl::checkWaitingQueueExtended(IN const BTSApp2BtsMessageCompareItem& compareItem)
{
   (void)(compareItem);
   // implement if necessary
}

void PhonebookControl::handleDoubledApp2BtsMessages(IN const ::std::vector<App2Bts_BaseMessage*>& msgList)
{
   // TODO: recheck what to do with doubled requests - answer or ignore? what to do with status?

   if(0 == msgList.size())
   {
      // should never happen else you have programmed something wrong
      // #error_indication
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   App2Bts_BaseMessage* ptrMessage;

   for(size_t i = 0; i < msgList.size(); i++)
   {
      ptrMessage = msgList[i];

      if(NULL == ptrMessage)
      {
         // should never happen else you have programmed something wrong
         // #error_indication
         FW_NORMAL_ASSERT_ALWAYS();
         continue;
      }

      if(_functionBlock != ptrMessage->getFunctionBlock())
      {
         // should never happen else you have programmed something wrong
         // #error_indication
         FW_NORMAL_ASSERT_ALWAYS();
         continue;
      }

      if(true == _handleDoubledRequests)
      {
         BTSApp2BtsOpcode opcode = ptrMessage->getOpCode();
         ::std::vector<Bts2App_BaseMessage*> sendBts2AppMsgList;
         sendBts2AppMsgList.reserve(10);

         ETG_TRACE_USR3((" handleDoubledApp2BtsMessages(): App2Bts 0x%04X to be handled", ptrMessage->getTraceOpCode()));

         // handle doubled messages depending on opcode
         // - create direct answer message (Bts2App) in sub-handler function
         // - handle any error in sub-handler function because there is the best place to handle
         // - do not delete message; this is done at the end of this function
         switch(opcode)
         {
            case App2BtsOC_SetPhonebookDummy:
               handleDoubledApp2BtsSetPhonebookDummy(sendBts2AppMsgList, static_cast<App2Bts_SetPhonebookDummy*>(ptrMessage), (BTSCommonEnumClass)BTS_REQ_SUCCESS, BTS_COMMON_ENUM_CLASS_DEFAULT_VALUE); /*lint !e1774: Could use dynamic_cast to downcast polymorphic type*/ // no RTTI, use static_cast instead of dynamic_cast, we know the type of the message
               break;
            // all other
            default:
               FW_NORMAL_ASSERT_ALWAYS();
               break;
         }

         sendBts2AppMessageList(sendBts2AppMsgList);
      }
   }
}

/*********** start here with specific class members + methods *****************/
/*********** end here with specific class members + methods *******************/

void PhonebookControl::handleDoubledApp2BtsSetPhonebookDummy(OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, IN App2Bts_SetPhonebookDummy* ptrMessage, IN const BTSCommonEnumClass resultCode, IN const BTSCommonEnumClass statusCode)
{
   (void)(bts2AppMsgList);
   (void)(resultCode);
   (void)(statusCode);

   if(NULL == ptrMessage)
   {
      // should never happen else you have programmed something wrong
      // #error_indication
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   // send answer directly because doubled request - set all to success
   // TODO
}

void PhonebookControl::handleApp2BtsSetPhonebookDummy(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT bool& deleteApp2BtsMessage, IN App2Bts_SetPhonebookDummy* ptrMessage)
{
   (void)(bts2AppMsgList);
   (void)(deleteApp2BtsMessage);

   if(NULL == ptrMessage)
   {
      // should never happen else you have programmed something wrong
      // #error_indication
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   // TODO: implement
   _app2BtsWorkingQueue.push(ptrMessage, false);

   bts2IpcMsgList.reserve(1);

   // TODO: implement
}

void PhonebookControl::handleIpc2BtsServiceAvailabilityPhonebook(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSApp2BtsMessageCompareItem& compareItem, OUT bool& deleteApp2BtsMessage, INOUT App2Bts_BaseMessage** app2BtsMessage, IN Ipc2Bts_ServiceAvailabilityPhonebook* ptrMessage)
{
   if(NULL == ptrMessage)
   {
      // should never happen else you have programmed something wrong
      // #error_indication
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   if(NULL == app2BtsMessage)
   {
      // should never happen else you have programmed something wrong
      // #error_indication
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   // *app2BtsMessage is allowed to be NULL e.g. in case of indication/signal message

   bts2AppMsgList.reserve(1);

   // TODO: implement
   (void)(bts2IpcMsgList);
   (void)(compareItem);
   (void)(deleteApp2BtsMessage);

   // do not delete App2BtsMessage because this is a status message
}

void PhonebookControl::handleIpc2BtsPimSessionSourceUpdate(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSApp2BtsMessageCompareItem& compareItem, OUT bool& deleteApp2BtsMessage, INOUT App2Bts_BaseMessage** app2BtsMessage, IN Ipc2Bts_PimSessionSourceUpdate* ptrMessage)
{
   if(NULL == ptrMessage)
   {
      // should never happen else you have programmed something wrong
      // #error_indication
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   if(NULL == app2BtsMessage)
   {
      // should never happen else you have programmed something wrong
      // #error_indication
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   // *app2BtsMessage is allowed to be NULL e.g. in case of indication/signal message

   bts2AppMsgList.reserve(1);

   // TODO: implement
   (void)(bts2IpcMsgList);
   (void)(compareItem);
   (void)(deleteApp2BtsMessage);

   // do not delete App2BtsMessage because this is a status message
}

void PhonebookControl::handleIpc2BtsPimSessionDestinationUpdate(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSApp2BtsMessageCompareItem& compareItem, OUT bool& deleteApp2BtsMessage, INOUT App2Bts_BaseMessage** app2BtsMessage, IN Ipc2Bts_PimSessionDestinationUpdate* ptrMessage)
{
   if(NULL == ptrMessage)
   {
      // should never happen else you have programmed something wrong
      // #error_indication
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   if(NULL == app2BtsMessage)
   {
      // should never happen else you have programmed something wrong
      // #error_indication
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   // *app2BtsMessage is allowed to be NULL e.g. in case of indication/signal message

   bts2AppMsgList.reserve(1);

   // TODO: implement
   (void)(bts2IpcMsgList);
   (void)(compareItem);
   (void)(deleteApp2BtsMessage);

   // do not delete App2BtsMessage because this is a status message
}

void PhonebookControl::handleIpc2BtsPimSessionChannelUpdate(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSApp2BtsMessageCompareItem& compareItem, OUT bool& deleteApp2BtsMessage, INOUT App2Bts_BaseMessage** app2BtsMessage, IN Ipc2Bts_PimSessionChannelUpdate* ptrMessage)
{
   if(NULL == ptrMessage)
   {
      // should never happen else you have programmed something wrong
      // #error_indication
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   if(NULL == app2BtsMessage)
   {
      // should never happen else you have programmed something wrong
      // #error_indication
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   // *app2BtsMessage is allowed to be NULL e.g. in case of indication/signal message

   bts2AppMsgList.reserve(1);

   // TODO: implement
   (void)(bts2IpcMsgList);
   (void)(compareItem);
   (void)(deleteApp2BtsMessage);

   // do not delete App2BtsMessage because this is a status message
}

void PhonebookControl::handleIpc2BtsPimSessionTargetUpdate(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSApp2BtsMessageCompareItem& compareItem, OUT bool& deleteApp2BtsMessage, INOUT App2Bts_BaseMessage** app2BtsMessage, IN Ipc2Bts_PimSessionTargetUpdate* ptrMessage)
{
   if(NULL == ptrMessage)
   {
      // should never happen else you have programmed something wrong
      // #error_indication
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   if(NULL == app2BtsMessage)
   {
      // should never happen else you have programmed something wrong
      // #error_indication
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   // *app2BtsMessage is allowed to be NULL e.g. in case of indication/signal message

   bts2AppMsgList.reserve(1);

   // TODO: implement
   (void)(bts2IpcMsgList);
   (void)(compareItem);
   (void)(deleteApp2BtsMessage);

   // do not delete App2BtsMessage because this is a status message
}

void PhonebookControl::handleIpc2BtsPimTransferCancelResult(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSApp2BtsMessageCompareItem& compareItem, OUT bool& deleteApp2BtsMessage, INOUT App2Bts_BaseMessage** app2BtsMessage, IN Ipc2Bts_PimTransferCancelResult* ptrMessage)
{
   if(NULL == ptrMessage)
   {
      // should never happen else you have programmed something wrong
      // #error_indication
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   if(NULL == app2BtsMessage)
   {
      // should never happen else you have programmed something wrong
      // #error_indication
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   // *app2BtsMessage is allowed to be NULL e.g. in case of indication/signal message

   bts2AppMsgList.reserve(1);

   // TODO: implement
   (void)(bts2IpcMsgList);
   (void)(compareItem);
   (void)(deleteApp2BtsMessage);

   deleteApp2BtsMessage = true;
}

void PhonebookControl::handleIpc2BtsPimTransferStatusUpdate(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSApp2BtsMessageCompareItem& compareItem, OUT bool& deleteApp2BtsMessage, INOUT App2Bts_BaseMessage** app2BtsMessage, IN Ipc2Bts_PimTransferStatusUpdate* ptrMessage)
{
   if(NULL == ptrMessage)
   {
      // should never happen else you have programmed something wrong
      // #error_indication
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   if(NULL == app2BtsMessage)
   {
      // should never happen else you have programmed something wrong
      // #error_indication
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   // *app2BtsMessage is allowed to be NULL e.g. in case of indication/signal message

   bts2AppMsgList.reserve(1);

   // TODO: implement
   (void)(bts2IpcMsgList);
   (void)(compareItem);
   (void)(deleteApp2BtsMessage);

   // do not delete App2BtsMessage because this is a status message
}

void PhonebookControl::handleIpc2BtsPimTransferSessionUpdate(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSApp2BtsMessageCompareItem& compareItem, OUT bool& deleteApp2BtsMessage, INOUT App2Bts_BaseMessage** app2BtsMessage, IN Ipc2Bts_PimTransferSessionUpdate* ptrMessage)
{
   if(NULL == ptrMessage)
   {
      // should never happen else you have programmed something wrong
      // #error_indication
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   if(NULL == app2BtsMessage)
   {
      // should never happen else you have programmed something wrong
      // #error_indication
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   // *app2BtsMessage is allowed to be NULL e.g. in case of indication/signal message

   bts2AppMsgList.reserve(1);

   // TODO: implement
   (void)(bts2IpcMsgList);
   (void)(compareItem);
   (void)(deleteApp2BtsMessage);

   // do not delete App2BtsMessage because this is a status message
}

void PhonebookControl::handleIpc2BtsPimTransferNameUpdate(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSApp2BtsMessageCompareItem& compareItem, OUT bool& deleteApp2BtsMessage, INOUT App2Bts_BaseMessage** app2BtsMessage, IN Ipc2Bts_PimTransferNameUpdate* ptrMessage)
{
   if(NULL == ptrMessage)
   {
      // should never happen else you have programmed something wrong
      // #error_indication
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   if(NULL == app2BtsMessage)
   {
      // should never happen else you have programmed something wrong
      // #error_indication
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   // *app2BtsMessage is allowed to be NULL e.g. in case of indication/signal message

   bts2AppMsgList.reserve(1);

   // TODO: implement
   (void)(bts2IpcMsgList);
   (void)(compareItem);
   (void)(deleteApp2BtsMessage);

   // do not delete App2BtsMessage because this is a status message
}

void PhonebookControl::handleIpc2BtsPimTransferTypeUpdate(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSApp2BtsMessageCompareItem& compareItem, OUT bool& deleteApp2BtsMessage, INOUT App2Bts_BaseMessage** app2BtsMessage, IN Ipc2Bts_PimTransferTypeUpdate* ptrMessage)
{
   if(NULL == ptrMessage)
   {
      // should never happen else you have programmed something wrong
      // #error_indication
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   if(NULL == app2BtsMessage)
   {
      // should never happen else you have programmed something wrong
      // #error_indication
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   // *app2BtsMessage is allowed to be NULL e.g. in case of indication/signal message

   bts2AppMsgList.reserve(1);

   // TODO: implement
   (void)(bts2IpcMsgList);
   (void)(compareItem);
   (void)(deleteApp2BtsMessage);

   // do not delete App2BtsMessage because this is a status message
}

void PhonebookControl::handleIpc2BtsPimTransferTimeUpdate(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSApp2BtsMessageCompareItem& compareItem, OUT bool& deleteApp2BtsMessage, INOUT App2Bts_BaseMessage** app2BtsMessage, IN Ipc2Bts_PimTransferTimeUpdate* ptrMessage)
{
   if(NULL == ptrMessage)
   {
      // should never happen else you have programmed something wrong
      // #error_indication
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   if(NULL == app2BtsMessage)
   {
      // should never happen else you have programmed something wrong
      // #error_indication
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   // *app2BtsMessage is allowed to be NULL e.g. in case of indication/signal message

   bts2AppMsgList.reserve(1);

   // TODO: implement
   (void)(bts2IpcMsgList);
   (void)(compareItem);
   (void)(deleteApp2BtsMessage);

   // do not delete App2BtsMessage because this is a status message
}

void PhonebookControl::handleIpc2BtsPimTransferSizeUpdate(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSApp2BtsMessageCompareItem& compareItem, OUT bool& deleteApp2BtsMessage, INOUT App2Bts_BaseMessage** app2BtsMessage, IN Ipc2Bts_PimTransferSizeUpdate* ptrMessage)
{
   if(NULL == ptrMessage)
   {
      // should never happen else you have programmed something wrong
      // #error_indication
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   if(NULL == app2BtsMessage)
   {
      // should never happen else you have programmed something wrong
      // #error_indication
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   // *app2BtsMessage is allowed to be NULL e.g. in case of indication/signal message

   bts2AppMsgList.reserve(1);

   // TODO: implement
   (void)(bts2IpcMsgList);
   (void)(compareItem);
   (void)(deleteApp2BtsMessage);

   // do not delete App2BtsMessage because this is a status message
}

void PhonebookControl::handleIpc2BtsPimTransferTransferredUpdate(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSApp2BtsMessageCompareItem& compareItem, OUT bool& deleteApp2BtsMessage, INOUT App2Bts_BaseMessage** app2BtsMessage, IN Ipc2Bts_PimTransferTransferredUpdate* ptrMessage)
{
   if(NULL == ptrMessage)
   {
      // should never happen else you have programmed something wrong
      // #error_indication
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   if(NULL == app2BtsMessage)
   {
      // should never happen else you have programmed something wrong
      // #error_indication
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   // *app2BtsMessage is allowed to be NULL e.g. in case of indication/signal message

   bts2AppMsgList.reserve(1);

   // TODO: implement
   (void)(bts2IpcMsgList);
   (void)(compareItem);
   (void)(deleteApp2BtsMessage);

   // do not delete App2BtsMessage because this is a status message
}

void PhonebookControl::handleIpc2BtsPimTransferFilenameUpdate(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSApp2BtsMessageCompareItem& compareItem, OUT bool& deleteApp2BtsMessage, INOUT App2Bts_BaseMessage** app2BtsMessage, IN Ipc2Bts_PimTransferFilenameUpdate* ptrMessage)
{
   if(NULL == ptrMessage)
   {
      // should never happen else you have programmed something wrong
      // #error_indication
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   if(NULL == app2BtsMessage)
   {
      // should never happen else you have programmed something wrong
      // #error_indication
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   // *app2BtsMessage is allowed to be NULL e.g. in case of indication/signal message

   bts2AppMsgList.reserve(1);

   // TODO: implement
   (void)(bts2IpcMsgList);
   (void)(compareItem);
   (void)(deleteApp2BtsMessage);

   // do not delete App2BtsMessage because this is a status message
}

} //genivi
} //btstackif
