/**
 * @file ConfigurationControl_GEN.cpp
 *
 * @par SW-Component
 * Main
 *
 * @brief Configuration Control.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Implementation of configuration control functionality.
 */

#include "ConfigurationControl_GEN.h"
#include "App2Bts_MessageWrapper.h"
#include "Bts2App_MessageWrapper.h"
#include "Bts2Ipc_MessageWrapper_GEN.h"
#include "Ipc2Bts_MessageWrapper_GEN.h"
#include "EvolutionGeniviStackTypes.h"
#include "EvolutionGeniviUtils.h"
#include "BtsUtils.h"
#include "TraceClasses.h"
#include "FwAssert.h"
#include "FwTrace.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_BTS_CONTROL
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/ConfigurationControl_GEN.cpp.trc.h"
#endif
#endif

namespace btstackif {
namespace genivi {

ConfigurationControl::ConfigurationControl()
: BasicControl(BTS_FB_CONFIG)
{
   /*********** start here with specific class members ***************************/
   /*********** end here with specific class members *****************************/
}

ConfigurationControl::~ConfigurationControl()
{
   /*********** start here with specific class members ***************************/
   /*********** end here with specific class members *****************************/
}

void ConfigurationControl::pushApp2BtsMessage(IN App2Bts_BaseMessage* ptrMessage)
{
   if(NULL == ptrMessage)
   {
      // should never happen else you have programmed something wrong
      // #error_indication
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   if(_functionBlock != ptrMessage->getFunctionBlock())
   {
      // should never happen else you have programmed something wrong
      // #error_indication
      FW_NORMAL_ASSERT_ALWAYS();
      delete ptrMessage;
      return;
   }

   // HINT: if any error happens now the error has to be handled within this function or within any called sub-function

   bool similarOpCodeInWorkingQueue = false;

   // check for skipping similar opcode check
   if(false == skipSimilarOpCodeCheck(ptrMessage))
   {
      // get similar opcodes
      ::std::vector<BTSApp2BtsMessageMasterCompareItem> itemList;
      getSimilarOpCodes(itemList, ptrMessage);

      // check if opcode is in working queue
      similarOpCodeInWorkingQueue = isSimilarOpCodeInWorkingQueue(itemList);
   }

   // check if opcode is in working queue
   if(true == similarOpCodeInWorkingQueue)
   {
      // push to waiting queue
      pushApp2BtsMsgToWaitingQueue(ptrMessage, false); // single worker thread
      ETG_TRACE_USR3((" pushApp2BtsMessage(): App2Bts 0x%04X pushed to waiting queue", ptrMessage->getTraceOpCode()));
   }
   else
   {
      BTSApp2BtsOpcode opcode = ptrMessage->getOpCode();
      ::std::vector<Bts2Ipc_BaseMessage*> sendBts2IpcMsgList;
      sendBts2IpcMsgList.reserve(10);
      ::std::vector<Bts2App_BaseMessage*> sendBts2AppMsgList;
      sendBts2AppMsgList.reserve(10);
      bool deleteApp2BtsMsg = false;
      BTSApp2BtsMessageCompareItem compareItem;
      ptrMessage->getCompareItem(compareItem);

      ETG_TRACE_USR3((" pushApp2BtsMessage(): App2Bts 0x%04X to be processed", ptrMessage->getTraceOpCode()));

      // handle message depending on opcode
      // - set marker to delete message in sub-handler function if necessary
      // - push message to working queue in sub-handler function if necessary
      // - create direct answer message (Bts2App) in sub-handler function if necessary
      // - create Bts2Ipc message in sub-handler function if necessary
      // - handle any error in sub-handler function because there is the best place to handle
      switch(opcode)
      {
         case App2BtsOC_SetDummy:
            handleApp2BtsSetDummy(sendBts2IpcMsgList, sendBts2AppMsgList, deleteApp2BtsMsg, static_cast<App2Bts_SetDummy*>(ptrMessage)); /*lint !e1774: Could use dynamic_cast to downcast polymorphic type*/ // no RTTI, use static_cast instead of dynamic_cast, we know the type of the message
            break;
         // all other
         default:
            FW_NORMAL_ASSERT_ALWAYS();
            deleteApp2BtsMsg = true;
            break;
      }

      if(0 < sendBts2IpcMsgList.size())
      {
         _bts2IpcMessageWasSent = true; // this works only if single thread handling TODO: recheck again
      }

      sendBts2IpcMessageList(sendBts2IpcMsgList, compareItem);

      sendBts2AppMessageList(sendBts2AppMsgList);

      if(true == deleteApp2BtsMsg)
      {
         delete ptrMessage;
      }
   }
}

void ConfigurationControl::pushIpc2BtsMessage(IN Ipc2Bts_BaseMessage* ptrMessage)
{
   if(NULL != ptrMessage)
   {
      delete ptrMessage;
   }
}

void ConfigurationControl::setStackConfiguration(IN const BTSFunctionBlock component, IN const BTSInterfaceType stackInterface, IN const BTSFunctionBlock subComponent, IN const BTSUserMode userMode,
         OUT ::std::vector<BTSDbusInterfaceItem>& dbusInterfaces, IN const BTSLocalConfigurationContainer& configuration)
{
   (void)(component);
   (void)(stackInterface);
   (void)(subComponent);
   (void)(userMode);
   (void)(dbusInterfaces);
   (void)(configuration);
}

void ConfigurationControl::triggerInitializedCallback(void)
{
}

void ConfigurationControl::createDbusServiceAvailabilityMessage(IN const BTSCommonEnumClass interface, IN const BTSDbusServiceAvailability availabilityEvent)
{
   (void)(interface);
   (void)(availabilityEvent);
}

void ConfigurationControl::createDbusServiceAvailabilityMessage(IN const BTSCommonEnumClass interface, IN const BTSDbusServiceAvailability availabilityEvent, IN const BTSBusName& busName, IN const BTSObjectPath& objPath, IN const BTSCommonEnumClass busType)
{
   (void)(interface);
   (void)(availabilityEvent);
   (void)(busName);
   (void)(objPath);
   (void)(busType);
}

void ConfigurationControl::setSubControlTestCommand(IN const char* testCommand, IN const unsigned int testData)
{
   if(NULL == testCommand)
   {
      return;
   }

   (void)(testData);
}

void ConfigurationControl::setSubControlTestCommand(IN const char* testCommand, IN const unsigned char* testData)
{
   if(NULL == testCommand)
   {
      return;
   }

   if(NULL == testData)
   {
      return;
   }
}

void ConfigurationControl::sendDirectAnswerForApp2BtsMessages(IN const ::std::vector<App2Bts_BaseMessage*>& msgList, IN const BTSCommonEnumClass resultCode, IN const BTSCommonEnumClass statusCode)
{
   (void)(resultCode);
   (void)(statusCode);

   // TODO: recheck what to do with doubled requests - answer or ignore? what to do with status?

   if(0 == msgList.size())
   {
      // should never happen else you have programmed something wrong
      // #error_indication
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   App2Bts_BaseMessage* ptrMessage;

   for(size_t i = 0; i < msgList.size(); i++)
   {
      ptrMessage = msgList[i];

      if(NULL == ptrMessage)
      {
         // should never happen else you have programmed something wrong
         // #error_indication
         FW_NORMAL_ASSERT_ALWAYS();
         continue;
      }

      if(_functionBlock != ptrMessage->getFunctionBlock())
      {
         // should never happen else you have programmed something wrong
         // #error_indication
         FW_NORMAL_ASSERT_ALWAYS();
         continue;
      }

      // if(true == _handleDoubledRequests) TODO: check after merging
      {
         BTSApp2BtsOpcode opcode = ptrMessage->getOpCode();
         ::std::vector<Bts2App_BaseMessage*> sendBts2AppMsgList;
         sendBts2AppMsgList.reserve(10);

         ETG_TRACE_USR3((" sendDirectAnswerForApp2BtsMessages(): App2Bts 0x%04X to be handled", ptrMessage->getTraceOpCode()));

         // send direct answer depending on opcode
         // - create direct answer message (Bts2App) in sub-handler function
         // - handle any error in sub-handler function because there is the best place to handle
         // - do not delete message; this is done outside of this function
         switch(opcode)
         {
            case App2BtsOC_SetDummy:
               handleDoubledApp2BtsSetDummy(sendBts2AppMsgList, static_cast<App2Bts_SetDummy*>(ptrMessage), resultCode, statusCode); /*lint !e1774: Could use dynamic_cast to downcast polymorphic type*/ // no RTTI, use static_cast instead of dynamic_cast, we know the type of the message
               break;
            // all other
            default:
               FW_NORMAL_ASSERT_ALWAYS();
               break;
         }

         sendBts2AppMessageList(sendBts2AppMsgList);
      }
   }
}

bool ConfigurationControl::doApp2BtsMsgPrecheck(OUT bool& rejectRequest, OUT BTSCommonEnumClass& resultCode, OUT BTSCommonEnumClass& statusCode, OUT bool& skipOpCodeCheck, IN App2Bts_BaseMessage* ptrMessage)
{
   if(NULL == ptrMessage)
   {
      // should never happen else you have programmed something wrong
      // #error_indication
      FW_NORMAL_ASSERT_ALWAYS();
      return false;
   }

   (void)(rejectRequest);
   (void)(resultCode);
   (void)(statusCode);
   (void)(skipOpCodeCheck);

   return true;
}

void ConfigurationControl::getSimilarOpCodes(OUT ::std::vector<BTSApp2BtsMessageMasterCompareItem>& itemList, IN const App2Bts_BaseMessage* ptrMessage)
{
   if(NULL == ptrMessage)
   {
      // should never happen else you have programmed something wrong
      // #error_indication
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   BTSApp2BtsOpcode opCode = ptrMessage->getOpCode();
   BTSApp2BtsMessageMasterCompareItem item;

   if((App2BtsOC_ConfigurationBlockStart < opCode) && (opCode < App2BtsOC_ConfigurationBlockEnd))
   {
      size_t reserveSize = 0;

      // TODO: complete switch
      switch(opCode)
      {
         case App2BtsOC_SetDummy:
            reserveSize += 1;
            itemList.reserve(reserveSize);
            item.opCode = opCode;
            itemList.push_back(item);
            break;
         // add all other here if necessary
         default:
            FW_NORMAL_ASSERT_ALWAYS();
            break;
      }
   }
   else
   {
      // opcode in wrong range
      FW_NORMAL_ASSERT_ALWAYS();

      // add at least input opcode
      item.opCode = opCode;
      itemList.push_back(item);
   }
}

void ConfigurationControl::getMatchingOpCodes(OUT ::std::vector<BTSApp2BtsMessageMasterCompareItem>& itemList, OUT ::std::vector<BTSApp2BtsMessageMasterCompareItem>& highPrioItemList, IN const App2Bts_BaseMessage* ptrMessage)
{
   if(NULL == ptrMessage)
   {
      // should never happen else you have programmed something wrong
      // #error_indication
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   (void)(highPrioItemList);
   BTSApp2BtsOpcode opCode = ptrMessage->getOpCode();
   BTSApp2BtsMessageMasterCompareItem item;
   ptrMessage->getCompareItem(item);

   if((App2BtsOC_ConfigurationBlockStart < opCode) && (opCode < App2BtsOC_ConfigurationBlockEnd))
   {
      size_t reserveSize;

      // TODO: complete switch
      switch(opCode)
      {
         case App2BtsOC_SetDummy:
            reserveSize = 1;
            itemList.reserve(reserveSize);
            item.opCode = opCode;
            itemList.push_back(item);
            break;
         // add all other here if necessary
         default:
            FW_NORMAL_ASSERT_ALWAYS();
            break;
      }
   }
   else
   {
      // opcode in wrong range
      FW_NORMAL_ASSERT_ALWAYS();

      // add at least input opcode
      item.opCode = opCode;
      itemList.push_back(item);
   }
}

bool ConfigurationControl::skipSimilarOpCodeCheck(IN const App2Bts_BaseMessage* ptrMessage)
{
   (void)(ptrMessage);
   // implement if necessary

   return false;
}

void ConfigurationControl::checkWaitingQueueExtended(IN const BTSApp2BtsMessageCompareItem& compareItem)
{
   (void)(compareItem);
   // implement if necessary
}

void ConfigurationControl::handleDoubledApp2BtsMessages(IN const ::std::vector<App2Bts_BaseMessage*>& msgList)
{
   // TODO: recheck what to do with doubled requests - answer or ignore? what to do with status?

   if(0 == msgList.size())
   {
      // should never happen else you have programmed something wrong
      // #error_indication
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   App2Bts_BaseMessage* ptrMessage;

   for(size_t i = 0; i < msgList.size(); i++)
   {
      ptrMessage = msgList[i];

      if(NULL == ptrMessage)
      {
         // should never happen else you have programmed something wrong
         // #error_indication
         FW_NORMAL_ASSERT_ALWAYS();
         continue;
      }

      if(_functionBlock != ptrMessage->getFunctionBlock())
      {
         // should never happen else you have programmed something wrong
         // #error_indication
         FW_NORMAL_ASSERT_ALWAYS();
         continue;
      }

      if(true == _handleDoubledRequests)
      {
         BTSApp2BtsOpcode opcode = ptrMessage->getOpCode();
         ::std::vector<Bts2App_BaseMessage*> sendBts2AppMsgList;
         sendBts2AppMsgList.reserve(10);

         ETG_TRACE_USR3((" handleDoubledApp2BtsMessages(): App2Bts 0x%04X to be handled", ptrMessage->getTraceOpCode()));

         // handle doubled messages depending on opcode
         // - create direct answer message (Bts2App) in sub-handler function
         // - handle any error in sub-handler function because there is the best place to handle
         // - do not delete message; this is done at the end of this function
         switch(opcode)
         {
            case App2BtsOC_SetDummy:
               handleDoubledApp2BtsSetDummy(sendBts2AppMsgList, static_cast<App2Bts_SetDummy*>(ptrMessage), (BTSCommonEnumClass)BTS_REQ_SUCCESS, BTS_COMMON_ENUM_CLASS_DEFAULT_VALUE); /*lint !e1774: Could use dynamic_cast to downcast polymorphic type*/ // no RTTI, use static_cast instead of dynamic_cast, we know the type of the message
               break;
            // all other
            default:
               FW_NORMAL_ASSERT_ALWAYS();
               break;
         }

         sendBts2AppMessageList(sendBts2AppMsgList);
      }
   }
}

/*********** start here with specific class members + methods *****************/
/*********** end here with specific class members + methods *******************/

void ConfigurationControl::handleDoubledApp2BtsSetDummy(OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, IN App2Bts_SetDummy* ptrMessage, IN const BTSCommonEnumClass resultCode, IN const BTSCommonEnumClass statusCode)
{
   (void)(bts2AppMsgList);
   (void)(resultCode);
   (void)(statusCode);

   if(NULL == ptrMessage)
   {
      // should never happen else you have programmed something wrong
      // #error_indication
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   // send answer directly because doubled request - set all to success
   // TODO
}

void ConfigurationControl::handleApp2BtsSetDummy(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT bool& deleteApp2BtsMessage, IN App2Bts_SetDummy* ptrMessage)
{
   (void)(bts2AppMsgList);
   (void)(deleteApp2BtsMessage);

   if(NULL == ptrMessage)
   {
      // should never happen else you have programmed something wrong
      // #error_indication
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   // TODO: implement
   _app2BtsWorkingQueue.push(ptrMessage, false);

   bts2IpcMsgList.reserve(1);

   // TODO: implement
}

} //genivi
} //btstackif
