/**
 * @file MessagingControl_GEN.h
 *
 * @par SW-Component
 * Main
 *
 * @brief Messaging Control.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Class for messaging control functionality.
 */

#ifndef _MESSAGING_CONTROL_GENIVI_H_
#define _MESSAGING_CONTROL_GENIVI_H_

#include "BtStackInternalTypes.h"
#include "BasicControl.h"
#include "DbusServiceAvailability.h"

/*********** start here with specific includes ********************************/
/*********** end here with specific includes **********************************/

namespace btstackif {

// class forward declarations
class Bts2Ipc_BaseMessage;
// App2Bts messages
class App2Bts_SetMessagingDummy;

namespace genivi {

// class forward declarations
// Ipc2Bts messages
class Ipc2Bts_ServiceAvailabilityMessaging;
class Ipc2Bts_MsgSessionSourceUpdate;
class Ipc2Bts_MsgSessionDestinationUpdate;
class Ipc2Bts_MsgSessionChannelUpdate;
class Ipc2Bts_MsgSessionTargetUpdate;
class Ipc2Bts_MsgTransferCancelResult;
class Ipc2Bts_MsgTransferStatusUpdate;
class Ipc2Bts_MsgTransferSessionUpdate;
class Ipc2Bts_MsgTransferNameUpdate;
class Ipc2Bts_MsgTransferTypeUpdate;
class Ipc2Bts_MsgTransferTimeUpdate;
class Ipc2Bts_MsgTransferSizeUpdate;
class Ipc2Bts_MsgTransferTransferredUpdate;
class Ipc2Bts_MsgTransferFilenameUpdate;

/**
 *
 */
class MessagingControl : public BasicControl
{
public:
   /**************************************************************************
    Constructor / destructor
    **************************************************************************/
   /**
    * Constructor.
    */
   MessagingControl();

   /**
    * Destructor.
    */
   virtual ~MessagingControl();

   virtual void pushApp2BtsMessage(IN App2Bts_BaseMessage* ptrMessage);

   virtual void pushIpc2BtsMessage(IN Ipc2Bts_BaseMessage* ptrMessage);

   virtual void setStackConfiguration(IN const BTSFunctionBlock component, IN const BTSInterfaceType stackInterface, IN const BTSFunctionBlock subComponent, IN const BTSUserMode userMode,
         OUT ::std::vector<BTSDbusInterfaceItem>& dbusInterfaces, IN const BTSLocalConfigurationContainer& configuration);

   virtual void triggerInitializedCallback(void);

   virtual void createDbusServiceAvailabilityMessage(IN const BTSCommonEnumClass interface, IN const BTSDbusServiceAvailability availabilityEvent);

   virtual void createDbusServiceAvailabilityMessage(IN const BTSCommonEnumClass interface, IN const BTSDbusServiceAvailability availabilityEvent, IN const BTSBusName& busName, IN const BTSObjectPath& objPath, IN const BTSCommonEnumClass busType);

   virtual void setSubControlTestCommand(IN const char* testCommand, IN const unsigned int testData);

   virtual void setSubControlTestCommand(IN const char* testCommand, IN const unsigned char* testData);

   virtual void sendDirectAnswerForApp2BtsMessages(IN const ::std::vector<App2Bts_BaseMessage*>& msgList, IN const BTSCommonEnumClass resultCode, IN const BTSCommonEnumClass statusCode); // TODO: later do a merge of handleDoubledApp2BtsMessages and sendDirectAnswerForApp2BtsMessages; rename also all handleDoubledApp2BtsXxx

protected:
   virtual bool doApp2BtsMsgPrecheck(OUT bool& rejectRequest, OUT BTSCommonEnumClass& resultCode, OUT BTSCommonEnumClass& statusCode, OUT bool& skipOpCodeCheck, IN App2Bts_BaseMessage* ptrMessage);

   virtual void getSimilarOpCodes(OUT ::std::vector<BTSApp2BtsMessageMasterCompareItem>& itemList, IN const App2Bts_BaseMessage* ptrMessage);

   virtual void getMatchingOpCodes(OUT ::std::vector<BTSApp2BtsMessageMasterCompareItem>& itemList, OUT ::std::vector<BTSApp2BtsMessageMasterCompareItem>& highPrioItemList, IN const App2Bts_BaseMessage* ptrMessage);

   virtual bool skipSimilarOpCodeCheck(IN const App2Bts_BaseMessage* ptrMessage);

   virtual void checkWaitingQueueExtended(IN const BTSApp2BtsMessageCompareItem& compareItem);

   virtual void handleDoubledApp2BtsMessages(IN const ::std::vector<App2Bts_BaseMessage*>& msgList);

private:
   /*********** start here with specific class members + methods *****************/
   /*********** end here with specific class members + methods *******************/

   /**
    * Handle doubled App2Bts message.
    * Handle any error within this function.
    *
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[in] ptrMessage: received App2Bts message
    * @param[in] resultCode: result code
    * @param[in] statusCode: status code
    */
   void handleDoubledApp2BtsSetMessagingDummy(OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, IN App2Bts_SetMessagingDummy* ptrMessage, IN const BTSCommonEnumClass resultCode, IN const BTSCommonEnumClass statusCode);

   /**
    * Handle received App2Bts message.
    * Handle any error within this function.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] deleteApp2BtsMessage: marker to delete related App2Bts message
    * @param[in] ptrMessage: received App2Bts message
    */
   void handleApp2BtsSetMessagingDummy(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT bool& deleteApp2BtsMessage, IN App2Bts_SetMessagingDummy* ptrMessage);

   /**
    * Handle received Ipc2Bts message.
    * Handle any error within this function.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] compareItem: compare item for special handling
    * @param[out] deleteApp2BtsMessage: marker to delete related App2Bts message; in case of a DBUS message sequence has to be passed we have to delete the related App2Bts message after sequence is finished
    * @param[in,out] app2BtsMessage: related App2Bts message from working queue
    * @param[in] ptrMessage: received Ipc2Bts message
    */
   void handleIpc2BtsServiceAvailabilityMessaging(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSApp2BtsMessageCompareItem& compareItem, OUT bool& deleteApp2BtsMessage, INOUT App2Bts_BaseMessage** app2BtsMessage, IN Ipc2Bts_ServiceAvailabilityMessaging* ptrMessage);
   void handleIpc2BtsMsgSessionSourceUpdate(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSApp2BtsMessageCompareItem& compareItem, OUT bool& deleteApp2BtsMessage, INOUT App2Bts_BaseMessage** app2BtsMessage, IN Ipc2Bts_MsgSessionSourceUpdate* ptrMessage);
   void handleIpc2BtsMsgSessionDestinationUpdate(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSApp2BtsMessageCompareItem& compareItem, OUT bool& deleteApp2BtsMessage, INOUT App2Bts_BaseMessage** app2BtsMessage, IN Ipc2Bts_MsgSessionDestinationUpdate* ptrMessage);
   void handleIpc2BtsMsgSessionChannelUpdate(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSApp2BtsMessageCompareItem& compareItem, OUT bool& deleteApp2BtsMessage, INOUT App2Bts_BaseMessage** app2BtsMessage, IN Ipc2Bts_MsgSessionChannelUpdate* ptrMessage);
   void handleIpc2BtsMsgSessionTargetUpdate(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSApp2BtsMessageCompareItem& compareItem, OUT bool& deleteApp2BtsMessage, INOUT App2Bts_BaseMessage** app2BtsMessage, IN Ipc2Bts_MsgSessionTargetUpdate* ptrMessage);
   void handleIpc2BtsMsgTransferCancelResult(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSApp2BtsMessageCompareItem& compareItem, OUT bool& deleteApp2BtsMessage, INOUT App2Bts_BaseMessage** app2BtsMessage, IN Ipc2Bts_MsgTransferCancelResult* ptrMessage);
   void handleIpc2BtsMsgTransferStatusUpdate(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSApp2BtsMessageCompareItem& compareItem, OUT bool& deleteApp2BtsMessage, INOUT App2Bts_BaseMessage** app2BtsMessage, IN Ipc2Bts_MsgTransferStatusUpdate* ptrMessage);
   void handleIpc2BtsMsgTransferSessionUpdate(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSApp2BtsMessageCompareItem& compareItem, OUT bool& deleteApp2BtsMessage, INOUT App2Bts_BaseMessage** app2BtsMessage, IN Ipc2Bts_MsgTransferSessionUpdate* ptrMessage);
   void handleIpc2BtsMsgTransferNameUpdate(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSApp2BtsMessageCompareItem& compareItem, OUT bool& deleteApp2BtsMessage, INOUT App2Bts_BaseMessage** app2BtsMessage, IN Ipc2Bts_MsgTransferNameUpdate* ptrMessage);
   void handleIpc2BtsMsgTransferTypeUpdate(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSApp2BtsMessageCompareItem& compareItem, OUT bool& deleteApp2BtsMessage, INOUT App2Bts_BaseMessage** app2BtsMessage, IN Ipc2Bts_MsgTransferTypeUpdate* ptrMessage);
   void handleIpc2BtsMsgTransferTimeUpdate(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSApp2BtsMessageCompareItem& compareItem, OUT bool& deleteApp2BtsMessage, INOUT App2Bts_BaseMessage** app2BtsMessage, IN Ipc2Bts_MsgTransferTimeUpdate* ptrMessage);
   void handleIpc2BtsMsgTransferSizeUpdate(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSApp2BtsMessageCompareItem& compareItem, OUT bool& deleteApp2BtsMessage, INOUT App2Bts_BaseMessage** app2BtsMessage, IN Ipc2Bts_MsgTransferSizeUpdate* ptrMessage);
   void handleIpc2BtsMsgTransferTransferredUpdate(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSApp2BtsMessageCompareItem& compareItem, OUT bool& deleteApp2BtsMessage, INOUT App2Bts_BaseMessage** app2BtsMessage, IN Ipc2Bts_MsgTransferTransferredUpdate* ptrMessage);
   void handleIpc2BtsMsgTransferFilenameUpdate(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSApp2BtsMessageCompareItem& compareItem, OUT bool& deleteApp2BtsMessage, INOUT App2Bts_BaseMessage** app2BtsMessage, IN Ipc2Bts_MsgTransferFilenameUpdate* ptrMessage);
};

} //genivi
} //btstackif

#endif //_MESSAGING_CONTROL_GENIVI_H_
