/**
 * @file ConnectionControl_GEN.h
 *
 * @par SW-Component
 * Main
 *
 * @brief Connection Control.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Class for connection control functionality.
 */

#ifndef _CONNECTION_CONTROL_GENIVI_H_
#define _CONNECTION_CONTROL_GENIVI_H_

#include "BtStackInternalTypes.h"
#include "BasicControl.h"

/*********** start here with specific includes ********************************/
#include "ConnectionControlTestCommand.h"
/*********** end here with specific includes **********************************/

namespace btstackif {

// class forward declarations
// generic SMs
class ITestMode;
class IRssi;
class IRestrictedPairingConnecting;
class IDeviceManager;
class ILinkKey;
class IInquiry;
class IPairing;
class IServiceSearch;
class IConfiguration;
class IBasicConfiguration;
class ILocalAdapterModes;
class ISppPool;
class IProtocolManager;
class IConnectionPriority;
class ILocalBluetoothName;
class IDeviceIdServiceRecord;
class IHfpDeviceCapabilities;
class IBluetoothStackErrorHandling;
class IStartup;
class ISwitchBluetooth;

class Bts2Ipc_BaseMessage;

// App2Bts messages
class App2Bts_StartInternalCommunication;
class App2Bts_TriggerFbConnectionInitialized;
class App2Bts_SwitchBluetoothOnOff;
class App2Bts_GetBluetoothOnOffStatus;
class App2Bts_SwitchBluetoothHardwareOnOff;
class App2Bts_SetLocalBtName;
class App2Bts_GetLocalBtName;
class App2Bts_SetDeviceIdServiceRecord;
class App2Bts_SetDeviceForRestrictedPairingConnecting;
class App2Bts_SetLocalAdapterModes;
class App2Bts_GetLocalAdapterModes;
class App2Bts_StartDiscovery;
class App2Bts_StopDiscovery;
class App2Bts_GetDiscoveringStatus;
class App2Bts_StartPairing;
class App2Bts_CancelPairing;
class App2Bts_SetPinCode;
class App2Bts_ConfirmSecureSimplePairing;
class App2Bts_SetLinkKey;
class App2Bts_ConnectProtocol;
class App2Bts_AcceptRemoteProtocolConnect;
class App2Bts_RejectRemoteProtocolConnect;
class App2Bts_DisconnectProtocol;
class App2Bts_DisconnectDevice;
class App2Bts_GetProtocolConnectionStatus;
class App2Bts_GetDeviceConnectionStatus;
class App2Bts_StartRemoteServiceSearch;
class App2Bts_CancelRemoteServiceSearch;
class App2Bts_QueryRemoteName;
class App2Bts_RequestLinkQuality;
class App2Bts_StartTestMode;
class App2Bts_StopTestMode;
class App2Bts_RequestTestModeLinkQuality;

namespace genivi {

// class forward declarations
class IConnectionSmExtension;

// Genivi SMs
class RssiGenivi;
class DeviceManagerGenivi;
class LinkKeyGenivi;
class InquiryGenivi;
class PairingGenivi;
class ServiceSearchGenivi;
class LocalAdapterModesGenivi;
class IHfpAvpConnectDisconnectGenivi;
class IObexConnectDisconnectGenivi;
class IPanConnectDisconnectGenivi;
class ISppPoolGenivi;
class ILocalBluetoothNameGenivi;
class IBasicConfigurationGenivi;
class IHfpDeviceCapabilitiesGenivi;
class IStartupGenivi;
class ISwitchBluetoothGenivi;

// Ipc2Bts messages
class Ipc2Bts_ServiceAvailabilityConnection;
class Ipc2Bts_GetAllPropertiesConnection;
class Ipc2Bts_AddDeviceObjectPathMapping;
class Ipc2Bts_DelDeviceObjectPathMapping;
class Ipc2Bts_AddProtocolObjectPathMapping;
class Ipc2Bts_DelProtocolObjectPathMapping;
class Ipc2Bts_AddSerialObjectPathMapping;
class Ipc2Bts_DelSerialObjectPathMapping;
class Ipc2Bts_Release;
class Ipc2Bts_RequestPinCode;
class Ipc2Bts_DisplayPinCode;
class Ipc2Bts_RequestPasskey;
class Ipc2Bts_RequestConfirmation;
class Ipc2Bts_Cancel;
class Ipc2Bts_CancelRequest;
class Ipc2Bts_AuthorizeService;
class Ipc2Bts_RequestAuthorization;
class Ipc2Bts_RequestLinkkey;
class Ipc2Bts_DisplayPasskey;
class Ipc2Bts_StartDiscoveryResult;
class Ipc2Bts_StopDiscoveryResult;
class Ipc2Bts_RemoveDeviceResult;
class Ipc2Bts_CreateDeviceResult;
class Ipc2Bts_SendHCICmdResult;
class Ipc2Bts_AdapterInitialization;
class Ipc2Bts_DeviceCreated;
class Ipc2Bts_DeviceRemoved;
class Ipc2Bts_BtStackLog;
class Ipc2Bts_HCIEvent;
class Ipc2Bts_DeviceFound;
class Ipc2Bts_AdapterAddressUpdate;
class Ipc2Bts_AdapterClassUpdate;
class Ipc2Bts_AdapterAliasUpdate;
class Ipc2Bts_AdapterPoweredUpdate;
class Ipc2Bts_AdapterDiscoverableUpdate;
class Ipc2Bts_AdapterDiscoverabletimeoutUpdate;
class Ipc2Bts_AdapterPairableUpdate;
class Ipc2Bts_AdapterPairabletimeoutUpdate;
class Ipc2Bts_AdapterDiscoveringUpdate;
class Ipc2Bts_AdapterUuidsUpdate;
class Ipc2Bts_AdapterWbsModeUpdate;
class Ipc2Bts_AdapterConnectableUpdate;
class Ipc2Bts_AdapterHcimodeUpdate;
class Ipc2Bts_AdapterVersionUpdate;
class Ipc2Bts_AdapterTraceLevelUpdate;
class Ipc2Bts_AdapterTraceCategoryUpdate;
class Ipc2Bts_AdapterAvailableSppServersUpdate;
class Ipc2Bts_RegisterAgentResult;
class Ipc2Bts_UnregisterAgentResult;
class Ipc2Bts_RequestDefaultAgentResult;
class Ipc2Bts_DeviceConnectResult;
class Ipc2Bts_DeviceDisconnectResult;
class Ipc2Bts_PairResult;
class Ipc2Bts_CancelPairingResult;
class Ipc2Bts_DiscoverServicesResult;
class Ipc2Bts_CancelDiscoveryResult;
class Ipc2Bts_ReportLinkkey;
class Ipc2Bts_IncomingPairingComp;
class Ipc2Bts_SDPProcessReport;
class Ipc2Bts_DeviceAddressUpdate;
class Ipc2Bts_DeviceClassUpdate;
class Ipc2Bts_DeviceIconUpdate;
class Ipc2Bts_DeviceUuidsUpdate;
class Ipc2Bts_DevicePairedUpdate;
class Ipc2Bts_DeviceConnectedUpdate;
class Ipc2Bts_DeviceTrustedUpdate;
class Ipc2Bts_DeviceBlockedUpdate;
class Ipc2Bts_DeviceAliasUpdate;
class Ipc2Bts_DeviceAdapterUpdate;
class Ipc2Bts_DeviceModaliasUpdate;
class Ipc2Bts_DeviceRssiUpdate;
class Ipc2Bts_DeviceVendorIdUpdate;
class Ipc2Bts_DeviceVendorIdSourceUpdate;
class Ipc2Bts_DeviceProductIdUpdate;
class Ipc2Bts_DeviceMapInfoUpdate;
class Ipc2Bts_DeviceConnectedServiceUpdate;
class Ipc2Bts_DeviceVersionUpdate;
class Ipc2Bts_DevicePbapInfoUpdate;
class Ipc2Bts_DeviceModeUpdate;
class Ipc2Bts_DeviceRoleUpdate;
class Ipc2Bts_DeviceEnableAvpPauseUpdate;
class Ipc2Bts_BtApplDeviceConnectReqResult;
class Ipc2Bts_BtApplDeviceDisconnectReqResult;
class Ipc2Bts_BtApplCancelConnectReqResult;
class Ipc2Bts_BtApplGetHwVersionReqResult;
class Ipc2Bts_BtApplSetConfigurationReqResult;
class Ipc2Bts_BtApplReadConfigurationReqResult;
class Ipc2Bts_BtApplDeviceConnectCfm;
class Ipc2Bts_BtApplDeviceConnectInd;
class Ipc2Bts_BtApplDeviceDisconnectCfm;
class Ipc2Bts_BtApplDeviceDisconnectInd;
class Ipc2Bts_BtApplDeviceDisconnectCompInd;
class Ipc2Bts_BtApplDeviceConnectCompInd;
class Ipc2Bts_BtApplCancelConnectCfm;
class Ipc2Bts_BtApplStatusNotificationInd;
class Ipc2Bts_BtApplGetHwVersionCfm;
class Ipc2Bts_BtApplSetConfigurationCfm;
class Ipc2Bts_BtApplReadConfigurationCfm;
class Ipc2Bts_ConnectResult;
class Ipc2Bts_DisconnectResult;
class Ipc2Bts_ServiceRemoteUuidUpdate;
class Ipc2Bts_ServiceStateUpdate;
class Ipc2Bts_ServiceDeviceUpdate;
class Ipc2Bts_ServiceVersionUpdate;
class Ipc2Bts_DebugTraceConfigResult;
class Ipc2Bts_ProtocolTraceConfigResult;
class Ipc2Bts_ReportDebugTrace;
class Ipc2Bts_ReportProtocolTrace;
class Ipc2Bts_FatalError;
class Ipc2Bts_GeneralError;
class Ipc2Bts_ReportGeniviDebugTrace;
class Ipc2Bts_GeniviFatalError;
class Ipc2Bts_BtApplDidSetReqResult;
class Ipc2Bts_BtApplDidSetCfm;
class Ipc2Bts_DunDeviceFilePathUpdate;
class Ipc2Bts_BtApplSppSetUuidReqResult;
class Ipc2Bts_BtApplSppSetUuidCfm;
class Ipc2Bts_BtApplSppVirtualDeviceInd;
class Ipc2Bts_CreateSessionResult;
class Ipc2Bts_RemoveSessionResult;
class Ipc2Bts_CancelSessionResult;
class Ipc2Bts_SessionCreated;
class Ipc2Bts_SessionRemoved;
class Ipc2Bts_AddTechnologyObjectPath;
class Ipc2Bts_DelTechnologyObjectPath;
class Ipc2Bts_AddNetworkObjectPathMapping;
class Ipc2Bts_DelNetworkObjectPathMapping;
class Ipc2Bts_GetConnManManagerPropertiesResult;
class Ipc2Bts_GetConnManTechnologiesResult;
class Ipc2Bts_GetConnManServicesResult;
class Ipc2Bts_ConnManTechnologyAdded;
class Ipc2Bts_ConnManTechnologyRemoved;
class Ipc2Bts_ConnManServicesChanged;
class Ipc2Bts_ConnManServicesRemoved;
class Ipc2Bts_GetConnManServicePropertiesResult;
class Ipc2Bts_ConnManServiceConnectResult;
class Ipc2Bts_ConnManServiceDisconnectResult;
class Ipc2Bts_ConnManServiceStateUpdate;
class Ipc2Bts_ConnManServiceInterfaceUpdate;
class Ipc2Bts_SerialDeviceFilePath;
class Ipc2Bts_CharacterDeviceAddedRemoved;
class Ipc2Bts_AddHfpInterfaceObjectPathMapping;
class Ipc2Bts_DelHfpInterfaceObjectPathMapping;
class Ipc2Bts_GetHandsfreeProperties;
class Ipc2Bts_HandsfreeFeaturesUpdate;
class Ipc2Bts_HandsfreeInbandRingingUpdate;
class Ipc2Bts_GetModems;
class Ipc2Bts_ModemAdded;
class Ipc2Bts_ModemRemoved;
class Ipc2Bts_GetModemProperties;
class Ipc2Bts_ModemInterfacesUpdate;

namespace alpsevolutionext {
// class forward declarations
// Alps Evolution Genivi SMs
class IBasicConfigurationAlpsEvolutionExt;
class ISppConnectDisconnectAlpsEvolutionExt;
class IDeviceIdServiceRecordAlpsEvolutionExt;
class IBluetoothStackErrorHandlingAlpsEvolutionExt;
} //alpsevolutionext

/**
 *
 */
class ConnectionControl : public BasicControl
{
public:
   /**************************************************************************
    Constructor / destructor
    **************************************************************************/
   /**
    * Constructor.
    */
   ConnectionControl();

private:
   /**
    * Copy constructor.
    *
    * DO NOT USE!!!
    *
    * @param[in] ref: reference of object to be copied
    */
   ConnectionControl(const ConnectionControl& ref);

   /**
    * Assignment operator.
    *
    * DO NOT USE!!!
    *
    * @param[in] ref: reference of object to be assigned
    */
   ConnectionControl& operator=(const ConnectionControl& ref);

public:
   /**
    * Destructor.
    */
   virtual ~ConnectionControl();

   virtual void pushApp2BtsMessage(IN App2Bts_BaseMessage* ptrMessage);

   virtual void pushIpc2BtsMessage(IN Ipc2Bts_BaseMessage* ptrMessage);

   virtual void setStackConfiguration(IN const BTSFunctionBlock component, IN const BTSInterfaceType stackInterface, IN const BTSFunctionBlock subComponent, IN const BTSUserMode userMode,
            OUT ::std::vector<BTSDbusInterfaceItem>& dbusInterfaces, IN const BTSLocalConfigurationContainer& configuration);

   virtual void triggerInitializedCallback(void);

   virtual void createDbusServiceAvailabilityMessage(IN const BTSCommonEnumClass interface, IN const BTSDbusServiceAvailability availabilityEvent);

   virtual void createDbusServiceAvailabilityMessage(IN const BTSCommonEnumClass interface, IN const BTSDbusServiceAvailability availabilityEvent, IN const BTSBusName& busName, IN const BTSObjectPath& objPath, IN const BTSCommonEnumClass busType);

   virtual void handleTimeout(IN IExtendedTimeoutHandler* handler, IN const BTSTimerId timerId);

   virtual void setSubControlTestCommand(IN const char* testCommand, IN const unsigned int testData);

   virtual void setSubControlTestCommand(IN const char* testCommand, IN const unsigned char* testData);

   virtual void sendDirectAnswerForApp2BtsMessages(IN const ::std::vector<App2Bts_BaseMessage*>& msgList, IN const BTSCommonEnumClass resultCode, IN const BTSCommonEnumClass statusCode); // TODO: later do a merge of handleDoubledApp2BtsMessages and sendDirectAnswerForApp2BtsMessages; rename also all handleDoubledApp2BtsXxx

   virtual void sendDirectAnswerForApp2BtsMessagesWrapper(OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, IN const ::std::vector<App2Bts_BaseMessage*>& msgList, IN const BTSCommonEnumClass resultCode, IN const BTSCommonEnumClass statusCode);

protected:
   virtual void reorderApp2BtsWaitingQueue(INOUT MessageQueue< App2Bts_BaseMessage >& waitingQueue);

   virtual bool doApp2BtsMsgPrecheck(OUT bool& rejectRequest, OUT BTSCommonEnumClass& resultCode, OUT BTSCommonEnumClass& statusCode, OUT bool& skipOpCodeCheck, IN App2Bts_BaseMessage* ptrMessage);

   virtual void getSimilarOpCodes(OUT ::std::vector<BTSApp2BtsMessageMasterCompareItem>& itemList, IN const App2Bts_BaseMessage* ptrMessage);

   virtual void getMatchingOpCodes(OUT ::std::vector<BTSApp2BtsMessageMasterCompareItem>& itemList, OUT ::std::vector<BTSApp2BtsMessageMasterCompareItem>& highPrioItemList, IN const App2Bts_BaseMessage* ptrMessage);

   virtual bool skipSimilarOpCodeCheck(IN const App2Bts_BaseMessage* ptrMessage);

   virtual void checkWaitingQueueExtended(IN const BTSApp2BtsMessageCompareItem& compareItem);

   virtual void handleDoubledApp2BtsMessages(IN const ::std::vector<App2Bts_BaseMessage*>& msgList);

private:
   /*********** start here with specific class members + methods *****************/

   // SMs
   IConnectionSmExtension* _smExtension; /**< connection SM extension TODO: to be improved; check WBL part to see a better solution */

   // generic SMs
   ITestMode* _testMode; /**< generic SM for test mode */
   IRssi* _rssi; /**< generic SM for RSSI */
   IRestrictedPairingConnecting* _restrictedPairingConnecting; /**< generic SM for restricted pairing and connecting */
   IDeviceManager* _deviceManager; /**< generic SM for device manager */
   ILinkKey* _linkKey; /**< generic SM for link key */
   IInquiry* _inquiry; /**< generic SM for inquiry */
   IPairing* _pairing; /**< generic SM for pairing */
   IServiceSearch* _serviceSearch; /**< generic SM for service search */
   IConfiguration* _configuration; /**< generic SM for configuration */
   IBasicConfiguration* _basicConfiguration; /**< generic SM for basic configuration */
   ILocalAdapterModes* _localAdapterModes; /**< generic SM for local adapter modes */
   ISppPool* _sppPool; /**< generic SM for SPP pool */
   IProtocolManager* _protocolManagerNew; /**< generic SM for protocol manager */
   IConnectionPriority* _connectionPriority; /**< generic SM for Connection priority */
   ILocalBluetoothName* _localBluetoothName; /**< generic SM for local Bluetooth name */
   IDeviceIdServiceRecord* _deviceIdServiceRecord; /**< generic SM for device id service record */
   IHfpDeviceCapabilities* _hfpDeviceCapabilities; /**< generic SM for HFP device capabilities */
   IBluetoothStackErrorHandling* _bluetoothStackErrorHandling; /**< generic SM for Bluetooth stack error handling */
   IStartup* _startupNew; /**< generic SM for startup */
   ISwitchBluetooth* _switchBluetooth; /**< generic SM for switch Bluetooth on/off */

   // Genivi SMs
   RssiGenivi* _geniviRssi; /**< Genivi SM for RSSI */
   DeviceManagerGenivi* _geniviDeviceManager; /**< Genivi SM for device manager */
   LinkKeyGenivi* _geniviLinkKey; /**< Genivi SM for link key */
   InquiryGenivi* _geniviInquiry; /**< Genivi SM for inquiry */
   PairingGenivi* _geniviPairing; /**< Genivi SM for pairing */
   ServiceSearchGenivi* _geniviServiceSearch; /**< Genivi SM for service search */
   LocalAdapterModesGenivi* _geniviLocalAdapterModes; /**< Genivi SM for local adapter modes */
   IHfpAvpConnectDisconnectGenivi* _geniviHfpAvpConnectDisconnect; /**< Genivi SM for HFP/AVP connect/disconnect */
   IObexConnectDisconnectGenivi* _geniviObexConnectDisconnect; /**< Genivi SM for OBEX connect/disconnect */
   IPanConnectDisconnectGenivi* _geniviPanConnectDisconnect; /**< Genivi SM for PAN connect/disconnect */
   ISppPoolGenivi* _geniviSppPool; /**< Genivi SM for SPP pool */
   ILocalBluetoothNameGenivi* _geniviLocalBluetoothName; /**< Genivi SM for local Bluetooth name */
   IBasicConfigurationGenivi* _geniviBasicConfiguration; /**< Genivi SM for basic configuration */
   IHfpDeviceCapabilitiesGenivi* _geniviHfpDeviceCapabilities; /**< Genivi SM for HFP device capabilities */
   IStartupGenivi* _geniviStartup; /**< Genivi SM for startup */
   ISwitchBluetoothGenivi* _geniviSwitchBluetooth; /**< Genivi SM for switch Bluetooth on/off */

   // Alps Evolution Genivi SMs TODO: check later if these SMs can be moved to Alps Evolution Genivi specific part
   alpsevolutionext::IBasicConfigurationAlpsEvolutionExt* _alpsEvolutionExtBasicConfiguration; /**< Alps Evolution Genivi SM for basic configuration */
   alpsevolutionext::ISppConnectDisconnectAlpsEvolutionExt* _alpsEvolutionExtSppConnectDisconnect; /**< Alps Evolution Genivi SM for SPP connect/disconnect */
   alpsevolutionext::IDeviceIdServiceRecordAlpsEvolutionExt* _alpsEvolutionExtDeviceIdServiceRecord; /**< Alps Evolution Genivi SM for device id service record */
   alpsevolutionext::IBluetoothStackErrorHandlingAlpsEvolutionExt* _alpsEvolutionExtBluetoothStackErrorHandling; /**< Alps Evolution Genivi SM for Bluetooth stack error handling */

   // test command
   ConnectionControlTestCommand _testCommand; /**< test command handler */

   /**
    * Check result of request processing.
    *
    * @param[in] messageHandlingType: message handling type (what to do with the request)
    * @param[in] ptrMessage: app message
    *
    * @return = true: message can be deleted,
    * @return = false: do not delete message
    */
   bool checkRequestResult(IN const BTSApp2BtsMessageHandlingType messageHandlingType, IN App2Bts_BaseMessage* ptrMessage);

   /**
    * Perform next configuration step.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    */
   void doNextConfigurationStep(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem);

   /**
    * Initialize generic SMs. These SMs are defined/implemented in generic SW part of BtStackIf.
    */
   void initializeGenericSMs(void);

   /**
    * Destroy generic SMs. These SMs are defined/implemented in generic SW part of BtStackIf.
    */
   void destroyGenericSMs(void);

   /**
    * Initialize GENIVI specific SMs.
    */
   void initializeGeniviSMs(void);

   /**
    * Destroy GENIVI specific SMs.
    */
   void destroyGeniviSMs(void);

   /**
    * Initialize ALPS EVOLUTION GENIVI specific SMs.
    */
   void initializeAlpsEvolutionGeniviSMs(void);

   /**
    * Destroy ALPS EVOLUTION GENIVI specific SMs.
    */
   void destroyAlpsEvolutionGeniviSMs(void);

   /**
    * Assign SMs.
    */
   void assignSMs(void);

   /**
    * Check for available Bluetooth Box.
    */
   void checkForAvailableBtBox(void);

   /**
    * Reorder message queue.
    *
    * @param[in,out] messageQueue: message queue
    * @param[in] reorderConnect: reorder flag
    */
   void reorderApp2BtsQueue(INOUT MessageQueue< App2Bts_BaseMessage >& messageQueue, IN const bool reorderConnect) const;

   /**
    * Reorder message queue.
    *
    * @param[out] queueCopy: copy of message queue
    * @param[out] pendingItems: list of pending items
    * @param[out] pendingFlags: list of pending flags
    * @param[in] messageQueue: message queue
    */
   void reorderApp2BtsQueue4Connect(OUT MessageQueue< App2Bts_BaseMessage >& queueCopy, OUT ::std::vector< App2Bts_BaseMessage* >& pendingItems, OUT ::std::vector< bool >& pendingFlags, IN const MessageQueue< App2Bts_BaseMessage >& messageQueue) const;

   /**
    * Reorder message queue.
    *
    * @param[out] queueCopy: copy of message queue
    * @param[out] pendingItems: list of pending items
    * @param[out] pendingFlags: list of pending flags
    * @param[in] messageQueue: message queue
    */
   void reorderApp2BtsQueue4Search(OUT MessageQueue< App2Bts_BaseMessage >& queueCopy, OUT ::std::vector< App2Bts_BaseMessage* >& pendingItems, OUT ::std::vector< bool >& pendingFlags, IN const MessageQueue< App2Bts_BaseMessage >& messageQueue) const;

   /**
    * Initialize extensions (can be Bluetooth stack related, can be related to other functionality e.g. WiFi access).
    *
    * @param[in] configuration: local stack configuration data
    * @param[in] userMode: user mode
    * @param[in] interface: stack interface type
    */
   void initializeExtensions(IN const BTSLocalStackConfiguration& configuration, IN const BTSUserMode userMode, IN const BTSInterfaceType interface);

   /**
    * Destroy extensions (can be Bluetooth stack related, can be related to other functionality e.g. WiFi access).
    */
   void destroyExtensions(void);

   /**
    * Set configuration data for local stack (extensions).
    *
    * @param[in] component: component
    * @param[in] stackInterface: stack interface
    * @param[in] subComponent: sub component
    * @param[in] userMode: user mode
    * @param[out] dbusInterfaces: list of needed DBUS interfaces
    * @param[in] configuration: local stack configuration data
    */
   void setExtensionStackConfiguration(IN const BTSFunctionBlock component, IN const BTSInterfaceType stackInterface, IN const BTSFunctionBlock subComponent, IN const BTSUserMode userMode,
            OUT ::std::vector<BTSDbusInterfaceItem>& dbusInterfaces, IN const BTSLocalConfigurationContainer& configuration);

   /**
    * Do a pre-check for App2Bts messages (extensions).
    *
    * @param[out] continueProcessing: flag indicating that processing of given message shall be continued or stopped
    * @param[out] rejectRequest: flag for rejecting request
    * @param[out] resultCode: result code to be used for rejecting request
    * @param[out] statusCode: status code to be used for rejecting request
    * @param[out] skipOpCodeCheck: flag for skipping opcode check
    * @param[in] message: App2Bts message to be checked
    *
    * @return = true: given message was handled ("extension is responsible for this message request"),
    * @return = false: given message was not handled ("extension is not responsible for this message request")
    */
   bool doApp2BtsExtensionMsgPrecheck(OUT bool& continueProcessing, OUT bool& rejectRequest, OUT BTSCommonEnumClass& resultCode, OUT BTSCommonEnumClass& statusCode, OUT bool& skipOpCodeCheck, IN App2Bts_BaseMessage* message);

   /**
    * Get list of similar items (opcodes) (extensions).
    *
    * @param[out] itemList: similar item list
    * @param[in] message: message we want to have a similar item list for
    *
    * @return = true: given message was handled ("extension is responsible for this message request"),
    * @return = false: given message was not handled ("extension is not responsible for this message request")
    */
   bool getSimilarExtensionOpCodes(OUT ::std::vector< BTSApp2BtsMessageMasterCompareItem >& itemList, IN const App2Bts_BaseMessage* message) const;

   /**
    * Get list of matching items (opcodes) (extensions).
    *
    * @param[out] itemList: matching item list
    * @param[out] highPrioItemList: high priority item list
    * @param[in] message: message we want to have a matching item list for
    *
    * @return = true: given message was handled ("extension is responsible for this message request"),
    * @return = false: given message was not handled ("extension is not responsible for this message request")
    */
   bool getMatchingExtensionOpCodes(OUT ::std::vector< BTSApp2BtsMessageMasterCompareItem >& itemList, OUT ::std::vector< BTSApp2BtsMessageMasterCompareItem >& highPrioItemList, IN const App2Bts_BaseMessage* message) const;

   /**
    * Handle doubled entries (messages that will be not processed via the normal mechanism) (extensions). A suitable answer shall be sent for such kind of messages.
    *
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[in] message: "doubled" message
    * @param[in] resultCode: result code to be used
    * @param[in] statusCode: status code to be used
    *
    * @return = true: given message was handled ("extension is responsible for this message request"),
    * @return = false: given message was not handled ("extension is not responsible for this message request")
    */
   bool handleDoubledApp2BtsExtensionMessages(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN App2Bts_BaseMessage* message, IN const BTSCommonEnumClass resultCode, IN const BTSCommonEnumClass statusCode);

   /**
    * Handle received App2Bts message (extensions).
    * Handle any error within this function.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] deleteApp2BtsMessage: marker to delete related App2Bts message
    * @param[in] message: received App2Bts message
    *
    * @return = true: given message was handled ("extension is responsible for this message request"),
    * @return = false: given message was not handled ("extension is not responsible for this message request")
    */
   bool handleApp2BtsExtensionMessage(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT bool& deleteApp2BtsMessage, IN App2Bts_BaseMessage* message);

   /**
    * Handle received Ipc2Bts message (extensions).
    * Handle any error within this function.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] message: received Ipc2Bts message
    *
    * @return = true: given message was handled ("extension is responsible for this message request"),
    * @return = false: given message was not handled ("extension is not responsible for this message request")
    */
   bool handleIpc2BtsExtensionMessage(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_BaseMessage* message);

   /**
    * Set a sub control test command (extensions).
    *
    * @param[in] testCommand: test command as string
    * @param[in] testData: test data
    *
    * @return = true: given message was handled ("extension is responsible for this message request"),
    * @return = false: given message was not handled ("extension is not responsible for this message request")
    */
   bool setSubControlExtensionTestCommand(IN const char* testCommand, IN const unsigned int testData);

   /**
    * Set a sub control test command (extensions).
    *
    * @param[in] testCommand: test command as string
    * @param[in] testData: test data
    *
    * @return = true: given message was handled ("extension is responsible for this message request"),
    * @return = false: given message was not handled ("extension is not responsible for this message request")
    */
   bool setSubControlExtensionTestCommand(IN const char* testCommand, IN const unsigned char* testData);

   /**
    * Handle TTFis command (extensions).
    *
    * @param[in] testData: test data
    *
    * @return = true: given message was handled ("extension is responsible for this message request"),
    * @return = false: given message was not handled ("extension is not responsible for this message request")
    */
   bool handleTTFisExtensionCommand(IN const unsigned char* testData);

   /**
    * Handle App2Bts test command (extensions).
    *
    * @param[in] testData: test data
    *
    * @return = true: given message was handled ("extension is responsible for this message request"),
    * @return = false: given message was not handled ("extension is not responsible for this message request")
    */
   bool handleApp2BtsExtensionTestCommand(IN const unsigned char* testData);
   /*********** end here with specific class members + methods *******************/

   /**
    * Handle doubled App2Bts message.
    * Handle any error within this function.
    *
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[in] ptrMessage: received App2Bts message
    * @param[in] resultCode: result code
    * @param[in] statusCode: status code
    */
   void handleDoubledApp2BtsStartInternalCommunication(OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, IN App2Bts_StartInternalCommunication* ptrMessage, IN const BTSCommonEnumClass resultCode, IN const BTSCommonEnumClass statusCode);
   void handleDoubledApp2BtsTriggerFbConnectionInitialized(OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, IN App2Bts_TriggerFbConnectionInitialized* ptrMessage, IN const BTSCommonEnumClass resultCode, IN const BTSCommonEnumClass statusCode);
   void handleDoubledApp2BtsSwitchBluetoothOnOff(OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, IN App2Bts_SwitchBluetoothOnOff* ptrMessage, IN const BTSCommonEnumClass resultCode, IN const BTSCommonEnumClass statusCode);
   void handleDoubledApp2BtsGetBluetoothOnOffStatus(OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, IN App2Bts_GetBluetoothOnOffStatus* ptrMessage, IN const BTSCommonEnumClass resultCode, IN const BTSCommonEnumClass statusCode);
   void handleDoubledApp2BtsSwitchBluetoothHardwareOnOff(OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, IN App2Bts_SwitchBluetoothHardwareOnOff* ptrMessage, IN const BTSCommonEnumClass resultCode, IN const BTSCommonEnumClass statusCode);
   void handleDoubledApp2BtsSetLocalBtName(OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, IN App2Bts_SetLocalBtName* ptrMessage, IN const BTSCommonEnumClass resultCode, IN const BTSCommonEnumClass statusCode);
   void handleDoubledApp2BtsGetLocalBtName(OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, IN App2Bts_GetLocalBtName* ptrMessage, IN const BTSCommonEnumClass resultCode, IN const BTSCommonEnumClass statusCode);
   void handleDoubledApp2BtsSetDeviceIdServiceRecord(OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, IN App2Bts_SetDeviceIdServiceRecord* ptrMessage, IN const BTSCommonEnumClass resultCode, IN const BTSCommonEnumClass statusCode);
   void handleDoubledApp2BtsSetDeviceForRestrictedPairingConnecting(OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, IN App2Bts_SetDeviceForRestrictedPairingConnecting* ptrMessage, IN const BTSCommonEnumClass resultCode, IN const BTSCommonEnumClass statusCode);
   void handleDoubledApp2BtsSetLocalAdapterModes(OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, IN App2Bts_SetLocalAdapterModes* ptrMessage, IN const BTSCommonEnumClass resultCode, IN const BTSCommonEnumClass statusCode);
   void handleDoubledApp2BtsGetLocalAdapterModes(OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, IN App2Bts_GetLocalAdapterModes* ptrMessage, IN const BTSCommonEnumClass resultCode, IN const BTSCommonEnumClass statusCode);
   void handleDoubledApp2BtsStartDiscovery(OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, IN App2Bts_StartDiscovery* ptrMessage, IN const BTSCommonEnumClass resultCode, IN const BTSCommonEnumClass statusCode);
   void handleDoubledApp2BtsStopDiscovery(OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, IN App2Bts_StopDiscovery* ptrMessage, IN const BTSCommonEnumClass resultCode, IN const BTSCommonEnumClass statusCode);
   void handleDoubledApp2BtsGetDiscoveringStatus(OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, IN App2Bts_GetDiscoveringStatus* ptrMessage, IN const BTSCommonEnumClass resultCode, IN const BTSCommonEnumClass statusCode);
   void handleDoubledApp2BtsStartPairing(OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, IN App2Bts_StartPairing* ptrMessage, IN const BTSCommonEnumClass resultCode, IN const BTSCommonEnumClass statusCode);
   void handleDoubledApp2BtsCancelPairing(OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, IN App2Bts_CancelPairing* ptrMessage, IN const BTSCommonEnumClass resultCode, IN const BTSCommonEnumClass statusCode);
   void handleDoubledApp2BtsSetPinCode(OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, IN App2Bts_SetPinCode* ptrMessage, IN const BTSCommonEnumClass resultCode, IN const BTSCommonEnumClass statusCode);
   void handleDoubledApp2BtsConfirmSecureSimplePairing(OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, IN App2Bts_ConfirmSecureSimplePairing* ptrMessage, IN const BTSCommonEnumClass resultCode, IN const BTSCommonEnumClass statusCode);
   void handleDoubledApp2BtsSetLinkKey(OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, IN App2Bts_SetLinkKey* ptrMessage, IN const BTSCommonEnumClass resultCode, IN const BTSCommonEnumClass statusCode);
   void handleDoubledApp2BtsConnectProtocol(OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, IN App2Bts_ConnectProtocol* ptrMessage, IN const BTSCommonEnumClass resultCode, IN const BTSCommonEnumClass statusCode);
   void handleDoubledApp2BtsAcceptRemoteProtocolConnect(OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, IN App2Bts_AcceptRemoteProtocolConnect* ptrMessage, IN const BTSCommonEnumClass resultCode, IN const BTSCommonEnumClass statusCode);
   void handleDoubledApp2BtsRejectRemoteProtocolConnect(OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, IN App2Bts_RejectRemoteProtocolConnect* ptrMessage, IN const BTSCommonEnumClass resultCode, IN const BTSCommonEnumClass statusCode);
   void handleDoubledApp2BtsDisconnectProtocol(OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, IN App2Bts_DisconnectProtocol* ptrMessage, IN const BTSCommonEnumClass resultCode, IN const BTSCommonEnumClass statusCode);
   void handleDoubledApp2BtsDisconnectDevice(OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, IN App2Bts_DisconnectDevice* ptrMessage, IN const BTSCommonEnumClass resultCode, IN const BTSCommonEnumClass statusCode);
   void handleDoubledApp2BtsGetProtocolConnectionStatus(OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, IN App2Bts_GetProtocolConnectionStatus* ptrMessage, IN const BTSCommonEnumClass resultCode, IN const BTSCommonEnumClass statusCode);
   void handleDoubledApp2BtsGetDeviceConnectionStatus(OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, IN App2Bts_GetDeviceConnectionStatus* ptrMessage, IN const BTSCommonEnumClass resultCode, IN const BTSCommonEnumClass statusCode);
   void handleDoubledApp2BtsStartRemoteServiceSearch(OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, IN App2Bts_StartRemoteServiceSearch* ptrMessage, IN const BTSCommonEnumClass resultCode, IN const BTSCommonEnumClass statusCode);
   void handleDoubledApp2BtsCancelRemoteServiceSearch(OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, IN App2Bts_CancelRemoteServiceSearch* ptrMessage, IN const BTSCommonEnumClass resultCode, IN const BTSCommonEnumClass statusCode);
   void handleDoubledApp2BtsQueryRemoteName(OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, IN App2Bts_QueryRemoteName* ptrMessage, IN const BTSCommonEnumClass resultCode, IN const BTSCommonEnumClass statusCode);
   void handleDoubledApp2BtsRequestLinkQuality(OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, IN App2Bts_RequestLinkQuality* ptrMessage, IN const BTSCommonEnumClass resultCode, IN const BTSCommonEnumClass statusCode);
   void handleDoubledApp2BtsStartTestMode(OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, IN App2Bts_StartTestMode* ptrMessage, IN const BTSCommonEnumClass resultCode, IN const BTSCommonEnumClass statusCode);
   void handleDoubledApp2BtsStopTestMode(OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, IN App2Bts_StopTestMode* ptrMessage, IN const BTSCommonEnumClass resultCode, IN const BTSCommonEnumClass statusCode);
   void handleDoubledApp2BtsRequestTestModeLinkQuality(OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, IN App2Bts_RequestTestModeLinkQuality* ptrMessage, IN const BTSCommonEnumClass resultCode, IN const BTSCommonEnumClass statusCode);

   /**
    * Handle received App2Bts message.
    * Handle any error within this function.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] deleteApp2BtsMessage: marker to delete related App2Bts message
    * @param[in] ptrMessage: received App2Bts message
    */
   void handleApp2BtsStartInternalCommunication(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT bool& deleteApp2BtsMessage, IN App2Bts_StartInternalCommunication* ptrMessage);
   void handleApp2BtsTriggerFbConnectionInitialized(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT bool& deleteApp2BtsMessage, IN App2Bts_TriggerFbConnectionInitialized* ptrMessage);
   void handleApp2BtsSwitchBluetoothOnOff(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT bool& deleteApp2BtsMessage, IN App2Bts_SwitchBluetoothOnOff* ptrMessage);
   void handleApp2BtsGetBluetoothOnOffStatus(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT bool& deleteApp2BtsMessage, IN App2Bts_GetBluetoothOnOffStatus* ptrMessage);
   void handleApp2BtsSwitchBluetoothHardwareOnOff(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT bool& deleteApp2BtsMessage, IN App2Bts_SwitchBluetoothHardwareOnOff* ptrMessage);
   void handleApp2BtsSetLocalBtName(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT bool& deleteApp2BtsMessage, IN App2Bts_SetLocalBtName* ptrMessage);
   void handleApp2BtsGetLocalBtName(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT bool& deleteApp2BtsMessage, IN App2Bts_GetLocalBtName* ptrMessage);
   void handleApp2BtsSetDeviceIdServiceRecord(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT bool& deleteApp2BtsMessage, IN App2Bts_SetDeviceIdServiceRecord* ptrMessage);
   void handleApp2BtsSetDeviceForRestrictedPairingConnecting(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT bool& deleteApp2BtsMessage, IN App2Bts_SetDeviceForRestrictedPairingConnecting* ptrMessage);
   void handleApp2BtsSetLocalAdapterModes(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT bool& deleteApp2BtsMessage, IN App2Bts_SetLocalAdapterModes* ptrMessage);
   void handleApp2BtsGetLocalAdapterModes(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT bool& deleteApp2BtsMessage, IN App2Bts_GetLocalAdapterModes* ptrMessage);
   void handleApp2BtsStartDiscovery(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT bool& deleteApp2BtsMessage, IN App2Bts_StartDiscovery* ptrMessage);
   void handleApp2BtsStopDiscovery(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT bool& deleteApp2BtsMessage, IN App2Bts_StopDiscovery* ptrMessage);
   void handleApp2BtsGetDiscoveringStatus(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT bool& deleteApp2BtsMessage, IN App2Bts_GetDiscoveringStatus* ptrMessage);
   void handleApp2BtsStartPairing(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT bool& deleteApp2BtsMessage, IN App2Bts_StartPairing* ptrMessage);
   void handleApp2BtsCancelPairing(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT bool& deleteApp2BtsMessage, IN App2Bts_CancelPairing* ptrMessage);
   void handleApp2BtsSetPinCode(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT bool& deleteApp2BtsMessage, IN App2Bts_SetPinCode* ptrMessage);
   void handleApp2BtsConfirmSecureSimplePairing(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT bool& deleteApp2BtsMessage, IN App2Bts_ConfirmSecureSimplePairing* ptrMessage);
   void handleApp2BtsSetLinkKey(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT bool& deleteApp2BtsMessage, IN App2Bts_SetLinkKey* ptrMessage);
   void handleApp2BtsConnectProtocol(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT bool& deleteApp2BtsMessage, IN App2Bts_ConnectProtocol* ptrMessage);
   void handleApp2BtsAcceptRemoteProtocolConnect(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT bool& deleteApp2BtsMessage, IN App2Bts_AcceptRemoteProtocolConnect* ptrMessage);
   void handleApp2BtsRejectRemoteProtocolConnect(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT bool& deleteApp2BtsMessage, IN App2Bts_RejectRemoteProtocolConnect* ptrMessage);
   void handleApp2BtsDisconnectProtocol(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT bool& deleteApp2BtsMessage, IN App2Bts_DisconnectProtocol* ptrMessage);
   void handleApp2BtsDisconnectDevice(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT bool& deleteApp2BtsMessage, IN App2Bts_DisconnectDevice* ptrMessage);
   void handleApp2BtsGetProtocolConnectionStatus(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT bool& deleteApp2BtsMessage, IN App2Bts_GetProtocolConnectionStatus* ptrMessage);
   void handleApp2BtsGetDeviceConnectionStatus(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT bool& deleteApp2BtsMessage, IN App2Bts_GetDeviceConnectionStatus* ptrMessage);
   void handleApp2BtsStartRemoteServiceSearch(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT bool& deleteApp2BtsMessage, IN App2Bts_StartRemoteServiceSearch* ptrMessage);
   void handleApp2BtsCancelRemoteServiceSearch(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT bool& deleteApp2BtsMessage, IN App2Bts_CancelRemoteServiceSearch* ptrMessage);
   void handleApp2BtsQueryRemoteName(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT bool& deleteApp2BtsMessage, IN App2Bts_QueryRemoteName* ptrMessage);
   void handleApp2BtsRequestLinkQuality(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT bool& deleteApp2BtsMessage, IN App2Bts_RequestLinkQuality* ptrMessage);
   void handleApp2BtsStartTestMode(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT bool& deleteApp2BtsMessage, IN App2Bts_StartTestMode* ptrMessage);
   void handleApp2BtsStopTestMode(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT bool& deleteApp2BtsMessage, IN App2Bts_StopTestMode* ptrMessage);
   void handleApp2BtsRequestTestModeLinkQuality(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT bool& deleteApp2BtsMessage, IN App2Bts_RequestTestModeLinkQuality* ptrMessage);

   /**
    * Handle received Ipc2Bts message.
    * Handle any error within this function.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] compareItem: compare item for special handling
    * @param[out] deleteApp2BtsMessage: marker to delete related App2Bts message; in case of a DBUS message sequence has to be passed we have to delete the related App2Bts message after sequence is finished
    * @param[in,out] app2BtsMessage: related App2Bts message from working queue
    * @param[in] ptrMessage: received Ipc2Bts message
    */
   void handleIpc2BtsServiceAvailabilityConnection(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_ServiceAvailabilityConnection& message);
   void handleIpc2BtsGetAllPropertiesConnection(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_GetAllPropertiesConnection& message);
   void handleIpc2BtsAddDeviceObjectPathMapping(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_AddDeviceObjectPathMapping& message);
   void handleIpc2BtsDelDeviceObjectPathMapping(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_DelDeviceObjectPathMapping& message);
   void handleIpc2BtsAddProtocolObjectPathMapping(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_AddProtocolObjectPathMapping& message);
   void handleIpc2BtsDelProtocolObjectPathMapping(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_DelProtocolObjectPathMapping& message);
   void handleIpc2BtsAddSerialObjectPathMapping(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_AddSerialObjectPathMapping& message);
   void handleIpc2BtsDelSerialObjectPathMapping(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_DelSerialObjectPathMapping& message);
   void handleIpc2BtsRelease(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_Release& message);
   void handleIpc2BtsRequestPinCode(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_RequestPinCode& message);
   void handleIpc2BtsDisplayPinCode(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_DisplayPinCode& message);
   void handleIpc2BtsRequestPasskey(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_RequestPasskey& message);
   void handleIpc2BtsRequestConfirmation(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_RequestConfirmation& message);
   void handleIpc2BtsCancel(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_Cancel& message);
   void handleIpc2BtsCancelRequest(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_CancelRequest& message);
   void handleIpc2BtsAuthorizeService(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_AuthorizeService& message);
   void handleIpc2BtsRequestAuthorization(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_RequestAuthorization& message);
   void handleIpc2BtsRequestLinkkey(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_RequestLinkkey& message);
   void handleIpc2BtsDisplayPasskey(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_DisplayPasskey& message);
   void handleIpc2BtsStartDiscoveryResult(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_StartDiscoveryResult& message);
   void handleIpc2BtsStopDiscoveryResult(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_StopDiscoveryResult& message);
   void handleIpc2BtsRemoveDeviceResult(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_RemoveDeviceResult& message);
   void handleIpc2BtsCreateDeviceResult(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_CreateDeviceResult& message);
   void handleIpc2BtsSendHCICmdResult(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_SendHCICmdResult& message);
   void handleIpc2BtsAdapterInitialization(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_AdapterInitialization& message);
   void handleIpc2BtsDeviceCreated(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_DeviceCreated& message);
   void handleIpc2BtsDeviceRemoved(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_DeviceRemoved& message);
   void handleIpc2BtsBtStackLog(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_BtStackLog& message);
   void handleIpc2BtsHCIEvent(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_HCIEvent& message);
   void handleIpc2BtsDeviceFound(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_DeviceFound& message);
   void handleIpc2BtsAdapterAddressUpdate(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_AdapterAddressUpdate& message);
   void handleIpc2BtsAdapterClassUpdate(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_AdapterClassUpdate& message);
   void handleIpc2BtsAdapterAliasUpdate(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_AdapterAliasUpdate& message);
   void handleIpc2BtsAdapterPoweredUpdate(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_AdapterPoweredUpdate& message);
   void handleIpc2BtsAdapterDiscoverableUpdate(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_AdapterDiscoverableUpdate& message);
   void handleIpc2BtsAdapterDiscoverabletimeoutUpdate(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_AdapterDiscoverabletimeoutUpdate& message);
   void handleIpc2BtsAdapterPairableUpdate(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_AdapterPairableUpdate& message);
   void handleIpc2BtsAdapterPairabletimeoutUpdate(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_AdapterPairabletimeoutUpdate& message);
   void handleIpc2BtsAdapterDiscoveringUpdate(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_AdapterDiscoveringUpdate& message);
   void handleIpc2BtsAdapterUuidsUpdate(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_AdapterUuidsUpdate& message);
   void handleIpc2BtsAdapterWbsModeUpdate(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_AdapterWbsModeUpdate& message);
   void handleIpc2BtsAdapterConnectableUpdate(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_AdapterConnectableUpdate& message);
   void handleIpc2BtsAdapterHcimodeUpdate(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_AdapterHcimodeUpdate& message);
   void handleIpc2BtsAdapterVersionUpdate(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_AdapterVersionUpdate& message);
   void handleIpc2BtsAdapterTraceLevelUpdate(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_AdapterTraceLevelUpdate& message);
   void handleIpc2BtsAdapterTraceCategoryUpdate(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_AdapterTraceCategoryUpdate& message);
   void handleIpc2BtsAdapterAvailableSppServersUpdate(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_AdapterAvailableSppServersUpdate& message);
   void handleIpc2BtsRegisterAgentResult(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_RegisterAgentResult& message);
   void handleIpc2BtsUnregisterAgentResult(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_UnregisterAgentResult& message);
   void handleIpc2BtsRequestDefaultAgentResult(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_RequestDefaultAgentResult& message);
   void handleIpc2BtsDeviceConnectResult(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_DeviceConnectResult& message);
   void handleIpc2BtsDeviceDisconnectResult(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_DeviceDisconnectResult& message);
   void handleIpc2BtsPairResult(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_PairResult& message);
   void handleIpc2BtsCancelPairingResult(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_CancelPairingResult& message);
   void handleIpc2BtsDiscoverServicesResult(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_DiscoverServicesResult& message);
   void handleIpc2BtsCancelDiscoveryResult(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_CancelDiscoveryResult& message);
   void handleIpc2BtsReportLinkkey(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_ReportLinkkey& message);
   void handleIpc2BtsIncomingPairingComp(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_IncomingPairingComp& message);
   void handleIpc2BtsSDPProcessReport(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_SDPProcessReport& message);
   void handleIpc2BtsDeviceAddressUpdate(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_DeviceAddressUpdate& message);
   void handleIpc2BtsDeviceClassUpdate(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_DeviceClassUpdate& message);
   void handleIpc2BtsDeviceIconUpdate(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_DeviceIconUpdate& message);
   void handleIpc2BtsDeviceUuidsUpdate(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_DeviceUuidsUpdate& message);
   void handleIpc2BtsDevicePairedUpdate(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_DevicePairedUpdate& message);
   void handleIpc2BtsDeviceConnectedUpdate(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_DeviceConnectedUpdate& message);
   void handleIpc2BtsDeviceTrustedUpdate(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_DeviceTrustedUpdate& message);
   void handleIpc2BtsDeviceBlockedUpdate(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_DeviceBlockedUpdate& message);
   void handleIpc2BtsDeviceAliasUpdate(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_DeviceAliasUpdate& message);
   void handleIpc2BtsDeviceAdapterUpdate(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_DeviceAdapterUpdate& message);
   void handleIpc2BtsDeviceModaliasUpdate(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_DeviceModaliasUpdate& message);
   void handleIpc2BtsDeviceRssiUpdate(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_DeviceRssiUpdate& message);
   void handleIpc2BtsDeviceVendorIdUpdate(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_DeviceVendorIdUpdate& message);
   void handleIpc2BtsDeviceVendorIdSourceUpdate(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_DeviceVendorIdSourceUpdate& message);
   void handleIpc2BtsDeviceProductIdUpdate(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_DeviceProductIdUpdate& message);
   void handleIpc2BtsDeviceMapInfoUpdate(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_DeviceMapInfoUpdate& message);
   void handleIpc2BtsDeviceConnectedServiceUpdate(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_DeviceConnectedServiceUpdate& message);
   void handleIpc2BtsDeviceVersionUpdate(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_DeviceVersionUpdate& message);
   void handleIpc2BtsDevicePbapInfoUpdate(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_DevicePbapInfoUpdate& message);
   void handleIpc2BtsDeviceModeUpdate(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_DeviceModeUpdate& message);
   void handleIpc2BtsDeviceRoleUpdate(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_DeviceRoleUpdate& message);
   void handleIpc2BtsDeviceEnableAvpPauseUpdate(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_DeviceEnableAvpPauseUpdate& message);
   void handleIpc2BtsBtApplDeviceConnectReqResult(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_BtApplDeviceConnectReqResult& message);
   void handleIpc2BtsBtApplDeviceDisconnectReqResult(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_BtApplDeviceDisconnectReqResult& message);
   void handleIpc2BtsBtApplCancelConnectReqResult(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_BtApplCancelConnectReqResult& message);
   void handleIpc2BtsBtApplGetHwVersionReqResult(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_BtApplGetHwVersionReqResult& message);
   void handleIpc2BtsBtApplSetConfigurationReqResult(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_BtApplSetConfigurationReqResult& message);
   void handleIpc2BtsBtApplReadConfigurationReqResult(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_BtApplReadConfigurationReqResult& message);
   void handleIpc2BtsBtApplDeviceConnectCfm(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_BtApplDeviceConnectCfm& message);
   void handleIpc2BtsBtApplDeviceConnectInd(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_BtApplDeviceConnectInd& message);
   void handleIpc2BtsBtApplDeviceDisconnectCfm(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_BtApplDeviceDisconnectCfm& message);
   void handleIpc2BtsBtApplDeviceDisconnectInd(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_BtApplDeviceDisconnectInd& message);
   void handleIpc2BtsBtApplDeviceDisconnectCompInd(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_BtApplDeviceDisconnectCompInd& message);
   void handleIpc2BtsBtApplDeviceConnectCompInd(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_BtApplDeviceConnectCompInd& message);
   void handleIpc2BtsBtApplCancelConnectCfm(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_BtApplCancelConnectCfm& message);
   void handleIpc2BtsBtApplStatusNotificationInd(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_BtApplStatusNotificationInd& message);
   void handleIpc2BtsBtApplGetHwVersionCfm(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_BtApplGetHwVersionCfm& message);
   void handleIpc2BtsBtApplSetConfigurationCfm(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_BtApplSetConfigurationCfm& message);
   void handleIpc2BtsBtApplReadConfigurationCfm(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_BtApplReadConfigurationCfm& message);
   void handleIpc2BtsConnectResult(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_ConnectResult& message);
   void handleIpc2BtsDisconnectResult(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_DisconnectResult& message);
   void handleIpc2BtsServiceRemoteUuidUpdate(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_ServiceRemoteUuidUpdate& message);
   void handleIpc2BtsServiceStateUpdate(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_ServiceStateUpdate& message);
   void handleIpc2BtsServiceDeviceUpdate(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_ServiceDeviceUpdate& message);
   void handleIpc2BtsServiceVersionUpdate(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_ServiceVersionUpdate& message);
   void handleIpc2BtsDebugTraceConfigResult(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_DebugTraceConfigResult& message);
   void handleIpc2BtsProtocolTraceConfigResult(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_ProtocolTraceConfigResult& message);
   void handleIpc2BtsReportDebugTrace(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_ReportDebugTrace& message);
   void handleIpc2BtsReportProtocolTrace(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_ReportProtocolTrace& message);
   void handleIpc2BtsFatalError(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_FatalError& message);
   void handleIpc2BtsGeneralError(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_GeneralError& message);
   void handleIpc2BtsReportGeniviDebugTrace(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_ReportGeniviDebugTrace& message);
   void handleIpc2BtsGeniviFatalError(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_GeniviFatalError& message);
   void handleIpc2BtsBtApplDidSetReqResult(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_BtApplDidSetReqResult& message);
   void handleIpc2BtsBtApplDidSetCfm(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_BtApplDidSetCfm& message);
   void handleIpc2BtsDunDeviceFilePathUpdate(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_DunDeviceFilePathUpdate& message);
   void handleIpc2BtsBtApplSppSetUuidReqResult(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_BtApplSppSetUuidReqResult& message);
   void handleIpc2BtsBtApplSppSetUuidCfm(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_BtApplSppSetUuidCfm& message);
   void handleIpc2BtsBtApplSppVirtualDeviceInd(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_BtApplSppVirtualDeviceInd& message);
   void handleIpc2BtsCreateSessionResult(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_CreateSessionResult& message);
   void handleIpc2BtsRemoveSessionResult(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_RemoveSessionResult& message);
   void handleIpc2BtsCancelSessionResult(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_CancelSessionResult& message);
   void handleIpc2BtsSessionCreated(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_SessionCreated& message);
   void handleIpc2BtsSessionRemoved(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_SessionRemoved& message);
   void handleIpc2BtsAddTechnologyObjectPath(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_AddTechnologyObjectPath& message);
   void handleIpc2BtsDelTechnologyObjectPath(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_DelTechnologyObjectPath& message);
   void handleIpc2BtsAddNetworkObjectPathMapping(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_AddNetworkObjectPathMapping& message);
   void handleIpc2BtsDelNetworkObjectPathMapping(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_DelNetworkObjectPathMapping& message);
   void handleIpc2BtsGetConnManManagerPropertiesResult(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_GetConnManManagerPropertiesResult& message);
   void handleIpc2BtsGetConnManTechnologiesResult(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_GetConnManTechnologiesResult& message);
   void handleIpc2BtsGetConnManServicesResult(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_GetConnManServicesResult& message);
   void handleIpc2BtsConnManTechnologyAdded(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_ConnManTechnologyAdded& message);
   void handleIpc2BtsConnManTechnologyRemoved(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_ConnManTechnologyRemoved& message);
   void handleIpc2BtsConnManServicesChanged(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_ConnManServicesChanged& message);
   void handleIpc2BtsConnManServicesRemoved(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_ConnManServicesRemoved& message);
   void handleIpc2BtsGetConnManServicePropertiesResult(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_GetConnManServicePropertiesResult& message);
   void handleIpc2BtsConnManServiceConnectResult(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_ConnManServiceConnectResult& message);
   void handleIpc2BtsConnManServiceDisconnectResult(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_ConnManServiceDisconnectResult& message);
   void handleIpc2BtsConnManServiceStateUpdate(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_ConnManServiceStateUpdate& message);
   void handleIpc2BtsConnManServiceInterfaceUpdate(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_ConnManServiceInterfaceUpdate& message);
   void handleIpc2BtsSerialDeviceFilePath(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_SerialDeviceFilePath& message);
   void handleIpc2BtsCharacterDeviceAddedRemoved(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_CharacterDeviceAddedRemoved& message);
   void handleIpc2BtsAddHfpInterfaceObjectPathMapping(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_AddHfpInterfaceObjectPathMapping& message);
   void handleIpc2BtsDelHfpInterfaceObjectPathMapping(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_DelHfpInterfaceObjectPathMapping& message);
   void handleIpc2BtsGetHandsfreeProperties(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_GetHandsfreeProperties& message);
   void handleIpc2BtsHandsfreeFeaturesUpdate(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_HandsfreeFeaturesUpdate& message);
   void handleIpc2BtsHandsfreeInbandRingingUpdate(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_HandsfreeInbandRingingUpdate& message);
   void handleIpc2BtsGetModems(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_GetModems& message);
   void handleIpc2BtsModemAdded(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_ModemAdded& message);
   void handleIpc2BtsModemRemoved(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_ModemRemoved& message);
   void handleIpc2BtsGetModemProperties(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_GetModemProperties& message);
   void handleIpc2BtsModemInterfacesUpdate(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_ModemInterfacesUpdate& message);

   /**
    * Handle test command for App2Bts messages.
    *
    * @param[in] testData: test data
    */
   void handleApp2BtsTestCommand(IN const unsigned char* testData);
};

} //genivi
} //btstackif

#endif //_CONNECTION_CONTROL_GENIVI_H_
