/**
 * @file EvolutionGeniviStackTypes.h
 *
 * @par SW-Component
 * Types
 *
 * @brief Basic types for Evolution Genivi stack.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Definitions of all internal types for Evolution Genivi stack.
 */

#ifndef _EVOLUTION_GENIVI_STACK_TYPES_H_
#define _EVOLUTION_GENIVI_STACK_TYPES_H_

#include "BtStackInternalTypes.h"
#include "EvolutionGeniviStackOpcodes.h"
#include "ConnManTypes.h"

#include "cc_dbus_if/EvolutionGeniviDbusTypes.h"

namespace btstackif {
namespace genivi {

/*
 * RULE 1: use a class instead of declaring a struct (avoiding problems with copy actions), implement copy constructor, assignment operator and equality/inequality operator
 *
 * keep following order:
 * - class forward declarations
 * - defines (if really necessary)
 * - typedefs
 * - enums
 * - structs => classes
 */

#define BTS_SET_AGENT_CAPABILITY "BTS_SET_AGENT_CAPABILITY" /**< connection control test command */

#define BTS_GET_DEVICE_VENDOR_ID "BTS_GET_DEVICE_VENDOR_ID" /**< connection control test command */

#define BTS_GET_ALL_SEARCH_RESULTS "BTS_GET_ALL_SEARCH_RESULTS" /**< connection control test command */

#define BTS_TRIGGER_DISCOVER_SERVICES_BUSY "BTS_TRIGGER_DISCOVER_SERVICES_BUSY" /**< connection control test command */

#define BTS_TRIGGER_CONNECT_RESULT_BUSY "BTS_TRIGGER_CONNECT_RESULT_BUSY" /**< connection control test command */

#define BTS_START_REMOTE_SERVICE_SEARCH "BTS_START_REMOTE_SERVICE_SEARCH" /**< connection control test command */

#define BTS_START_DUN_CONNECT "BTS_START_DUN_CONNECT" /**< connection control test command */

#define BTS_SET_PAN_CONNECT_MODE "BTS_SET_PAN_CONNECT_MODE" /**< connection control test command */

#define BTS_SET_TEST_TRIGGER_DEVICE_CONNECTED_UPDATE "BTS_SET_TEST_TRIGGER_DEVICE_CONNECTED_UPDATE" /**< connection control test command */

#define BTS_SET_TEST_TRIGGER_REMOTE_CONNECT_DURING_DEV_DISCONNECT "BTS_SET_TEST_TRIGGER_REMOTE_CONNECT_DURING_DEV_DISCONNECT" /**< connection control test command */

#define BTS_SET_SIMULATE_FAILED_DISCONNECT_RESULT "BTS_SET_SIMULATE_FAILED_DISCONNECT_RESULT" /**< connection control test command */

#define BTS_LIST_CHARACTER_DEVICES "BTS_LIST_CHARACTER_DEVICES" /**< connection control test command */

#define BTS_OPEN_SPP_TEST_APP "BTS_OPEN_SPP_TEST_APP" /**< connection control test command */

#define BTS_CLOSE_SPP_TEST_APP "BTS_CLOSE_SPP_TEST_APP" /**< connection control test command */

#define BTS_SET_SPP_RAW_MODE "BTS_SET_SPP_RAW_MODE" /**< connection control test command */

#define BTS_REMOVE_ALL_DEVICES "BTS_REMOVE_ALL_DEVICES" /**< connection control test command */

#define BTS_START_REMOTE_SERVICE_SEARCH_2 "BTS_START_REMOTE_SERVICE_SEARCH_2" /**< connection control test command */

#define BTS_BLOCK_REQUEST_CONFIRMATION "BTS_BLOCK_REQUEST_CONFIRMATION" /**< connection control test command */

#define BTS_SIMULATE_REQUEST_CONFIRMATION "BTS_SIMULATE_REQUEST_CONFIRMATION" /**< connection control test command */

#define BTS_SIMULATE_OBEX_OBJECT_PUSH_AUTHORIZE "BTS_SIMULATE_OBEX_OBJECT_PUSH_AUTHORIZE" /**< connection control test command */

#define BTS_SIMULATE_HFP_AUTHORIZE "BTS_SIMULATE_HFP_AUTHORIZE" /**< connection control test command */

#define BTS_SEND_DISCONNECT_DURING_PAIRING "BTS_SEND_DISCONNECT_DURING_PAIRING" /**< connection control test command */

#define BTS_START_CONNECT_AVP "BTS_START_CONNECT_AVP" /**< connection control test command */

#define BTS_SET_TEST_TRIGGER_CANCEL_SERVICE_SEARCH "BTS_SET_TEST_TRIGGER_CANCEL_SERVICE_SEARCH" /**< connection control test command */

#define BTS_BLOCK_AUTHORIZE_SERVICE_RESPONSE "BTS_BLOCK_AUTHORIZE_SERVICE_RESPONSE" /**< connection control test command */

#define BTS_SIMULATE_AVP_AUTHORIZE "BTS_SIMULATE_AVP_AUTHORIZE" /**< connection control test command */

#define BTS_SIMULATE_SERVICE_STATE "BTS_SIMULATE_SERVICE_STATE" /**< connection control test command */

#define BTS_CREATE_DEVICE "BTS_CREATE_DEVICE" /**< connection control test command */

#define BTS_START_CONNECT_PIM "BTS_START_CONNECT_PIM" /**< connection control test command */

#define BTS_SEND_VIRTUAL_IPC2BTS_MSG "BTS_SEND_VIRTUAL_IPC2BTS_MSG" /**< connection control test command */

#define BTS_TRIGGER_IGNORE_OBEX_PROTOCOL_ADDED "BTS_TRIGGER_IGNORE_OBEX_PROTOCOL_ADDED" /**< connection control test command */

#define BTS_TRIGGER_IGNORE_SDP_PROCESS_REPORT "BTS_TRIGGER_IGNORE_SDP_PROCESS_REPORT" /**< connection control test command */

#define BTS_TRIGGER_SIMULATE_SDP_PROCESS_REPORT "BTS_TRIGGER_SIMULATE_SDP_PROCESS_REPORT" /**< connection control test command */

#define BTS_TRIGGER_SIMULATE_STACK_RESET "BTS_TRIGGER_SIMULATE_STACK_RESET" /**< connection control test command */

#define BTS_TRIGGER_SIMULATE_STACK_RESET_DURING_PAIRING "BTS_TRIGGER_SIMULATE_STACK_RESET_DURING_PAIRING" /**< connection control test command */

#define BTS_TRIGGER_SIMULATE_STACK_RESET_DURING_APP2BTS "BTS_TRIGGER_SIMULATE_STACK_RESET_DURING_APP2BTS" /**< connection control test command */

/**
 * Type definition for log opcode of BtStack.
 */
typedef unsigned short int BTSLogOpcode;

/**
 * Type definition for length of BtStack log data.
 */
typedef unsigned short int BTSLogLength;

/**
 * Type definition for BtStack log data.
 */
typedef ::std::vector< unsigned char > BTSLogData;

/**
 * Type definition for instance index. TODO: replace by BTSSppInstanceId
 */
typedef unsigned char BTSInstanceIndex;

/**
 * Type definition for debug trace level.
 */
typedef unsigned char BTSDebugTraceLevel;

/**
 * Type definition for debug time. Current time form 0 to 23H59m59s999ms e.g. 01:30 0x0000D2F0
 */
typedef unsigned int BTSDebugTime;

/**
 * Type definition for debug trace category of Bluetooth application.
 */
typedef unsigned int BTSDebugBtAppCategory;

/**
 * Type definition for debug trace category of Bluetooth stack.
 */
typedef unsigned int BTSDebugBtStackCategory;

/**
 * Type definition for reserved debug trace category.
 */
typedef unsigned int BTSDebugReservedCategory;

/**
 * Type definition for debug trace filter.
 */
typedef unsigned short int BTSDebugTraceFilter;

/**
 * Type definition for debug trace location.
 */
typedef unsigned int BTSDebugTraceLocation;

/**
 * Type definition for debug trace length.
 */
typedef unsigned short int BTSDebugTraceLength;

/**
 * Type definition for debug trace data.
 */
typedef ::std::vector< unsigned char > BTSDebugTraceData;

/**
 * Type definition for fatal error location.
 */
typedef unsigned int BTSFatalErrorLocation;

/**
 * Type definition for fatal error type.
 */
typedef unsigned int BTSFatalErrorType;

/**
 * Type definition for internal opcode.
 */
typedef unsigned short int BTSInternalOpcode;

/**
 * Type definition for debug trace error code.
 */
typedef unsigned short int BTSDebugTraceErrorCode;

/**
 * Type definition for icon name.
 */
typedef ::std::string BTSIconName;

/**
 * Type definition for modalias.
 */
typedef ::std::string BTSModalias;

/**
 * Type definition for length of HCI data.
 */
typedef unsigned short int BTSHciLength;

/**
 * Type definition for HCI data.
 */
typedef ::std::vector< unsigned char > BTSHciData;

/**
 * Type definition for DID index.
 */
typedef unsigned char BTSDidIndex;

/**
 * Type definition for length of DID data.
 */
typedef unsigned short int BTSDidLength;

/**
 * Type definition for DID data.
 */
typedef ::std::vector< unsigned char > BTSDidData;

/**
 * Type definition for length of DUN data.
 */
typedef unsigned short int BTSDunLength;

/**
 * Type definition for DUN data.
 */
typedef ::std::vector< unsigned char > BTSDunData;

/**
 * Type definition for length of PAN data.
 */
typedef unsigned short int BTSPanLength;

/**
 * Type definition for PAN data.
 */
typedef ::std::vector< unsigned char > BTSPanData;

/**
 * Type definition for length of SPP data.
 */
typedef unsigned short int BTSSppLength;

/**
 * Type definition for SPP data.
 */
typedef ::std::vector< unsigned char > BTSSppData;

/**
 * Type definition for OBEX session channel.
 */
typedef unsigned char BTSSessionChannel;

/**
 * Type definition for transfer size.
 */
typedef uint64_t BTSTransferSize;

/**
 * Type definition for transfer time.
 */
typedef unsigned char BTSTransferTime;

/**
 * Type definition for transfer name.
 */
typedef ::std::string BTSTransferName;

/**
 * Type definition for transfer type. TODO: check later if enum is better or not
 */
typedef ::std::string BTSTransferType;

/**
 * Type definition for Manufacturer Name of the BR/EDR Controller. See Bluetooth Assigned Numbers.
 */
typedef unsigned short int BTSManufacturerNameController;

/**
 * Type definition for Subversion of the Current LMP or PAL in the Controller. This value is implementation dependent.
 */
typedef unsigned short int BTSSubVersionController;

/**
 * Type definition for Extension information. If using the UGKZ7 module, Firmware version and ROM version will be included in this extension information.
 */
typedef ::fw::Array<unsigned char, 8> BTSExtensionInfo;

/**
 * Type definition for configuration data stream.
 */
typedef ::std::vector< unsigned char > BTSEvoConfigDataStream;

/**
 * Type definition for HFP Handsfree supported features.
 */
typedef BTSBitfield< unsigned int, ::ccdbusif::evolution::HfpHandsfreeFeatures, ::ccdbusif::evolution::HFP_HANDSFREE_FEATURES_UNKNOWN > BTSHfpSupportedFeatures;

/**
 * Type definition for DBUS interface list.
 */
typedef ::std::vector< ::ccdbusif::evolution::Interface > BTSDbusInterfaceList;

/**
 * Type definition for Genivi trace related values e.g. file number, line number, ....
 */
typedef unsigned short int BTSGeniviTraceNumber;

/**
 * Type definition for Genivi trace category.
 */
typedef unsigned int BTSGeniviDebugTraceCategory;

/**
 * Type definition for Genivi trace data.
 */
typedef ::std::vector< unsigned char > BTSGeniviTraceData;

/**
 * Type definition for Genivi trace error.
 */
typedef ::std::string BTSGeniviTraceError;

/**
 * Type definition for Bluetooth low energy advertising timeout (BLE).
 * HINT: Value 0 means unlimited time advertising.
 */
typedef unsigned char BTSBLEAdvertisingTimeout;

/**
 * Type definition for Bluetooth low energy error code (BLE).
 * HINT: The maximum or error code length is 0x14. TODO: double check with ALPS
 */
typedef ::std::string BTSBLEErrorCode;

/**
 * Type definition for Bluetooth low energy pairing data (BLE).
 */
typedef ::std::vector< unsigned char > BTSBLEPairingData;

/**
 * Type definition for Bluetooth low energy advertising data (BLE).
 */
typedef ::std::vector< unsigned char > BTSBLEAdvertisingData;

/**
 * Type definition for operation identification.
 */
typedef unsigned int BTSOperationId;

/**
 * Type definition for number of typed keys.
 */
typedef unsigned short int BTSNumberTypedKeys;

/**
 * Type definition for log status of BtStack.
 */
enum BTSLogStatus
{
   BTS_LOG_WRITE = 0, /**< write */
   BTS_LOG_READ, /**< read */
   // -------------------------------------------------------------------------
   BTS_LOG_LAST /**< end marker, must be the last one */
};

/**
 * Type definition for status code.
 */
enum BTSStatusCode
{
   BTS_STATUS_CODE_SUCCESS = 0x00, /**< EVTRES_SUCCESS - Operation succeeded. */
   BTS_STATUS_CODE_UNKNOWN_OPCODE = 0x01, /**< EVTRES_UNKNOWN_OPCODE - Opcode is unknown or not supported. Please check command syntax and resend command. */
   BTS_STATUS_CODE_ILLEGAL_FORMAT = 0x02, /**< EVTRES_ILLEGAL_FORMAT - Message parameters are not in correct format. Please check parameter length and resend command. */
   BTS_STATUS_CODE_ILLEGAL_PARAM = 0x03, /**< EVTRES_ILLEGAL_PARAM - Message parameter(s) are out of range. Please check parameter values and resend command. */
   BTS_STATUS_CODE_OPERATION_FAIL = 0x04, /**< EVTRES_OPERATION_FAIL - Operation failed. If failure is caused by remote Bluetooth device, in general no action will be required. */
   BTS_STATUS_CODE_SYSTEM_BUSY = 0x05, /**< EVTRES_SYSTEM_BUSY - System is currently busy. Please wait a moment and try again. */
   BTS_STATUS_CODE_ILLEGAL_OPERATION = 0x06, /**< EVTRES_ILLEGAL_OPERATION - Operation cannot be carried out. Review host application and check whether the command can be executed in this situation. */
   BTS_STATUS_CODE_ILLEGAL_STATE = 0x07, /**< EVTRES_ILLEGAL_STATE - Operation cannot be carried out in the current state. Wait for appropriate state and try again. */
   BTS_STATUS_CODE_PS_FAILURE = 0x08, /**< EVTRES_PS_FAILURE */
   BTS_STATUS_CODE_SDP_FAILURE = 0x09, /**< EVTRES_SDP_FAILURE - Operation failed due to an SDP problem. Please check that the service is available in remote device, then issue re-connection attempt. */
   BTS_STATUS_CODE_ABNORMAL = 0x0A, /**< EVTRES_ABNORMAL - Abnormal operation e.g. link loss causing disconnection. Please issue re-connection attempt if appropriate. */
   BTS_STATUS_CODE_ABORTED = 0x0B, /**< EVTRES_ABORTED - Operation has been aborted. In general no special action will be required. */
   BTS_STATUS_CODE_EXTENDED_ERROR = 0x0C, /**< EVTRES_EXTENDED_ERROR - Operation failed with Extended error. The action depends on the specific error code. */
   BTS_STATUS_CODE_SDP_CONNECTION_ERROR = 0x0D, /**< EVTRES_SDP_CONNECTION_ERROR - SDP Connection error. Please adjust the environment then try again. */
   BTS_STATUS_CODE_TIMEOUT_ERROR = 0x0E, /**< EVTRES_TIMEOUT_ERROR - Operation Timeout. In general no special action will be required. */
   BTS_STATUS_CODE_RESTRICTED_CONNECTION = 0x0F, /**< EVTRES_RESTRICTED_CONNECTION - ... */

   BTS_STATUS_CODE_RAWDATA_ILLEGAL = 0x10, /**< EVTRES_RAWDATA_ILLEGAL - Format of raw data sent from remote device is nonstandard. In general no special action will be required. */
   BTS_STATUS_CODE_SERVICE_NOTSUPPORTED = 0x11, /**< EVTRES_SERVICE_NOTSUPPORTED - Command is not supported. In general no special action will be required. */
   BTS_STATUS_CODE_MOREDATA_EXPECTED = 0x12, /**< EVTRES_MOREDATA_EXPECTED - More packages are expected to be received from the Host. Please send remained packages. */
   BTS_STATUS_CODE_PAGE_TIMEOUT = 0x14, /**< EVTRES_PAGE_TIMEOUT - Page time out in name request and connection request. */
   BTS_STATUS_CODE_LMP_TIMEOUT = 0x16, /**< EVTRES_LMP_TIMEOUT - An LMP transaction failed to respond within the LMP response timeout. */
   BTS_STATUS_CODE_SERVICE_CONNECT_TIMEOUT = 0x17, /**< EVTRES_SERVICE_CONNECT_TIMEOUT - The service connection timed out. */
   BTS_STATUS_CODE_AUTHENTICATION_FAILURE = 0x19, /**< EVTRES_AUTHENTICATION_FAILURE - ... */
   BTS_STATUS_CODE_PIN_OR_KEY_MISSING = 0x1A, /**< EVTRES_PIN_OR_KEY_MISSING - ... */
   BTS_STATUS_CODE_ROLE_CHANGE_NOT_ALLOWED = 0x1B, /**< EVTRES_ROLE_CHANGE_NOT_ALLOWED - HCI error code role change not allowed(0x21). In general no special action will be required. */
   BTS_STATUS_CODE_CONNECT_REJECT_BDADDR = 0x1C, /**< EVTRES_CONNECT_REJECT_BDADDR - HCI error code connection rejected due to unacceptable BD_ADDR (0x0F). In general no special action will be required. */
   BTS_STATUS_CODE_ACL_EXISTS = 0x1D, /**< EVTRES_ACL_EXISTS - An attempt to create a new ACL Connection to a device when there is already a connection to this device. */
   BTS_STATUS_CODE_ACL_NOT_CONNECTED = 0x1E, /**< EVTRES_ACL_NOT_CONNECTED - ACL is not connected. */

   BTS_STATUS_CODE_HW_FAILURE = 0x20, /**< EVTRES_HW_FAILURE - HCI error code hardware failure(0x03). In general no special action will be required. */
   BTS_STATUS_CODE_CONNECT_REJECT_LIMITED_RESOURCES = 0x21, /**< EVTRES_CONNECT_REJECT_LIMITED_RESOURCES - HCI error code connection rejected due to limited resources(0x0D). Please check remote device connection information. */
   BTS_STATUS_CODE_CONNECT_REJECT_SECURITY_REASONS = 0x22, /**< EVTRES_CONNECT_REJECT_SECURITY_REASONS - HCI error code connection rejected due to security reasons(0x0E). Please reset local and remote device security setting and try again. */
   BTS_STATUS_CODE_REMOTE_DISCONNECT_LOW_RESOURCES = 0x23, /**< EVTRES_REMOTE_DISCONNECT_LOW_RESOURCES - HCI error code remote device terminated connection due to low resources(0x14). In general no special action will be required. */
   BTS_STATUS_CODE_REMOTE_DISCONNECT_POWEROFF = 0x24, /**< EVTRES_REMOTE_DISCONNECT_POWEROFF - HCI error code remote device terminated connection due to power off(0x15). In general no special action will be required. */
   BTS_STATUS_CODE_PAIRING_NOT_ALLOWED = 0x25, /**< EVTRES_PAIRING_NOT_ALLOWED - HCI error code remote device does not allow pairing(0x18). Please check the remote device configuration. */
   BTS_STATUS_CODE_CONNECTION_ACCEPT_TIMEOUT = 0x26, /**< EVTRES_CONNECTION_ACCEPT_TIMEOUT - HCI error code the Connection Accept Timeout has been exceeded for this connection attempt(0x10). Please check the remote device configuration. */
   BTS_STATUS_CODE_LOCAL_REJECT = 0x27, /**< EVTRES_LOCAL_REJECT - Module local reject command. In general no special action will be required. */
   BTS_STATUS_CODE_REMOTE_REJECT = 0x28, /**< EVTRES_REMOTE_REJECT - ... */
   BTS_STATUS_CODE_INTERFACE_REMOVED = 0x29, /**< EVTRES_INTERFACE_REMOVED - ... */
   BTS_STATUS_CODE_REMOTE_DISCONNECT_ACL = 0x2C, /**< EVTRES_REMOTE_DISCONNECT_ACL - Remote device disconnects ACL directly during outgoing connection. */

   BTS_STATUS_CODE_ONLY_RSSI_SUCCESS = 0x30, /**< ONLY_RSSI_SUCCESS - for BtApplTestModeLinkQualityCfm: only RSSI value is valid */
   BTS_STATUS_CODE_ONLY_LINK_QUALITY_SUCCESS = 0x31, /**< ONLY_LINK_QULITY_SUCCESS - for BtApplTestModeLinkQualityCfm: only link quality value is valid */

   BTS_STATUS_CODE_FAULT_LC_LMP_IN_INVALID_PACKET = 0x44, /**< Indicates a wrong LMP packet is received */

   BTS_STATUS_CODE_FAULT_HOSTIO_FROZEN_HANDLES = 0x58, /**< UGXZE-104A/UGXZE-304A: All available HCI handles are currently in use. This can result if a large number of recently-closed ACL links are not yet cleaned-up. */

   BTS_STATUS_CODE_DSP_LOAD_SEQUENCING_ERROR = 0x76, /**< UGXZE-104A: A Sequencing error has occurred during the DSP loading process. */

   BTS_STATUS_CODE_SUCCESS_SLAVE = 0x80, /**< EVTRES_SUCCESS_SLAVE - Connected as slave role. In general no special action will be required. */
   BTS_STATUS_CODE_SERVICE_INCOMING = 0x81, /**< EVTRES_SERVICE_INCOMING - Same address with same service is under incoming connection. Please wait for the service incoming connection complete. */
   BTS_STATUS_CODE_INSTANCE_COLLISION = 0x82, /**< EVTRES_INSTANCE_COLLISION */
   BTS_STATUS_CODE_CONNECT_COLLISION = 0x83, /**< EVTRES_CONNECT_COLLISION */

   BTS_STATUS_CODE_START_OTHER_CODES = 0xA1, /**< start of other codes */
   BTS_STATUS_CODE_INVALID_ARGUMENTS, /**<  Invalid arguments in method call. Please check parameter and resend the method call. */
   BTS_STATUS_CODE_IN_PROGRESS, /**<  Operation already in progress. Please wait a moment and try again. */
   BTS_STATUS_CODE_ALREADY_EXISTS, /**<  Already Exists. Please check whether the object has been exist. */
   BTS_STATUS_CODE_NOT_SUPPORTED, /**<  Operation is not supported. Please check whether this operation can be done in this situation. */
   BTS_STATUS_CODE_NOT_CONNECTED, /**<  Not Connected. Please check whether the service has not been connected. */
   BTS_STATUS_CODE_ALREADY_CONNECTED, /**<  Already Connected. Please check whether the service has already been connected. */
   BTS_STATUS_CODE_NOT_AVAILABLE, /**<  Operation currently not available. Please check whether this operation can be done in this situation. */
   BTS_STATUS_CODE_NOT_AUTHORIZED, /**<  Operation Not Authorized. Please check whether this operation can be done in this situation. */
   BTS_STATUS_CODE_AGENT_NOT_AVAILABLE, /**<  Agent Not Available. Please check whether this Agent has been registered. */
   BTS_STATUS_CODE_NOT_READY, /**<  Resource Not Ready. Please wait a moment and try again. */
   BTS_STATUS_CODE_FAILED, /**<  Operation failed. Please check the remote device state and try again. */
   BTS_STATUS_CODE_INVALID_FORMAT, /**<  Argument format is not recognized. Please check parameter format and resend the method call. */
   BTS_STATUS_CODE_NOT_IMPLEMENTED, /**<  Implementation not provided. Please check whether this operation is permitted. */
   BTS_STATUS_CODE_NOT_FOUND, /**<  Object is not found or not valid for this operation. Please check whether the object is exist and try again. */
   BTS_STATUS_CODE_NOT_ACTIVE, /**<  Operation is not active or in progress. Please wait a moment and try again. */
   BTS_STATUS_CODE_TIMED_OUT, /**<  Operation failure due to timeout. Please check the remote device state and try again. */
   BTS_STATUS_CODE_IN_USE, /**<  The resource is currently in use. Please wait a moment and try again. */
   BTS_STATUS_CODE_FORBIDDEN, /**<  Operation is forbidden. Please check whether this operation is permitted. */
   BTS_STATUS_CODE_CONNECT_FAILED, /**<  ... */
   BTS_STATUS_CODE_DOES_NOT_EXIST, /**<  ... */
   BTS_STATUS_CODE_CONNECTION_LOST, /**<  connection lost during pairing */
   BTS_STATUS_CODE_AUTHENTICATION_ERROR, /**< authentication error during pairing */
   BTS_STATUS_CODE_PAIRING_CANCELLED, /**< pairing canceled */
   BTS_STATUS_CODE_UNKNOWN_REASON, /**< unknown reason for pairing failed */
   BTS_STATUS_CODE_RETRY_ABORTED, /**< any internal retry action was aborted (internal status code) */
   // -------------------------------------------------------------------------
   BTS_STATUS_CODE_LAST /**< end marker, must be the last one */
};

/**
 * Type definition for disconnect reason.
 */
enum BTSBtApplDisconnectReason
{
   BTS_BT_APPL_DISCONNECT_REASON_LOCAL_DEVICE = 0, /**< disconnected by local device */
   BTS_BT_APPL_DISCONNECT_REASON_REMOTE_DEVICE, /**< disconnected by remote device */
   BTS_BT_APPL_DISCONNECT_REASON_EXCEPTION, /**< disconnected by exception */
   BTS_BT_APPL_DISCONNECT_REASON_LINK_LOSS, /**< disconnected by link loss */
   // -------------------------------------------------------------------------
   BTS_BT_APPL_DISCONNECT_REASON_LAST /**< end marker, must be the last one */
};

/**
 * Type definition for service state.
 */
enum BTSServiceState
{
   BTS_SERVICE_STATE_DISCONNECTED = 0, /**< disconnected */
   BTS_SERVICE_STATE_CONNECTING, /**< connecting */
   BTS_SERVICE_STATE_CONNECTED, /**< connected */
   BTS_SERVICE_STATE_DISCONNECTING, /**< disconnecting */
   BTS_SERVICE_STATE_ERROR, /**< error */
   // -------------------------------------------------------------------------
   BTS_SERVICE_STATE_LAST /**< end marker, must be the last one */
};

/**
 * Type definition for debug trace level (bit definition).
 */
enum BTSDebugTraceLevelBit
{
   BTS_DEBUG_TRACE_LEVEL_FATAL = 0, /**< Bit0 – FATAL: system is unusable */
   BTS_DEBUG_TRACE_LEVEL_ERROR = 1, /**< Bit1 – ERROR: error conditions */
   BTS_DEBUG_TRACE_LEVEL_WARN = 2, /**< Bit2 – WARN: warning conditions */
   BTS_DEBUG_TRACE_LEVEL_INFO = 3, /**< Bit3 – INFO: informational */
   BTS_DEBUG_TRACE_LEVEL_DEBUG = 4, /**< Bit4 – DEBUG: debug-level messages */
   BTS_DEBUG_TRACE_LEVEL_LOG = 5, /**< Bit5 – LOG: normal messages */
   BTS_DEBUG_TRACE_LEVEL_RESERVED = 6, /**< Bit6 – 0: reserved for future use */
   BTS_DEBUG_TRACE_LEVEL_TIME = 7, /**< Bit7 – TIME: need time information */
   // -------------------------------------------------------------------------
   BTS_DEBUG_TRACE_LEVEL_LAST /**< end marker, must be the last one */
};

/**
 * Type definition for debug trace mode.
 */
enum BTSDebugTraceMode
{
   BTS_DEBUG_TRACE_MODE_DISABLE = 0, /**< 0x00 – Disable: do not output log */
   BTS_DEBUG_TRACE_MODE_AUTO, /**< 0x01 – Auto: to specify log categories by module */
   BTS_DEBUG_TRACE_MODE_MANUAL, /**< 0x02 – Manual: to specify log category by host */
   // -------------------------------------------------------------------------
   BTS_DEBUG_TRACE_MODE_LAST /**< end marker, must be the last one */
};

/**
 * Type definition for debug trace category of Bluetooth application (bit definition). To specify category while in manual mode. Set bit as 0/1 to enable/disable the corresponding trace.
 */
enum BTSDebugBtAppCategoryBit
{
   BTS_DEBUG_BT_APP_CAT_GAP = 0, /**< Bit0 - GAP */
   BTS_DEBUG_BT_APP_CAT_CM_DM = 1, /**< Bit1 - CM_DM */
   BTS_DEBUG_BT_APP_CAT_SPP = 2, /**< Bit2 - SPP */
   BTS_DEBUG_BT_APP_CAT_HFP = 3, /**< Bit3 - HFP */
   BTS_DEBUG_BT_APP_CAT_AVP = 4, /**< Bit4 - AVP */
   BTS_DEBUG_BT_APP_CAT_AVP_RESERVED = 5, /**< Bit5 - AVP Reserved */
   BTS_DEBUG_BT_APP_CAT_PBDL_HFP = 6, /**< Bit6 - PBDL_HFP */
   BTS_DEBUG_BT_APP_CAT_PBDL_SPP = 7, /**< Bit7 - PBDL_SPP */
   BTS_DEBUG_BT_APP_CAT_PBDL_PBAP = 8, /**< Bit8 - PBDL_PBAP */
   BTS_DEBUG_BT_APP_CAT_PBDL_SYNCML = 9, /**< Bit9 - PBDL_SYNCML */
   BTS_DEBUG_BT_APP_CAT_MSG_HFP = 10, /**< Bit10 - MSG_HFP */
   BTS_DEBUG_BT_APP_CAT_MSG_MAP = 11, /**< Bit11 - MSG_MAP */
   BTS_DEBUG_BT_APP_CAT_MSG_SPP = 12, /**< Bit12 - MSG_SPP */
   BTS_DEBUG_BT_APP_CAT_DUN = 13, /**< Bit13 - DUN */
   BTS_DEBUG_BT_APP_CAT_OPPS = 14, /**< Bit14 - OPPS */
   BTS_DEBUG_BT_APP_CAT_OPPC = 15, /**< Bit15 - OPPC */
   BTS_DEBUG_BT_APP_CAT_BIPR = 16, /**< Bit16 - BIPR */
   BTS_DEBUG_BT_APP_CAT_BIPI = 17, /**< Bit17 - BIPI */
   BTS_DEBUG_BT_APP_CAT_PAN = 18, /**< Bit18 - PAN */
   BTS_DEBUG_BT_APP_CAT_DI = 19, /**< Bit19 - DI */
   BTS_DEBUG_BT_APP_CAT_HS = 20, /**< Bit20 - HS */
   BTS_DEBUG_BT_APP_CAT_MISC = 21, /**< Bit21 - MISC */
   BTS_DEBUG_BT_APP_CAT_SAP = 22, /**< Bit22 - SAP */
   // -------------------------------------------------------------------------
   BTS_DEBUG_BT_APP_CAT_LAST /**< end marker, must be the last one */
};

/**
 * Type definition for debug trace category of Bluetooth stack (bit definition).
 */
enum BTSDebugBtStackCategoryBit
{
   BTS_DEBUG_BT_STACK_CAT_HCI = 0, /**< Bit0 - HCI */
   BTS_DEBUG_BT_STACK_CAT_L2CAP = 1, /**< Bit1 - L2CAP */
   BTS_DEBUG_BT_STACK_CAT_RFCOMM = 2, /**< Bit2 - RFCOMM */
   BTS_DEBUG_BT_STACK_CAT_OBEX = 3, /**< Bit3 - OBEX */
   BTS_DEBUG_BT_STACK_CAT_SDP = 4, /**< Bit4 - SDP */
   BTS_DEBUG_BT_STACK_CAT_GAP = 5, /**< Bit5 - GAP */
   BTS_DEBUG_BT_STACK_CAT_SPP = 6, /**< Bit6 - SPP */
   BTS_DEBUG_BT_STACK_CAT_BTMNG = 7, /**< Bit7 - BTMNG */
   BTS_DEBUG_BT_STACK_CAT_SEC = 8, /**< Bit8 - SEC */
   BTS_DEBUG_BT_STACK_CAT_AVRCP = 9, /**< Bit9 - AVRCP */
   BTS_DEBUG_BT_STACK_CAT_A2DP = 10, /**< Bit10 - A2DP */
   BTS_DEBUG_BT_STACK_CAT_DUN = 11, /**< Bit11 - DUN */
   BTS_DEBUG_BT_STACK_CAT_OPP = 12, /**< Bit12 - OPP */
   BTS_DEBUG_BT_STACK_CAT_HANDSFREE = 13, /**< Bit13 - HANDSFREE */
   BTS_DEBUG_BT_STACK_CAT_BIP = 14, /**< Bit14 - BIP */
   BTS_DEBUG_BT_STACK_CAT_PBAP = 15, /**< Bit15 - PBAP */
   BTS_DEBUG_BT_STACK_CAT_3GPP = 16, /**< Bit16 - 3GPP */
   BTS_DEBUG_BT_STACK_CAT_SYNCML = 17, /**< Bit17 - SYNCML */
   BTS_DEBUG_BT_STACK_CAT_MAP = 18, /**< Bit18 - MAP */
   BTS_DEBUG_BT_STACK_CAT_BNEP = 19, /**< Bit19 - BNEP */
   BTS_DEBUG_BT_STACK_CAT_PAN = 20, /**< Bit20 - PAN */
   BTS_DEBUG_BT_STACK_CAT_DI = 21, /**< Bit21 - DI */
   BTS_DEBUG_BT_STACK_CAT_A2MP = 22, /**< Bit22 - A2MP */
   BTS_DEBUG_BT_STACK_CAT_LAYER_SPP_TRANS = 23, /**< Bit23 - LAYER_SPP_TRANS */
   BTS_DEBUG_BT_STACK_CAT_BTCOMMON = 24, /**< Bit24 - BTCOMMON */
   BTS_DEBUG_BT_STACK_CAT_SAP = 25, /**< Bit25 - SAP */
   // -------------------------------------------------------------------------
   BTS_DEBUG_BT_STACK_CAT_LAST /**< end marker, must be the last one */
};

/**
 * Type definition for debug trace filter (bit definition).
 */
enum BTSDebugTraceFilterBit
{
   BTS_DEBUG_TRACE_FILTER_HCI_CMD = 0, /**< Bit0 - HCI command packet */
   BTS_DEBUG_TRACE_FILTER_ACL = 1, /**< Bit1 - ACL packet, AV music data is controlled by bit10 */
   BTS_DEBUG_TRACE_FILTER_SCO = 2, /**< Bit2 - SCO data packet(TBD) */
   BTS_DEBUG_TRACE_FILTER_HCI_EVT = 3, /**< Bit3 - HCI event packet */
   BTS_DEBUG_TRACE_FILTER_L2CAP = 4, /**< Bit4 - L2CAP signaling information */
   BTS_DEBUG_TRACE_FILTER_RFCOMM_SABM = 5, /**< Bit5 - RFCOMM SABM/DISC/UA/DM/UIH(DLCI0) */
   BTS_DEBUG_TRACE_FILTER_RFCOMM_UIH = 6, /**< Bit6 - RFCOMM UIH(except DLCI0) */
   BTS_DEBUG_TRACE_FILTER_AVDTP = 7, /**< Bit7 - AVDTP Signaling */
   BTS_DEBUG_TRACE_FILTER_AVCTP = 8, /**< Bit8 - AVCTP */
   BTS_DEBUG_TRACE_FILTER_BNEP = 9, /**< Bit9 - BNEP control packet. */
   BTS_DEBUG_TRACE_FILTER_AV = 10, /**< Bit10 - AV music data in ACL packet. */
   // -------------------------------------------------------------------------
   BTS_DEBUG_TRACE_FILTER_LAST /**< end marker, must be the last one */
};

/**
 * Type definition for debug trace direction.
 */
enum BTSDebugTraceDirection
{
   BTS_DEBUG_TRACE_DIRECTION_RECEIVE = 0, /**< receive direction */
   BTS_DEBUG_TRACE_DIRECTION_SEND, /**< send direction */
   // -------------------------------------------------------------------------
   BTS_DEBUG_TRACE_DIRECTION_LAST /**< end marker, must be the last one */
};

/**
 * Type definition for debug trace category.
 */
enum BTSDebugTraceCategory
{
   BTS_DEBUG_TRACE_CAT_GAP_APP = 0, /**< GAP */
   BTS_DEBUG_TRACE_CAT_CM_DM, /**< CM_DM */
   BTS_DEBUG_TRACE_CAT_SPP_APP, /**< SPP */
   BTS_DEBUG_TRACE_CAT_HFP, /**< HFP */
   BTS_DEBUG_TRACE_CAT_AVP, /**< AVP */
   BTS_DEBUG_TRACE_CAT_AVP_RESERVED, /**< AVP Reserved */
   BTS_DEBUG_TRACE_CAT_PBDL_HFP, /**< PBDL_HFP */
   BTS_DEBUG_TRACE_CAT_PBDL_SPP, /**< PBDL_SPP */
   BTS_DEBUG_TRACE_CAT_PBDL_PBAP, /**< PBDL_PBAP */
   BTS_DEBUG_TRACE_CAT_PBDL_SYNCML, /**< PBDL_SYNCML */
   BTS_DEBUG_TRACE_CAT_MSG_HFP, /**< MSG_HFP */
   BTS_DEBUG_TRACE_CAT_MSG_MAP, /**< MSG_MAP */
   BTS_DEBUG_TRACE_CAT_MSG_SPP, /**< MSG_SPP */
   BTS_DEBUG_TRACE_CAT_DUN_APP, /**< DUN */
   BTS_DEBUG_TRACE_CAT_OPPS, /**< OPPS */
   BTS_DEBUG_TRACE_CAT_OPPC, /**< OPPC */
   BTS_DEBUG_TRACE_CAT_BIPR, /**< BIPR */
   BTS_DEBUG_TRACE_CAT_BIPI, /**< BIPI */
   BTS_DEBUG_TRACE_CAT_PAN_APP, /**< PAN */
   BTS_DEBUG_TRACE_CAT_DI_APP, /**< DI */
   BTS_DEBUG_TRACE_CAT_HS, /**< HS */
   BTS_DEBUG_TRACE_CAT_MISC, /**< MISC */
   BTS_DEBUG_TRACE_CAT_SAP_APP, /**< SAP */
   BTS_DEBUG_TRACE_CAT_HCI = 0x20, /**< HCI */
   BTS_DEBUG_TRACE_CAT_L2CAP, /**< L2CAP */
   BTS_DEBUG_TRACE_CAT_RFCOMM, /**< RFCOMM */
   BTS_DEBUG_TRACE_CAT_OBEX, /**< OBEX */
   BTS_DEBUG_TRACE_CAT_SDP, /**< SDP */
   BTS_DEBUG_TRACE_CAT_GAP_STACK, /**< GAP */
   BTS_DEBUG_TRACE_CAT_SPP_STACK, /**< SPP */
   BTS_DEBUG_TRACE_CAT_BTMNG, /**< BTMNG */
   BTS_DEBUG_TRACE_CAT_SEC, /**< SEC */
   BTS_DEBUG_TRACE_CAT_AVRCP, /**< AVRCP */
   BTS_DEBUG_TRACE_CAT_A2DP, /**< A2DP */
   BTS_DEBUG_TRACE_CAT_DUN_STACK, /**< DUN */
   BTS_DEBUG_TRACE_CAT_OPP, /**< OPP */
   BTS_DEBUG_TRACE_CAT_HANDSFREE, /**< HANDSFREE */
   BTS_DEBUG_TRACE_CAT_BIP, /**< BIP */
   BTS_DEBUG_TRACE_CAT_PBAP, /**< PBAP */
   BTS_DEBUG_TRACE_CAT_3GPP, /**< 3GPP */
   BTS_DEBUG_TRACE_CAT_SYNCML, /**< SYNCML */
   BTS_DEBUG_TRACE_CAT_MAP, /**< MAP */
   BTS_DEBUG_TRACE_CAT_BNEP, /**< BNEP */
   BTS_DEBUG_TRACE_CAT_PAN_STACK, /**< PAN */
   BTS_DEBUG_TRACE_CAT_DI_STACK, /**< DI */
   BTS_DEBUG_TRACE_CAT_A2MP, /**< A2MP */
   BTS_DEBUG_TRACE_CAT_LAYER_SPP_TRANS, /**< LAYER_SPP_TRANS */
   BTS_DEBUG_TRACE_CAT_BTCOMMON, /**< BTCOMMON */
   BTS_DEBUG_TRACE_CAT_SAP_STACK, /**< SAP */
   // -------------------------------------------------------------------------
   BTS_DEBUG_TRACE_CAT_LAST /**< end marker, must be the last one */
};

/**
 * Type definition for DUN flow control.
 */
enum BTSDunFlowControl
{
   BTS_DUN_FLOW_CONTROL_GO = 0, /**< go */
   BTS_DUN_FLOW_CONTROL_STOP, /**< stop */
   // -------------------------------------------------------------------------
   BTS_DUN_FLOW_CONTROL_LAST /**< end marker, must be the last one */
};

/**
 * Type definition for SPP flow control.
 */
enum BTSSppFlowControl
{
   BTS_SPP_FLOW_CONTROL_GO = 0, /**< go */
   BTS_SPP_FLOW_CONTROL_STOP, /**< stop */
   // -------------------------------------------------------------------------
   BTS_SPP_FLOW_CONTROL_LAST /**< end marker, must be the last one */
};

/**
 * Type definition for transfer status.
 */
enum BTSTransferStatus
{
   BTS_TRANSFER_STATUS_QUEUED = 0, /**< queued */
   BTS_TRANSFER_STATUS_ACTIVE, /**< active */
   BTS_TRANSFER_STATUS_COMPLETE, /**< complete */
   BTS_TRANSFER_STATUS_ERROR, /**< error */
   // -------------------------------------------------------------------------
   BTS_TRANSFER_STATUS_LAST /**< end marker, must be the last one */
};

/**
 * Type definition for interface/component/service.
 */
enum BTSGenDbusServiceInterface
{
   BTS_GEN_DBUS_SERVICE_OBJECT_MANAGER_SYSTEM = 0, /**< object manager system bus */
   BTS_GEN_DBUS_SERVICE_OBJECT_MANAGER_SESSION, /**< object manager session bus */
   BTS_GEN_DBUS_SERVICE_INTROSPECTABLE, /**< introspectable */
   BTS_GEN_DBUS_SERVICE_PROPERTIES_GET_ALL, /**< properties get all */
   BTS_GEN_DBUS_SERVICE_GEN_INITIALIZATION, /**< virtual service for adapter initialization (GENIVI part) */
   BTS_GEN_DBUS_SERVICE_HLI_INITIALIZATION, /**< virtual service for adapter initialization (HLI part) */
   BTS_GEN_DBUS_SERVICE_ADAPTER, /**< adapter */
   BTS_GEN_DBUS_SERVICE_AGENT, /**< agent */
   BTS_GEN_DBUS_SERVICE_AGENT_MANAGER, /**< agent manager */
   BTS_GEN_DBUS_SERVICE_DEVICE, /**< device */
   BTS_GEN_DBUS_SERVICE_DM, /**< DM */
   BTS_GEN_DBUS_SERVICE_SERVICE, /**< service */
   BTS_GEN_DBUS_SERVICE_TRACE, /**< trace */
   BTS_GEN_DBUS_SERVICE_GENIVI_TRACE, /**< genivi trace */
   BTS_GEN_DBUS_SERVICE_DID, /**< DID */
   BTS_GEN_DBUS_SERVICE_SPP, /**< SPP */
   BTS_GEN_DBUS_SERVICE_OBEX_CLIENT, /**< obex client */
   BTS_GEN_DBUS_SERVICE_SERIAL, /**< serial */
   BTS_GEN_DBUS_SERVICE_CONNMAN_MANAGER, /**< ConnMan: Manager */
   BTS_GEN_DBUS_SERVICE_CONNMAN_TECHNOLOGY, /**< ConnMan: Technology */
   BTS_GEN_DBUS_SERVICE_CONNMAN_SERVICE, /**< ConnMan: Service */
   BTS_GEN_DBUS_SERVICE_HFP_HANDSFREE, /**< HFP: Handsfree */
   BTS_GEN_DBUS_SERVICE_HFP_MANAGER, /**< HFP: Manager */
   BTS_GEN_DBUS_SERVICE_HFP_MODEM, /**< HFP: Modem */
   BTS_GEN_DBUS_SERVICE_BLE_OBJECT_MANAGER_SYSTEM, /**< BLE: object manager system bus */
   BTS_GEN_DBUS_SERVICE_BLE_PROPERTIES_GET_ALL, /**< BLE: properties get all */
   BTS_GEN_DBUS_SERVICE_BLE_GEN_INITIALIZATION, /**< BLE: virtual service for adapter initialization (GENIVI part) */
   BTS_GEN_DBUS_SERVICE_BLE_HLI_INITIALIZATION, /**< BLE: virtual service for adapter initialization (HLI part) */
   BTS_GEN_DBUS_SERVICE_BLE_ADAPTER, /**< BLE: adapter */
   BTS_GEN_DBUS_SERVICE_BLE_DEVICE, /**< BLE: device */
   BTS_GEN_DBUS_SERVICE_BLE_SEAMLESS_PAIRING, /**< BLE: seamless pairing */
   BTS_GEN_DBUS_SERVICE_DISABLED_BT_STACK_COMM, /**< disabled BT stack communication */
   // -------------------------------------------------------------------------
   BTS_GEN_DBUS_SERVICE_LAST /**< end marker, must be the last one */
};

/**
 * Type definition for configuration IDs.
 */
enum BTSEvoConfigIdEnum
{
   // BT setup block
   BTS_EVO_CONFIG_ID_inquiry_item = 0x0304, /**< Configure how the remote device info will be notified to the host. */
   BTS_EVO_CONFIG_ID_page_timeout = 0x0360, /**< Page timeout. */
   BTS_EVO_CONFIG_ID_page_timeout_2nd = 0x0369, /**< Page timeout second device. */

   // TODO: add remaining ones

   // -------------------------------------------------------------------------
   BTS_EVO_CONFIG_ID_LAST /**< end marker, must be the last one */
};

/**
 * Type definition for function and service bits (profiles/services).
 */
enum BTSServiceFunctionBit
{
   BTS_SRV_FUNC_SPP = 0, /**< SPP */
   BTS_SRV_FUNC_HFP = 1, /**< HFP */
   BTS_SRV_FUNC_AVP = 2, /**< AVP */
   BTS_SRV_FUNC_A2DP = 3, /**< A2DP */
   BTS_SRV_FUNC_AVRCP = 4, /**< AVRCP */
   BTS_SRV_FUNC_PIM = 5, /**< PIM */
   BTS_SRV_FUNC_PIM_HFP = 6, /**< PIM HFP */
   BTS_SRV_FUNC_PIM_SPP = 7, /**< PIM SPP */
   BTS_SRV_FUNC_PIM_PBAP = 8, /**< PIM PBAP */
   BTS_SRV_FUNC_PIM_SYNCML = 9, /**< PIM SyncML */
   BTS_SRV_FUNC_MSG = 10, /**< MSG */
   BTS_SRV_FUNC_MSG_HFP = 11, /**< MSG HFP */
   BTS_SRV_FUNC_MSG_MAP = 12, /**< MSG MAP */
   BTS_SRV_FUNC_MSG_SPP = 13, /**< MSG SPP */
   BTS_SRV_FUNC_DUN = 14, /**< DUN */
   BTS_SRV_FUNC_OPPS = 15, /**< OPPS */
   BTS_SRV_FUNC_OPPC = 16, /**< OPPC */
   BTS_SRV_FUNC_BIPR = 17, /**< BIPR */
   BTS_SRV_FUNC_BIPI = 18, /**< BIPI */
   BTS_SRV_FUNC_PANU = 19, /**< PANU */
   BTS_SRV_FUNC_DID = 20, /**< DID */
   BTS_SRV_FUNC_SAP = 21, /**< SAP */
   BTS_SRV_FUNC_CTN = 22, /**< CTN */
   BTS_SRV_FUNC_CTN1 = 23, /**< CTN */
   BTS_SRV_FUNC_CTN2 = 24, /**< CTN */
   BTS_SRV_FUNC_SYNCML = 25, /**< SyncML */
   BTS_SRV_FUNC_AVP_COVERART = 27, /**< AVP COVERART (BIPI) */
   // -------------------------------------------------------------------------
   BTS_SRV_FUNC_LAST /**< end marker, must be the last one */
};

/**
 * Type definition for function and service bit representation.
 */
typedef BTSBitfield<uint64_t, BTSServiceFunctionBit, BTS_SRV_FUNC_LAST> BTSServiceFunction;

/**
 * Type definition for debug trace direction.
 */
enum BTSTestMode
{
   BTS_TEST_MODE_REMOTE_LOOPBACK = 1, /**< Remote Loopback Mode */
   BTS_TEST_MODE_READ_LINK_QUALITY = 4, /**< Read Link Quality Mode */
   // -------------------------------------------------------------------------
   BTS_TEST_MODE_LAST /**< end marker, must be the last one */
};

/**
 * Type definition for Bluetooth low energy device address type (BLE).
 */
enum BTSBLEDeviceAddressType
{
   BTS_BLE_DEV_ADDRESS_TYPE_PUBLIC = 0, /**< public address */
   BTS_BLE_DEV_ADDRESS_TYPE_RANDOM, /**< random address */
   // -------------------------------------------------------------------------
   BTS_BLE_DEV_ADDRESS_TYPE_LAST /**< end marker, must be the last one */
};

/**
 * Type definition for Bluetooth low energy discoverable mode (BLE).
 */
enum BTSBLEDiscoverableMode
{
   BTS_BLE_DISCOVERABLE_MODE_GENERAL = 0, /**< LE general discoverable mode (default) */
   BTS_BLE_DISCOVERABLE_MODE_LIMITED, /**< LE limited discoverable mode */
   // -------------------------------------------------------------------------
   BTS_BLE_DISCOVERABLE_MODE_LAST /**< end marker, must be the last one */
};

/**
 * Type definition for Bluetooth low energy status (BLE).
 */
enum BTSBLEStatus
{
   BTS_BLE_STATUS_SUCCESS = 0, /**< success */
   BTS_BLE_STATUS_TIMEOUT, /**< processing timeout */
   BTS_BLE_STATUS_NO_RESPONSE, /**< remote no response */
   BTS_BLE_STATUS_UNKNOWN = 255, /**< unknown reason */
   // -------------------------------------------------------------------------
   BTS_BLE_STATUS_LAST /**< end marker, must be the last one */
};

/**
 * Type definition for Evolution stack version.
 */
class BTSEvolutionVersion
{
public:
   BTSEvolutionVersion();
   BTSEvolutionVersion(IN const BTSEvolutionVersion& ref);
   BTSEvolutionVersion& operator=(IN const BTSEvolutionVersion& ref);
   bool operator==(IN const BTSEvolutionVersion& ref) const;
   bool operator!=(IN const BTSEvolutionVersion& ref) const;
   virtual ~BTSEvolutionVersion();
   // member with public access
   unsigned short int appCode;
   unsigned short int geniviVersion; // appVersion
   unsigned short int evolutionVersion; // coreVersion
};

} //genivi
} //btstackif

#endif //_EVOLUTION_GENIVI_STACK_TYPES_H_
