/**
 * @file SppConnectDisconnectAlpsEvolutionExt.cpp
 *
 * @par SW-Component
 * State machine for SPP connect/disconnect
 *
 * @brief Implementation of Alps Evolution Genivi SPP connect/disconnect state machine.
 *
 * @copyright (C) 2017 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Source file for implementation of Alps Evolution Genivi SPP connect/disconnect state machine.
 */

#include "SppConnectDisconnectAlpsEvolutionExt.h"
#include "IProtocolManagerCallback.h"
#include "ProtocolManagerData.h"
#include "IBasicControl.h"
#include "IDeviceManager.h"
#include "IObjectPathManagerGenivi.h"
#include "Bts2Ipc_MessageWrapper_GEN.h"
#include "Ipc2Bts_MessageWrapper_GEN.h"
#include "BtsUtils.h"
#include "FwErrmemPrint.h"
#include "TraceClasses.h"
#include "FwTrace.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_BTS_CONTROL
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/SppConnectDisconnectAlpsEvolutionExt.cpp.trc.h"
#endif
#endif

// if enabled connect and disconnect SMs shall be updated and tested
// #define ENABLE_BUSY
#undef ENABLE_BUSY

// if ALPS has introduced operation id then activate the define
// #define ENABLE_ALPS_OP_ID
#undef ENABLE_ALPS_OP_ID

namespace btstackif {
namespace genivi {
namespace alpsevolutionext {

SppConnectDisconnectAlpsEvolutionExt::SppConnectDisconnectAlpsEvolutionExt() :
_callback(0),
_controlIf(0),
_timerPoolIf(0),
_deviceManagerIf(0),
_objectPathManagerIf(0)
{
}

SppConnectDisconnectAlpsEvolutionExt::~SppConnectDisconnectAlpsEvolutionExt()
{
   _callback = 0;
   _controlIf = 0;
   _timerPoolIf = 0;
   _deviceManagerIf = 0;
   _objectPathManagerIf = 0;
}

void SppConnectDisconnectAlpsEvolutionExt::reset(void)
{
}

void SppConnectDisconnectAlpsEvolutionExt::setCallback(IN IProtocolManagerCallback* callback)
{
   _callback = callback;

   FW_ERRMEM_ASSERT(0 != _callback);
}

void SppConnectDisconnectAlpsEvolutionExt::setControlIf(IN IBasicControl* control)
{
   _controlIf = control;

   FW_ERRMEM_ASSERT(0 != _controlIf);
}

void SppConnectDisconnectAlpsEvolutionExt::setTimerPoolIf(IN ITimerPool* timerPool)
{
   _timerPoolIf = timerPool;

   FW_ERRMEM_ASSERT(0 != _timerPoolIf);
}

void SppConnectDisconnectAlpsEvolutionExt::prepareProtocolConnect(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName)
{
   FW_ERRMEM_IF_NULL_PTR_RETURN(_callback);

   // simulate some states: protocol added + status updated
   // DBUS service availability is not needed
   BTSHandleIpc2BtsMessageItem messageItem;
   _callback->protocolAdded(bts2IpcMsgList, bts2AppMsgList, messageItem, address, protocol, sppInstance, uuid, masInstance, masName);
   FW_ERRMEM_ASSERT(false == messageItem.deleteMessage);
   FW_ERRMEM_ASSERT(0 == messageItem.message);

   // initial status is always disconnected => no action required

   // check if all data is complete
   checkForDataComplete(bts2IpcMsgList, bts2AppMsgList, messageItem, *_callback, address, protocol, sppInstance, uuid, masInstance, masName);
   FW_ERRMEM_ASSERT(false == messageItem.deleteMessage);
   FW_ERRMEM_ASSERT(0 == messageItem.message);
}

void SppConnectDisconnectAlpsEvolutionExt::setDeviceAvailability(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName, IN const bool available)
{
   (void)(available);

   FW_ERRMEM_IF_NULL_PTR_RETURN(_callback);

   checkForDataComplete(bts2IpcMsgList, bts2AppMsgList, messageItem, *_callback, address, protocol, sppInstance, uuid, masInstance, masName);
}

bool SppConnectDisconnectAlpsEvolutionExt::isServiceAvailable(IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName) const
{
   (void)(address);
   (void)(protocol);
   (void)(sppInstance);
   (void)(uuid);
   (void)(masInstance);
   (void)(masName);

   // all needed DBUS services are available (DBUS server application is up and running)
   return true;
}

void SppConnectDisconnectAlpsEvolutionExt::waitForServiceAvailable(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName)
{
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);
   (void)(address);
   (void)(protocol);
   (void)(sppInstance);
   (void)(uuid);
   (void)(masInstance);
   (void)(masName);

   // isServiceAvailable() was answered with true, therefore this function shall never be called
   FW_ERRMEM_ASSERT_ALWAYS();
}

bool SppConnectDisconnectAlpsEvolutionExt::isServiceSearchNeeded(OUT BTSSearchType& searchType, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName) const
{
   (void)(searchType);
   (void)(address);
   (void)(protocol);
   (void)(sppInstance);
   (void)(uuid);
   (void)(masInstance);
   (void)(masName);

   // service search shall be triggered by BtStackIf client before initiating any SPP connect
   return false;
}

bool SppConnectDisconnectAlpsEvolutionExt::isSettingUuidNeeded(IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName) const
{
   (void)(address);
   (void)(protocol);
   (void)(sppInstance);
   (void)(uuid);
   (void)(masInstance);
   (void)(masName);

   // needed
   return true;
}

void SppConnectDisconnectAlpsEvolutionExt::waitForProtocolAvailable(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName)
{
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);
   (void)(address);
   (void)(protocol);
   (void)(sppInstance);
   (void)(uuid);
   (void)(masInstance);
   (void)(masName);

   // not necessary
   FW_ERRMEM_ASSERT_ALWAYS();
}

void SppConnectDisconnectAlpsEvolutionExt::connect(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName)
{
   (void)(bts2AppMsgList);
   (void)(uuid);
   (void)(masInstance);
   (void)(masName);

   createBtApplDeviceConnectRequest(bts2IpcMsgList, address, protocol, sppInstance);
}

void SppConnectDisconnectAlpsEvolutionExt::disconnect(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName, IN const bool pauseBtStreaming)
{
   (void)(bts2AppMsgList);
   (void)(uuid);
   (void)(masInstance);
   (void)(masName);
   (void)(pauseBtStreaming);

   createBtApplDeviceDisconnectRequest(bts2IpcMsgList, address, protocol, sppInstance);
}

void SppConnectDisconnectAlpsEvolutionExt::cancel(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName)
{
   (void)(bts2AppMsgList);
   (void)(uuid);
   (void)(masInstance);
   (void)(masName);

   // sending of disconnect during ongoing connect disabled because extension of SM is necessary
   createBtApplDeviceDisconnectRequest(bts2IpcMsgList, address, protocol, sppInstance);
}

void SppConnectDisconnectAlpsEvolutionExt::accept(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName, IN const bool accept)
{
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);
   (void)(address);
   (void)(protocol);
   (void)(sppInstance);
   (void)(uuid);
   (void)(masInstance);
   (void)(masName);
   (void)(accept);

   // no remote connect for SPP at the moment
   FW_ERRMEM_ASSERT_ALWAYS();
}

void SppConnectDisconnectAlpsEvolutionExt::sendVirtualFailedConnectResult(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName, IN const BTSIpcCommonErrorCode errorCode) const
{
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);

   FW_ERRMEM_IF_NULL_PTR_RETURN(_callback);
   FW_ERRMEM_IF_NULL_PTR_RETURN(_controlIf);

   BTSOperationId opId(0);
   if(false == getOperationId(opId, *_callback, address, protocol, sppInstance, uuid, masInstance, masName))
   {
      FW_ERRMEM_ASSERT_ALWAYS();
   }

   Ipc2Bts_BtApplDeviceConnectCfm* msg = ptrNew_Ipc2Bts_BtApplDeviceConnectCfm();
   if(0 != msg)
   {
      msg->setOpId(opId);
      if(BTS_IPC_RETRY_ABORTED == errorCode)
      {
         msg->setStatus(BTS_STATUS_CODE_RETRY_ABORTED);
      }
      else if(BTS_IPC_SUCCESS == errorCode)
      {
         msg->setStatus(BTS_STATUS_CODE_SUCCESS);
      }
      else
      {
         msg->setStatus(BTS_STATUS_CODE_OPERATION_FAIL);
      }
      msg->setIpcCommonErrorCode(BTS_IPC_SUCCESS);
   }

   _controlIf->sendInternalIpc2BtsMessage(msg, true);
}

void SppConnectDisconnectAlpsEvolutionExt::sendVirtualFailedDisconnectResult(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName, IN const BTSIpcCommonErrorCode errorCode) const
{
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);

   FW_ERRMEM_IF_NULL_PTR_RETURN(_callback);
   FW_ERRMEM_IF_NULL_PTR_RETURN(_controlIf);

   BTSOperationId opId(0);
   if(false == getOperationId(opId, *_callback, address, protocol, sppInstance, uuid, masInstance, masName))
   {
      FW_ERRMEM_ASSERT_ALWAYS();
   }

   Ipc2Bts_BtApplDeviceDisconnectCfm* msg = ptrNew_Ipc2Bts_BtApplDeviceDisconnectCfm();
   if(0 != msg)
   {
      msg->setOpId(opId);
      if(BTS_IPC_RETRY_ABORTED == errorCode)
      {
         msg->setStatus(BTS_STATUS_CODE_RETRY_ABORTED);
      }
      else if(BTS_IPC_SUCCESS == errorCode)
      {
         msg->setStatus(BTS_STATUS_CODE_SUCCESS);
      }
      else
      {
         msg->setStatus(BTS_STATUS_CODE_OPERATION_FAIL);
      }
      msg->setIpcCommonErrorCode(BTS_IPC_SUCCESS);
   }

   _controlIf->sendInternalIpc2BtsMessage(msg, true);
}

void SppConnectDisconnectAlpsEvolutionExt::sendVirtualConnectedUpdate(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName, IN const bool connected, IN const BTSIpcCommonErrorCode errorCode)
{
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);

   FW_ERRMEM_IF_NULL_PTR_RETURN(_callback);
   FW_ERRMEM_IF_NULL_PTR_RETURN(_controlIf);

   BTSOperationId opId(0);
   if(false == getOperationId(opId, *_callback, address, protocol, sppInstance, uuid, masInstance, masName))
   {
      // store operation id
      opId = 0;
      setOperationId(*_callback, address, protocol, sppInstance, uuid, masInstance, masName, opId);
   }

   if(true == connected)
   {
      Ipc2Bts_BtApplDeviceConnectInd* msg = ptrNew_Ipc2Bts_BtApplDeviceConnectInd();
      if(0 != msg)
      {
         msg->setStatus(BTS_STATUS_CODE_SUCCESS);
         BTSServiceFunction serviceFunction;
         serviceFunction.setBit(BTS_SRV_FUNC_SPP);
         msg->setServiceFunction(serviceFunction);
         msg->setInstance(sppInstance);
         msg->setBDAddress(address);
         msg->setIpcCommonErrorCode(errorCode);
      }

      Ipc2Bts_BtApplDeviceConnectCompInd* msg2 = ptrNew_Ipc2Bts_BtApplDeviceConnectCompInd();
      if(0 != msg2)
      {
         msg2->setOpId(opId);
      }

      _controlIf->sendInternalIpc2BtsMessage(msg, true);
      _controlIf->sendInternalIpc2BtsMessage(msg2, true);
   }
   else
   {
      Ipc2Bts_BtApplDeviceDisconnectInd* msg = ptrNew_Ipc2Bts_BtApplDeviceDisconnectInd();
      if(0 != msg)
      {
         msg->setStatus(BTS_STATUS_CODE_SUCCESS);
         msg->setReason(BTS_BT_APPL_DISCONNECT_REASON_LOCAL_DEVICE);
         BTSServiceFunction serviceFunction;
         serviceFunction.setBit(BTS_SRV_FUNC_SPP);
         msg->setServiceFunction(serviceFunction);
         msg->setInstance(sppInstance);
         msg->setBDAddress(address);
         msg->setIpcCommonErrorCode(errorCode);
      }

      Ipc2Bts_BtApplDeviceDisconnectCompInd* msg2 = ptrNew_Ipc2Bts_BtApplDeviceDisconnectCompInd();
      if(0 != msg2)
      {
         msg2->setOpId(opId);
      }

      _controlIf->sendInternalIpc2BtsMessage(msg, true);
      _controlIf->sendInternalIpc2BtsMessage(msg2, true);
   }
}

IProtocolManagerRequest* SppConnectDisconnectAlpsEvolutionExt::getRequestIf(void)
{
   return this;
}

void SppConnectDisconnectAlpsEvolutionExt::setDeviceManagerIf(IN IDeviceManager* deviceManagerIf)
{
   _deviceManagerIf = deviceManagerIf;

   FW_ERRMEM_ASSERT(0 != _deviceManagerIf);
}

void SppConnectDisconnectAlpsEvolutionExt::setObjectPathManagerIf(IN IObjectPathManagerGenivi* objectPathManager)
{
   _objectPathManagerIf = objectPathManager;

   FW_ERRMEM_ASSERT(0 != _objectPathManagerIf);
}

bool SppConnectDisconnectAlpsEvolutionExt::handleCancelRequest(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const bool sent, IN const act_t token)
{
   (void)(bts2AppMsgList);
   (void)(messageItem);

   /*
    * Cancel request will be triggered by Evolution in following scenarios:
    * - timeout AutorizeService: 10s => will be handled by failed connect
    */

   // but ensure that answer is sent
   if(false == sent)
   {
      createCancelRes(bts2IpcMsgList, token, true);
   }

   return true;
}

bool SppConnectDisconnectAlpsEvolutionExt::handleCancelRequestRequest(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSObjectPath& device, IN const bool sent, IN const act_t token)
{
   (void)(bts2AppMsgList);
   (void)(messageItem);
   (void)(device);

   /*
    * Cancel request will be triggered by Evolution in following scenarios:
    * - timeout AutorizeService: 10s => will be handled by failed connect
    */

   // but ensure that answer is sent
   if(false == sent)
   {
      createCancelRequestRes(bts2IpcMsgList, token, true);
   }

   return true;
}

bool SppConnectDisconnectAlpsEvolutionExt::handleRemoteConnect(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSObjectPath& device, IN const BTSProtocolId protocol, IN const BTSUuid& uuid, IN const act_t token)
{
   (void)(bts2AppMsgList);
   (void)(messageItem);
   (void)(uuid);

   FW_ERRMEM_IF_NULL_PTR_RETURN_FALSE(_callback);
   FW_ERRMEM_IF_NULL_PTR_RETURN_FALSE(_objectPathManagerIf);

   if(BTS_PROTO_SPP != protocol)
   {
      return false;
   }

   ETG_TRACE_USR3((" handleRemoteConnect: device=%s", device.c_str()));

   BTSBDAddress address;
   if(false == _objectPathManagerIf->getAddress4ObjectPath(address, device))
   {
      FW_ERRMEM_ASSERT_ALWAYS();
      return false;
   }

   // TODO: [low]: remote SPP connect feature is currently disabled; implement if needed
   FW_ERRMEM_ASSERT_ALWAYS();
   // intermediate solution
   createAuthorizeServiceResponseMsg(bts2IpcMsgList, address, token, BTS_CONFIRM_REJECT);

   return true;
}

void SppConnectDisconnectAlpsEvolutionExt::handleConnectResult(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId instance, IN const BTSOperationId opId, IN const BTSIpcCommonErrorCode errorCode)
{
   FW_ERRMEM_IF_NULL_PTR_RETURN(_callback);

   if((BTS_PROTO_SPP != protocol) && (BTS_PROTO_DUN != protocol) && (BTS_PROTO_PAN != protocol))
   {
      return;
   }

   if((BTS_PROTO_PAN == protocol) && (true == _callback->isPanViaConnMan()))
   {
      return;
   }

   ETG_TRACE_USR3((" handleConnectResult"));

   BTSUuid uuid;
   if(false == getUuid4Instance(uuid, address, protocol, instance, *_callback))
   {
      FW_ERRMEM_ASSERT_ALWAYS();
      return;
   }

   const BTSMasInstanceId masInstance(0);
   const BTSMasInstanceName masName;

   // store operation id
   setOperationId(*_callback, address, protocol, instance, uuid, masInstance, masName, opId);

#ifdef ENABLE_ALPS_OP_ID
   // new handling in Evolution stack
   if(BTS_IPC_SUCCESS == errorCode)
   {
      // wait for next message
   }
   else
   {
      BTSRequestResult result(BTS_REQ_FAILED);

      // check error code
      if(BTS_IPC_BUSY == errorCode)
      {
#ifdef ENABLE_BUSY
         result = BTS_REQ_BUSY;
#endif
      }
      else if(BTS_IPC_RETRY_ABORTED == errorCode)
      {
         result = BTS_REQ_CONNECT_ABORTED;
      }
      else
      {
         // other error codes
      }

      // forward connect result
      (void)_callback->connectResult(bts2IpcMsgList, bts2AppMsgList, messageItem, address, protocol, instance, uuid, masInstance, masName, result);
   }
#else
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);
   (void)(messageItem);
   (void)(errorCode);

   // old handling in Evolution stack
   /*
    * current stack behavior is as follows:
    * - either DBUS method return or DBUS error message is sent as answer
    * - DBUS signal message containing "confirmation" data is sent in both cases; this message indicates same error as answer messages
    * => answer can be ignored
    */
#endif
}

void SppConnectDisconnectAlpsEvolutionExt::handleConnectCfm(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSOperationId opId, IN const BTSStatusCode status)
{
   FW_ERRMEM_IF_NULL_PTR_RETURN(_callback);

   ETG_TRACE_USR3((" handleConnectCfm"));

   BTSBDAddress address;
   BTSProtocolId protocol(BTS_PROTO_LAST);
   BTSSppInstanceId instance(0);
   BTSUuid uuid;

   if(false == getProtocol4OperationId(address, protocol, instance, uuid, *_callback, opId))
   {
      FW_ERRMEM_ASSERT_ALWAYS();
      return;
   }

   if((BTS_PROTO_SPP != protocol) && (BTS_PROTO_DUN != protocol) && (BTS_PROTO_PAN != protocol))
   {
      FW_ERRMEM_ASSERT_ALWAYS();
      return;
   }

   if((BTS_PROTO_PAN == protocol) && (true == _callback->isPanViaConnMan()))
   {
      FW_ERRMEM_ASSERT_ALWAYS();
      return;
   }

   // check status
   if(BTS_STATUS_CODE_SUCCESS == status)
   {
      // wait for next message
   }
   else
   {
      const BTSMasInstanceId masInstance(0);
      const BTSMasInstanceName masName;
      BTSRequestResult result(BTS_REQ_FAILED);

      // check status code
      if(BTS_STATUS_CODE_SYSTEM_BUSY == status)
      {
#ifdef ENABLE_BUSY
         result = BTS_REQ_BUSY;
#endif
      }
      else if(BTS_STATUS_CODE_RETRY_ABORTED == status)
      {
         result = BTS_REQ_CONNECT_ABORTED;
      }
      else
      {
         // other status codes
      }

      // forward connect result
      (void)_callback->connectResult(bts2IpcMsgList, bts2AppMsgList, messageItem, address, protocol, instance, uuid, masInstance, masName, result);
   }
}

void SppConnectDisconnectAlpsEvolutionExt::handleDeviceNameInd(IN const BTSSppInstanceId instance, IN const BTSDeviceName& deviceName)
{
   FW_ERRMEM_IF_NULL_PTR_RETURN(_callback);

   ETG_TRACE_USR3((" handleDeviceNameInd"));

#ifdef ENABLE_ALPS_OP_ID
   const BTSSppInstanceId sppInstance(instance);
#else
   (void)(instance);

   // given SPP instance is invalid

   BTSSppInstanceId sppInstance(0);
   if(false == getSppInstance(sppInstance, *_callback))
   {
      // in case of PAN via ConnMan the function getSppInstance() will return false
      return;
   }
#endif

   BTSBDAddress address;
   BTSProtocolId protocol(BTS_PROTO_LAST);
   BTSUuid uuid;

   if(false == getProtocol4Instance(address, protocol, uuid, *_callback, sppInstance))
   {
      FW_ERRMEM_ASSERT_ALWAYS();
      return;
   }

   if((BTS_PROTO_SPP != protocol) && (BTS_PROTO_DUN != protocol) && (BTS_PROTO_PAN != protocol))
   {
      FW_ERRMEM_ASSERT_ALWAYS();
      return;
   }

   if((BTS_PROTO_PAN == protocol) && (true == _callback->isPanViaConnMan()))
   {
      FW_ERRMEM_ASSERT_ALWAYS();
      return;
   }

   const BTSMasInstanceId masInstance(0);
   const BTSMasInstanceName masName;

   // set device name
   BTSDeviceName virtualDeviceName;
   if((BTS_PROTO_SPP == protocol) ||
      (BTS_PROTO_DUN == protocol))
   {
      virtualDeviceName = "/dev/";
      virtualDeviceName += deviceName;
   }
   else
   {
      virtualDeviceName = deviceName;
   }

   // forward device name
   ::std::vector< Bts2Ipc_BaseMessage* > tmpBts2IpcMsgList;
   ::std::vector< Bts2App_BaseMessage* > tmpBts2AppMsgList;
    BTSHandleIpc2BtsMessageItem messageItem;
   _callback->updateVirtualDeviceName(tmpBts2IpcMsgList, tmpBts2AppMsgList, messageItem, address, protocol, instance, uuid, masInstance, masName, virtualDeviceName);
}

void SppConnectDisconnectAlpsEvolutionExt::updateConnectStatus(IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId instance, IN const BTSStatusCode status)
{
   FW_ERRMEM_IF_NULL_PTR_RETURN(_callback);

   if((BTS_PROTO_DUN != protocol) && (BTS_PROTO_PAN != protocol))
   {
      return;
   }

   if((BTS_PROTO_PAN == protocol) && (true == _callback->isPanViaConnMan()))
   {
      return;
   }

   ETG_TRACE_USR3((" updateConnectStatus: address=%s", address.c_str()));

   // optional information, contains e.g. page timeout information

   // forward only in case of failed connect
   if(BTS_STATUS_CODE_SUCCESS == status)
   {
      return;
   }

   BTSUuid uuid;
   if(false == getUuid4Instance(uuid, address, protocol, instance, *_callback))
   {
      FW_ERRMEM_ASSERT_ALWAYS();
      return;
   }

   // forward failed connect reason
   const BTSMasInstanceId masInstance(0);
   const BTSMasInstanceName masName;
   BTSDisconnectReason reason(BTS_DISCONNECT_REASON_PROTO_CONNECT_FAILED);

   // check for page timeout, aborted, missing key
   if(BTS_STATUS_CODE_PAGE_TIMEOUT == status)
   {
      reason = BTS_DISCONNECT_REASON_OUT_OF_RANGE; // e.g. device is out of range
   }
   else if(BTS_STATUS_CODE_ABORTED == status)
   {
      reason = BTS_DISCONNECT_REASON_PROTO_CONNECT_ABORTED; // e.g. connect was aborted/cancelled
   }
   else if(BTS_STATUS_CODE_PIN_OR_KEY_MISSING == status)
   {
      reason = BTS_DISCONNECT_REASON_MISSING_LINK_KEY; // e.g. pairing information on BT device was removed
   }

   _callback->updateFailedConnectReason(address, protocol, instance, uuid, masInstance, masName, reason);
}

void SppConnectDisconnectAlpsEvolutionExt::handleConnectInd(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId instance, IN const BTSStatusCode status)
{
   FW_ERRMEM_IF_NULL_PTR_RETURN(_callback);

   if((BTS_PROTO_DUN != protocol) && (BTS_PROTO_PAN != protocol))
   {
      return;
   }

   if((BTS_PROTO_PAN == protocol) && (true == _callback->isPanViaConnMan()))
   {
      return;
   }

   ETG_TRACE_USR3((" handleConnectInd: address=%s", address.c_str()));

   BTSUuid uuid;
   if(false == getUuid4Instance(uuid, address, protocol, instance, *_callback))
   {
      FW_ERRMEM_ASSERT_ALWAYS();
      return;
   }

   const BTSMasInstanceId masInstance(0);
   const BTSMasInstanceName masName;

   // optional information w.r.t. connectIndication() call to upper layer

   // store status
   setStatus(*_callback, address, protocol, instance, uuid, masInstance, masName, status);

   // forward connecting
   forwardIntermediateState(bts2IpcMsgList, bts2AppMsgList, messageItem, *_callback, address, protocol, instance, uuid, masInstance, masName, BTS_SERVICE_STATE_CONNECTING);

   // forward connect indication, use success or failed as result, reason for failed connect was forwarded via updateFailedConnectReason() before
   const BTSRequestResult result(((BTS_STATUS_CODE_SUCCESS == status) || (BTS_STATUS_CODE_SUCCESS_SLAVE == status)) ? BTS_REQ_SUCCESS : BTS_REQ_FAILED);
   _callback->connectIndication(bts2IpcMsgList, bts2AppMsgList, messageItem, address, protocol, instance, uuid, masInstance, masName, result);

   // forward new connection status
   const BTSConnectionStatus connStatus(((BTS_STATUS_CODE_SUCCESS == status) || (BTS_STATUS_CODE_SUCCESS_SLAVE == status)) ? BTS_CONN_CONNECTED : BTS_CONN_DISCONNECTED);
   const BTSDisconnectReason reason(BTS_DISCONNECT_REASON_LAST);
   (void)_callback->updateStatus(bts2IpcMsgList, bts2AppMsgList, messageItem, address, protocol, instance, uuid, masInstance, masName, connStatus, reason);
}

void SppConnectDisconnectAlpsEvolutionExt::handleConnectCompInd(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, OUT bool& connectFailed, OUT BTSBDAddress& failedAddress, IN const BTSOperationId opId)
{
   FW_ERRMEM_IF_NULL_PTR_RETURN(_callback);

   ETG_TRACE_USR3((" handleConnectCompInd"));

   BTSBDAddress address;
   BTSProtocolId protocol(BTS_PROTO_LAST);
   BTSSppInstanceId instance(0);
   BTSUuid uuid;

   if(false == getProtocol4OperationId(address, protocol, instance, uuid, *_callback, opId))
   {
      FW_ERRMEM_ASSERT_ALWAYS();
      return;
   }

   if((BTS_PROTO_SPP != protocol) && (BTS_PROTO_DUN != protocol) && (BTS_PROTO_PAN != protocol))
   {
      FW_ERRMEM_ASSERT_ALWAYS();
      return;
   }

   if((BTS_PROTO_PAN == protocol) && (true == _callback->isPanViaConnMan()))
   {
      FW_ERRMEM_ASSERT_ALWAYS();
      return;
   }

   const BTSMasInstanceId masInstance(0);
   const BTSMasInstanceName masName;
   const BTSStatusCode status(getStatus(*_callback, address, protocol, instance, uuid, masInstance, masName));
   BTSRequestResult result(BTS_REQ_FAILED);

   // check error code
   if((BTS_STATUS_CODE_SUCCESS == status) || (BTS_STATUS_CODE_SUCCESS_SLAVE == status))
   {
      result = BTS_REQ_SUCCESS;
   }
   else if(BTS_STATUS_CODE_SYSTEM_BUSY == status)
   {
#ifdef ENABLE_BUSY
      result = BTS_REQ_BUSY;
#endif
   }
   else if(BTS_STATUS_CODE_RETRY_ABORTED == status)
   {
      result = BTS_REQ_CONNECT_ABORTED;
   }
   else
   {
      // other error codes
   }

   // forward connect result
   if(true == _callback->connectResult(bts2IpcMsgList, bts2AppMsgList, messageItem, address, protocol, instance, uuid, masInstance, masName, result))
   {
      connectFailed = true;
      failedAddress = address;
   }
}

void SppConnectDisconnectAlpsEvolutionExt::handleDisconnectResult(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId instance, IN const BTSOperationId opId, IN const BTSIpcCommonErrorCode errorCode)
{
   FW_ERRMEM_IF_NULL_PTR_RETURN(_callback);

   if((BTS_PROTO_SPP != protocol) && (BTS_PROTO_DUN != protocol) && (BTS_PROTO_PAN != protocol))
   {
      return;
   }

   if((BTS_PROTO_PAN == protocol) && (true == _callback->isPanViaConnMan()))
   {
      return;
   }

   ETG_TRACE_USR3((" handleDisconnectResult"));

   BTSUuid uuid;
   if(false == getUuid4Instance(uuid, address, protocol, instance, *_callback))
   {
      FW_ERRMEM_ASSERT_ALWAYS();
      return;
   }

   const BTSMasInstanceId masInstance(0);
   const BTSMasInstanceName masName;

   // store operation id
   setOperationId(*_callback, address, protocol, instance, uuid, masInstance, masName, opId);

#ifdef ENABLE_ALPS_OP_ID
   // new handling in Evolution stack
   if(BTS_IPC_SUCCESS == errorCode)
   {
      // wait for next message
   }
   else
   {
      BTSRequestResult result(BTS_REQ_FAILED);

      // check error code
      if(BTS_IPC_NOT_CONNECTED == errorCode)
      {
         result = BTS_REQ_SUCCESS;
      }
      else if(BTS_IPC_BUSY == errorCode)
      {
#ifdef ENABLE_BUSY
         result = BTS_REQ_BUSY;
#endif
      }
      else if(BTS_IPC_RETRY_ABORTED == errorCode)
      {
         result = BTS_REQ_DISCONNECT_ABORTED;
      }
      else
      {
         // other error codes
      }

      // forward disconnect result
      _callback->disconnectResult(bts2IpcMsgList, bts2AppMsgList, messageItem, address, protocol, instance, uuid, masInstance, masName, result);
   }
#else
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);
   (void)(messageItem);
   (void)(errorCode);

   // old handling in Evolution stack
   /*
    * current stack behavior is as follows:
    * - either DBUS method return or DBUS error message is sent as answer
    * - DBUS signal message containing "confirmation" data is sent in both cases; this message indicates same error as answer messages
    * => answer can be ignored
    */
#endif
}

void SppConnectDisconnectAlpsEvolutionExt::handleDisconnectCfm(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSOperationId opId, IN const BTSStatusCode status)
{
   FW_ERRMEM_IF_NULL_PTR_RETURN(_callback);

   ETG_TRACE_USR3((" handleDisconnectCfm"));

   BTSBDAddress address;
   BTSProtocolId protocol(BTS_PROTO_LAST);
   BTSSppInstanceId instance(0);
   BTSUuid uuid;

   if(false == getProtocol4OperationId(address, protocol, instance, uuid, *_callback, opId))
   {
      FW_ERRMEM_ASSERT_ALWAYS();
      return;
   }

   if((BTS_PROTO_SPP != protocol) && (BTS_PROTO_DUN != protocol) && (BTS_PROTO_PAN != protocol))
   {
      FW_ERRMEM_ASSERT_ALWAYS();
      return;
   }

   if((BTS_PROTO_PAN == protocol) && (true == _callback->isPanViaConnMan()))
   {
      FW_ERRMEM_ASSERT_ALWAYS();
      return;
   }

   // check status
   if(BTS_STATUS_CODE_SUCCESS == status)
   {
      // wait for next message
   }
   else
   {
      const BTSMasInstanceId masInstance(0);
      const BTSMasInstanceName masName;
      BTSRequestResult result(BTS_REQ_FAILED);

      // check status code
      if(BTS_STATUS_CODE_SYSTEM_BUSY == status)
      {
#ifdef ENABLE_BUSY
         result = BTS_REQ_BUSY;
#endif
      }
      else if(BTS_STATUS_CODE_RETRY_ABORTED == status)
      {
         result = BTS_REQ_DISCONNECT_ABORTED;
      }
      else
      {
         // other status codes
      }

      // forward disconnect result
      _callback->disconnectResult(bts2IpcMsgList, bts2AppMsgList, messageItem, address, protocol, instance, uuid, masInstance, masName, result);
   }
}

void SppConnectDisconnectAlpsEvolutionExt::updateDisconnectReason(IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId instance, IN const BTSInternalDisconnectReason reason)
{
   FW_ERRMEM_IF_NULL_PTR_RETURN(_callback);

   if((BTS_PROTO_DUN != protocol) && (BTS_PROTO_PAN != protocol))
   {
      return;
   }

   if((BTS_PROTO_PAN == protocol) && (true == _callback->isPanViaConnMan()))
   {
      return;
   }

   ETG_TRACE_USR3((" updateDisconnectReason: address=%s", address.c_str()));

   // optional information, only called in case of successful disconnect

   BTSUuid uuid;
   if(false == getUuid4Instance(uuid, address, protocol, instance, *_callback))
   {
      FW_ERRMEM_ASSERT_ALWAYS();
      return;
   }

   // forward disconnect reason
   const BTSMasInstanceId masInstance(0);
   const BTSMasInstanceName masName;

   _callback->updateDisconnectReason(address, protocol, instance, uuid, masInstance, masName, convertInternalDisconnectReason2Public(reason));
}

void SppConnectDisconnectAlpsEvolutionExt::handleDisconnectInd(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId instance, IN const BTSStatusCode status)
{
   FW_ERRMEM_IF_NULL_PTR_RETURN(_callback);

   if((BTS_PROTO_DUN != protocol) && (BTS_PROTO_PAN != protocol))
   {
      return;
   }

   if((BTS_PROTO_PAN == protocol) && (true == _callback->isPanViaConnMan()))
   {
      return;
   }

   ETG_TRACE_USR3((" handleDisconnectInd: address=%s", address.c_str()));

   /*
    * Following situation can happen during cancel local SPP connect:
    * - SPP connect finished on Evolution side
    * - SPP connect failed
    * - cancel (BtApplDeviceDisconnectReq) request is sent in parallel to last message in connect sequence from Evolution stack
    * - BtApplDeviceDisconnectInd is received with address 000000000000 and status ILLEGAL_STATE
    * - example: [3537603504]BTS_IPC2BTS,TRACE_LEVEL_USER1,Function:doInputTrace,LINE:2577, Ipc2Bts: 0x02202858 (BtApplDeviceDisconnectInd): errCode=SUCCESS btAddr=000000000000 status=ILLEGAL_STATE reason=LOCAL_DEVICE serviceFunction=0x0000000000000001 instance=0
    *
    * handle as follows:
    * - ignore and do nothing
    * - note: last status is set during call of handleConnectInd(), in case of failed connect the status is "not connected", therefore no need to update the status now
    * - note: no need to update disconnecting state
    * - BtApplDeviceDisconnectCompInd message will follow anyway, this one triggers disconnect result to higher layer
    */
   const BTSBDAddress illegalStateAddress("000000000000");

   if((illegalStateAddress == address) && (BTS_STATUS_CODE_ILLEGAL_STATE == status))
   {
      ETG_TRACE_USR3((" handleDisconnectInd: ignore message"));
      return;
   }

   BTSUuid uuid;
   if(false == getUuid4Instance(uuid, address, protocol, instance, *_callback))
   {
      FW_ERRMEM_ASSERT_ALWAYS();
      return;
   }

   const BTSMasInstanceId masInstance(0);
   const BTSMasInstanceName masName;

   // optional information w.r.t. disconnectIndication() call to upper layer

   // store status
   setStatus(*_callback, address, protocol, instance, uuid, masInstance, masName, status);

   // forward disconnecting
   forwardIntermediateState(bts2IpcMsgList, bts2AppMsgList, messageItem, *_callback, address, protocol, instance, uuid, masInstance, masName, BTS_SERVICE_STATE_DISCONNECTING);

   // forward disconnect indication, use success or failed as result, disconnect reason was forwarded via updateDisconnectReason() before
   const BTSRequestResult result((BTS_STATUS_CODE_SUCCESS == status) ? BTS_REQ_SUCCESS : BTS_REQ_FAILED);
   _callback->disconnectIndication(bts2IpcMsgList, bts2AppMsgList, messageItem, address, protocol, instance, uuid, masInstance, masName, result);

   // forward new connection status
   const BTSConnectionStatus connStatus((BTS_STATUS_CODE_SUCCESS == status) ? BTS_CONN_DISCONNECTED : BTS_CONN_CONNECTED);
   const BTSDisconnectReason reason(BTS_DISCONNECT_REASON_LAST);
   (void)_callback->updateStatus(bts2IpcMsgList, bts2AppMsgList, messageItem, address, protocol, instance, uuid, masInstance, masName, connStatus, reason);
}

void SppConnectDisconnectAlpsEvolutionExt::handleDisconnectCompInd(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSOperationId opId)
{
   FW_ERRMEM_IF_NULL_PTR_RETURN(_callback);

   ETG_TRACE_USR3((" handleDisconnectCompInd"));

   BTSBDAddress address;
   BTSProtocolId protocol(BTS_PROTO_LAST);
   BTSSppInstanceId instance(0);
   BTSUuid uuid;

   if(false == getProtocol4OperationId(address, protocol, instance, uuid, *_callback, opId))
   {
      FW_ERRMEM_ASSERT_ALWAYS();
      return;
   }

   if((BTS_PROTO_SPP != protocol) && (BTS_PROTO_DUN != protocol) && (BTS_PROTO_PAN != protocol))
   {
      FW_ERRMEM_ASSERT_ALWAYS();
      return;
   }

   if((BTS_PROTO_PAN == protocol) && (true == _callback->isPanViaConnMan()))
   {
      FW_ERRMEM_ASSERT_ALWAYS();
      return;
   }

   const BTSMasInstanceId masInstance(0);
   const BTSMasInstanceName masName;
   const BTSStatusCode status(getStatus(*_callback, address, protocol, instance, uuid, masInstance, masName));
   BTSRequestResult result(BTS_REQ_FAILED);

   // check error code
   if(BTS_STATUS_CODE_SUCCESS == status)
   {
      result = BTS_REQ_SUCCESS;
   }
   else if(BTS_STATUS_CODE_SYSTEM_BUSY == status)
   {
#ifdef ENABLE_BUSY
      result = BTS_REQ_BUSY;
#endif
   }
   else if(BTS_STATUS_CODE_RETRY_ABORTED == status)
   {
      result = BTS_REQ_DISCONNECT_ABORTED;
   }
   else
   {
      // other error codes
   }

   // forward disconnect result
   _callback->disconnectResult(bts2IpcMsgList, bts2AppMsgList, messageItem, address, protocol, instance, uuid, masInstance, masName, result);
}

void SppConnectDisconnectAlpsEvolutionExt::createCancelRes(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, IN const act_t token, IN const bool responseFlag /*= false*/) const
{
   Bts2Ipc_CancelRes* msg = ptrNew_Bts2Ipc_CancelRes();
   if(0 != msg)
   {
      msg->setDbusToken(token);
      msg->setResponseMessageFlag(responseFlag);

      bts2IpcMsgList.push_back(msg);
   }
}

void SppConnectDisconnectAlpsEvolutionExt::createCancelRequestRes(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, IN const act_t token, IN const bool responseFlag /*= false*/) const
{
   Bts2Ipc_CancelRequestRes* msg = ptrNew_Bts2Ipc_CancelRequestRes();
   if(0 != msg)
   {
      msg->setDbusToken(token);
      msg->setResponseMessageFlag(responseFlag);

      bts2IpcMsgList.push_back(msg);
   }
}

void SppConnectDisconnectAlpsEvolutionExt::createBtApplDeviceConnectRequest(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, IN const BTSBDAddress& deviceAddress, IN const BTSProtocolId protocolId, IN const BTSSppInstanceId instance) const
{
   Bts2Ipc_BtApplDeviceConnectReq* msg = ptrNew_Bts2Ipc_BtApplDeviceConnectReq();
   if(0 != msg)
   {
      msg->setBDAddress(deviceAddress);
      BTSServiceFunction& function = msg->getServiceFunctionMutable();
      if(BTS_PROTO_SPP == protocolId)
      {
         function.setBit(BTS_SRV_FUNC_SPP);
      }
      else if(BTS_PROTO_DUN == protocolId)
      {
         function.setBit(BTS_SRV_FUNC_DUN);
      }
      else if(BTS_PROTO_PAN == protocolId)
      {
         function.setBit(BTS_SRV_FUNC_PANU);
      }
      else
      {
         FW_ERRMEM_ASSERT_ALWAYS();
      }
      msg->setInstance(instance);
      msg->setResponseMessageFlag(true);

      bts2IpcMsgList.push_back(msg);
   }
}

void SppConnectDisconnectAlpsEvolutionExt::createBtApplDeviceDisconnectRequest(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, IN const BTSBDAddress& deviceAddress, IN const BTSProtocolId protocolId, IN const BTSSppInstanceId instance) const
{
   Bts2Ipc_BtApplDeviceDisconnectReq* msg = ptrNew_Bts2Ipc_BtApplDeviceDisconnectReq();
   if(0 != msg)
   {
      msg->setBDAddress(deviceAddress);
      BTSServiceFunction& function = msg->getServiceFunctionMutable();
      if(BTS_PROTO_SPP == protocolId)
      {
         function.setBit(BTS_SRV_FUNC_SPP);
      }
      else if(BTS_PROTO_DUN == protocolId)
      {
         function.setBit(BTS_SRV_FUNC_DUN);
      }
      else if(BTS_PROTO_PAN == protocolId)
      {
         function.setBit(BTS_SRV_FUNC_PANU);
      }
      else
      {
         FW_ERRMEM_ASSERT_ALWAYS();
      }
      msg->setInstance(instance);
      msg->setResponseMessageFlag(true);

      bts2IpcMsgList.push_back(msg);
   }
}

void SppConnectDisconnectAlpsEvolutionExt::createAuthorizeServiceResponseMsg(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, IN const BTSBDAddress& address, IN const act_t token, IN const BTSConfirmationMode mode) const
{
   Bts2Ipc_AuthorizeServiceRes* msg = ptrNew_Bts2Ipc_AuthorizeServiceRes();
   if(0 != msg)
   {
      msg->setBDAddress(address);
      msg->setAccept(mode);
      msg->setDbusToken(token);
      msg->setResponseMessageFlag(true);

      bts2IpcMsgList.push_back(msg);
   }
}

void SppConnectDisconnectAlpsEvolutionExt::checkForDataComplete(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN IProtocolManagerCallback& callback, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName) const
{
   // get data entry
   ProtocolManagerData& entry = callback.getEntry(address, protocol, sppInstance, uuid, masInstance, masName);

   // get stored availability information
   const bool oldAvailable(entry.info.getBit(ProtocolManagerData::PROTOCOL_AVAILABLE));

   // get new availability information
   bool newAvailable(true);

   if(false == isDeviceAvailable(address))
   {
      newAvailable = false;
   }
   else if(false == entry.info.getBit(ProtocolManagerData::PROTOCOL_ADDED))
   {
      newAvailable = false;
   }

   // indicate only changed states
   if(oldAvailable != newAvailable)
   {
      ProtocolManagerData::ProtocolAvailableInfo protocolAvailableInfo;
      protocolAvailableInfo.setBit(ProtocolManagerData::PROTOCOL_AVAILABLE_LEVEL_1);
      protocolAvailableInfo.setBit(ProtocolManagerData::PROTOCOL_AVAILABLE_LEVEL_2);

      if(true == newAvailable)
      {
         callback.protocolAvailable(bts2IpcMsgList, bts2AppMsgList, messageItem, address, protocol, sppInstance, uuid, masInstance, masName, protocolAvailableInfo);
      }
      else
      {
         callback.protocolUnavailable(bts2IpcMsgList, bts2AppMsgList, messageItem, address, protocol, sppInstance, uuid, masInstance, masName, protocolAvailableInfo);
      }
   }
}

bool SppConnectDisconnectAlpsEvolutionExt::isDeviceAvailable(IN const BTSBDAddress& address) const
{
   FW_ERRMEM_IF_NULL_PTR_RETURN_FALSE(_deviceManagerIf);

   return _deviceManagerIf->isDeviceAvailable(address);
}

bool SppConnectDisconnectAlpsEvolutionExt::getUuid4Instance(OUT BTSUuid& uuid, IN const BTSBDAddress& deviceAddress, IN const BTSProtocolId protocolId, IN const BTSSppInstanceId instance, IN IProtocolManagerCallback& callback) const
{
   if(BTS_PROTO_SPP != protocolId)
   {
      // SPP UUID not needed
      return true;
   }

   const ::std::map< BTSProtocolBaseEntry, ProtocolManagerData >& protocolList = callback.getProtocolList();

   // check list
   for(::std::map< BTSProtocolBaseEntry, ProtocolManagerData >::const_iterator it = protocolList.begin(); it != protocolList.end(); ++it)
   {
      const BTSProtocolBaseEntry& key = it->first;

      if((deviceAddress == key.deviceAddress) && (protocolId == key.protocolId))
      {
         const ProtocolManagerData& data = it->second;

         if(instance == data.sppInstance)
         {
            if(0 != data.sm.getConnectSm())
            {
               // only 1 SPP connect to the same device at the same time is allowed
               uuid = key.sppUuid;
               return true;
            }
            else if(true == data.connected)
            {
               // SPP instance connected
               uuid = key.sppUuid;
               return true;
            }
            else
            {
               // no connect ongoing, not connected => there should be no match for SPP instance
               FW_ERRMEM_ASSERT_ALWAYS();
            }
         }
      }
   }

   return false;
}

bool SppConnectDisconnectAlpsEvolutionExt::getSppInstance(OUT BTSSppInstanceId& instance, IN IProtocolManagerCallback& callback) const
{
   const ::std::map< BTSProtocolBaseEntry, ProtocolManagerData >& protocolList = callback.getProtocolList();

   // check list
   for(::std::map< BTSProtocolBaseEntry, ProtocolManagerData >::const_iterator it = protocolList.begin(); it != protocolList.end(); ++it)
   {
      const BTSProtocolBaseEntry& key = it->first;

      if((BTS_PROTO_SPP == key.protocolId) ||
         (BTS_PROTO_DUN == key.protocolId) ||
         ((BTS_PROTO_PAN == key.protocolId) && (false == callback.isPanViaConnMan())))
      {
         const ProtocolManagerData& data = it->second;

         if(0 != data.sm.getConnectSm())
         {
            // NOTE: old handling: only 1 connect is active at the same time => entry found
            instance = data.sppInstance;
            return true;
         }
      }
   }

   return false;
}

void SppConnectDisconnectAlpsEvolutionExt::setStatus(IN IProtocolManagerCallback& callback, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName, IN const BTSStatusCode status)
{
   ProtocolManagerData& entry = callback.getEntry(address, protocol, sppInstance, uuid, masInstance, masName);

   setStatus(entry.secondaryInfo, status);
}

void SppConnectDisconnectAlpsEvolutionExt::setStatus(OUT unsigned int& secondaryData, IN const BTSStatusCode status) const
{
   // use bits 0 .. 15
   const unsigned int stateMask(0x0000FFFF);
   const unsigned int stateValue((const unsigned int)status);

   secondaryData &= ~stateMask;
   secondaryData |= (stateValue & stateMask);
}

BTSStatusCode SppConnectDisconnectAlpsEvolutionExt::getStatus(IN IProtocolManagerCallback& callback, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName) const
{
   ProtocolManagerData& entry = callback.getEntry(address, protocol, sppInstance, uuid, masInstance, masName);

   return getStatus(entry.secondaryInfo);
}

BTSStatusCode SppConnectDisconnectAlpsEvolutionExt::getStatus(IN const unsigned int secondaryData) const
{
   // use bits 0 .. 15
   const unsigned int stateMask(0x0000FFFF);

   return (BTSStatusCode)(secondaryData & stateMask);
}

void SppConnectDisconnectAlpsEvolutionExt::setOperationId(IN IProtocolManagerCallback& callback, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName, IN const BTSOperationId opId)
{
   ProtocolManagerData& entry = callback.getEntry(address, protocol, sppInstance, uuid, masInstance, masName);

   setOperationId(entry.operationId, opId);
}

void SppConnectDisconnectAlpsEvolutionExt::setOperationId(OUT unsigned int& secondaryData, IN const BTSOperationId opId) const
{
   // value provided by Evolution stack is a 16 bit value, add a marker that indicates that the operation id was set
   secondaryData = opId | 0x80000000;
}

bool SppConnectDisconnectAlpsEvolutionExt::getOperationId(OUT BTSOperationId& opId, IN const unsigned int secondaryData) const
{
   // check marker
   if(0x80000000 == (0x80000000 & secondaryData))
   {
      opId = secondaryData & ~0x80000000;
      return true;
   }

   return false;
}

bool SppConnectDisconnectAlpsEvolutionExt::getOperationId(OUT BTSOperationId& opId, IN IProtocolManagerCallback& callback, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName) const
{
   ProtocolManagerData& entry = callback.getEntry(address, protocol, sppInstance, uuid, masInstance, masName);

   return getOperationId(opId, entry.operationId);
}

bool SppConnectDisconnectAlpsEvolutionExt::getProtocol4OperationId(OUT BTSBDAddress& address, OUT BTSProtocolId& protocol, OUT BTSSppInstanceId& sppInstance, OUT BTSUuid& uuid, IN IProtocolManagerCallback& callback, IN const BTSOperationId opId) const
{
   const ::std::map< BTSProtocolBaseEntry, ProtocolManagerData >& protocolList = callback.getProtocolList();

   // check list
   for(::std::map< BTSProtocolBaseEntry, ProtocolManagerData >::const_iterator it = protocolList.begin(); it != protocolList.end(); ++it)
   {
      const ProtocolManagerData& data = it->second;
      BTSOperationId storedOpId(0);

      if(true == getOperationId(storedOpId, data.operationId))
      {
         if(opId == storedOpId)
         {
            const BTSProtocolBaseEntry& key = it->first;

            address = key.deviceAddress;
            protocol = key.protocolId;
            sppInstance = data.sppInstance;
            uuid = key.sppUuid;

            return true;
         }
      }
   }

   return false;
}

bool SppConnectDisconnectAlpsEvolutionExt::getProtocol4Instance(OUT BTSBDAddress& address, OUT BTSProtocolId& protocol, OUT BTSUuid& uuid, IN IProtocolManagerCallback& callback, IN const BTSSppInstanceId sppInstance) const
{
   const ::std::map< BTSProtocolBaseEntry, ProtocolManagerData >& protocolList = callback.getProtocolList();

   // check list
   for(::std::map< BTSProtocolBaseEntry, ProtocolManagerData >::const_iterator it = protocolList.begin(); it != protocolList.end(); ++it)
   {
      const BTSProtocolBaseEntry& key = it->first;

      // TODO: [low]: the approach to provide device name together with instance works only for SPP; for 2 parallel DUN/PAN connections we need more data e.g. device address + service function + instance (=> similar to BtApplDeviceConnectInd)
      // TODO: [low]: workaround: consider only SPP because DUN is not supported by head unit and PAN is handled via ConnMan

      if(BTS_PROTO_SPP == key.protocolId)
      {
         const ProtocolManagerData& data = it->second;

         if((0 != data.sm.getConnectSm()) && (sppInstance == data.sppInstance))
         {
            address = key.deviceAddress;
            protocol = key.protocolId;
            uuid = key.sppUuid;
            return true;
         }
      }
   }

   return false;
}

void SppConnectDisconnectAlpsEvolutionExt::forwardIntermediateState(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN IProtocolManagerCallback& callback, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName, IN const BTSServiceState serviceState) const
{
   if(BTS_SERVICE_STATE_CONNECTING == serviceState)
   {
      // SM shall exist during this moment
      callback.connecting(bts2IpcMsgList, bts2AppMsgList, messageItem, address, protocol, sppInstance, uuid, masInstance, masName);
   }
   else if(BTS_SERVICE_STATE_DISCONNECTING == serviceState)
   {
      // it can happen that SM does not exist during this moment e.g. connection loss
      const ProtocolManagerData& protocolData = callback.getEntry(address, protocol, sppInstance, uuid, masInstance, masName);
      if((0 != protocolData.sm.getDisconnectSm()) || (0 != protocolData.sm.getConnectSm()))
      {
         callback.disconnecting(bts2IpcMsgList, bts2AppMsgList, messageItem, address, protocol, sppInstance, uuid, masInstance, masName);
      }
   }
}

} //alpsevolutionext
} //genivi
} //btstackif
