/**
 * @file BluetoothStackErrorHandlingAlpsEvolutionExt.cpp
 *
 * @par SW-Component
 * State machine for Bluetooth stack error handling
 *
 * @brief Implementation of Alps Evolution Genivi Bluetooth stack error handling state machine.
 *
 * @copyright (C) 2018 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Source file for implementation of Alps Evolution Genivi Bluetooth stack error handling state machine.
 */

#include "BluetoothStackErrorHandlingAlpsEvolutionExt.h"
#include "IBluetoothStackErrorHandlingCallback.h"
#include "ISwitchBluetooth.h"
#include "FwErrmemPrint.h"
#include "Bts2Ipc_MessageWrapper_GEN.h"
#include "Ipc2Bts_MessageWrapper_GEN.h"

namespace btstackif {
namespace genivi {
namespace alpsevolutionext {

BluetoothStackErrorHandlingAlpsEvolutionExt::BluetoothStackErrorHandlingAlpsEvolutionExt() :
_callback(0),
_switchBluetoothIf(0),
_stackErrorOpcode(0xFFFE),
_hardwareFailureOpcode(0xFFFF),
_ignoreStatusCodeConnectRejectSecurityReasons(true),
_stackType("AlpsEvolutionGenivi"),
_fatalError("FatalError"),
_generalError("GeneralError"),
_debugTrace("DebugTrace"),
_geniviFatalError("GeniviFatalError"),
_geniviDebugTrace("GeniviDebugTrace"),
_stackError("StackError"),
_hardwareFailure("HardwareFailure")
{
}

BluetoothStackErrorHandlingAlpsEvolutionExt::~BluetoothStackErrorHandlingAlpsEvolutionExt()
{
   _callback = 0;
   _switchBluetoothIf = 0;
}

void BluetoothStackErrorHandlingAlpsEvolutionExt::reset(void)
{
}

void BluetoothStackErrorHandlingAlpsEvolutionExt::setCallback(IN IBluetoothStackErrorHandlingCallback* callback)
{
   _callback = callback;

   FW_ERRMEM_ASSERT(0 != _callback);
}

IBluetoothStackErrorHandlingRequest* BluetoothStackErrorHandlingAlpsEvolutionExt::getRequestIf(void)
{
   return this;
}

void BluetoothStackErrorHandlingAlpsEvolutionExt::setSwitchBluetoothIf(IN ISwitchBluetooth* switchBluetoothIf)
{
   _switchBluetoothIf = switchBluetoothIf;

   FW_ERRMEM_ASSERT(0 != _switchBluetoothIf);
}

void BluetoothStackErrorHandlingAlpsEvolutionExt::handleFatalError(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSIpcCommonErrorCode errorCode, IN const BTSFatalErrorLocation location, IN const BTSFatalErrorType type)
{
   // signal Trace::FatalError

   if(BTS_IPC_SUCCESS == errorCode)
   {
      /* according ARN:
       * When Host application receive below signal , Host application should reset genivi Stack.
       * * Trace::GeneralError with opcode = 0xFFFE or opcode = 0xFFFF
       * * Trace::FatalError
       */
      handleCriticalErrorInternal(bts2IpcMsgList, bts2AppMsgList, messageItem, _fatalError, BTS_STATUS_CODE_SUCCESS, location, type);
   }
   else
   {
      // somehow other use case => write entry to error memory
      handleOtherErrorInternal(bts2IpcMsgList, bts2AppMsgList, messageItem, _fatalError, errorCode);
   }
}

void BluetoothStackErrorHandlingAlpsEvolutionExt::handleGeneralError(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSIpcCommonErrorCode errorCode, IN const BTSStatusCode status, IN const BTSInternalOpcode binaryStatus, IN const BTSInternalOpcode opcode)
{
   // signal Trace::GeneralError

   if(BTS_IPC_SUCCESS == errorCode)
   {
      BTSStatusCode tmpStatus(status);

      if((BTS_STATUS_CODE_SUCCESS != status) && (binaryStatus != (BTSInternalOpcode)status))
      {
         // converted value does not match binary value => status code is unknown => print binary value
         tmpStatus = (BTSStatusCode)binaryStatus;
      }

      /* according ARN:
       * When Host application receive below signal , Host application should reset genivi Stack.
       * * Trace::GeneralError with opcode = 0xFFFE or opcode = 0xFFFF
       * * Trace::FatalError
       */
      if(_stackErrorOpcode == opcode)
      {
         handleCriticalErrorInternal(bts2IpcMsgList, bts2AppMsgList, messageItem, _stackError, tmpStatus);
      }
      else if(_hardwareFailureOpcode == opcode)
      {
         handleCriticalErrorInternal(bts2IpcMsgList, bts2AppMsgList, messageItem, _hardwareFailure, tmpStatus);
      }
      else
      {
         // other uncritical opcode
         handleGeneralErrorInternal(bts2IpcMsgList, bts2AppMsgList, messageItem, tmpStatus, opcode);
      }
   }
   else
   {
      // somehow other use case => write entry to error memory
      handleOtherErrorInternal(bts2IpcMsgList, bts2AppMsgList, messageItem, _generalError, errorCode);
   }
}

void BluetoothStackErrorHandlingAlpsEvolutionExt::handleGeniviDebugTrace(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSIpcCommonErrorCode errorCode, IN const BTSGeniviTraceNumber errFile, IN const BTSGeniviTraceNumber errLine, IN const BTSGeniviTraceNumber level, IN const BTSGeniviDebugTraceCategory category, IN const BTSGeniviTraceNumber errCode, IN const BTSGeniviTraceData& data)
{
   FW_ERRMEM_IF_NULL_PTR_RETURN(_callback);

   // signal GeniviTrace::ReportDebugTrace

   if(BTS_IPC_SUCCESS == errorCode)
   {
      /* ALPS: 20160804:
         >>Yes, there is no configuration item for ReportDebugTrace. And also it needn't these configuration items.
         Because it only works for BTLOG_IMMEDIATELY,BTLOG_CRITICAL,BTLOG_ERROR.
         Also, FatalError only works for BTLOG_FATAL.
         Only when the corresponding error occurred, the trace will reported, otherwise, there will be no traces.
       *
         >>ReportDebugTrace and FatalError(GeniviTrace) are only worked in genivi level. They are not designed for Evolution.
         When you see these d-bus signals on host, it means that there are errors occurred in evo_genivi_bt.
         Also it doesn't work for Debug level, but it can be used for debugging when errors occurred.
         So it means, normally, there will not be any ReportDebugTrace or FatalError(GeniviTrace)  reported unless error occurred. Stack will not flooding the communication channel.
       *
         >>ReportDebugTrace will only report BTLOG_IMMEDIATELY,BTLOG_CRITICAL,BTLOG_ERROR.
         If you want to filter these d-bus signal information, you can filter these as you needed through your host.
       */
      const size_t binSize = data.size();
      const unsigned char* binData = &(data[0]);
      char tmpBuffer[201] = { 0 };
      for(size_t i = 0; (i < binSize) && (i < ((sizeof(tmpBuffer) - 1) / 2)); i++)
      {
         sprintf(&tmpBuffer[2 * i], "%02X", binData[i]);
      }

      // build error message
      char buffer[301];

      (void)snprintf(buffer, sizeof(buffer), "errFile=%u errLine=%u level=%u category=%u errCode=%u data=%s", errFile, errLine, level, category, errCode, tmpBuffer);

      _callback->handleDebugTrace(bts2IpcMsgList, bts2AppMsgList, messageItem, _stackType, _geniviDebugTrace, buffer, true);
   }
   else
   {
      // somehow other use case => write entry to error memory
      handleOtherErrorInternal(bts2IpcMsgList, bts2AppMsgList, messageItem, _geniviDebugTrace, errorCode);
   }
}

void BluetoothStackErrorHandlingAlpsEvolutionExt::handleGeniviFatalError(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSIpcCommonErrorCode errorCode, IN const BTSGeniviTraceNumber errFile, IN const BTSGeniviTraceNumber errLine, IN const BTSGeniviTraceNumber errCode, IN const BTSGeniviTraceError& fmt, IN const BTSGeniviTraceError& errValue)
{
   FW_ERRMEM_IF_NULL_PTR_RETURN(_callback);

   // signal GeniviTrace::FatalError

   if(BTS_IPC_SUCCESS == errorCode)
   {
      /* ALPS: 20160804:
         >>Yes, there is no configuration item for ReportDebugTrace. And also it needn't these configuration items.
         Because it only works for BTLOG_IMMEDIATELY,BTLOG_CRITICAL,BTLOG_ERROR.
         Also, FatalError only works for BTLOG_FATAL.
         Only when the corresponding error occurred, the trace will reported, otherwise, there will be no traces.
       *
         >>ReportDebugTrace and FatalError(GeniviTrace) are only worked in genivi level. They are not designed for Evolution.
         When you see these d-bus signals on host, it means that there are errors occurred in evo_genivi_bt.
         Also it doesn't work for Debug level, but it can be used for debugging when errors occurred.
         So it means, normally, there will not be any ReportDebugTrace or FatalError(GeniviTrace)  reported unless error occurred. Stack will not flooding the communication channel.
       *
         >>ReportDebugTrace will only report BTLOG_IMMEDIATELY,BTLOG_CRITICAL,BTLOG_ERROR.
         If you want to filter these d-bus signal information, you can filter these as you needed through your host.
       */

      // build error message
      char buffer[301];

      (void)snprintf(buffer, sizeof(buffer), "errFile=%u errLine=%u errCode=%u fmt=%s errValue=%s", errFile, errLine, errCode, fmt.c_str(), errValue.c_str());

      _callback->handleCriticalError(bts2IpcMsgList, bts2AppMsgList, messageItem, _stackType, _geniviFatalError, buffer, false);
   }
   else
   {
      // somehow other use case => write entry to error memory
      handleOtherErrorInternal(bts2IpcMsgList, bts2AppMsgList, messageItem, _geniviFatalError, errorCode);
   }
}

void BluetoothStackErrorHandlingAlpsEvolutionExt::handleReportDebugTrace(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSIpcCommonErrorCode errorCode, IN const BTSDebugTraceLocation location, IN const BTSDebugTraceLevel level, IN const BTSDebugTraceCategory category, IN const BTSDebugTraceErrorCode errCode, IN const BTSDebugTraceData& data)
{
   FW_ERRMEM_IF_NULL_PTR_RETURN(_callback);

   // signal Trace::ReportDebugTrace

   if(BTS_IPC_SUCCESS == errorCode)
   {
      // signal Trace::ReportDebugTrace might be emitted before signal Trace::FatalError to provide more data useful for investigation purpose
      // received data:
      // - location: 32bit, location of output this debug trace
      // - level: 8bit, debug trace level of this message, allow only FATAL (bit 0) or ERROR (bit 1)
      // - category: 8bit, category of trace
      // - errCode: 16bit, error code of this trace log
      // - data: array<8bit>, debug trace data

      BTSBitfield<BTSDebugTraceLevel, BTSDebugTraceLevelBit, BTS_DEBUG_TRACE_LEVEL_LAST> checkLevel;
      checkLevel.setBit(BTS_DEBUG_TRACE_LEVEL_FATAL);
      checkLevel.setBit(BTS_DEBUG_TRACE_LEVEL_ERROR);

      char tmpBuffer[201] = { 0 };
      for(size_t i = 0; (i < data.size()) && (i < ((sizeof(tmpBuffer) - 1) / 2)); i++)
      {
         sprintf(&tmpBuffer[2 * i], "%02X", data[i]);
      }

      // build error message
      bool writeToErrorMemory;
      char buffer[301];

      (void)snprintf(buffer, sizeof(buffer), "location=%u level=%u category=%d errCode=%u data=%s", location, level, category, errCode, tmpBuffer);

      if(0 != (level & checkLevel.getData()))
      {
         // write to error memory
         writeToErrorMemory = true;
      }
      else
      {
         // do fatal trace only
         writeToErrorMemory = false;
      }

      _callback->handleDebugTrace(bts2IpcMsgList, bts2AppMsgList, messageItem, _stackType, _debugTrace, buffer, writeToErrorMemory);
   }
   else
   {
      // somehow other use case => write entry to error memory
      handleOtherErrorInternal(bts2IpcMsgList, bts2AppMsgList, messageItem, _debugTrace, errorCode);
   }
}

void BluetoothStackErrorHandlingAlpsEvolutionExt::handleCriticalErrorInternal(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const ::std::string& errorType, IN const BTSStatusCode status, IN const BTSFatalErrorLocation location /*= 0*/, IN const BTSFatalErrorType type /*= 0*/)
{
   FW_ERRMEM_IF_NULL_PTR_RETURN(_callback);
   FW_ERRMEM_IF_NULL_PTR_RETURN(_switchBluetoothIf);

   // build error message
   bool doReset;
   char buffer[101];

   if(BTS_STATUS_CODE_SUCCESS == status)
   {
      (void)snprintf(buffer, sizeof(buffer), "location=%u type=%u", location, type);
   }
   else
   {
      (void)snprintf(buffer, sizeof(buffer), "status=0x%04X", (unsigned short int)status);
   }

   // reset stack
   if((true == _ignoreStatusCodeConnectRejectSecurityReasons) && (BTS_STATUS_CODE_CONNECT_REJECT_SECURITY_REASONS == status))
   {
      /*
       * comment from Marvell:
       * The only impact is particular data packet will be lost. i.e. If HW error occurred after sending AT+CLCC data , this data packet
       * will not sent over air, otherwise there is no impact on the link, host can ignore the error. The packets sent after HW error will have no impact.
       *
       * comment from ALPS:
       * Evolution stack has an internal timeout handling for missing answers, for example the AT command timeout is about 5 seconds.
       * Timeout handling is also included in other layer e.g. HCI, L2CAP,RFCOMM OBEX,etc.
       * From this point of view, we think it is ok not to reset the target when this hardware error happened. But just in cases, it is better to store this hardware error information in the target system log for issues analysis purpose.
       */
      doReset = false;
   }
   else if((BTS_STATUS_CODE_FAULT_HOSTIO_FROZEN_HANDLES == status) && (BTS_HCI_CHIP_VENDOR_CSR == _switchBluetoothIf->getHciChipVendorId()))
   {
      /*
       * Can happen with CSR based BT modules in case of rapid connects/disconnects or rapid failed connects.
       * Do not reset the stack / system.
       */
      doReset = false;
   }
   else if((BTS_STATUS_CODE_FAULT_LC_LMP_IN_INVALID_PACKET == status) && (BTS_HCI_CHIP_VENDOR_CSR == _switchBluetoothIf->getHciChipVendorId()))
   {
      /*
       * comment from ALPS:
       * This error code is from controller and indicates a wrong LMP packet is received.
       * Usually LMP error will be NAKed to the peer device and re-transmitted, thus, the impact is minor. Host can ignore it.
       */
      doReset = false;
   }
   else
   {
      doReset = true;
   }

   _callback->handleCriticalError(bts2IpcMsgList, bts2AppMsgList, messageItem, _stackType, errorType, buffer, doReset);
}

void BluetoothStackErrorHandlingAlpsEvolutionExt::handleGeneralErrorInternal(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSStatusCode status, IN const BTSInternalOpcode opcode)
{
   FW_ERRMEM_IF_NULL_PTR_RETURN(_callback);

   /*
    * filter following messages:
    * - status=0x0004 opcode=0x0402 => EVTRES_OPERATION_FAIL for BT_APPL_AUTHORISE --- happens if we reject remote protocol connection request
    * - status=0x0007 opcode=0x040B => EVTRES_ILLEGAL_STATE for BT_APPL_LINK_KEY --- happens if phone disconnects ACL during local profile connect
    */
   if(((BTS_STATUS_CODE_OPERATION_FAIL == status) && (0x0402 == opcode)) ||
      ((BTS_STATUS_CODE_ILLEGAL_STATE == status) && (0x040B == opcode)))
   {
      return;
   }

   // build error message
   char buffer[101];
   (void)snprintf(buffer, sizeof(buffer), "status=0x%04X opcode=0x%04X", (unsigned short int)status, opcode);

   _callback->handleGeneralError(bts2IpcMsgList, bts2AppMsgList, messageItem, _stackType, _generalError, buffer);
}

void BluetoothStackErrorHandlingAlpsEvolutionExt::handleOtherErrorInternal(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const ::std::string& errorType, IN const BTSIpcCommonErrorCode errorCode)
{
   FW_ERRMEM_IF_NULL_PTR_RETURN(_callback);

   // build error message
   char buffer[101];
   (void)snprintf(buffer, sizeof(buffer), "errorCode=%d", errorCode);

   _callback->handleOtherError(bts2IpcMsgList, bts2AppMsgList, messageItem, _stackType, errorType, buffer);
}

} //alpsevolutionext
} //genivi
} //btstackif
