/**
 * @file BluetoothLowEnergyAlpsEvolutionExt.cpp
 *
 * @par SW-Component
 * State machine for Bluetooth Low Energy
 *
 * @brief Implementation of Alps Evolution Genivi Bluetooth Low Energy state machine.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Source file for implementation of Alps Evolution Genivi Bluetooth Low Energy state machine.
 */

#include "BluetoothLowEnergyAlpsEvolutionExt.h"
#include "FwAssert.h"
#include "cc_dbus_if/EvolutionGeniviDbusParser.h"

#include "TraceClasses.h"
#include "FwTrace.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_BTS_CONTROL
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/BluetoothLowEnergyAlpsEvolutionExt.cpp.trc.h"
#endif
#endif

namespace btstackif {
namespace genivi {
namespace alpsevolutionext {

BluetoothLowEnergyAlpsEvolutionExt::BluetoothLowEnergyAlpsEvolutionExt() :
_controlIf(0),
_timerPoolIf(0),
_enabled(false),
_serviceGroupBLEGeneral(),
_serviceStatusBLEObjectManagerSystem(),
_serviceStatusBLEPropertiesGetAll(),
_serviceStatusBLEGenInitialization(),
_serviceStatusBLEHliInitialization(),
_serviceStatusBLEAdapter(),
_serviceStatusBLEDevice(),
_serviceStatusBLESeamlessPairing()
{
   _serviceGroupBLEGeneral.setEnabled(false); // per default BLE is disabled
}

BluetoothLowEnergyAlpsEvolutionExt::~BluetoothLowEnergyAlpsEvolutionExt()
{
   _controlIf = 0;
   _timerPoolIf = 0;
}

void BluetoothLowEnergyAlpsEvolutionExt::setControlIf(IN IBasicControl* control)
{
   _controlIf = control;

   FW_NORMAL_ASSERT(0 != _controlIf);
}

void BluetoothLowEnergyAlpsEvolutionExt::setTimerPoolIf(IN ITimerPool* timerPool)
{
   _timerPoolIf = timerPool;

   FW_NORMAL_ASSERT(0 != _timerPoolIf);
}

void BluetoothLowEnergyAlpsEvolutionExt::enable(IN const bool enable)
{
   // store new state
   _enabled = enable;
}

bool BluetoothLowEnergyAlpsEvolutionExt::getEnabled(void) const
{
   return _enabled;
}

void BluetoothLowEnergyAlpsEvolutionExt::setStackConfiguration(OUT ::std::vector< BTSDbusInterfaceItem >& dbusInterfaces, IN const BTSLocalStackConfiguration& configuration, IN const BTSUserMode userMode, IN const BTSInterfaceType interface)
{
   (void)(configuration);

   ETG_TRACE_USR2((" setStackConfiguration(): _enabled=%d", _enabled));

   if(BTS_USER_MODE_LAST <= userMode)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   if(BTS_IF_ALPS_EVOLUTION_GENIVI != interface)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   if(false == _enabled)
   {
      // do not process
      return;
   }

   // check user mode
   if((BTS_USER_MODE_CONNECTION == userMode) ||
      (BTS_USER_MODE_CONN_WO_AGENT == userMode) ||
      (BTS_USER_MODE_ALL_WI_AGENT == userMode) ||
      (BTS_USER_MODE_ALL_WO_AGENT == userMode))
   {
      // check configuration not necessary because _enabled flag is set to configuration value before
      // enable BLE
      _serviceGroupBLEGeneral.setEnabled(true);
   }

   // check if BLE is enabled
   if(true == _serviceGroupBLEGeneral.getEnabled())
   {
      FW_IF_NULL_PTR_RETURN(_controlIf);

      BTSDbusInterfaceItem dbusItem;
      ::ccdbusif::evolution::Interface evoInterface;
      ::ccdbusif::evolution::EvolutionGeniviDbusParser evoParser;

      _serviceGroupBLEGeneral.disableTimer();
      _serviceGroupBLEGeneral.addServiceInfo("");
      _serviceGroupBLEGeneral.setName("BLE_GroupGeneral");

      _serviceStatusBLEObjectManagerSystem.setName("BLE_ObjectManagerSystem");
      _serviceStatusBLEPropertiesGetAll.setName("BLE_PropertiesGetAll");
      _serviceStatusBLEGenInitialization.setName("BLE_GenInitialization");
      _serviceStatusBLEHliInitialization.setName("BLE_HliInitialization");
      _serviceStatusBLEAdapter.setName("BLE_Adapter");
      _serviceStatusBLEDevice.setName("BLE_Device");
      _serviceStatusBLESeamlessPairing.setName("BLE_SeamlessPairing");

      _serviceStatusBLEObjectManagerSystem.setIpc2BtsSendIf(_controlIf);
      _serviceStatusBLEPropertiesGetAll.setIpc2BtsSendIf(_controlIf);
      _serviceStatusBLEGenInitialization.setIpc2BtsSendIf(_controlIf);
      _serviceStatusBLEHliInitialization.setIpc2BtsSendIf(_controlIf);
      _serviceStatusBLEAdapter.setIpc2BtsSendIf(_controlIf);
      _serviceStatusBLEDevice.setIpc2BtsSendIf(_controlIf);
      _serviceStatusBLESeamlessPairing.setIpc2BtsSendIf(_controlIf);

      _serviceStatusBLEObjectManagerSystem.setInterface((BTSCommonEnumClass)BTS_GEN_DBUS_SERVICE_BLE_OBJECT_MANAGER_SYSTEM);
      _serviceStatusBLEPropertiesGetAll.setInterface((BTSCommonEnumClass)BTS_GEN_DBUS_SERVICE_BLE_PROPERTIES_GET_ALL);
      _serviceStatusBLEGenInitialization.setInterface((BTSCommonEnumClass)BTS_GEN_DBUS_SERVICE_BLE_GEN_INITIALIZATION);
      _serviceStatusBLEHliInitialization.setInterface((BTSCommonEnumClass)BTS_GEN_DBUS_SERVICE_BLE_HLI_INITIALIZATION);
      _serviceStatusBLEAdapter.setInterface((BTSCommonEnumClass)BTS_GEN_DBUS_SERVICE_BLE_ADAPTER);
      _serviceStatusBLEDevice.setInterface((BTSCommonEnumClass)BTS_GEN_DBUS_SERVICE_BLE_DEVICE);
      _serviceStatusBLESeamlessPairing.setInterface((BTSCommonEnumClass)BTS_GEN_DBUS_SERVICE_BLE_SEAMLESS_PAIRING);

      _serviceGroupBLEGeneral.addServiceStatus(&_serviceStatusBLEObjectManagerSystem);
      _serviceGroupBLEGeneral.addServiceStatus(&_serviceStatusBLEPropertiesGetAll);
      _serviceGroupBLEGeneral.addServiceStatus(&_serviceStatusBLEGenInitialization);
      _serviceGroupBLEGeneral.addServiceStatus(&_serviceStatusBLEHliInitialization);
      _serviceGroupBLEGeneral.addServiceStatus(&_serviceStatusBLEAdapter);

      _serviceStatusBLEObjectManagerSystem.setEnabled(true);
      _serviceStatusBLEPropertiesGetAll.setEnabled(true);
      _serviceStatusBLEGenInitialization.setEnabled(true);
      _serviceStatusBLEHliInitialization.setEnabled(true);
      _serviceStatusBLEAdapter.setEnabled(true);
      _serviceStatusBLEDevice.setEnabled(true);
      _serviceStatusBLESeamlessPairing.setEnabled(true);

      _serviceStatusBLEObjectManagerSystem.addServiceInfo(evoParser.getRootObjectPath());
      evoInterface = ::ccdbusif::evolution::IF_LE_ADAPTER;
      _serviceStatusBLEPropertiesGetAll.addServiceInfo(evoParser.getInterface2BusType(evoInterface), evoParser.getInterface2BusName(evoInterface), evoParser.getInterface2ObjectPath(evoInterface)); // LE_Adapter1
      _serviceStatusBLEGenInitialization.addServiceInfo("<gen_initialization>");
      _serviceStatusBLEHliInitialization.addServiceInfo("<hli_initialization>");
      _serviceStatusBLEAdapter.addServiceInfo(evoParser.getInterface2ObjectPath(::ccdbusif::evolution::IF_LE_ADAPTER));

      _serviceStatusBLEObjectManagerSystem.setAvailability(BTS_DBUS_SERVICE_WAITING);
      // evoInterface = ::ccdbusif::evolution::IF_LE_ADAPTER;
      _serviceStatusBLEPropertiesGetAll.setAvailability(evoParser.getInterface2BusType(evoInterface), evoParser.getInterface2BusName(evoInterface), evoParser.getInterface2ObjectPath(evoInterface), BTS_DBUS_SERVICE_AVAILABLE); // LE_Adapter1
      _serviceStatusBLEGenInitialization.setAvailability(BTS_DBUS_SERVICE_WAITING);
      _serviceStatusBLEHliInitialization.setAvailability(BTS_DBUS_SERVICE_WAITING);
      _serviceStatusBLEAdapter.setAvailability(BTS_DBUS_SERVICE_WAITING);

      dbusItem.dbusInterface = (BTSCommonEnumClass)BTS_GEN_DBUS_SERVICE_BLE_OBJECT_MANAGER_SYSTEM;
      dbusItem.busType = (BTSCommonEnumClass)evoParser.getInterface2BusType(::ccdbusif::evolution::IF_LE_ADAPTER);
      dbusItem.busName = evoParser.getInterface2BusName(::ccdbusif::evolution::IF_LE_ADAPTER);
      dbusItem.objPath = evoParser.getRootObjectPath();
      dbusInterfaces.push_back(dbusItem);

      // evoInterface = ::ccdbusif::evolution::IF_LE_ADAPTER;
      dbusItem.dbusInterface = (BTSCommonEnumClass)BTS_GEN_DBUS_SERVICE_BLE_PROPERTIES_GET_ALL;
      dbusItem.busType = (BTSCommonEnumClass)evoParser.getInterface2BusType(evoInterface);
      dbusItem.busName = evoParser.getInterface2BusName(evoInterface);
      dbusItem.objPath = evoParser.getInterface2ObjectPath(evoInterface);
      dbusInterfaces.push_back(dbusItem);

      dbusItem.dbusInterface = (BTSCommonEnumClass)BTS_GEN_DBUS_SERVICE_BLE_ADAPTER;
      dbusInterfaces.push_back(dbusItem);

      _serviceGroupBLEGeneral.checkAvailability();

      ETG_TRACE_USR1((" ### BLE: startup started ###"));
   }
}

bool BluetoothLowEnergyAlpsEvolutionExt::handleIpc2BtsServiceAvailabilityConnection(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_ServiceAvailabilityConnection* message)
{
   FW_IF_NULL_PTR_RETURN_FALSE(message);

   // TODO
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);
   (void)(messageItem);

   return true;
}

bool BluetoothLowEnergyAlpsEvolutionExt::handleIpc2BtsGetAllPropertiesConnection(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_GetAllPropertiesConnection* message)
{
   FW_IF_NULL_PTR_RETURN_FALSE(message);

   // TODO
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);
   (void)(messageItem);

   return true;
}

bool BluetoothLowEnergyAlpsEvolutionExt::handleIpc2BtsCreateLeDeviceResult(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_CreateLeDeviceResult* message)
{
   FW_IF_NULL_PTR_RETURN_FALSE(message);

   // TODO
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);
   (void)(messageItem);

   return true;
}

bool BluetoothLowEnergyAlpsEvolutionExt::handleIpc2BtsRemoveLeDeviceResult(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_RemoveLeDeviceResult* message)
{
   FW_IF_NULL_PTR_RETURN_FALSE(message);

   // TODO
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);
   (void)(messageItem);

   return true;
}

bool BluetoothLowEnergyAlpsEvolutionExt::handleIpc2BtsLeAdapterInitialization(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_LeAdapterInitialization* message)
{
   FW_IF_NULL_PTR_RETURN_FALSE(message);

   // TODO
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);
   (void)(messageItem);

   return true;
}

bool BluetoothLowEnergyAlpsEvolutionExt::handleIpc2BtsLeDeviceCreated(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_LeDeviceCreated* message)
{
   FW_IF_NULL_PTR_RETURN_FALSE(message);

   // TODO
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);
   (void)(messageItem);

   return true;
}

bool BluetoothLowEnergyAlpsEvolutionExt::handleIpc2BtsLeDeviceRemoved(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_LeDeviceRemoved* message)
{
   FW_IF_NULL_PTR_RETURN_FALSE(message);

   // TODO
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);
   (void)(messageItem);

   return true;
}

bool BluetoothLowEnergyAlpsEvolutionExt::handleIpc2BtsLeAdapterAddressTypeUpdate(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_LeAdapterAddressTypeUpdate* message)
{
   FW_IF_NULL_PTR_RETURN_FALSE(message);

   // TODO
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);
   (void)(messageItem);

   return true;
}

bool BluetoothLowEnergyAlpsEvolutionExt::handleIpc2BtsLeAdapterAddressUpdate(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_LeAdapterAddressUpdate* message)
{
   FW_IF_NULL_PTR_RETURN_FALSE(message);

   // TODO
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);
   (void)(messageItem);

   return true;
}

bool BluetoothLowEnergyAlpsEvolutionExt::handleIpc2BtsLeAdapterAliasUpdate(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_LeAdapterAliasUpdate* message)
{
   FW_IF_NULL_PTR_RETURN_FALSE(message);

   // TODO
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);
   (void)(messageItem);

   return true;
}

bool BluetoothLowEnergyAlpsEvolutionExt::handleIpc2BtsLeAdapterAdvertisingTimeoutUpdate(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_LeAdapterAdvertisingTimeoutUpdate* message)
{
   FW_IF_NULL_PTR_RETURN_FALSE(message);

   // TODO
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);
   (void)(messageItem);

   return true;
}

bool BluetoothLowEnergyAlpsEvolutionExt::handleIpc2BtsLeAdapterDiscoverableModeUpdate(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_LeAdapterDiscoverableModeUpdate* message)
{
   FW_IF_NULL_PTR_RETURN_FALSE(message);

   // TODO
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);
   (void)(messageItem);

   return true;
}

bool BluetoothLowEnergyAlpsEvolutionExt::handleIpc2BtsLeAdapterAdvertisingDataUpdate(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_LeAdapterAdvertisingDataUpdate* message)
{
   FW_IF_NULL_PTR_RETURN_FALSE(message);

   // TODO
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);
   (void)(messageItem);

   return true;
}

bool BluetoothLowEnergyAlpsEvolutionExt::handleIpc2BtsLeAdapterAdvertiseEnableUpdate(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_LeAdapterAdvertiseEnableUpdate* message)
{
   FW_IF_NULL_PTR_RETURN_FALSE(message);

   // TODO
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);
   (void)(messageItem);

   return true;
}

bool BluetoothLowEnergyAlpsEvolutionExt::handleIpc2BtsAddLeDeviceObjectPathMapping(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_AddLeDeviceObjectPathMapping* message)
{
   FW_IF_NULL_PTR_RETURN_FALSE(message);

   // TODO
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);
   (void)(messageItem);

   return true;
}

bool BluetoothLowEnergyAlpsEvolutionExt::handleIpc2BtsDelLeDeviceObjectPathMapping(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_DelLeDeviceObjectPathMapping* message)
{
   FW_IF_NULL_PTR_RETURN_FALSE(message);

   // TODO
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);
   (void)(messageItem);

   return true;
}

bool BluetoothLowEnergyAlpsEvolutionExt::handleIpc2BtsLeDeviceDisconnectResult(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_LeDeviceDisconnectResult* message)
{
   FW_IF_NULL_PTR_RETURN_FALSE(message);

   // TODO
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);
   (void)(messageItem);

   return true;
}

bool BluetoothLowEnergyAlpsEvolutionExt::handleIpc2BtsLeDisconnectionComp(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_LeDisconnectionComp* message)
{
   FW_IF_NULL_PTR_RETURN_FALSE(message);

   // TODO
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);
   (void)(messageItem);

   return true;
}

bool BluetoothLowEnergyAlpsEvolutionExt::handleIpc2BtsLeDeviceAddressTypeUpdate(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_LeDeviceAddressTypeUpdate* message)
{
   FW_IF_NULL_PTR_RETURN_FALSE(message);

   // TODO
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);
   (void)(messageItem);

   return true;
}

bool BluetoothLowEnergyAlpsEvolutionExt::handleIpc2BtsLeDeviceAddressUpdate(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_LeDeviceAddressUpdate* message)
{
   FW_IF_NULL_PTR_RETURN_FALSE(message);

   // TODO
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);
   (void)(messageItem);

   return true;
}

bool BluetoothLowEnergyAlpsEvolutionExt::handleIpc2BtsLeDeviceConnectedUpdate(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_LeDeviceConnectedUpdate* message)
{
   FW_IF_NULL_PTR_RETURN_FALSE(message);

   // TODO
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);
   (void)(messageItem);

   return true;
}

bool BluetoothLowEnergyAlpsEvolutionExt::handleIpc2BtsSendSeamlessPairingErrorCodeResult(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_SendSeamlessPairingErrorCodeResult* message)
{
   FW_IF_NULL_PTR_RETURN_FALSE(message);

   // TODO
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);
   (void)(messageItem);

   return true;
}

bool BluetoothLowEnergyAlpsEvolutionExt::handleIpc2BtsSeamlessPairingAllowAccessResult(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_SeamlessPairingAllowAccessResult* message)
{
   FW_IF_NULL_PTR_RETURN_FALSE(message);

   // TODO
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);
   (void)(messageItem);

   return true;
}

bool BluetoothLowEnergyAlpsEvolutionExt::handleIpc2BtsSeamlessPairingErrorCodeReceived(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_SeamlessPairingErrorCodeReceived* message)
{
   FW_IF_NULL_PTR_RETURN_FALSE(message);

   // TODO
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);
   (void)(messageItem);

   return true;
}

bool BluetoothLowEnergyAlpsEvolutionExt::handleIpc2BtsSeamlessPairingAccessRequest(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_SeamlessPairingAccessRequest* message)
{
   FW_IF_NULL_PTR_RETURN_FALSE(message);

   // TODO
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);
   (void)(messageItem);

   return true;
}

bool BluetoothLowEnergyAlpsEvolutionExt::handleIpc2BtsSeamlessPairingPairingDataUpdate(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_SeamlessPairingPairingDataUpdate* message)
{
   FW_IF_NULL_PTR_RETURN_FALSE(message);

   // TODO
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);
   (void)(messageItem);

   return true;
}

} //alpsevolutionext
} //genivi
} //btstackif
