/**
 * @file BasicConfigurationAlpsEvolutionExt.cpp
 *
 * @par SW-Component
 * State machine for basic configuration
 *
 * @brief Implementation of Alps Evolution Genivi basic configuration state machine.
 *
 * @copyright (C) 2017 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Source file for implementation of Alps Evolution Genivi basic configuration state machine.
 */

#include "BasicConfigurationAlpsEvolutionExt.h"
#include "IBasicConfigurationCallback.h"
#include "IBasicControl.h"
#include "Bts2Ipc_MessageWrapper_GEN.h"
#include "Ipc2Bts_MessageWrapper_GEN.h"
#include "EvolutionGeniviUtils.h"

#include "cc_dbus_if/EvolutionGeniviUtility.h"

#include "TraceClasses.h"
#include "FwAssert.h"
#include "FwTrace.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_BTS_CONTROL
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/BasicConfigurationAlpsEvolutionExt.cpp.trc.h"
#endif
#endif

namespace btstackif {
namespace genivi {
namespace alpsevolutionext {

BasicConfigurationAlpsEvolutionExt::BasicConfigurationAlpsEvolutionExt() :
_callback(0),
_controlIf(0),
_requestedData()
{
}

BasicConfigurationAlpsEvolutionExt::~BasicConfigurationAlpsEvolutionExt()
{
   _callback = 0;
   _controlIf = 0;
}

void BasicConfigurationAlpsEvolutionExt::reset(void)
{
   _requestedData.clear();
}

void BasicConfigurationAlpsEvolutionExt::setCallback(IN IBasicConfigurationCallback* callback)
{
   _callback = callback;

   FW_NORMAL_ASSERT(0 != _callback);
}

void BasicConfigurationAlpsEvolutionExt::setControlIf(IN IBasicControl* control)
{
   _controlIf = control;

   FW_NORMAL_ASSERT(0 != _controlIf);
}

void BasicConfigurationAlpsEvolutionExt::getConfigurationParameter(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSConfigurationParameter configurationParameter)
{
   (void)(bts2AppMsgList);

   switch(configurationParameter)
   {
      case BTS_CONFIG_PARAM_PAGE_TIMEOUT:
      case BTS_CONFIG_PARAM_PAGE_TIMEOUT_SECOND:
      {
         createBtApplReadConfigurationReq(bts2IpcMsgList, 0, convert2EvolutionId(configurationParameter));
         break;
      }
      default:
      {
         FW_NORMAL_ASSERT_ALWAYS();
         break;
      }
   }
}

void BasicConfigurationAlpsEvolutionExt::setConfigurationParameter(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSConfigurationParameter configurationParameter, IN const uint64_t data)
{
   (void)(bts2AppMsgList);

   switch(configurationParameter)
   {
      case BTS_CONFIG_PARAM_PAGE_TIMEOUT:
      case BTS_CONFIG_PARAM_PAGE_TIMEOUT_SECOND:
      {
         addRequestedData(_requestedData, configurationParameter, data);
         BTSEvoConfigDataStream streamData;
         ::ccdbusif::evolution_genivi_utility::writeUInt16(streamData, convertPageTimeout2Evolution((uint16_t)data));
         createBtApplSetConfigurationReq(bts2IpcMsgList, 0, convert2EvolutionId(configurationParameter), streamData);
         break;
      }
      default:
      {
         FW_NORMAL_ASSERT_ALWAYS();
         break;
      }
   }
}

void BasicConfigurationAlpsEvolutionExt::setConfigurationParameter(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSConfigurationParameter configurationParameter, IN const ::std::string& data)
{
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);
   (void)(configurationParameter);
   (void)(data);

   FW_NORMAL_ASSERT_ALWAYS();
}

void BasicConfigurationAlpsEvolutionExt::sendVirtualConfigurationParameterUpdate(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSConfigurationParameter configurationParameter, IN const uint64_t data, IN const BTSIpcCommonErrorCode errorCode)
{
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);

   FW_IF_NULL_PTR_RETURN(_controlIf);

   Ipc2Bts_BaseMessage* msg(0);

   switch(configurationParameter)
   {
      case BTS_CONFIG_PARAM_PAGE_TIMEOUT:
      case BTS_CONFIG_PARAM_PAGE_TIMEOUT_SECOND:
      {
         addRequestedData(_requestedData, configurationParameter, data);
         msg = createVirtualBtApplSetConfigurationCfmMsg(0, convert2EvolutionId(configurationParameter), errorCode);
         break;
      }
      default:
      {
         FW_NORMAL_ASSERT_ALWAYS();
         break;
      }
   }

   if(0 != msg)
   {
      _controlIf->sendInternalIpc2BtsMessage(msg);
   }
}

void BasicConfigurationAlpsEvolutionExt::sendVirtualConfigurationParameterUpdate(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSConfigurationParameter configurationParameter, IN const ::std::string& data, IN const BTSIpcCommonErrorCode errorCode)
{
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);
   (void)(configurationParameter);
   (void)(data);
   (void)(errorCode);

   FW_NORMAL_ASSERT_ALWAYS();
}

IBasicConfigurationRequest* BasicConfigurationAlpsEvolutionExt::getRequestIf(void)
{
   return this;
}

void BasicConfigurationAlpsEvolutionExt::handleSetConfigurationReqResult(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSIpcCommonErrorCode errorCode, IN const BTSInstanceIndex instance, IN const BTSEvoConfigIdEnum configId)
{
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);
   (void)(messageItem);
   (void)(errorCode);
   (void)(instance);
   (void)(configId);

   /*
    * current stack behavior is as follows:
    * - either DBUS method return or DBUS error message is sent as answer
    * - DBUS signal message containing "confirmation" data is sent in both cases; this message indicates same error as answer messages
    * => answer can be ignored
    */
}

void BasicConfigurationAlpsEvolutionExt::handleSetConfigurationCfm(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSStatusCode status, IN const BTSInstanceIndex instance, IN const BTSEvoConfigIdEnum configId)
{
   FW_IF_NULL_PTR_RETURN(_callback);

   (void)(instance);

   const BTSConfigurationParameter configurationParameter(convert2InternalId(configId));

   switch(configurationParameter)
   {
      case BTS_CONFIG_PARAM_PAGE_TIMEOUT:
      case BTS_CONFIG_PARAM_PAGE_TIMEOUT_SECOND:
      {
         const uint64_t data(getRequestedIntData(_requestedData, configurationParameter));
         delRequestedData(_requestedData, configurationParameter);
         _callback->updateConfigurationParameter(bts2IpcMsgList, bts2AppMsgList, messageItem, configurationParameter, data, (BTS_STATUS_CODE_SUCCESS == status));
         break;
      }
      default:
      {
         FW_NORMAL_ASSERT_ALWAYS();
         break;
      }
   }
}

void BasicConfigurationAlpsEvolutionExt::handleReadConfigurationReqResult(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSIpcCommonErrorCode errorCode, IN const BTSInstanceIndex instance, IN const BTSEvoConfigIdEnum configId)
{
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);
   (void)(messageItem);
   (void)(errorCode);
   (void)(instance);
   (void)(configId);

   /*
    * current stack behavior is as follows:
    * - either DBUS method return or DBUS error message is sent as answer
    * - DBUS signal message containing "confirmation" data is sent in both cases; this message indicates same error as answer messages
    * => answer can be ignored
    */
}

void BasicConfigurationAlpsEvolutionExt::handleReadConfigurationCfm(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSStatusCode status, IN const BTSInstanceIndex instance, IN const BTSEvoConfigIdEnum configId, IN const BTSEvoConfigDataStream& data)
{
   FW_IF_NULL_PTR_RETURN(_callback);

   (void)(instance);

   const BTSConfigurationParameter configurationParameter(convert2InternalId(configId));

   switch(configurationParameter)
   {
      case BTS_CONFIG_PARAM_PAGE_TIMEOUT:
      case BTS_CONFIG_PARAM_PAGE_TIMEOUT_SECOND:
      {
         uint16_t configData(0);
         bool success(false);
         if((BTS_STATUS_CODE_SUCCESS == status) && (2 == data.size()))
         {
            ::ccdbusif::evolution_genivi_utility::readUInt16(configData, data);
            success = true;
         }
         _callback->updateConfigurationParameter(bts2IpcMsgList, bts2AppMsgList, messageItem, configurationParameter, convertPageTimeout2Internal(configData), success);
         break;
      }
      default:
      {
         FW_NORMAL_ASSERT_ALWAYS();
         break;
      }
   }
}

BTSEvoConfigIdEnum BasicConfigurationAlpsEvolutionExt::convert2EvolutionId(IN const BTSConfigurationParameter configurationParameter) const
{
   BTSEvoConfigIdEnum configId(BTS_EVO_CONFIG_ID_LAST);

   switch(configurationParameter)
   {
      case BTS_CONFIG_PARAM_PAGE_TIMEOUT:
         configId = BTS_EVO_CONFIG_ID_page_timeout;
         break;
      case BTS_CONFIG_PARAM_PAGE_TIMEOUT_SECOND:
         configId = BTS_EVO_CONFIG_ID_page_timeout_2nd;
         break;
      default:
         FW_NORMAL_ASSERT_ALWAYS();
         break;
   }

   return configId;
}

BTSConfigurationParameter BasicConfigurationAlpsEvolutionExt::convert2InternalId(IN const BTSEvoConfigIdEnum configId) const
{
   BTSConfigurationParameter configurationParameter(BTS_CONFIG_PARAM_LAST);

   switch(configId)
   {
      case BTS_EVO_CONFIG_ID_page_timeout:
         configurationParameter = BTS_CONFIG_PARAM_PAGE_TIMEOUT;
         break;
      case BTS_EVO_CONFIG_ID_page_timeout_2nd:
         configurationParameter = BTS_CONFIG_PARAM_PAGE_TIMEOUT_SECOND;
         break;
      default:
         FW_NORMAL_ASSERT_ALWAYS();
         break;
   }

   return configurationParameter;
}

uint16_t BasicConfigurationAlpsEvolutionExt::convertPageTimeout2Evolution(IN const uint16_t data) const
{
   // configuration value shall provide number of baseband slots (0.625 msec)
   // max allowed value in seconds is 29

   uint32_t returnValue(data);

   if(29000 < returnValue)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      returnValue = 29000;
   }

   returnValue = (uint32_t)(returnValue * 1000);
   returnValue = (uint32_t)(returnValue / 625);

   return (uint16_t)(returnValue);
}

uint16_t BasicConfigurationAlpsEvolutionExt::convertPageTimeout2Internal(IN const uint16_t data) const
{
   // received configuration value provides number of baseband slots (0.625 msec)

   uint32_t returnValue(data);

   returnValue = (uint32_t)(returnValue * 625);
   returnValue = (uint32_t)(returnValue / 1000);

   return (uint16_t)(returnValue);
}

void BasicConfigurationAlpsEvolutionExt::addRequestedData(OUT ::std::map< BTSConfigurationParameter, uint64_t >& requestedDataList, IN const BTSConfigurationParameter configurationParameter, IN const uint64_t data) const
{
   requestedDataList[configurationParameter] = data;
}

void BasicConfigurationAlpsEvolutionExt::delRequestedData(OUT ::std::map< BTSConfigurationParameter, uint64_t >& requestedDataList, IN const BTSConfigurationParameter configurationParameter) const
{
   ::std::map< BTSConfigurationParameter, uint64_t >::iterator it = requestedDataList.find(configurationParameter);
   if(requestedDataList.end() != it)
   {
      requestedDataList.erase(it);
   }
}

uint64_t BasicConfigurationAlpsEvolutionExt::getRequestedIntData(IN const ::std::map< BTSConfigurationParameter, uint64_t >& requestedDataList, IN const BTSConfigurationParameter configurationParameter) const
{
   ::std::map< BTSConfigurationParameter, uint64_t >::const_iterator it = requestedDataList.find(configurationParameter);
   if(requestedDataList.end() != it)
   {
      return it->second;
   }
   else
   {
      FW_NORMAL_ASSERT_ALWAYS();
      return 0;
   }
}

void BasicConfigurationAlpsEvolutionExt::createBtApplReadConfigurationReq(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, IN const BTSInstanceIndex instance, IN const BTSEvoConfigIdEnum configId) const
{
   Bts2Ipc_BtApplReadConfigurationReq* msg = ptrNew_Bts2Ipc_BtApplReadConfigurationReq();
   if(0 != msg)
   {
      msg->setInstance(instance);
      msg->setConfigId(configId);
      msg->setResponseMessageFlag(true);

      bts2IpcMsgList.push_back(msg);
   }
}

void BasicConfigurationAlpsEvolutionExt::createBtApplSetConfigurationReq(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, IN const BTSInstanceIndex instance, IN const BTSEvoConfigIdEnum configId, IN const BTSEvoConfigDataStream& data) const
{
   Bts2Ipc_BtApplSetConfigurationReq* msg = ptrNew_Bts2Ipc_BtApplSetConfigurationReq();
   if(0 != msg)
   {
      msg->setInstance(instance);
      msg->setConfigId(configId);
      msg->setData(data);
      msg->setResponseMessageFlag(true);

      bts2IpcMsgList.push_back(msg);
   }
}

Ipc2Bts_BaseMessage* BasicConfigurationAlpsEvolutionExt::createVirtualBtApplSetConfigurationCfmMsg(IN const BTSInstanceIndex instance, IN const BTSEvoConfigIdEnum configId, IN const BTSIpcCommonErrorCode errorCode) const
{
   Ipc2Bts_BtApplSetConfigurationCfm* msg = ptrNew_Ipc2Bts_BtApplSetConfigurationCfm();
   if(0 != msg)
   {
      msg->setInstance(instance);
      msg->setConfigId(configId);
      if(BTS_IPC_SUCCESS == errorCode)
      {
         msg->setStatus(BTS_STATUS_CODE_SUCCESS);
      }
      else
      {
         msg->setStatus(BTS_STATUS_CODE_FAILED);
      }
      msg->setIpcCommonErrorCode(BTS_IPC_SUCCESS);
   }

   return msg;
}


} //alpsevolutionext
} //genivi
} //btstackif
