/**
 * @file TestModeAlpsEvolutionExt.h
 *
 * @par SW-Component
 * State machine for test mode
 *
 * @brief Implementation of Alps Evolution Genivi test mode state machine.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Header file for implementation of Alps Evolution Genivi test mode state machine.
 */

#ifndef _TEST_MODE_ALPS_EVOLUTION_EXT_H_
#define _TEST_MODE_ALPS_EVOLUTION_EXT_H_

#include "ITestModeRequest.h"
#include "EvolutionGeniviStackTypes.h"
#include "BtsTimer.h"

namespace btstackif {
namespace genivi {
namespace alpsevolutionext {

/**
 * Test mode class.
 */
class TestModeAlpsEvolutionExt : public ITestModeRequest
{
public:
   /**
    * Default constructor.
    */
   TestModeAlpsEvolutionExt();

   /**
    * Destructor.
    */
   virtual ~TestModeAlpsEvolutionExt();

   virtual void reset(void);

   virtual void setCallback(IN ITestModeCallback* callback);

   virtual void setControlIf(IN IBasicControl* control);

   virtual void start(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address);

   virtual void stop(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address);

   virtual void getLinkQuality(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList);

   virtual void handleSetHciModeStatus(IN const bool hciMode);

   virtual void handleSetHciModeResult(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const bool hciMode, IN const bool success);

   /**
    * Handle test mode confirmation message.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] status: status
    */
   void handleBtApplTestModeCfm(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSStatusCode status);

   /**
    * Handle test mode complete indication message.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] enabled: enabled flag
    * @param[in] status: status
    */
   void handleBtApplTestModeCompInd(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const bool enabled, IN const BTSStatusCode status);

   /**
    * Handle test mode link quality confirmation message.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] status: status
    * @param[in] rssi: RSSI
    * @param[in] linkQuality: link quality
    */
   void handleBtApplTestModeLinkQualityCfm(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSStatusCode status, IN const BTSRssi rssi, IN const BTSLinkQuality linkQuality);

private:
   static TestModeAlpsEvolutionExt* _exemplar; /**< own instance */
   ITestModeCallback* _callback; /**< callback interface */
   IBasicControl* _controlIf; /**< basic control interface */
   bool _hci; /**< HCI/APP mode */
   bool _enable; /**< enable flag for test mode */
   BTSBDAddress _address; /**< BT device address */
   Timer _timerHciMode; /**< timer for HCI mode timeout handling */
   const BTSTimeValue _timeoutHciMode; /**< timeout for HCI mode update */

   void createSetAdapterHciModeMsg(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, IN const bool enable) const;

   void createBtApplTestModeReqMsg(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, IN const BTSBDAddress& address, IN const bool enable, IN const BTSTestMode mode, IN const BTSBtRole role) const;

   void createBtApplTestModeLinkQualityReqMsg(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, IN const BTSTestMode mode) const;

   /**
    * Get instance.
    *
    * @return = pointer to own instance
    */
   static TestModeAlpsEvolutionExt* getInstance(void);

   /**
    * Timer callback function.
    */
   static void timeoutHciModeWrapper(void);

   /**
    * Timer callback function.
    */
   void timeoutHciMode(void);
};

/*
 * inline methods
 */
inline TestModeAlpsEvolutionExt* TestModeAlpsEvolutionExt::getInstance(void)
{
   return _exemplar;
}

inline void TestModeAlpsEvolutionExt::timeoutHciModeWrapper(void)
{
   TestModeAlpsEvolutionExt* instance = TestModeAlpsEvolutionExt::getInstance();
   if(0 != instance)
   {
      instance->timeoutHciMode();
   }
}

} //alpsevolutionext
} //genivi
} //btstackif

#endif //_TEST_MODE_ALPS_EVOLUTION_EXT_H_
