/**
 * @file ISppConnectDisconnectAlpsEvolutionExt.h
 *
 * @par SW-Component
 * State machine for SPP connect/disconnect
 *
 * @brief Interface definition for Alps Evolution Genivi SPP connect/disconnect state machine.
 *
 * @copyright (C) 2017 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details This file defines the Alps Evolution Genivi interface for SPP connect/disconnect state machine.
 */

#ifndef _I_SPP_CONNECT_DISCONNECT_ALPS_EVOLUTION_EXT_H_
#define _I_SPP_CONNECT_DISCONNECT_ALPS_EVOLUTION_EXT_H_

#include "EvolutionGeniviStackTypes.h"

namespace btstackif {

// class forward declarations
class IProtocolManagerRequest;
class IDeviceManager;
class Bts2Ipc_BaseMessage;
class Bts2App_BaseMessage;

namespace genivi {

// class forward declarations
class IObjectPathManagerGenivi;

namespace alpsevolutionext {

/**
 * Interface definition.
 */
class ISppConnectDisconnectAlpsEvolutionExt
{
public:
   /**
    * Destructor.
    */
   virtual ~ISppConnectDisconnectAlpsEvolutionExt() {}

   /**
    * Get request interface.
    *
    * @return = request interface
    */
   virtual IProtocolManagerRequest* getRequestIf(void) = 0;

   /**
    * Set device manager interface.
    *
    * @param[in] deviceManagerIf: device manager interface
    */
   virtual void setDeviceManagerIf(IN IDeviceManager* deviceManagerIf) = 0;

   /**
    * Set object path manager interface.
    *
    * @param[in] objectPathManager: object path manager interface
    */
   virtual void setObjectPathManagerIf(IN IObjectPathManagerGenivi* objectPathManager) = 0;

   /**
    * Handle cancel request from stack.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] sent: flag indicating that response message was already sent
    * @param[in] token: DBUS token
    *
    * @return = true: request was answered,
    * @return = false: request was not answered
    */
   virtual bool handleCancelRequest(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const bool sent, IN const act_t token) = 0;

   /**
    * Handle cancel request from stack.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] device: object path of device
    * @param[in] sent: flag indicating that response message was already sent
    * @param[in] token: DBUS token
    *
    * @return = true: request was answered,
    * @return = false: request was not answered
    */
   virtual bool handleCancelRequestRequest(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSObjectPath& device, IN const bool sent, IN const act_t token) = 0;

   /**
    * Handle remote connect request.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] device: object path of device
    * @param[in] protocol: protocol id
    * @param[in] uuid: uuid
    * @param[in] token: DBUS token
    *
    * @return = true: request was handled,
    * @return = false: request was not handled
    */
   virtual bool handleRemoteConnect(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSObjectPath& device, IN const BTSProtocolId protocol, IN const BTSUuid& uuid, IN const act_t token) = 0;

   /**
    * Handle connect result.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] address: device address
    * @param[in] protocol: protocol id
    * @param[in] instance: SPP instance id
    * @param[in] opId: operation id
    * @param[in] errorCode: error code (result)
    */
   virtual void handleConnectResult(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId instance, IN const BTSOperationId opId, IN const BTSIpcCommonErrorCode errorCode) = 0;

   /**
    * Handle connect confirmation.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] opId: operation id
    * @param[in] status: status
    */
   virtual void handleConnectCfm(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSOperationId opId, IN const BTSStatusCode status) = 0;

   /**
    * Handle device name indication.
    *
    * @param[in] instance: SPP instance id
    * @param[in] deviceName: device name
    */
   virtual void handleDeviceNameInd(IN const BTSSppInstanceId instance, IN const BTSDeviceName& deviceName) = 0;

   /**
    * Set connect status.
    *
    * @param[in] address: device address
    * @param[in] protocol: protocol id
    * @param[in] instance: SPP instance id
    * @param[in] status: status
    */
   virtual void updateConnectStatus(IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId instance, IN const BTSStatusCode status) = 0;

   /**
    * Handle connect indication.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] address: device address
    * @param[in] protocol: protocol id
    * @param[in] instance: SPP instance id
    * @param[in] status: status
    */
   virtual void handleConnectInd(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId instance, IN const BTSStatusCode status) = 0;

   /**
    * Handle connect complete indication.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[out] connectFailed: flag indicating that ACL connect failed
    * @param[out] failedAddress: address of device the ACL connect failed to
    * @param[in] opId: operation id
    */
   virtual void handleConnectCompInd(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, OUT bool& connectFailed, OUT BTSBDAddress& failedAddress, IN const BTSOperationId opId) = 0;

   /**
    * Handle disconnect result.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] address: device address
    * @param[in] protocol: protocol id
    * @param[in] instance: SPP instance id
    * @param[in] opId: operation id
    * @param[in] errorCode: error code (result)
    */
   virtual void handleDisconnectResult(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId instance, IN const BTSOperationId opId, IN const BTSIpcCommonErrorCode errorCode) = 0;

   /**
    * Handle disconnect confirmation.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] opId: operation id
    * @param[in] status: status
    */
   virtual void handleDisconnectCfm(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSOperationId opId, IN const BTSStatusCode status) = 0;

   /**
    * Set disconnect reason.
    *
    * @param[in] address: device address
    * @param[in] protocol: protocol id
    * @param[in] instance: SPP instance id
    * @param[in] reason: disconnect reason
    */
   virtual void updateDisconnectReason(IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId instance, IN const BTSInternalDisconnectReason reason) = 0;

   /**
    * Handle disconnect indication.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] address: device address
    * @param[in] protocol: protocol id
    * @param[in] instance: SPP instance id
    * @param[in] status: status
    */
   virtual void handleDisconnectInd(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId instance, IN const BTSStatusCode status) = 0;

   /**
    * Handle disconnect complete indication.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] opId: operation id
    */
   virtual void handleDisconnectCompInd(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSOperationId opId) = 0;
};

} //alpsevolutionext
} //genivi
} //btstackif

#endif //_I_SPP_CONNECT_DISCONNECT_ALPS_EVOLUTION_EXT_H_
