/**
 * @file IBluetoothLowEnergyAlpsEvolutionExt.h
 *
 * @par SW-Component
 * State machine for Bluetooth Low Energy
 *
 * @brief Interface definition for Alps Evolution Genivi Bluetooth Low Energy state machine.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details This file defines the interface for Alps Evolution Genivi Bluetooth Low Energy state machine.
 */

#ifndef _I_BLUETOOTH_LOW_ENERGY_ALPS_EVOLUTION_EXT_H_
#define _I_BLUETOOTH_LOW_ENERGY_ALPS_EVOLUTION_EXT_H_

#include "EvolutionGeniviStackTypes.h"

namespace btstackif {

// class forward declarations
class IBasicControl;
class ITimerPool;
class Bts2Ipc_BaseMessage;
class Bts2App_BaseMessage;

namespace genivi {

// class forward declarations
class Ipc2Bts_ServiceAvailabilityConnection;
class Ipc2Bts_GetAllPropertiesConnection;
class Ipc2Bts_CreateLeDeviceResult;
class Ipc2Bts_RemoveLeDeviceResult;
class Ipc2Bts_LeAdapterInitialization;
class Ipc2Bts_LeDeviceCreated;
class Ipc2Bts_LeDeviceRemoved;
class Ipc2Bts_LeAdapterAddressTypeUpdate;
class Ipc2Bts_LeAdapterAddressUpdate;
class Ipc2Bts_LeAdapterAliasUpdate;
class Ipc2Bts_LeAdapterAdvertisingTimeoutUpdate;
class Ipc2Bts_LeAdapterDiscoverableModeUpdate;
class Ipc2Bts_LeAdapterAdvertisingDataUpdate;
class Ipc2Bts_LeAdapterAdvertiseEnableUpdate;
class Ipc2Bts_AddLeDeviceObjectPathMapping;
class Ipc2Bts_DelLeDeviceObjectPathMapping;
class Ipc2Bts_LeDeviceDisconnectResult;
class Ipc2Bts_LeDisconnectionComp;
class Ipc2Bts_LeDeviceAddressTypeUpdate;
class Ipc2Bts_LeDeviceAddressUpdate;
class Ipc2Bts_LeDeviceConnectedUpdate;
class Ipc2Bts_SendSeamlessPairingErrorCodeResult;
class Ipc2Bts_SeamlessPairingAllowAccessResult;
class Ipc2Bts_SeamlessPairingErrorCodeReceived;
class Ipc2Bts_SeamlessPairingAccessRequest;
class Ipc2Bts_SeamlessPairingPairingDataUpdate;

namespace alpsevolutionext {

/**
 * Interface definition.
 */
class IBluetoothLowEnergyAlpsEvolutionExt
{
public:
   /**
    * Destructor.
    */
   virtual ~IBluetoothLowEnergyAlpsEvolutionExt() {}

   /**
    * Set basic control interface.
    *
    * @param[in] control: basic control interface
    */
   virtual void setControlIf(IN IBasicControl* control) = 0;

   /**
    * Set timer pool interface.
    *
    * @param[in] timerPool: timer pool interface
    */
   virtual void setTimerPoolIf(IN ITimerPool* timerPool) = 0;

   /**
    * Enable/disable BLE feature.
    *
    * @param[in] enable: enable flag
    */
   virtual void enable(IN const bool enable) = 0;

   /**
    * Get enabled flag.
    *
    * @return = enabled flag
    */
   virtual bool getEnabled(void) const = 0;

   /**
    * Set configuration data for local stack.
    *
    * @param[out] dbusInterfaces: list of needed DBUS interfaces
    * @param[in] configuration: local stack configuration data
    * @param[in] userMode: user mode
    * @param[in] interface: stack interface type
    */
   virtual void setStackConfiguration(OUT ::std::vector< BTSDbusInterfaceItem >& dbusInterfaces, IN const BTSLocalStackConfiguration& configuration, IN const BTSUserMode userMode, IN const BTSInterfaceType interface) = 0;

   /**
    * Handle received Ipc2Bts message.
    * Handle any error within this function.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] message: received Ipc2Bts message
    *
    * @return = true: given message was handled ("extension is responsible for this message request"),
    * @return = false: given message was not handled ("extension is not responsible for this message request")
    */
   virtual bool handleIpc2BtsServiceAvailabilityConnection(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_ServiceAvailabilityConnection* message) = 0;
   virtual bool handleIpc2BtsGetAllPropertiesConnection(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_GetAllPropertiesConnection* message) = 0;
   virtual bool handleIpc2BtsCreateLeDeviceResult(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_CreateLeDeviceResult* message) = 0;
   virtual bool handleIpc2BtsRemoveLeDeviceResult(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_RemoveLeDeviceResult* message) = 0;
   virtual bool handleIpc2BtsLeAdapterInitialization(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_LeAdapterInitialization* message) = 0;
   virtual bool handleIpc2BtsLeDeviceCreated(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_LeDeviceCreated* message) = 0;
   virtual bool handleIpc2BtsLeDeviceRemoved(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_LeDeviceRemoved* message) = 0;
   virtual bool handleIpc2BtsLeAdapterAddressTypeUpdate(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_LeAdapterAddressTypeUpdate* message) = 0;
   virtual bool handleIpc2BtsLeAdapterAddressUpdate(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_LeAdapterAddressUpdate* message) = 0;
   virtual bool handleIpc2BtsLeAdapterAliasUpdate(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_LeAdapterAliasUpdate* message) = 0;
   virtual bool handleIpc2BtsLeAdapterAdvertisingTimeoutUpdate(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_LeAdapterAdvertisingTimeoutUpdate* message) = 0;
   virtual bool handleIpc2BtsLeAdapterDiscoverableModeUpdate(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_LeAdapterDiscoverableModeUpdate* message) = 0;
   virtual bool handleIpc2BtsLeAdapterAdvertisingDataUpdate(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_LeAdapterAdvertisingDataUpdate* message) = 0;
   virtual bool handleIpc2BtsLeAdapterAdvertiseEnableUpdate(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_LeAdapterAdvertiseEnableUpdate* message) = 0;
   virtual bool handleIpc2BtsAddLeDeviceObjectPathMapping(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_AddLeDeviceObjectPathMapping* message) = 0;
   virtual bool handleIpc2BtsDelLeDeviceObjectPathMapping(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_DelLeDeviceObjectPathMapping* message) = 0;
   virtual bool handleIpc2BtsLeDeviceDisconnectResult(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_LeDeviceDisconnectResult* message) = 0;
   virtual bool handleIpc2BtsLeDisconnectionComp(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_LeDisconnectionComp* message) = 0;
   virtual bool handleIpc2BtsLeDeviceAddressTypeUpdate(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_LeDeviceAddressTypeUpdate* message) = 0;
   virtual bool handleIpc2BtsLeDeviceAddressUpdate(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_LeDeviceAddressUpdate* message) = 0;
   virtual bool handleIpc2BtsLeDeviceConnectedUpdate(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_LeDeviceConnectedUpdate* message) = 0;
   virtual bool handleIpc2BtsSendSeamlessPairingErrorCodeResult(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_SendSeamlessPairingErrorCodeResult* message) = 0;
   virtual bool handleIpc2BtsSeamlessPairingAllowAccessResult(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_SeamlessPairingAllowAccessResult* message) = 0;
   virtual bool handleIpc2BtsSeamlessPairingErrorCodeReceived(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_SeamlessPairingErrorCodeReceived* message) = 0;
   virtual bool handleIpc2BtsSeamlessPairingAccessRequest(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_SeamlessPairingAccessRequest* message) = 0;
   virtual bool handleIpc2BtsSeamlessPairingPairingDataUpdate(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_SeamlessPairingPairingDataUpdate* message) = 0;
};

} //alpsevolutionext
} //genivi
} //btstackif

#endif //_I_BLUETOOTH_LOW_ENERGY_ALPS_EVOLUTION_EXT_H_
