/**
 * @file BluetoothStackErrorHandlingAlpsEvolutionExt.h
 *
 * @par SW-Component
 * State machine for Bluetooth stack error handling
 *
 * @brief Implementation of Alps Evolution Genivi Bluetooth stack error handling state machine.
 *
 * @copyright (C) 2018 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Header file for implementation of Alps Evolution Genivi Bluetooth stack error handling state machine.
 */

#ifndef _BLUETOOTH_STACK_ERROR_HANDLING_ALPS_EVOLUTION_EXT_H_
#define _BLUETOOTH_STACK_ERROR_HANDLING_ALPS_EVOLUTION_EXT_H_

#include "IBluetoothStackErrorHandlingRequest.h"
#include "IBluetoothStackErrorHandlingAlpsEvolutionExt.h"
#include "EvolutionGeniviStackTypes.h"

namespace btstackif {
namespace genivi {
namespace alpsevolutionext {

/**
 * Bluetooth stack error handling class.
 */
class BluetoothStackErrorHandlingAlpsEvolutionExt : public IBluetoothStackErrorHandlingRequest, public IBluetoothStackErrorHandlingAlpsEvolutionExt
{
public:
   /**
    * Default constructor.
    */
   BluetoothStackErrorHandlingAlpsEvolutionExt();

   /**
    * Destructor.
    */
   virtual ~BluetoothStackErrorHandlingAlpsEvolutionExt();

   virtual void reset(void);

   virtual void setCallback(IN IBluetoothStackErrorHandlingCallback* callback);

   virtual IBluetoothStackErrorHandlingRequest* getRequestIf(void);

   virtual void setSwitchBluetoothIf(IN ISwitchBluetooth* switchBluetoothIf);

   virtual void handleFatalError(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSIpcCommonErrorCode errorCode, IN const BTSFatalErrorLocation location, IN const BTSFatalErrorType type);

   virtual void handleGeneralError(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSIpcCommonErrorCode errorCode, IN const BTSStatusCode status, IN const BTSInternalOpcode binaryStatus, IN const BTSInternalOpcode opcode);

   virtual void handleGeniviDebugTrace(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSIpcCommonErrorCode errorCode, IN const BTSGeniviTraceNumber errFile, IN const BTSGeniviTraceNumber errLine, IN const BTSGeniviTraceNumber level, IN const BTSGeniviDebugTraceCategory category, IN const BTSGeniviTraceNumber errCode, IN const BTSGeniviTraceData& data);

   virtual void handleGeniviFatalError(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSIpcCommonErrorCode errorCode, IN const BTSGeniviTraceNumber errFile, IN const BTSGeniviTraceNumber errLine, IN const BTSGeniviTraceNumber errCode, IN const BTSGeniviTraceError& fmt, IN const BTSGeniviTraceError& errValue);

   virtual void handleReportDebugTrace(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSIpcCommonErrorCode errorCode, IN const BTSDebugTraceLocation location, IN const BTSDebugTraceLevel level, IN const BTSDebugTraceCategory category, IN const BTSDebugTraceErrorCode errCode, IN const BTSDebugTraceData& data);

private:
   IBluetoothStackErrorHandlingCallback* _callback; /**< callback interface */
   ISwitchBluetooth* _switchBluetoothIf; /**< switch Bluetooth interface */
   const BTSInternalOpcode _stackErrorOpcode; /**< stack error opcode */
   const BTSInternalOpcode _hardwareFailureOpcode; /**< hardware failure opcode */
   const bool _ignoreStatusCodeConnectRejectSecurityReasons; /**< ignore flag */
   const ::std::string _stackType; /**< stack type */
   const ::std::string _fatalError; /**< fatal error */
   const ::std::string _generalError; /**< general error */
   const ::std::string _debugTrace; /**< debug trace */
   const ::std::string _geniviFatalError; /**< genivi fatal error */
   const ::std::string _geniviDebugTrace; /**< genivi debug trace */
   const ::std::string _stackError; /**< stack error */
   const ::std::string _hardwareFailure; /**< hardware failure */

   void handleCriticalErrorInternal(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const ::std::string& errorType, IN const BTSStatusCode status, IN const BTSFatalErrorLocation location = 0, IN const BTSFatalErrorType type = 0);

   void handleGeneralErrorInternal(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSStatusCode status, IN const BTSInternalOpcode opcode);

   void handleOtherErrorInternal(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const ::std::string& errorType, IN const BTSIpcCommonErrorCode errorCode);
};

} //alpsevolutionext
} //genivi
} //btstackif

#endif //_BLUETOOTH_STACK_ERROR_HANDLING_ALPS_EVOLUTION_EXT_H_
