/**
 * @file AlpsEvolutionBluetoothLowEnergyDbusIf.cpp
 *
 * @par SW-Component
 * DBus Extension
 *
 * @brief Implementation of Alps Evolution Genivi Bluetooth low energy DBus interface extension.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Source file for implementation of Alps Evolution Genivi Bluetooth low energy DBus interface extension.
 */

#include "AlpsEvolutionBluetoothLowEnergyDbusIf.h"
#include "FwAssert.h"
#include "IDbusBase.h"
#include "Bts2Ipc_MessageWrapper_GEN.h"
#include "Ipc2Bts_MessageWrapper_GEN.h"

#include "cc_dbus_if/CcDbusIfControllerFactory.h"
#include "cc_dbus_if/CcDbusIfLaunchType.h"
#include "cc_dbus_if/ICcDbusIfController.h"
#include "cc_dbus_if/ICcDbusIfControllerClient.h"
#include "cc_dbus_if/IOrgFreedesktopDbusObjectManagerSendRequestIf.h"
#include "cc_dbus_if/ICcDbusIfControllerObjectManager.h"
#include "cc_dbus_if/IPropertiesGetAllDbusSendRequestIf.h"
#include "cc_dbus_if/ICcDbusIfControllerPropertiesGetAll.h"

#include "cc_dbus_if/EvolutionGeniviUtility.h"
#include "cc_dbus_if/EvolutionGeniviDbusTypes.h"
#include "cc_dbus_if/EvolutionGeniviDbusParser.h"

#include "cc_dbus_if/ICcDbusIfControllerEvolutionGenivi.h"

#include "cc_dbus_if/IEvolutionGeniviDbusLEAdapterSendRequestIf.h"
#include "cc_dbus_if/IEvolutionGeniviDbusLEDeviceSendRequestIf.h"
#include "cc_dbus_if/IEvolutionGeniviDbusLESeamlessPairingSendRequestIf.h"

#include "IAlpsEvolutionOrgFreedesktopObjectManagerCallbackIf.h"
#include "AlpsEvolutionOrgFreedesktopObjectManagerFactory.h"
#include "PropertiesGetAllDbusCallbackIf.h"
#include "PropertiesGetAllDbusFactory.h"
#include "EvolutionGeniviDbusLEAdapterFactory.h"
#include "EvolutionGeniviDbusLEDeviceFactory.h"
#include "EvolutionGeniviDbusLESeamlessPairingFactory.h"

#include "TraceClasses.h"
#include "FwTrace.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_BTS_CONTROL
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/AlpsEvolutionBluetoothLowEnergyDbusIf.cpp.trc.h"
#endif
#endif

namespace btstackif {
namespace genivi {
namespace alpsevolutionext {

AlpsEvolutionBluetoothLowEnergyDbusIf::AlpsEvolutionBluetoothLowEnergyDbusIf() :
_dbusBase(0),
_dbusRecHandler(0),
_enabled(false),
_initialized(false),
_controllerClient(0),
_objectManagerIf(0),
_propertiesGetAllIf(0),
_adapterIf(0),
_deviceIf(0),
_seamlessPairingIf(0),
_objectManagerCb(0),
_propertiesGetAllCb(0),
_adapterCb(0),
_deviceCb(0),
_seamlessPairingCb(0)
{
}

AlpsEvolutionBluetoothLowEnergyDbusIf::~AlpsEvolutionBluetoothLowEnergyDbusIf()
{
   _dbusBase = 0;
   _dbusRecHandler = 0;
   _controllerClient = 0;

   _objectManagerIf = 0;
   _propertiesGetAllIf = 0;

   _adapterIf = 0;
   _deviceIf = 0;
   _seamlessPairingIf = 0;

   AlpsEvolutionOrgFreedesktopObjectManagerFactory::getInstance().destroyCb();
   _objectManagerCb = 0;
   PropertiesGetAllDbusFactory::getInstance().destroyCb();
   _propertiesGetAllCb = 0;
   EvolutionGeniviDbusLEAdapterFactory::getInstance().destroyCb();
   _adapterCb = 0;
   EvolutionGeniviDbusLEDeviceFactory::getInstance().destroyCb();
   _deviceCb = 0;
   EvolutionGeniviDbusLESeamlessPairingFactory::getInstance().destroyCb();
   _seamlessPairingCb = 0;
}

void AlpsEvolutionBluetoothLowEnergyDbusIf::setDbusBaseIf(IN IDbusBase* dbusBase)
{
   _dbusBase = dbusBase;

   FW_NORMAL_ASSERT(0 != _dbusBase);
}

void AlpsEvolutionBluetoothLowEnergyDbusIf::setDbusRecHandlerIf(IN IDbusRecHandler* dbusRecHandler)
{
   _dbusRecHandler = dbusRecHandler;

   FW_NORMAL_ASSERT(0 != _dbusRecHandler);
}

void AlpsEvolutionBluetoothLowEnergyDbusIf::enable(IN const bool enable)
{
   if(enable == _enabled)
   {
      // required state already set
      return;
   }

   if(true == _initialized)
   {
      // already initialized => different state not allowed
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   // store new state
   _enabled = enable;

   if(true == _enabled)
   {
      // all stuff necessary for enabled state
   }
   else
   {
      // all stuff necessary for disabled state
   }
}

void AlpsEvolutionBluetoothLowEnergyDbusIf::initialize(void)
{
   ETG_TRACE_USR2((" initialize(): _enabled=%d _initialized=%d", _enabled, _initialized));

   if(false == _enabled)
   {
      // do not initialize
      return;
   }

   if(true == _initialized)
   {
      // already initialized
      return;
   }

   // all stuff necessary for initialization
   // will be done by setCcDbusIfControllerIf()

   _initialized = true;
}

void AlpsEvolutionBluetoothLowEnergyDbusIf::destroy(void)
{
   ETG_TRACE_USR2((" destroy(): _enabled=%d _initialized=%d", _enabled, _initialized));

   if(false == _enabled)
   {
      // nothing to do
      return;
   }

   if(false == _initialized)
   {
      // not initialized
      return;
   }

   // all stuff necessary for destruction
   // was done by resetCcDbusIfControllerIf()

   _initialized = false;
}

void AlpsEvolutionBluetoothLowEnergyDbusIf::setCcDbusIfControllerIf(IN const BTSFunctionBlock component, IN const BTSInterfaceType stackInterface, IN const BTSFunctionBlock subComponent, IN const BTSUserMode userMode,
         IN ::ccdbusif::ICcDbusIfController* controller, IN const ::std::vector< BTSDbusInterfaceItem >& dbusInterfaces, IN const BTSLocalConfigurationContainer& configuration)
{
   (void)(component);
   (void)(subComponent);
   (void)(configuration);

   ETG_TRACE_USR2((" setCcDbusIfControllerIf(): _enabled=%d _initialized=%d", _enabled, _initialized));

   if(false == _enabled)
   {
      // nothing to do
      return;
   }

   if(false == _initialized)
   {
      // not initialized
      return;
   }

   FW_IF_NULL_PTR_RETURN(_dbusBase);
   FW_IF_NULL_PTR_RETURN(_dbusRecHandler);
   FW_IF_NULL_PTR_RETURN(controller);

   if(BTS_USER_MODE_LAST <= userMode)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   if(BTS_IF_ALPS_EVOLUTION_GENIVI != stackInterface)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   // check if interfaces are already set (checking _controllerClient is enough)
   if(0 != _controllerClient)
   {
      // already set
      return;
   }

   BTSErrorCode errCode = BTS_OK;

   _controllerClient = controller->getCcDbusIfControllerClient();

   ::ccdbusif::ICcDbusIfControllerEvolutionGenivi* evolutionIf = controller->getCcDbusIfControllerEvolutionGenivi();
   ::ccdbusif::ICcDbusIfControllerObjectManager* objectManagerMainIf = controller->getCcDbusIfControllerObjectManager();
   ::ccdbusif::ICcDbusIfControllerPropertiesGetAll* propertiesGetAllMainIf = controller->getCcDbusIfControllerPropertiesGetAll();

   if((0 == _controllerClient) ||
      (0 == evolutionIf) ||
      (0 == objectManagerMainIf) ||
      (0 == propertiesGetAllMainIf))
   {
      FW_NORMAL_ASSERT_ALWAYS();
      errCode = BTS_INVALID_PARAM;
   }
   else
   {
      //---------------------------------------------------------------------
      _objectManagerIf = objectManagerMainIf->getObjectManagerProxy();
      if(0 == _objectManagerIf)
      {
         FW_NORMAL_ASSERT_ALWAYS();
         errCode = BTS_INVALID_PARAM;
      }
      // create callback handler
      if(0 == _objectManagerCb)
      {
         _objectManagerCb = AlpsEvolutionOrgFreedesktopObjectManagerFactory::getInstance().getCb(_dbusRecHandler);
      }
      if(0 == _objectManagerCb)
      {
         FW_NORMAL_ASSERT_ALWAYS();
         errCode = BTS_INVALID_PARAM;
      }
      // get callback id
      if(0 != _objectManagerCb)
      {
         if(0 == _objectManagerCb->getCallbackId())
         {
            _objectManagerCb->setCallbackId(controller->getCallbackId());
         }
         if(0 == _objectManagerCb->getCallbackId())
         {
            FW_NORMAL_ASSERT_ALWAYS();
            errCode = BTS_INVALID_PARAM;
         }
      }
      //---------------------------------------------------------------------
      _propertiesGetAllIf = propertiesGetAllMainIf->getPropertiesGetAllProxy();
      if(0 == _propertiesGetAllIf)
      {
         FW_NORMAL_ASSERT_ALWAYS();
         errCode = BTS_INVALID_PARAM;
      }
      // create callback handler
      if(0 == _propertiesGetAllCb)
      {
         _propertiesGetAllCb = PropertiesGetAllDbusFactory::getInstance().getCb(_dbusRecHandler);
      }
      if(0 == _propertiesGetAllCb)
      {
         FW_NORMAL_ASSERT_ALWAYS();
         errCode = BTS_INVALID_PARAM;
      }
      //---------------------------------------------------------------------
      _adapterIf = evolutionIf->getEvolutionGeniviLEAdapterProxy();
      if(0 == _adapterIf)
      {
         FW_NORMAL_ASSERT_ALWAYS();
         errCode = BTS_INVALID_PARAM;
      }
      // create callback handler
      if(0 == _adapterCb)
      {
         _adapterCb = EvolutionGeniviDbusLEAdapterFactory::getInstance().getCb(_dbusRecHandler);
      }
      if(0 == _adapterCb)
      {
         FW_NORMAL_ASSERT_ALWAYS();
         errCode = BTS_INVALID_PARAM;
      }
      //---------------------------------------------------------------------
      _deviceIf = evolutionIf->getEvolutionGeniviLEDeviceProxy();
      if(0 == _deviceIf)
      {
         FW_NORMAL_ASSERT_ALWAYS();
         errCode = BTS_INVALID_PARAM;
      }
      // create callback handler
      if(0 == _deviceCb)
      {
         _deviceCb = EvolutionGeniviDbusLEDeviceFactory::getInstance().getCb(_dbusRecHandler);
      }
      if(0 == _deviceCb)
      {
         FW_NORMAL_ASSERT_ALWAYS();
         errCode = BTS_INVALID_PARAM;
      }
      //---------------------------------------------------------------------
      _seamlessPairingIf = evolutionIf->getEvolutionGeniviLESeamlessPairingProxy();
      if(0 == _seamlessPairingIf)
      {
         FW_NORMAL_ASSERT_ALWAYS();
         errCode = BTS_INVALID_PARAM;
      }
      // create callback handler
      if(0 == _seamlessPairingCb)
      {
         _seamlessPairingCb = EvolutionGeniviDbusLESeamlessPairingFactory::getInstance().getCb(_dbusRecHandler);
      }
      if(0 == _seamlessPairingCb)
      {
         FW_NORMAL_ASSERT_ALWAYS();
         errCode = BTS_INVALID_PARAM;
      }
      //---------------------------------------------------------------------
   }

   if(BTS_OK != errCode)
   {
      // reset all
      _controllerClient = 0;
      _objectManagerIf = 0;
      _propertiesGetAllIf = 0;
      _adapterIf = 0;
      _deviceIf = 0;
      _seamlessPairingIf = 0;

      // do not delete callback handler
   }
   else
   {
      // set callback handler (if no send interface and no callback handler were created nothing will happen during next step)
      for(size_t i = 0; i < dbusInterfaces.size(); i++)
      {
         switch(dbusInterfaces[i].dbusInterface)
         {
            case (BTSCommonEnumClass)BTS_GEN_DBUS_SERVICE_BLE_OBJECT_MANAGER_SYSTEM:
               if((0 != _objectManagerIf) && (0 != _objectManagerCb))
               {
                  _dbusBase->createDbusServiceAvailabilityMessage(dbusInterfaces[i].dbusInterface, BTS_DBUS_SERVICE_WAITING);
                  _objectManagerIf->setCallbackIf(_objectManagerCb, true, _objectManagerCb->getCallbackId(), dbusInterfaces[i].objPath, dbusInterfaces[i].busName, (::ccdbusif::DbusBusType)dbusInterfaces[i].busType);
                  _objectManagerIf->createProxyIf(_objectManagerCb->getCallbackId(), false, dbusInterfaces[i].objPath, dbusInterfaces[i].busName, (::ccdbusif::DbusBusType)dbusInterfaces[i].busType);

                  // set device interface for handle created and removed devices
                  _objectManagerCb->setLEDeviceSendIf(_deviceIf);
                  // set device interface for handle created and removed seamless pairing objects
                  _objectManagerCb->setLESeamlessPairingSendIf(_seamlessPairingIf);
               }
               break;
            case (BTSCommonEnumClass)BTS_GEN_DBUS_SERVICE_BLE_PROPERTIES_GET_ALL:
               if((0 != _propertiesGetAllIf) && (0 != _propertiesGetAllCb))
               {
                  _dbusBase->createDbusServiceAvailabilityMessage(dbusInterfaces[i].dbusInterface, BTS_DBUS_SERVICE_WAITING, dbusInterfaces[i].busName, dbusInterfaces[i].objPath, dbusInterfaces[i].busType);
                  _propertiesGetAllIf->setCallbackIf(_propertiesGetAllCb, true, dbusInterfaces[i].busName, dbusInterfaces[i].objPath, (::ccdbusif::DbusBusType)dbusInterfaces[i].busType, 0);
               }
               break;
            case (BTSCommonEnumClass)BTS_GEN_DBUS_SERVICE_BLE_ADAPTER:
               if((0 != _adapterIf) && (0 != _adapterCb))
               {
                  _dbusBase->createDbusServiceAvailabilityMessage(dbusInterfaces[i].dbusInterface, BTS_DBUS_SERVICE_WAITING);
                  _adapterIf->setCallbackIf(_adapterCb, true);
               }
               break;
            case (BTSCommonEnumClass)BTS_GEN_DBUS_SERVICE_BLE_DEVICE:
               // created during runtime
               break;
            case (BTSCommonEnumClass)BTS_GEN_DBUS_SERVICE_BLE_SEAMLESS_PAIRING:
               // created during runtime
               break;
            default:
               break;
         }
      }

      // set callback for proxies created during runtime
      if((0 != _deviceIf) && (0 != _deviceCb))
      {
         _deviceIf->setCallbackIf(_deviceCb, true);
      }
      if((0 != _seamlessPairingIf) && (0 != _seamlessPairingCb))
      {
         _seamlessPairingIf->setCallbackIf(_seamlessPairingCb, true);
      }
   }
}

void AlpsEvolutionBluetoothLowEnergyDbusIf::resetCcDbusIfControllerIf(void)
{
   ETG_TRACE_USR2((" resetCcDbusIfControllerIf(): _enabled=%d _initialized=%d", _enabled, _initialized));

   if(false == _enabled)
   {
      // nothing to do
      return;
   }

   if(false == _initialized)
   {
      // not initialized
      return;
   }

   // check if interfaces are already reset (checking _controllerClient is enough)
   if(0 == _controllerClient)
   {
      // already reset
      return;
   }

   // reset all
   _controllerClient = 0;
   _objectManagerIf = 0;
   _propertiesGetAllIf = 0;
   _adapterIf = 0;
   _deviceIf = 0;
   _seamlessPairingIf = 0;

   // do not delete callback handler
}

bool AlpsEvolutionBluetoothLowEnergyDbusIf::setSimulationTestCommand(IN const char* testCommand, IN const unsigned int testData)
{
   (void)(testCommand);
   (void)(testData);
   return false;
}

bool AlpsEvolutionBluetoothLowEnergyDbusIf::setSimulationTestCommand(IN const char* testCommand, IN const unsigned char* testData)
{
   (void)(testCommand);
   (void)(testData);
   return false;
}

bool AlpsEvolutionBluetoothLowEnergyDbusIf::setSimulationTestCommand(IN const char* testCommand, IN const Ipc2Bts_BaseMessage& testData)
{
   (void)(testCommand);
   (void)(testData);
   return false;
}

bool AlpsEvolutionBluetoothLowEnergyDbusIf::sendMessage(OUT bool& deleteMsg, OUT bool& sendErrorMsg, IN Bts2Ipc_BaseMessage* message)
{
   if(false == _enabled)
   {
      // do not process
      return false;
   }

   if(false == _initialized)
   {
      // not initialized
      return false;
   }

   FW_IF_NULL_PTR_RETURN_FALSE(message);

   const BTSGenBts2IpcOpcode opcode((BTSGenBts2IpcOpcode)message->getOpCode());
   bool handled(true);

   switch(opcode)
   {
      case Bts2IpcOC_CreateLeDevice:
         handleBts2IpcCreateLeDevice(deleteMsg, sendErrorMsg, static_cast<Bts2Ipc_CreateLeDevice*>(message));
         break;
      case Bts2IpcOC_RemoveLeDevice:
         handleBts2IpcRemoveLeDevice(deleteMsg, sendErrorMsg, static_cast<Bts2Ipc_RemoveLeDevice*>(message));
         break;
      case Bts2IpcOC_GetLeAdapterAddressType:
         handleBts2IpcGetLeAdapterAddressType(deleteMsg, sendErrorMsg, static_cast<Bts2Ipc_GetLeAdapterAddressType*>(message));
         break;
      case Bts2IpcOC_GetLeAdapterAddress:
         handleBts2IpcGetLeAdapterAddress(deleteMsg, sendErrorMsg, static_cast<Bts2Ipc_GetLeAdapterAddress*>(message));
         break;
      case Bts2IpcOC_GetLeAdapterAlias:
         handleBts2IpcGetLeAdapterAlias(deleteMsg, sendErrorMsg, static_cast<Bts2Ipc_GetLeAdapterAlias*>(message));
         break;
      case Bts2IpcOC_SetLeAdapterAlias:
         handleBts2IpcSetLeAdapterAlias(deleteMsg, sendErrorMsg, static_cast<const Bts2Ipc_SetLeAdapterAlias*>(message));
         break;
      case Bts2IpcOC_GetLeAdapterAdvertisingTimeout:
         handleBts2IpcGetLeAdapterAdvertisingTimeout(deleteMsg, sendErrorMsg, static_cast<Bts2Ipc_GetLeAdapterAdvertisingTimeout*>(message));
         break;
      case Bts2IpcOC_SetLeAdapterAdvertisingTimeout:
         handleBts2IpcSetLeAdapterAdvertisingTimeout(deleteMsg, sendErrorMsg, static_cast<const Bts2Ipc_SetLeAdapterAdvertisingTimeout*>(message));
         break;
      case Bts2IpcOC_GetLeAdapterDiscoverableMode:
         handleBts2IpcGetLeAdapterDiscoverableMode(deleteMsg, sendErrorMsg, static_cast<Bts2Ipc_GetLeAdapterDiscoverableMode*>(message));
         break;
      case Bts2IpcOC_SetLeAdapterDiscoverableMode:
         handleBts2IpcSetLeAdapterDiscoverableMode(deleteMsg, sendErrorMsg, static_cast<const Bts2Ipc_SetLeAdapterDiscoverableMode*>(message));
         break;
      case Bts2IpcOC_GetLeAdapterAdvertisingData:
         handleBts2IpcGetLeAdapterAdvertisingData(deleteMsg, sendErrorMsg, static_cast<Bts2Ipc_GetLeAdapterAdvertisingData*>(message));
         break;
      case Bts2IpcOC_SetLeAdapterAdvertisingData:
         handleBts2IpcSetLeAdapterAdvertisingData(deleteMsg, sendErrorMsg, static_cast<const Bts2Ipc_SetLeAdapterAdvertisingData*>(message));
         break;
      case Bts2IpcOC_GetLeAdapterAdvertiseEnable:
         handleBts2IpcGetLeAdapterAdvertiseEnable(deleteMsg, sendErrorMsg, static_cast<Bts2Ipc_GetLeAdapterAdvertiseEnable*>(message));
         break;
      case Bts2IpcOC_SetLeAdapterAdvertiseEnable:
         handleBts2IpcSetLeAdapterAdvertiseEnable(deleteMsg, sendErrorMsg, static_cast<const Bts2Ipc_SetLeAdapterAdvertiseEnable*>(message));
         break;
      case Bts2IpcOC_LeDeviceDisconnect:
         handleBts2IpcLeDeviceDisconnect(deleteMsg, sendErrorMsg, static_cast<Bts2Ipc_LeDeviceDisconnect*>(message));
         break;
      case Bts2IpcOC_GetLeDeviceAddressType:
         handleBts2IpcGetLeDeviceAddressType(deleteMsg, sendErrorMsg, static_cast<Bts2Ipc_GetLeDeviceAddressType*>(message));
         break;
      case Bts2IpcOC_GetLeDeviceAddress:
         handleBts2IpcGetLeDeviceAddress(deleteMsg, sendErrorMsg, static_cast<Bts2Ipc_GetLeDeviceAddress*>(message));
         break;
      case Bts2IpcOC_GetLeDeviceConnected:
         handleBts2IpcGetLeDeviceConnected(deleteMsg, sendErrorMsg, static_cast<Bts2Ipc_GetLeDeviceConnected*>(message));
         break;
      case Bts2IpcOC_SendSeamlessPairingErrorCode:
         handleBts2IpcSendSeamlessPairingErrorCode(deleteMsg, sendErrorMsg, static_cast<Bts2Ipc_SendSeamlessPairingErrorCode*>(message));
         break;
      case Bts2IpcOC_SeamlessPairingAllowAccess:
         handleBts2IpcSeamlessPairingAllowAccess(deleteMsg, sendErrorMsg, static_cast<Bts2Ipc_SeamlessPairingAllowAccess*>(message));
         break;
      case Bts2IpcOC_GetSeamlessPairingPairingData:
         handleBts2IpcGetSeamlessPairingPairingData(deleteMsg, sendErrorMsg, static_cast<Bts2Ipc_GetSeamlessPairingPairingData*>(message));
         break;
      case Bts2IpcOC_SetSeamlessPairingPairingData:
         handleBts2IpcSetSeamlessPairingPairingData(deleteMsg, sendErrorMsg, static_cast<const Bts2Ipc_SetSeamlessPairingPairingData*>(message));
         break;
      // all other
      default:
         handled = false;
         break;
   }

   return handled;
}

bool AlpsEvolutionBluetoothLowEnergyDbusIf::createErrorMessage(OUT Ipc2Bts_BaseMessage** ipc2BtsMessage, IN const Bts2Ipc_BaseMessage* bts2IpcMessage, IN const BTSIpcCommonErrorCode errorCode) const
{
   (void)(errorCode);

   if(false == _enabled)
   {
      // do not process
      return false;
   }

   if(false == _initialized)
   {
      // not initialized
      return false;
   }

   FW_IF_NULL_PTR_RETURN_FALSE(ipc2BtsMessage);
   FW_IF_NULL_PTR_RETURN_FALSE(bts2IpcMessage);

   const Bts2Ipc_BaseMessage* message = bts2IpcMessage;
   const BTSGenBts2IpcOpcode opcode((BTSGenBts2IpcOpcode)message->getOpCode());
   bool handled(true);

   switch(opcode)
   {
      case Bts2IpcOC_CreateLeDevice:
      {
         *ipc2BtsMessage = ptrNew_Ipc2Bts_CreateLeDeviceResult();
         break;
      }
      case Bts2IpcOC_RemoveLeDevice:
      {
         *ipc2BtsMessage = ptrNew_Ipc2Bts_RemoveLeDeviceResult();
         break;
      }
      case Bts2IpcOC_GetLeAdapterAddressType:
      {
         *ipc2BtsMessage = ptrNew_Ipc2Bts_LeAdapterAddressTypeUpdate();
         break;
      }
      case Bts2IpcOC_GetLeAdapterAddress:
      {
         *ipc2BtsMessage = ptrNew_Ipc2Bts_LeAdapterAddressUpdate();
         break;
      }
      case Bts2IpcOC_GetLeAdapterAlias:
      {
         *ipc2BtsMessage = ptrNew_Ipc2Bts_LeAdapterAliasUpdate();
         break;
      }
      case Bts2IpcOC_SetLeAdapterAlias:
      {
         // no method return on ASF level
         break;
      }
      case Bts2IpcOC_GetLeAdapterAdvertisingTimeout:
      {
         *ipc2BtsMessage = ptrNew_Ipc2Bts_LeAdapterAdvertisingTimeoutUpdate();
         break;
      }
      case Bts2IpcOC_SetLeAdapterAdvertisingTimeout:
      {
         // no method return on ASF level
         break;
      }
      case Bts2IpcOC_GetLeAdapterDiscoverableMode:
      {
         *ipc2BtsMessage = ptrNew_Ipc2Bts_LeAdapterDiscoverableModeUpdate();
         break;
      }
      case Bts2IpcOC_SetLeAdapterDiscoverableMode:
      {
         // no method return on ASF level
         break;
      }
      case Bts2IpcOC_GetLeAdapterAdvertisingData:
      {
         *ipc2BtsMessage = ptrNew_Ipc2Bts_LeAdapterAdvertisingDataUpdate();
         break;
      }
      case Bts2IpcOC_SetLeAdapterAdvertisingData:
      {
         // no method return on ASF level
         break;
      }
      case Bts2IpcOC_GetLeAdapterAdvertiseEnable:
      {
         *ipc2BtsMessage = ptrNew_Ipc2Bts_LeAdapterAdvertiseEnableUpdate();
         break;
      }
      case Bts2IpcOC_SetLeAdapterAdvertiseEnable:
      {
         // no method return on ASF level
         break;
      }
      case Bts2IpcOC_LeDeviceDisconnect:
      {
         Ipc2Bts_LeDeviceDisconnectResult* outMsg = ptrNew_Ipc2Bts_LeDeviceDisconnectResult();
         if(0 != outMsg)
         {
            const Bts2Ipc_LeDeviceDisconnect* inMsg = static_cast<const Bts2Ipc_LeDeviceDisconnect*>(message);
            outMsg->setDevice(inMsg->getDevice());
            *ipc2BtsMessage = outMsg;
         }
         break;
      }
      case Bts2IpcOC_GetLeDeviceAddressType:
      {
         Ipc2Bts_LeDeviceAddressTypeUpdate* outMsg = ptrNew_Ipc2Bts_LeDeviceAddressTypeUpdate();
         if(0 != outMsg)
         {
            const Bts2Ipc_GetLeDeviceAddressType* inMsg = static_cast<const Bts2Ipc_GetLeDeviceAddressType*>(message);
            outMsg->setDevice(inMsg->getDevice());
            *ipc2BtsMessage = outMsg;
         }
         break;
      }
      case Bts2IpcOC_GetLeDeviceAddress:
      {
         Ipc2Bts_LeDeviceAddressUpdate* outMsg = ptrNew_Ipc2Bts_LeDeviceAddressUpdate();
         if(0 != outMsg)
         {
            const Bts2Ipc_GetLeDeviceAddress* inMsg = static_cast<const Bts2Ipc_GetLeDeviceAddress*>(message);
            outMsg->setDevice(inMsg->getDevice());
            *ipc2BtsMessage = outMsg;
         }
         break;
      }
      case Bts2IpcOC_GetLeDeviceConnected:
      {
         Ipc2Bts_LeDeviceConnectedUpdate* outMsg = ptrNew_Ipc2Bts_LeDeviceConnectedUpdate();
         if(0 != outMsg)
         {
            const Bts2Ipc_GetLeDeviceConnected* inMsg = static_cast<const Bts2Ipc_GetLeDeviceConnected*>(message);
            outMsg->setDevice(inMsg->getDevice());
            *ipc2BtsMessage = outMsg;
         }
         break;
      }
      case Bts2IpcOC_SendSeamlessPairingErrorCode:
      {
         Ipc2Bts_SendSeamlessPairingErrorCodeResult* outMsg = ptrNew_Ipc2Bts_SendSeamlessPairingErrorCodeResult();
         if(0 != outMsg)
         {
            const Bts2Ipc_SendSeamlessPairingErrorCode* inMsg = static_cast<const Bts2Ipc_SendSeamlessPairingErrorCode*>(message);
            outMsg->setDevice(inMsg->getDevice());
            *ipc2BtsMessage = outMsg;
         }
         break;
      }
      case Bts2IpcOC_SeamlessPairingAllowAccess:
      {
         Ipc2Bts_SeamlessPairingAllowAccessResult* outMsg = ptrNew_Ipc2Bts_SeamlessPairingAllowAccessResult();
         if(0 != outMsg)
         {
            const Bts2Ipc_SeamlessPairingAllowAccess* inMsg = static_cast<const Bts2Ipc_SeamlessPairingAllowAccess*>(message);
            outMsg->setDevice(inMsg->getDevice());
            *ipc2BtsMessage = outMsg;
         }
         break;
      }
      case Bts2IpcOC_GetSeamlessPairingPairingData:
      {
         Ipc2Bts_SeamlessPairingPairingDataUpdate* outMsg = ptrNew_Ipc2Bts_SeamlessPairingPairingDataUpdate();
         if(0 != outMsg)
         {
            const Bts2Ipc_GetSeamlessPairingPairingData* inMsg = static_cast<const Bts2Ipc_GetSeamlessPairingPairingData*>(message);
            outMsg->setDevice(inMsg->getDevice());
            *ipc2BtsMessage = outMsg;
         }
         break;
      }
      case Bts2IpcOC_SetSeamlessPairingPairingData:
      {
         // no method return on ASF level
         break;
      }
      // all other
      default:
      {
         handled = false;
         break;
      }
   }

   return handled;
}

bool AlpsEvolutionBluetoothLowEnergyDbusIf::transferMessageData(OUT Ipc2Bts_BaseMessage* ipc2BtsMessage, IN const Bts2Ipc_BaseMessage* bts2IpcMessage) const
{
   // TODO: implement after rework is done
   (void)(ipc2BtsMessage);
   (void)(bts2IpcMessage);

   return false;
}

void AlpsEvolutionBluetoothLowEnergyDbusIf::handleBts2IpcCreateLeDevice(OUT bool& deleteMsg, OUT bool& sendErrorMsg, IN Bts2Ipc_CreateLeDevice* message) const
{
   FW_IF_NULL_PTR_RETURN(message);

   if(0 == _adapterIf)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      deleteMsg = true;
      sendErrorMsg = true;
   }
   else
   {
      act_t token = _adapterIf->sendCreateLeDeviceRequest(message->getBDAddress());

      if(::ccdbusif::DEFAULT_ACT == token)
      {
         // sending failed (valid use case)
         FW_NORMAL_ASSERT_ALWAYS();
         deleteMsg = true;
         sendErrorMsg = true;
      }
      else
      {
         // sending success
         message->setDbusToken(token);
         ::ccdbusif::ActInfo info(token);
         ETG_TRACE_USR3((" handleBts2IpcCreateLeDevice(): Bts2Ipc 0x%04X success: token=%s", message->getTraceOpCode(), info.getInfo()));
      }
   }
}

void AlpsEvolutionBluetoothLowEnergyDbusIf::handleBts2IpcRemoveLeDevice(OUT bool& deleteMsg, OUT bool& sendErrorMsg, IN Bts2Ipc_RemoveLeDevice* message) const
{
   FW_IF_NULL_PTR_RETURN(message);

   if(0 == _adapterIf)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      deleteMsg = true;
      sendErrorMsg = true;
   }
   else
   {
      act_t token = _adapterIf->sendRemoveLeDeviceRequest(message->getDevice());

      if(::ccdbusif::DEFAULT_ACT == token)
      {
         // sending failed (valid use case)
         FW_NORMAL_ASSERT_ALWAYS();
         deleteMsg = true;
         sendErrorMsg = true;
      }
      else
      {
         // sending success
         message->setDbusToken(token);
         ::ccdbusif::ActInfo info(token);
         ETG_TRACE_USR3((" handleBts2IpcRemoveLeDevice(): Bts2Ipc 0x%04X success: token=%s", message->getTraceOpCode(), info.getInfo()));
      }
   }
}

void AlpsEvolutionBluetoothLowEnergyDbusIf::handleBts2IpcGetLeAdapterAddressType(OUT bool& deleteMsg, OUT bool& sendErrorMsg, IN Bts2Ipc_GetLeAdapterAddressType* message) const
{
   FW_IF_NULL_PTR_RETURN(message);

   if(0 == _adapterIf)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      deleteMsg = true;
      sendErrorMsg = true;
   }
   else
   {
      act_t token = _adapterIf->sendAddressTypeGet();

      if(::ccdbusif::DEFAULT_ACT == token)
      {
         // sending failed (valid use case)
         FW_NORMAL_ASSERT_ALWAYS();
         deleteMsg = true;
         sendErrorMsg = true;
      }
      else
      {
         // sending success
         message->setDbusToken(token);
         ::ccdbusif::ActInfo info(token);
         ETG_TRACE_USR3((" handleBts2IpcGetLeAdapterAddressType(): Bts2Ipc 0x%04X success: token=%s", message->getTraceOpCode(), info.getInfo()));
      }
   }
}

void AlpsEvolutionBluetoothLowEnergyDbusIf::handleBts2IpcGetLeAdapterAddress(OUT bool& deleteMsg, OUT bool& sendErrorMsg, IN Bts2Ipc_GetLeAdapterAddress* message) const
{
   FW_IF_NULL_PTR_RETURN(message);

   if(0 == _adapterIf)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      deleteMsg = true;
      sendErrorMsg = true;
   }
   else
   {
      act_t token = _adapterIf->sendAddressGet();

      if(::ccdbusif::DEFAULT_ACT == token)
      {
         // sending failed (valid use case)
         FW_NORMAL_ASSERT_ALWAYS();
         deleteMsg = true;
         sendErrorMsg = true;
      }
      else
      {
         // sending success
         message->setDbusToken(token);
         ::ccdbusif::ActInfo info(token);
         ETG_TRACE_USR3((" handleBts2IpcGetLeAdapterAddress(): Bts2Ipc 0x%04X success: token=%s", message->getTraceOpCode(), info.getInfo()));
      }
   }
}

void AlpsEvolutionBluetoothLowEnergyDbusIf::handleBts2IpcGetLeAdapterAlias(OUT bool& deleteMsg, OUT bool& sendErrorMsg, IN Bts2Ipc_GetLeAdapterAlias* message) const
{
   FW_IF_NULL_PTR_RETURN(message);

   if(0 == _adapterIf)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      deleteMsg = true;
      sendErrorMsg = true;
   }
   else
   {
      act_t token = _adapterIf->sendAliasGet();

      if(::ccdbusif::DEFAULT_ACT == token)
      {
         // sending failed (valid use case)
         FW_NORMAL_ASSERT_ALWAYS();
         deleteMsg = true;
         sendErrorMsg = true;
      }
      else
      {
         // sending success
         message->setDbusToken(token);
         ::ccdbusif::ActInfo info(token);
         ETG_TRACE_USR3((" handleBts2IpcGetLeAdapterAlias(): Bts2Ipc 0x%04X success: token=%s", message->getTraceOpCode(), info.getInfo()));
      }
   }
}

void AlpsEvolutionBluetoothLowEnergyDbusIf::handleBts2IpcSetLeAdapterAlias(OUT bool& deleteMsg, OUT bool& sendErrorMsg, IN const Bts2Ipc_SetLeAdapterAlias* message) const
{
   FW_IF_NULL_PTR_RETURN(message);

   if(0 == _adapterIf)
   {
      FW_NORMAL_ASSERT_ALWAYS();
   }
   else
   {
      _adapterIf->sendAliasSet(message->getAlias());

      ETG_TRACE_USR3((" handleBts2IpcSetLeAdapterAlias(): Bts2Ipc 0x%04X success: token=<none>", message->getTraceOpCode()));
   }

   // no method return on ASF level
   deleteMsg = true;
   sendErrorMsg = false;
}

void AlpsEvolutionBluetoothLowEnergyDbusIf::handleBts2IpcGetLeAdapterAdvertisingTimeout(OUT bool& deleteMsg, OUT bool& sendErrorMsg, IN Bts2Ipc_GetLeAdapterAdvertisingTimeout* message) const
{
   FW_IF_NULL_PTR_RETURN(message);

   if(0 == _adapterIf)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      deleteMsg = true;
      sendErrorMsg = true;
   }
   else
   {
      act_t token = _adapterIf->sendAdvertisingTimeoutGet();

      if(::ccdbusif::DEFAULT_ACT == token)
      {
         // sending failed (valid use case)
         FW_NORMAL_ASSERT_ALWAYS();
         deleteMsg = true;
         sendErrorMsg = true;
      }
      else
      {
         // sending success
         message->setDbusToken(token);
         ::ccdbusif::ActInfo info(token);
         ETG_TRACE_USR3((" handleBts2IpcGetLeAdapterAdvertisingTimeout(): Bts2Ipc 0x%04X success: token=%s", message->getTraceOpCode(), info.getInfo()));
      }
   }
}

void AlpsEvolutionBluetoothLowEnergyDbusIf::handleBts2IpcSetLeAdapterAdvertisingTimeout(OUT bool& deleteMsg, OUT bool& sendErrorMsg, IN const Bts2Ipc_SetLeAdapterAdvertisingTimeout* message) const
{
   FW_IF_NULL_PTR_RETURN(message);

   if(0 == _adapterIf)
   {
      FW_NORMAL_ASSERT_ALWAYS();
   }
   else
   {
      _adapterIf->sendAdvertisingTimeoutSet(message->getAdvertisingTimeout());

      ETG_TRACE_USR3((" handleBts2IpcSetLeAdapterAdvertisingTimeout(): Bts2Ipc 0x%04X success: token=<none>", message->getTraceOpCode()));
   }

   // no method return on ASF level
   deleteMsg = true;
   sendErrorMsg = false;
}

void AlpsEvolutionBluetoothLowEnergyDbusIf::handleBts2IpcGetLeAdapterDiscoverableMode(OUT bool& deleteMsg, OUT bool& sendErrorMsg, IN Bts2Ipc_GetLeAdapterDiscoverableMode* message) const
{
   FW_IF_NULL_PTR_RETURN(message);

   if(0 == _adapterIf)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      deleteMsg = true;
      sendErrorMsg = true;
   }
   else
   {
      act_t token = _adapterIf->sendDiscoverableModeGet();

      if(::ccdbusif::DEFAULT_ACT == token)
      {
         // sending failed (valid use case)
         FW_NORMAL_ASSERT_ALWAYS();
         deleteMsg = true;
         sendErrorMsg = true;
      }
      else
      {
         // sending success
         message->setDbusToken(token);
         ::ccdbusif::ActInfo info(token);
         ETG_TRACE_USR3((" handleBts2IpcGetLeAdapterDiscoverableMode(): Bts2Ipc 0x%04X success: token=%s", message->getTraceOpCode(), info.getInfo()));
      }
   }
}

void AlpsEvolutionBluetoothLowEnergyDbusIf::handleBts2IpcSetLeAdapterDiscoverableMode(OUT bool& deleteMsg, OUT bool& sendErrorMsg, IN const Bts2Ipc_SetLeAdapterDiscoverableMode* message) const
{
   FW_IF_NULL_PTR_RETURN(message);

   if(0 == _adapterIf)
   {
      FW_NORMAL_ASSERT_ALWAYS();
   }
   else
   {
      _adapterIf->sendDiscoverableModeSet((uint8)message->getDiscoverableMode());

      ETG_TRACE_USR3((" handleBts2IpcSetLeAdapterDiscoverableMode(): Bts2Ipc 0x%04X success: token=<none>", message->getTraceOpCode()));
   }

   // no method return on ASF level
   deleteMsg = true;
   sendErrorMsg = false;
}

void AlpsEvolutionBluetoothLowEnergyDbusIf::handleBts2IpcGetLeAdapterAdvertisingData(OUT bool& deleteMsg, OUT bool& sendErrorMsg, IN Bts2Ipc_GetLeAdapterAdvertisingData* message) const
{
   FW_IF_NULL_PTR_RETURN(message);

   if(0 == _adapterIf)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      deleteMsg = true;
      sendErrorMsg = true;
   }
   else
   {
      act_t token = _adapterIf->sendAdvertisingDataGet();

      if(::ccdbusif::DEFAULT_ACT == token)
      {
         // sending failed (valid use case)
         FW_NORMAL_ASSERT_ALWAYS();
         deleteMsg = true;
         sendErrorMsg = true;
      }
      else
      {
         // sending success
         message->setDbusToken(token);
         ::ccdbusif::ActInfo info(token);
         ETG_TRACE_USR3((" handleBts2IpcGetLeAdapterAdvertisingData(): Bts2Ipc 0x%04X success: token=%s", message->getTraceOpCode(), info.getInfo()));
      }
   }
}

void AlpsEvolutionBluetoothLowEnergyDbusIf::handleBts2IpcSetLeAdapterAdvertisingData(OUT bool& deleteMsg, OUT bool& sendErrorMsg, IN const Bts2Ipc_SetLeAdapterAdvertisingData* message) const
{
   FW_IF_NULL_PTR_RETURN(message);

   if(0 == _adapterIf)
   {
      FW_NORMAL_ASSERT_ALWAYS();
   }
   else
   {
      _adapterIf->sendAdvertisingDataSet(message->getAdvertisingData());

      ETG_TRACE_USR3((" handleBts2IpcSetLeAdapterAdvertisingData(): Bts2Ipc 0x%04X success: token=<none>", message->getTraceOpCode()));
   }

   // no method return on ASF level
   deleteMsg = true;
   sendErrorMsg = false;
}

void AlpsEvolutionBluetoothLowEnergyDbusIf::handleBts2IpcGetLeAdapterAdvertiseEnable(OUT bool& deleteMsg, OUT bool& sendErrorMsg, IN Bts2Ipc_GetLeAdapterAdvertiseEnable* message) const
{
   FW_IF_NULL_PTR_RETURN(message);

   if(0 == _adapterIf)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      deleteMsg = true;
      sendErrorMsg = true;
   }
   else
   {
      act_t token = _adapterIf->sendAdvertiseEnableGet();

      if(::ccdbusif::DEFAULT_ACT == token)
      {
         // sending failed (valid use case)
         FW_NORMAL_ASSERT_ALWAYS();
         deleteMsg = true;
         sendErrorMsg = true;
      }
      else
      {
         // sending success
         message->setDbusToken(token);
         ::ccdbusif::ActInfo info(token);
         ETG_TRACE_USR3((" handleBts2IpcGetLeAdapterAdvertiseEnable(): Bts2Ipc 0x%04X success: token=%s", message->getTraceOpCode(), info.getInfo()));
      }
   }
}

void AlpsEvolutionBluetoothLowEnergyDbusIf::handleBts2IpcSetLeAdapterAdvertiseEnable(OUT bool& deleteMsg, OUT bool& sendErrorMsg, IN const Bts2Ipc_SetLeAdapterAdvertiseEnable* message) const
{
   FW_IF_NULL_PTR_RETURN(message);

   if(0 == _adapterIf)
   {
      FW_NORMAL_ASSERT_ALWAYS();
   }
   else
   {
      _adapterIf->sendAdvertiseEnableSet(message->getAdvertiseEnable());

      ETG_TRACE_USR3((" handleBts2IpcSetLeAdapterAdvertiseEnable(): Bts2Ipc 0x%04X success: token=<none>", message->getTraceOpCode()));
   }

   // no method return on ASF level
   deleteMsg = true;
   sendErrorMsg = false;
}

void AlpsEvolutionBluetoothLowEnergyDbusIf::handleBts2IpcLeDeviceDisconnect(OUT bool& deleteMsg, OUT bool& sendErrorMsg, IN Bts2Ipc_LeDeviceDisconnect* message) const
{
   FW_IF_NULL_PTR_RETURN(message);

   if(0 == _deviceIf)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      deleteMsg = true;
      sendErrorMsg = true;
   }
   else
   {
      act_t token = _deviceIf->sendDisconnectRequest(message->getDevice());

      if(::ccdbusif::DEFAULT_ACT == token)
      {
         // sending failed (valid use case)
         FW_NORMAL_ASSERT_ALWAYS();
         deleteMsg = true;
         sendErrorMsg = true;
      }
      else
      {
         // sending success
         message->setDbusToken(token);
         ::ccdbusif::ActInfo info(token);
         ETG_TRACE_USR3((" handleBts2IpcLeDeviceDisconnect(): Bts2Ipc 0x%04X success: token=%s", message->getTraceOpCode(), info.getInfo()));
      }
   }
}

void AlpsEvolutionBluetoothLowEnergyDbusIf::handleBts2IpcGetLeDeviceAddressType(OUT bool& deleteMsg, OUT bool& sendErrorMsg, IN Bts2Ipc_GetLeDeviceAddressType* message) const
{
   FW_IF_NULL_PTR_RETURN(message);

   if(0 == _deviceIf)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      deleteMsg = true;
      sendErrorMsg = true;
   }
   else
   {
      act_t token = _deviceIf->sendAddressTypeGet(message->getDevice());

      if(::ccdbusif::DEFAULT_ACT == token)
      {
         // sending failed (valid use case)
         FW_NORMAL_ASSERT_ALWAYS();
         deleteMsg = true;
         sendErrorMsg = true;
      }
      else
      {
         // sending success
         message->setDbusToken(token);
         ::ccdbusif::ActInfo info(token);
         ETG_TRACE_USR3((" handleBts2IpcGetLeDeviceAddressType(): Bts2Ipc 0x%04X success: token=%s", message->getTraceOpCode(), info.getInfo()));
      }
   }
}

void AlpsEvolutionBluetoothLowEnergyDbusIf::handleBts2IpcGetLeDeviceAddress(OUT bool& deleteMsg, OUT bool& sendErrorMsg, IN Bts2Ipc_GetLeDeviceAddress* message) const
{
   FW_IF_NULL_PTR_RETURN(message);

   if(0 == _deviceIf)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      deleteMsg = true;
      sendErrorMsg = true;
   }
   else
   {
      act_t token = _deviceIf->sendAddressGet(message->getDevice());

      if(::ccdbusif::DEFAULT_ACT == token)
      {
         // sending failed (valid use case)
         FW_NORMAL_ASSERT_ALWAYS();
         deleteMsg = true;
         sendErrorMsg = true;
      }
      else
      {
         // sending success
         message->setDbusToken(token);
         ::ccdbusif::ActInfo info(token);
         ETG_TRACE_USR3((" handleBts2IpcGetLeDeviceAddress(): Bts2Ipc 0x%04X success: token=%s", message->getTraceOpCode(), info.getInfo()));
      }
   }
}

void AlpsEvolutionBluetoothLowEnergyDbusIf::handleBts2IpcGetLeDeviceConnected(OUT bool& deleteMsg, OUT bool& sendErrorMsg, IN Bts2Ipc_GetLeDeviceConnected* message) const
{
   FW_IF_NULL_PTR_RETURN(message);

   if(0 == _deviceIf)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      deleteMsg = true;
      sendErrorMsg = true;
   }
   else
   {
      act_t token = _deviceIf->sendConnectedGet(message->getDevice());

      if(::ccdbusif::DEFAULT_ACT == token)
      {
         // sending failed (valid use case)
         FW_NORMAL_ASSERT_ALWAYS();
         deleteMsg = true;
         sendErrorMsg = true;
      }
      else
      {
         // sending success
         message->setDbusToken(token);
         ::ccdbusif::ActInfo info(token);
         ETG_TRACE_USR3((" handleBts2IpcGetLeDeviceConnected(): Bts2Ipc 0x%04X success: token=%s", message->getTraceOpCode(), info.getInfo()));
      }
   }
}

void AlpsEvolutionBluetoothLowEnergyDbusIf::handleBts2IpcSendSeamlessPairingErrorCode(OUT bool& deleteMsg, OUT bool& sendErrorMsg, IN Bts2Ipc_SendSeamlessPairingErrorCode* message) const
{
   FW_IF_NULL_PTR_RETURN(message);

   if(0 == _seamlessPairingIf)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      deleteMsg = true;
      sendErrorMsg = true;
   }
   else
   {
      act_t token = _seamlessPairingIf->sendSendErrorCodeRequest(message->getDevice(), message->getErrorCode());

      if(::ccdbusif::DEFAULT_ACT == token)
      {
         // sending failed (valid use case)
         FW_NORMAL_ASSERT_ALWAYS();
         deleteMsg = true;
         sendErrorMsg = true;
      }
      else
      {
         // sending success
         message->setDbusToken(token);
         ::ccdbusif::ActInfo info(token);
         ETG_TRACE_USR3((" handleBts2IpcSendSeamlessPairingErrorCode(): Bts2Ipc 0x%04X success: token=%s", message->getTraceOpCode(), info.getInfo()));
      }
   }
}

void AlpsEvolutionBluetoothLowEnergyDbusIf::handleBts2IpcSeamlessPairingAllowAccess(OUT bool& deleteMsg, OUT bool& sendErrorMsg, IN Bts2Ipc_SeamlessPairingAllowAccess* message) const
{
   FW_IF_NULL_PTR_RETURN(message);

   if(0 == _seamlessPairingIf)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      deleteMsg = true;
      sendErrorMsg = true;
   }
   else
   {
      act_t token = _seamlessPairingIf->sendAllowAccessRequest(message->getDevice(), message->getAccept());

      if(::ccdbusif::DEFAULT_ACT == token)
      {
         // sending failed (valid use case)
         FW_NORMAL_ASSERT_ALWAYS();
         deleteMsg = true;
         sendErrorMsg = true;
      }
      else
      {
         // sending success
         message->setDbusToken(token);
         ::ccdbusif::ActInfo info(token);
         ETG_TRACE_USR3((" handleBts2IpcSeamlessPairingAllowAccess(): Bts2Ipc 0x%04X success: token=%s", message->getTraceOpCode(), info.getInfo()));
      }
   }
}

void AlpsEvolutionBluetoothLowEnergyDbusIf::handleBts2IpcGetSeamlessPairingPairingData(OUT bool& deleteMsg, OUT bool& sendErrorMsg, IN Bts2Ipc_GetSeamlessPairingPairingData* message) const
{
   FW_IF_NULL_PTR_RETURN(message);

   if(0 == _seamlessPairingIf)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      deleteMsg = true;
      sendErrorMsg = true;
   }
   else
   {
      act_t token = _seamlessPairingIf->sendPairingDataGet(message->getDevice());

      if(::ccdbusif::DEFAULT_ACT == token)
      {
         // sending failed (valid use case)
         FW_NORMAL_ASSERT_ALWAYS();
         deleteMsg = true;
         sendErrorMsg = true;
      }
      else
      {
         // sending success
         message->setDbusToken(token);
         ::ccdbusif::ActInfo info(token);
         ETG_TRACE_USR3((" handleBts2IpcGetSeamlessPairingPairingData(): Bts2Ipc 0x%04X success: token=%s", message->getTraceOpCode(), info.getInfo()));
      }
   }
}

void AlpsEvolutionBluetoothLowEnergyDbusIf::handleBts2IpcSetSeamlessPairingPairingData(OUT bool& deleteMsg, OUT bool& sendErrorMsg, IN const Bts2Ipc_SetSeamlessPairingPairingData* message) const
{
   FW_IF_NULL_PTR_RETURN(message);

   if(0 == _seamlessPairingIf)
   {
      FW_NORMAL_ASSERT_ALWAYS();
   }
   else
   {
      _seamlessPairingIf->sendPairingDataSet(message->getDevice(), message->getPairingData());

      ETG_TRACE_USR3((" handleBts2IpcSetSeamlessPairingPairingData(): Bts2Ipc 0x%04X success: token=<none>", message->getTraceOpCode()));
   }

   // no method return on ASF level
   deleteMsg = true;
   sendErrorMsg = false;
}

} //alpsevolutionext
} //genivi
} //btstackif
