/**
 * @file AlpsEvolutionConnectionControl.cpp
 *
 * @par SW-Component
 * Control Extension
 *
 * @brief Implementation of Alps Evolution Genivi connection control extension.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Source file for implementation of Alps Evolution Genivi connection control extension.
 */

#include "AlpsEvolutionConnectionControl.h"
#include "FwAssert.h"
#include "Ipc2Bts_MessageWrapper_GEN.h"

// SM includes
#include "TestModeAlpsEvolutionExt.h"
#include "BluetoothLowEnergyAlpsEvolutionExt.h"

#include "TraceClasses.h"
#include "FwTrace.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_BTS_CONTROL
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/AlpsEvolutionConnectionControl.cpp.trc.h"
#endif
#endif

namespace btstackif {
namespace genivi {
namespace alpsevolutionext {

AlpsEvolutionConnectionControl::AlpsEvolutionConnectionControl() :
_controlIf(0),
_timerPoolIf(0),
_enabled(false),
_initialized(false),
_testMode(0),
_ble(0)
{
}

AlpsEvolutionConnectionControl::~AlpsEvolutionConnectionControl()
{
   _controlIf = 0;
   _timerPoolIf = 0;
   _testMode = 0;
   _ble = 0;
}

void AlpsEvolutionConnectionControl::setControlIf(IN IBasicControl* control)
{
   _controlIf = control;

   FW_NORMAL_ASSERT(0 != _controlIf);
}

void AlpsEvolutionConnectionControl::setTimerPoolIf(IN ITimerPool* timerPool)
{
   _timerPoolIf = timerPool;

   FW_NORMAL_ASSERT(0 != _timerPoolIf);
}

void AlpsEvolutionConnectionControl::enable(IN const bool enable)
{
   if(enable == _enabled)
   {
      // required state already set
      return;
   }

   if(true == _initialized)
   {
      // already initialized => different state not allowed
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   // store new state
   _enabled = enable;

   if(true == _enabled)
   {
      // all stuff necessary for enabled state
   }
   else
   {
      // all stuff necessary for disabled state
   }
}

void AlpsEvolutionConnectionControl::initialize(void)
{
   ETG_TRACE_USR2((" initialize(): _enabled=%d _initialized=%d", _enabled, _initialized));

   FW_IF_NULL_PTR_RETURN(_controlIf);
   FW_IF_NULL_PTR_RETURN(_timerPoolIf);

   if(false == _enabled)
   {
      // do not initialize
      return;
   }

   if(true == _initialized)
   {
      // already initialized
      return;
   }

   // all stuff necessary for initialization

   // create SMs
   if(0 == _testMode)
   {
      _testMode = new TestModeAlpsEvolutionExt();
   }
   FW_NORMAL_ASSERT(0 != _testMode);

   if(0 == _ble)
   {
      _ble = new BluetoothLowEnergyAlpsEvolutionExt();
   }
   FW_NORMAL_ASSERT(0 != _ble);

   if(0 != _ble)
   {
      _ble->setControlIf(_controlIf);
      _ble->setTimerPoolIf(_timerPoolIf);
   }

   _initialized = true;
}

void AlpsEvolutionConnectionControl::destroy(void)
{
   ETG_TRACE_USR2((" destroy(): _enabled=%d _initialized=%d", _enabled, _initialized));

   if(false == _enabled)
   {
      // nothing to do
      return;
   }

   if(false == _initialized)
   {
      // not initialized
      return;
   }

   // all stuff necessary for destruction

   // destroy SMs
   if(0 != _testMode)
   {
      delete _testMode;
      _testMode = 0;
   }

   if(0 != _ble)
   {
      delete _ble;
      _ble = 0;
   }

   _initialized = false;
}

void AlpsEvolutionConnectionControl::setStackConfiguration(IN const BTSFunctionBlock component, IN const BTSInterfaceType stackInterface, IN const BTSFunctionBlock subComponent, IN const BTSUserMode userMode,
         OUT ::std::vector<BTSDbusInterfaceItem>& dbusInterfaces, IN const BTSLocalConfigurationContainer& configuration)
{
   (void)(component);
   (void)(subComponent);

   ETG_TRACE_USR2((" setStackConfiguration(): _enabled=%d _initialized=%d", _enabled, _initialized));

   if(BTS_USER_MODE_LAST <= userMode)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   if(BTS_IF_ALPS_EVOLUTION_GENIVI != stackInterface)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   if(false == _enabled)
   {
      // do not process
      return;
   }

   if(false == _initialized)
   {
      // not initialized
      return;
   }

   FW_IF_NULL_PTR_RETURN(_ble);
   IBluetoothLowEnergyAlpsEvolutionExt& ble = *_ble;
   ble.enable(configuration.connectionConfiguration.configuration.bluetoothLowEnergyEnabled);
   ble.setStackConfiguration(dbusInterfaces, configuration.connectionConfiguration.configuration, userMode, stackInterface);
}

bool AlpsEvolutionConnectionControl::doApp2BtsMsgPrecheck(OUT bool& continueProcessing, OUT bool& rejectRequest, OUT BTSCommonEnumClass& resultCode, OUT BTSCommonEnumClass& statusCode, OUT bool& skipOpCodeCheck, IN App2Bts_BaseMessage* message)
{
   FW_IF_NULL_PTR_RETURN_FALSE(message);

   // currently there are no App2Bts messages
   (void)(continueProcessing);
   (void)(rejectRequest);
   (void)(resultCode);
   (void)(statusCode);
   (void)(skipOpCodeCheck);

   return false;
}

bool AlpsEvolutionConnectionControl::getSimilarOpCodes(OUT ::std::vector< BTSApp2BtsMessageMasterCompareItem >& itemList, IN const App2Bts_BaseMessage* message) const
{
   FW_IF_NULL_PTR_RETURN_FALSE(message);

   // currently there are no App2Bts messages
   (void)(itemList);

   return false;
}

bool AlpsEvolutionConnectionControl::getMatchingOpCodes(OUT ::std::vector< BTSApp2BtsMessageMasterCompareItem >& itemList, OUT ::std::vector< BTSApp2BtsMessageMasterCompareItem >& highPrioItemList, IN const App2Bts_BaseMessage* message) const
{
   FW_IF_NULL_PTR_RETURN_FALSE(message);

   // currently there are no App2Bts messages
   (void)(itemList);
   (void)(highPrioItemList);

   return false;
}

bool AlpsEvolutionConnectionControl::triggerInitializedCallback(void)
{
   return false;
}

bool AlpsEvolutionConnectionControl::handleDoubledApp2BtsMessages(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN App2Bts_BaseMessage* message, IN const BTSCommonEnumClass resultCode, IN const BTSCommonEnumClass statusCode)
{
   FW_IF_NULL_PTR_RETURN_FALSE(message);

   // currently there are no App2Bts messages
   (void)(bts2AppMsgList);
   (void)(resultCode);
   (void)(statusCode);

   return false;
}

bool AlpsEvolutionConnectionControl::handleApp2BtsMessage(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT bool& deleteApp2BtsMessage, IN App2Bts_BaseMessage* message)
{
   FW_IF_NULL_PTR_RETURN_FALSE(message);

   // currently there are no App2Bts messages
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);
   (void)(deleteApp2BtsMessage);

   return false;
}

bool AlpsEvolutionConnectionControl::handleIpc2BtsMessage(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_BaseMessage* message)
{
   if(false == _enabled)
   {
      // do not process
      return false;
   }

   if(false == _initialized)
   {
      // not initialized
      return false;
   }

   FW_IF_NULL_PTR_RETURN_FALSE(message);

   const BTSGenIpc2BtsOpcode opcode((BTSGenIpc2BtsOpcode)message->getOpCode());
   bool handled(false);

   // handle message depending on opcode
   // - do not delete Ipc2Bts message in sub-handler function because this done at the end of this function
   // - set marker to delete related App2Bts message in sub-handler function if necessary
   // - create Bts2App message in sub-handler function if necessary
   // - create Bts2Ipc message in sub-handler function if necessary
   // - handle any error in sub-handler function because there is the best place to handle
   switch(opcode)
   {
      case Ipc2BtsOC_BtApplTestModeReqResult:
         handled = handleIpc2BtsBtApplTestModeReqResult(bts2IpcMsgList, bts2AppMsgList, messageItem, static_cast< Ipc2Bts_BtApplTestModeReqResult* >(message)); /*lint !e1774: Could use dynamic_cast to downcast polymorphic type*/ // no RTTI, use static_cast instead of dynamic_cast, we know the type of the message
         break;
      case Ipc2BtsOC_BtApplTestModeLinkQualityReqResult:
         handled = handleIpc2BtsBtApplTestModeLinkQualityReqResult(bts2IpcMsgList, bts2AppMsgList, messageItem, static_cast< Ipc2Bts_BtApplTestModeLinkQualityReqResult* >(message)); /*lint !e1774: Could use dynamic_cast to downcast polymorphic type*/ // no RTTI, use static_cast instead of dynamic_cast, we know the type of the message
         break;
      case Ipc2BtsOC_BtApplTestModeCfm:
         handled = handleIpc2BtsBtApplTestModeCfm(bts2IpcMsgList, bts2AppMsgList, messageItem, static_cast< Ipc2Bts_BtApplTestModeCfm* >(message)); /*lint !e1774: Could use dynamic_cast to downcast polymorphic type*/ // no RTTI, use static_cast instead of dynamic_cast, we know the type of the message
         break;
      case Ipc2BtsOC_BtApplTestModeCompInd:
         handled = handleIpc2BtsBtApplTestModeCompInd(bts2IpcMsgList, bts2AppMsgList, messageItem, static_cast< Ipc2Bts_BtApplTestModeCompInd* >(message)); /*lint !e1774: Could use dynamic_cast to downcast polymorphic type*/ // no RTTI, use static_cast instead of dynamic_cast, we know the type of the message
         break;
      case Ipc2BtsOC_BtApplTestModeLinkQualityCfm:
         handled = handleIpc2BtsBtApplTestModeLinkQualityCfm(bts2IpcMsgList, bts2AppMsgList, messageItem, static_cast< Ipc2Bts_BtApplTestModeLinkQualityCfm* >(message)); /*lint !e1774: Could use dynamic_cast to downcast polymorphic type*/ // no RTTI, use static_cast instead of dynamic_cast, we know the type of the message
         break;
      case Ipc2BtsOC_ServiceAvailabilityConnection:
         handled = handleIpc2BtsServiceAvailabilityConnection(bts2IpcMsgList, bts2AppMsgList, messageItem, static_cast<Ipc2Bts_ServiceAvailabilityConnection*>(message)); /*lint !e1774: Could use dynamic_cast to downcast polymorphic type*/ // no RTTI, use static_cast instead of dynamic_cast, we know the type of the message
         break;
      case Ipc2BtsOC_GetAllPropertiesConnection:
         handled = handleIpc2BtsGetAllPropertiesConnection(bts2IpcMsgList, bts2AppMsgList, messageItem, static_cast<Ipc2Bts_GetAllPropertiesConnection*>(message)); /*lint !e1774: Could use dynamic_cast to downcast polymorphic type*/ // no RTTI, use static_cast instead of dynamic_cast, we know the type of the message
         break;
      case Ipc2BtsOC_CreateLeDeviceResult:
         handled = handleIpc2BtsCreateLeDeviceResult(bts2IpcMsgList, bts2AppMsgList, messageItem, static_cast<Ipc2Bts_CreateLeDeviceResult*>(message)); /*lint !e1774: Could use dynamic_cast to downcast polymorphic type*/ // no RTTI, use static_cast instead of dynamic_cast, we know the type of the message
         break;
      case Ipc2BtsOC_RemoveLeDeviceResult:
         handled = handleIpc2BtsRemoveLeDeviceResult(bts2IpcMsgList, bts2AppMsgList, messageItem, static_cast<Ipc2Bts_RemoveLeDeviceResult*>(message)); /*lint !e1774: Could use dynamic_cast to downcast polymorphic type*/ // no RTTI, use static_cast instead of dynamic_cast, we know the type of the message
         break;
      case Ipc2BtsOC_LeAdapterInitialization:
         handled = handleIpc2BtsLeAdapterInitialization(bts2IpcMsgList, bts2AppMsgList, messageItem, static_cast<Ipc2Bts_LeAdapterInitialization*>(message)); /*lint !e1774: Could use dynamic_cast to downcast polymorphic type*/ // no RTTI, use static_cast instead of dynamic_cast, we know the type of the message
         break;
      case Ipc2BtsOC_LeDeviceCreated:
         handled = handleIpc2BtsLeDeviceCreated(bts2IpcMsgList, bts2AppMsgList, messageItem, static_cast<Ipc2Bts_LeDeviceCreated*>(message)); /*lint !e1774: Could use dynamic_cast to downcast polymorphic type*/ // no RTTI, use static_cast instead of dynamic_cast, we know the type of the message
         break;
      case Ipc2BtsOC_LeDeviceRemoved:
         handled = handleIpc2BtsLeDeviceRemoved(bts2IpcMsgList, bts2AppMsgList, messageItem, static_cast<Ipc2Bts_LeDeviceRemoved*>(message)); /*lint !e1774: Could use dynamic_cast to downcast polymorphic type*/ // no RTTI, use static_cast instead of dynamic_cast, we know the type of the message
         break;
      case Ipc2BtsOC_LeAdapterAddressTypeUpdate:
         handled = handleIpc2BtsLeAdapterAddressTypeUpdate(bts2IpcMsgList, bts2AppMsgList, messageItem, static_cast<Ipc2Bts_LeAdapterAddressTypeUpdate*>(message)); /*lint !e1774: Could use dynamic_cast to downcast polymorphic type*/ // no RTTI, use static_cast instead of dynamic_cast, we know the type of the message
         break;
      case Ipc2BtsOC_LeAdapterAddressUpdate:
         handled = handleIpc2BtsLeAdapterAddressUpdate(bts2IpcMsgList, bts2AppMsgList, messageItem, static_cast<Ipc2Bts_LeAdapterAddressUpdate*>(message)); /*lint !e1774: Could use dynamic_cast to downcast polymorphic type*/ // no RTTI, use static_cast instead of dynamic_cast, we know the type of the message
         break;
      case Ipc2BtsOC_LeAdapterAliasUpdate:
         handled = handleIpc2BtsLeAdapterAliasUpdate(bts2IpcMsgList, bts2AppMsgList, messageItem, static_cast<Ipc2Bts_LeAdapterAliasUpdate*>(message)); /*lint !e1774: Could use dynamic_cast to downcast polymorphic type*/ // no RTTI, use static_cast instead of dynamic_cast, we know the type of the message
         break;
      case Ipc2BtsOC_LeAdapterAdvertisingTimeoutUpdate:
         handled = handleIpc2BtsLeAdapterAdvertisingTimeoutUpdate(bts2IpcMsgList, bts2AppMsgList, messageItem, static_cast<Ipc2Bts_LeAdapterAdvertisingTimeoutUpdate*>(message)); /*lint !e1774: Could use dynamic_cast to downcast polymorphic type*/ // no RTTI, use static_cast instead of dynamic_cast, we know the type of the message
         break;
      case Ipc2BtsOC_LeAdapterDiscoverableModeUpdate:
         handled = handleIpc2BtsLeAdapterDiscoverableModeUpdate(bts2IpcMsgList, bts2AppMsgList, messageItem, static_cast<Ipc2Bts_LeAdapterDiscoverableModeUpdate*>(message)); /*lint !e1774: Could use dynamic_cast to downcast polymorphic type*/ // no RTTI, use static_cast instead of dynamic_cast, we know the type of the message
         break;
      case Ipc2BtsOC_LeAdapterAdvertisingDataUpdate:
         handled = handleIpc2BtsLeAdapterAdvertisingDataUpdate(bts2IpcMsgList, bts2AppMsgList, messageItem, static_cast<Ipc2Bts_LeAdapterAdvertisingDataUpdate*>(message)); /*lint !e1774: Could use dynamic_cast to downcast polymorphic type*/ // no RTTI, use static_cast instead of dynamic_cast, we know the type of the message
         break;
      case Ipc2BtsOC_LeAdapterAdvertiseEnableUpdate:
         handled = handleIpc2BtsLeAdapterAdvertiseEnableUpdate(bts2IpcMsgList, bts2AppMsgList, messageItem, static_cast<Ipc2Bts_LeAdapterAdvertiseEnableUpdate*>(message)); /*lint !e1774: Could use dynamic_cast to downcast polymorphic type*/ // no RTTI, use static_cast instead of dynamic_cast, we know the type of the message
         break;
      case Ipc2BtsOC_AddLeDeviceObjectPathMapping:
         handled = handleIpc2BtsAddLeDeviceObjectPathMapping(bts2IpcMsgList, bts2AppMsgList, messageItem, static_cast<Ipc2Bts_AddLeDeviceObjectPathMapping*>(message)); /*lint !e1774: Could use dynamic_cast to downcast polymorphic type*/ // no RTTI, use static_cast instead of dynamic_cast, we know the type of the message
         break;
      case Ipc2BtsOC_DelLeDeviceObjectPathMapping:
         handled = handleIpc2BtsDelLeDeviceObjectPathMapping(bts2IpcMsgList, bts2AppMsgList, messageItem, static_cast<Ipc2Bts_DelLeDeviceObjectPathMapping*>(message)); /*lint !e1774: Could use dynamic_cast to downcast polymorphic type*/ // no RTTI, use static_cast instead of dynamic_cast, we know the type of the message
         break;
      case Ipc2BtsOC_LeDeviceDisconnectResult:
         handled = handleIpc2BtsLeDeviceDisconnectResult(bts2IpcMsgList, bts2AppMsgList, messageItem, static_cast<Ipc2Bts_LeDeviceDisconnectResult*>(message)); /*lint !e1774: Could use dynamic_cast to downcast polymorphic type*/ // no RTTI, use static_cast instead of dynamic_cast, we know the type of the message
         break;
      case Ipc2BtsOC_LeDisconnectionComp:
         handled = handleIpc2BtsLeDisconnectionComp(bts2IpcMsgList, bts2AppMsgList, messageItem, static_cast<Ipc2Bts_LeDisconnectionComp*>(message)); /*lint !e1774: Could use dynamic_cast to downcast polymorphic type*/ // no RTTI, use static_cast instead of dynamic_cast, we know the type of the message
         break;
      case Ipc2BtsOC_LeDeviceAddressTypeUpdate:
         handled = handleIpc2BtsLeDeviceAddressTypeUpdate(bts2IpcMsgList, bts2AppMsgList, messageItem, static_cast<Ipc2Bts_LeDeviceAddressTypeUpdate*>(message)); /*lint !e1774: Could use dynamic_cast to downcast polymorphic type*/ // no RTTI, use static_cast instead of dynamic_cast, we know the type of the message
         break;
      case Ipc2BtsOC_LeDeviceAddressUpdate:
         handled = handleIpc2BtsLeDeviceAddressUpdate(bts2IpcMsgList, bts2AppMsgList, messageItem, static_cast<Ipc2Bts_LeDeviceAddressUpdate*>(message)); /*lint !e1774: Could use dynamic_cast to downcast polymorphic type*/ // no RTTI, use static_cast instead of dynamic_cast, we know the type of the message
         break;
      case Ipc2BtsOC_LeDeviceConnectedUpdate:
         handled = handleIpc2BtsLeDeviceConnectedUpdate(bts2IpcMsgList, bts2AppMsgList, messageItem, static_cast<Ipc2Bts_LeDeviceConnectedUpdate*>(message)); /*lint !e1774: Could use dynamic_cast to downcast polymorphic type*/ // no RTTI, use static_cast instead of dynamic_cast, we know the type of the message
         break;
      case Ipc2BtsOC_SendSeamlessPairingErrorCodeResult:
         handled = handleIpc2BtsSendSeamlessPairingErrorCodeResult(bts2IpcMsgList, bts2AppMsgList, messageItem, static_cast<Ipc2Bts_SendSeamlessPairingErrorCodeResult*>(message)); /*lint !e1774: Could use dynamic_cast to downcast polymorphic type*/ // no RTTI, use static_cast instead of dynamic_cast, we know the type of the message
         break;
      case Ipc2BtsOC_SeamlessPairingAllowAccessResult:
         handled = handleIpc2BtsSeamlessPairingAllowAccessResult(bts2IpcMsgList, bts2AppMsgList, messageItem, static_cast<Ipc2Bts_SeamlessPairingAllowAccessResult*>(message)); /*lint !e1774: Could use dynamic_cast to downcast polymorphic type*/ // no RTTI, use static_cast instead of dynamic_cast, we know the type of the message
         break;
      case Ipc2BtsOC_SeamlessPairingErrorCodeReceived:
         handled = handleIpc2BtsSeamlessPairingErrorCodeReceived(bts2IpcMsgList, bts2AppMsgList, messageItem, static_cast<Ipc2Bts_SeamlessPairingErrorCodeReceived*>(message)); /*lint !e1774: Could use dynamic_cast to downcast polymorphic type*/ // no RTTI, use static_cast instead of dynamic_cast, we know the type of the message
         break;
      case Ipc2BtsOC_SeamlessPairingAccessRequest:
         handled = handleIpc2BtsSeamlessPairingAccessRequest(bts2IpcMsgList, bts2AppMsgList, messageItem, static_cast<Ipc2Bts_SeamlessPairingAccessRequest*>(message)); /*lint !e1774: Could use dynamic_cast to downcast polymorphic type*/ // no RTTI, use static_cast instead of dynamic_cast, we know the type of the message
         break;
      case Ipc2BtsOC_SeamlessPairingPairingDataUpdate:
         handled = handleIpc2BtsSeamlessPairingPairingDataUpdate(bts2IpcMsgList, bts2AppMsgList, messageItem, static_cast<Ipc2Bts_SeamlessPairingPairingDataUpdate*>(message)); /*lint !e1774: Could use dynamic_cast to downcast polymorphic type*/ // no RTTI, use static_cast instead of dynamic_cast, we know the type of the message
         break;
      // all other
      default:
         break;
   }

   return handled;
}

bool AlpsEvolutionConnectionControl::setSubControlTestCommand(IN const char* testCommand, IN const unsigned int testData)
{
   (void)(testCommand);
   (void)(testData);
   return false;
}

bool AlpsEvolutionConnectionControl::setSubControlTestCommand(IN const char* testCommand, IN const unsigned char* testData)
{
   (void)(testCommand);
   (void)(testData);
   return false;
}

bool AlpsEvolutionConnectionControl::handleTTFisCommand(IN const unsigned char* testData)
{
   (void)(testData);
   return false;
}

bool AlpsEvolutionConnectionControl::handleApp2BtsTestCommand(IN const unsigned char* testData)
{
   (void)(testData);
   return false;
}

ITestModeRequest* AlpsEvolutionConnectionControl::getTestMode(void)
{
   return _testMode;
}

bool AlpsEvolutionConnectionControl::handleIpc2BtsBtApplTestModeReqResult(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_BtApplTestModeReqResult* message)
{
   FW_IF_NULL_PTR_RETURN_FALSE(message);

   // messageItem.message is allowed to be NULL e.g. in case of indication/signal message

   /*
    * current stack behavior is as follows:
    * - either DBUS method return or DBUS error message is sent as answer
    * - DBUS signal message containing "confirmation" data is sent in both cases; this message indicates same error as answer messages
    * => answer can be ignored
    */
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);
   (void)(messageItem);

   return true;
}

bool AlpsEvolutionConnectionControl::handleIpc2BtsBtApplTestModeLinkQualityReqResult(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_BtApplTestModeLinkQualityReqResult* message)
{
   FW_IF_NULL_PTR_RETURN_FALSE(message);

   // messageItem.message is allowed to be NULL e.g. in case of indication/signal message

   /*
    * current stack behavior is as follows:
    * - either DBUS method return or DBUS error message is sent as answer
    * - DBUS signal message containing "confirmation" data is sent in both cases; this message indicates same error as answer messages
    * => answer can be ignored
    */
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);
   (void)(messageItem);

   return true;
}

bool AlpsEvolutionConnectionControl::handleIpc2BtsBtApplTestModeCfm(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_BtApplTestModeCfm* message)
{
   FW_IF_NULL_PTR_RETURN_FALSE(message);
   FW_IF_NULL_PTR_RETURN_FALSE(_testMode);
   TestModeAlpsEvolutionExt& testMode = *_testMode;

   // messageItem.message is allowed to be NULL e.g. in case of indication/signal message

   testMode.handleBtApplTestModeCfm(bts2IpcMsgList, bts2AppMsgList, messageItem, message->getStatus());

   return true;
}

bool AlpsEvolutionConnectionControl::handleIpc2BtsBtApplTestModeCompInd(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_BtApplTestModeCompInd* message)
{
   FW_IF_NULL_PTR_RETURN_FALSE(message);
   FW_IF_NULL_PTR_RETURN_FALSE(_testMode);
   TestModeAlpsEvolutionExt& testMode = *_testMode;

   // messageItem.message is allowed to be NULL e.g. in case of indication/signal message

   testMode.handleBtApplTestModeCompInd(bts2IpcMsgList, bts2AppMsgList, messageItem, message->getEnable(), message->getStatus());

   return true;
}

bool AlpsEvolutionConnectionControl::handleIpc2BtsBtApplTestModeLinkQualityCfm(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_BtApplTestModeLinkQualityCfm* message)
{
   FW_IF_NULL_PTR_RETURN_FALSE(message);
   FW_IF_NULL_PTR_RETURN_FALSE(_testMode);
   TestModeAlpsEvolutionExt& testMode = *_testMode;

   // messageItem.message is allowed to be NULL e.g. in case of indication/signal message

   testMode.handleBtApplTestModeLinkQualityCfm(bts2IpcMsgList, bts2AppMsgList, messageItem, message->getStatus(), message->getRssi(), message->getLinkQuality());

   return true;
}

bool AlpsEvolutionConnectionControl::handleIpc2BtsServiceAvailabilityConnection(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_ServiceAvailabilityConnection* message)
{
   FW_IF_NULL_PTR_RETURN_FALSE(message);
   FW_IF_NULL_PTR_RETURN_FALSE(_ble);
   IBluetoothLowEnergyAlpsEvolutionExt& ble = *_ble;

   // messageItem.message is allowed to be NULL e.g. in case of indication/signal message

   if(true == ble.getEnabled())
   {
      return ble.handleIpc2BtsServiceAvailabilityConnection(bts2IpcMsgList, bts2AppMsgList, messageItem, message);
   }

   return false;
}

bool AlpsEvolutionConnectionControl::handleIpc2BtsGetAllPropertiesConnection(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_GetAllPropertiesConnection* message)
{
   FW_IF_NULL_PTR_RETURN_FALSE(message);
   FW_IF_NULL_PTR_RETURN_FALSE(_ble);
   IBluetoothLowEnergyAlpsEvolutionExt& ble = *_ble;

   // messageItem.message is allowed to be NULL e.g. in case of indication/signal message

   if(true == ble.getEnabled())
   {
      return ble.handleIpc2BtsGetAllPropertiesConnection(bts2IpcMsgList, bts2AppMsgList, messageItem, message);
   }

   return false;
}

bool AlpsEvolutionConnectionControl::handleIpc2BtsCreateLeDeviceResult(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_CreateLeDeviceResult* message)
{
   FW_IF_NULL_PTR_RETURN_FALSE(message);
   FW_IF_NULL_PTR_RETURN_FALSE(_ble);
   IBluetoothLowEnergyAlpsEvolutionExt& ble = *_ble;

   // messageItem.message is allowed to be NULL e.g. in case of indication/signal message

   if(true == ble.getEnabled())
   {
      return ble.handleIpc2BtsCreateLeDeviceResult(bts2IpcMsgList, bts2AppMsgList, messageItem, message);
   }

   return false;
}

bool AlpsEvolutionConnectionControl::handleIpc2BtsRemoveLeDeviceResult(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_RemoveLeDeviceResult* message)
{
   FW_IF_NULL_PTR_RETURN_FALSE(message);
   FW_IF_NULL_PTR_RETURN_FALSE(_ble);
   IBluetoothLowEnergyAlpsEvolutionExt& ble = *_ble;

   // messageItem.message is allowed to be NULL e.g. in case of indication/signal message

   if(true == ble.getEnabled())
   {
      return ble.handleIpc2BtsRemoveLeDeviceResult(bts2IpcMsgList, bts2AppMsgList, messageItem, message);
   }

   return false;
}

bool AlpsEvolutionConnectionControl::handleIpc2BtsLeAdapterInitialization(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_LeAdapterInitialization* message)
{
   FW_IF_NULL_PTR_RETURN_FALSE(message);
   FW_IF_NULL_PTR_RETURN_FALSE(_ble);
   IBluetoothLowEnergyAlpsEvolutionExt& ble = *_ble;

   // messageItem.message is allowed to be NULL e.g. in case of indication/signal message

   if(true == ble.getEnabled())
   {
      return ble.handleIpc2BtsLeAdapterInitialization(bts2IpcMsgList, bts2AppMsgList, messageItem, message);
   }

   return false;
}

bool AlpsEvolutionConnectionControl::handleIpc2BtsLeDeviceCreated(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_LeDeviceCreated* message)
{
   FW_IF_NULL_PTR_RETURN_FALSE(message);
   FW_IF_NULL_PTR_RETURN_FALSE(_ble);
   IBluetoothLowEnergyAlpsEvolutionExt& ble = *_ble;

   // messageItem.message is allowed to be NULL e.g. in case of indication/signal message

   if(true == ble.getEnabled())
   {
      return ble.handleIpc2BtsLeDeviceCreated(bts2IpcMsgList, bts2AppMsgList, messageItem, message);
   }

   return false;
}

bool AlpsEvolutionConnectionControl::handleIpc2BtsLeDeviceRemoved(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_LeDeviceRemoved* message)
{
   FW_IF_NULL_PTR_RETURN_FALSE(message);
   FW_IF_NULL_PTR_RETURN_FALSE(_ble);
   IBluetoothLowEnergyAlpsEvolutionExt& ble = *_ble;

   // messageItem.message is allowed to be NULL e.g. in case of indication/signal message

   if(true == ble.getEnabled())
   {
      return ble.handleIpc2BtsLeDeviceRemoved(bts2IpcMsgList, bts2AppMsgList, messageItem, message);
   }

   return false;
}

bool AlpsEvolutionConnectionControl::handleIpc2BtsLeAdapterAddressTypeUpdate(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_LeAdapterAddressTypeUpdate* message)
{
   FW_IF_NULL_PTR_RETURN_FALSE(message);
   FW_IF_NULL_PTR_RETURN_FALSE(_ble);
   IBluetoothLowEnergyAlpsEvolutionExt& ble = *_ble;

   // messageItem.message is allowed to be NULL e.g. in case of indication/signal message

   if(true == ble.getEnabled())
   {
      return ble.handleIpc2BtsLeAdapterAddressTypeUpdate(bts2IpcMsgList, bts2AppMsgList, messageItem, message);
   }

   return false;
}

bool AlpsEvolutionConnectionControl::handleIpc2BtsLeAdapterAddressUpdate(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_LeAdapterAddressUpdate* message)
{
   FW_IF_NULL_PTR_RETURN_FALSE(message);
   FW_IF_NULL_PTR_RETURN_FALSE(_ble);
   IBluetoothLowEnergyAlpsEvolutionExt& ble = *_ble;

   // messageItem.message is allowed to be NULL e.g. in case of indication/signal message

   if(true == ble.getEnabled())
   {
      return ble.handleIpc2BtsLeAdapterAddressUpdate(bts2IpcMsgList, bts2AppMsgList, messageItem, message);
   }

   return false;
}

bool AlpsEvolutionConnectionControl::handleIpc2BtsLeAdapterAliasUpdate(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_LeAdapterAliasUpdate* message)
{
   FW_IF_NULL_PTR_RETURN_FALSE(message);
   FW_IF_NULL_PTR_RETURN_FALSE(_ble);
   IBluetoothLowEnergyAlpsEvolutionExt& ble = *_ble;

   // messageItem.message is allowed to be NULL e.g. in case of indication/signal message

   if(true == ble.getEnabled())
   {
      return ble.handleIpc2BtsLeAdapterAliasUpdate(bts2IpcMsgList, bts2AppMsgList, messageItem, message);
   }

   return false;
}

bool AlpsEvolutionConnectionControl::handleIpc2BtsLeAdapterAdvertisingTimeoutUpdate(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_LeAdapterAdvertisingTimeoutUpdate* message)
{
   FW_IF_NULL_PTR_RETURN_FALSE(message);
   FW_IF_NULL_PTR_RETURN_FALSE(_ble);
   IBluetoothLowEnergyAlpsEvolutionExt& ble = *_ble;

   // messageItem.message is allowed to be NULL e.g. in case of indication/signal message

   if(true == ble.getEnabled())
   {
      return ble.handleIpc2BtsLeAdapterAdvertisingTimeoutUpdate(bts2IpcMsgList, bts2AppMsgList, messageItem, message);
   }

   return false;
}

bool AlpsEvolutionConnectionControl::handleIpc2BtsLeAdapterDiscoverableModeUpdate(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_LeAdapterDiscoverableModeUpdate* message)
{
   FW_IF_NULL_PTR_RETURN_FALSE(message);
   FW_IF_NULL_PTR_RETURN_FALSE(_ble);
   IBluetoothLowEnergyAlpsEvolutionExt& ble = *_ble;

   // messageItem.message is allowed to be NULL e.g. in case of indication/signal message

   if(true == ble.getEnabled())
   {
      return ble.handleIpc2BtsLeAdapterDiscoverableModeUpdate(bts2IpcMsgList, bts2AppMsgList, messageItem, message);
   }

   return false;
}

bool AlpsEvolutionConnectionControl::handleIpc2BtsLeAdapterAdvertisingDataUpdate(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_LeAdapterAdvertisingDataUpdate* message)
{
   FW_IF_NULL_PTR_RETURN_FALSE(message);
   FW_IF_NULL_PTR_RETURN_FALSE(_ble);
   IBluetoothLowEnergyAlpsEvolutionExt& ble = *_ble;

   // messageItem.message is allowed to be NULL e.g. in case of indication/signal message

   if(true == ble.getEnabled())
   {
      return ble.handleIpc2BtsLeAdapterAdvertisingDataUpdate(bts2IpcMsgList, bts2AppMsgList, messageItem, message);
   }

   return false;
}

bool AlpsEvolutionConnectionControl::handleIpc2BtsLeAdapterAdvertiseEnableUpdate(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_LeAdapterAdvertiseEnableUpdate* message)
{
   FW_IF_NULL_PTR_RETURN_FALSE(message);
   FW_IF_NULL_PTR_RETURN_FALSE(_ble);
   IBluetoothLowEnergyAlpsEvolutionExt& ble = *_ble;

   // messageItem.message is allowed to be NULL e.g. in case of indication/signal message

   if(true == ble.getEnabled())
   {
      return ble.handleIpc2BtsLeAdapterAdvertiseEnableUpdate(bts2IpcMsgList, bts2AppMsgList, messageItem, message);
   }

   return false;
}

bool AlpsEvolutionConnectionControl::handleIpc2BtsAddLeDeviceObjectPathMapping(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_AddLeDeviceObjectPathMapping* message)
{
   FW_IF_NULL_PTR_RETURN_FALSE(message);
   FW_IF_NULL_PTR_RETURN_FALSE(_ble);
   IBluetoothLowEnergyAlpsEvolutionExt& ble = *_ble;

   // messageItem.message is allowed to be NULL e.g. in case of indication/signal message

   if(true == ble.getEnabled())
   {
      return ble.handleIpc2BtsAddLeDeviceObjectPathMapping(bts2IpcMsgList, bts2AppMsgList, messageItem, message);
   }

   return false;
}

bool AlpsEvolutionConnectionControl::handleIpc2BtsDelLeDeviceObjectPathMapping(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_DelLeDeviceObjectPathMapping* message)
{
   FW_IF_NULL_PTR_RETURN_FALSE(message);
   FW_IF_NULL_PTR_RETURN_FALSE(_ble);
   IBluetoothLowEnergyAlpsEvolutionExt& ble = *_ble;

   // messageItem.message is allowed to be NULL e.g. in case of indication/signal message

   if(true == ble.getEnabled())
   {
      return ble.handleIpc2BtsDelLeDeviceObjectPathMapping(bts2IpcMsgList, bts2AppMsgList, messageItem, message);
   }

   return false;
}

bool AlpsEvolutionConnectionControl::handleIpc2BtsLeDeviceDisconnectResult(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_LeDeviceDisconnectResult* message)
{
   FW_IF_NULL_PTR_RETURN_FALSE(message);
   FW_IF_NULL_PTR_RETURN_FALSE(_ble);
   IBluetoothLowEnergyAlpsEvolutionExt& ble = *_ble;

   // messageItem.message is allowed to be NULL e.g. in case of indication/signal message

   if(true == ble.getEnabled())
   {
      return ble.handleIpc2BtsLeDeviceDisconnectResult(bts2IpcMsgList, bts2AppMsgList, messageItem, message);
   }

   return false;
}

bool AlpsEvolutionConnectionControl::handleIpc2BtsLeDisconnectionComp(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_LeDisconnectionComp* message)
{
   FW_IF_NULL_PTR_RETURN_FALSE(message);
   FW_IF_NULL_PTR_RETURN_FALSE(_ble);
   IBluetoothLowEnergyAlpsEvolutionExt& ble = *_ble;

   // messageItem.message is allowed to be NULL e.g. in case of indication/signal message

   if(true == ble.getEnabled())
   {
      return ble.handleIpc2BtsLeDisconnectionComp(bts2IpcMsgList, bts2AppMsgList, messageItem, message);
   }

   return false;
}

bool AlpsEvolutionConnectionControl::handleIpc2BtsLeDeviceAddressTypeUpdate(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_LeDeviceAddressTypeUpdate* message)
{
   FW_IF_NULL_PTR_RETURN_FALSE(message);
   FW_IF_NULL_PTR_RETURN_FALSE(_ble);
   IBluetoothLowEnergyAlpsEvolutionExt& ble = *_ble;

   // messageItem.message is allowed to be NULL e.g. in case of indication/signal message

   if(true == ble.getEnabled())
   {
      return ble.handleIpc2BtsLeDeviceAddressTypeUpdate(bts2IpcMsgList, bts2AppMsgList, messageItem, message);
   }

   return false;
}

bool AlpsEvolutionConnectionControl::handleIpc2BtsLeDeviceAddressUpdate(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_LeDeviceAddressUpdate* message)
{
   FW_IF_NULL_PTR_RETURN_FALSE(message);
   FW_IF_NULL_PTR_RETURN_FALSE(_ble);
   IBluetoothLowEnergyAlpsEvolutionExt& ble = *_ble;

   // messageItem.message is allowed to be NULL e.g. in case of indication/signal message

   if(true == ble.getEnabled())
   {
      return ble.handleIpc2BtsLeDeviceAddressUpdate(bts2IpcMsgList, bts2AppMsgList, messageItem, message);
   }

   return false;
}

bool AlpsEvolutionConnectionControl::handleIpc2BtsLeDeviceConnectedUpdate(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_LeDeviceConnectedUpdate* message)
{
   FW_IF_NULL_PTR_RETURN_FALSE(message);
   FW_IF_NULL_PTR_RETURN_FALSE(_ble);
   IBluetoothLowEnergyAlpsEvolutionExt& ble = *_ble;

   // messageItem.message is allowed to be NULL e.g. in case of indication/signal message

   if(true == ble.getEnabled())
   {
      return ble.handleIpc2BtsLeDeviceConnectedUpdate(bts2IpcMsgList, bts2AppMsgList, messageItem, message);
   }

   return false;
}

bool AlpsEvolutionConnectionControl::handleIpc2BtsSendSeamlessPairingErrorCodeResult(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_SendSeamlessPairingErrorCodeResult* message)
{
   FW_IF_NULL_PTR_RETURN_FALSE(message);
   FW_IF_NULL_PTR_RETURN_FALSE(_ble);
   IBluetoothLowEnergyAlpsEvolutionExt& ble = *_ble;

   // messageItem.message is allowed to be NULL e.g. in case of indication/signal message

   if(true == ble.getEnabled())
   {
      return ble.handleIpc2BtsSendSeamlessPairingErrorCodeResult(bts2IpcMsgList, bts2AppMsgList, messageItem, message);
   }

   return false;
}

bool AlpsEvolutionConnectionControl::handleIpc2BtsSeamlessPairingAllowAccessResult(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_SeamlessPairingAllowAccessResult* message)
{
   FW_IF_NULL_PTR_RETURN_FALSE(message);
   FW_IF_NULL_PTR_RETURN_FALSE(_ble);
   IBluetoothLowEnergyAlpsEvolutionExt& ble = *_ble;

   // messageItem.message is allowed to be NULL e.g. in case of indication/signal message

   if(true == ble.getEnabled())
   {
      return ble.handleIpc2BtsSeamlessPairingAllowAccessResult(bts2IpcMsgList, bts2AppMsgList, messageItem, message);
   }

   return false;
}

bool AlpsEvolutionConnectionControl::handleIpc2BtsSeamlessPairingErrorCodeReceived(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_SeamlessPairingErrorCodeReceived* message)
{
   FW_IF_NULL_PTR_RETURN_FALSE(message);
   FW_IF_NULL_PTR_RETURN_FALSE(_ble);
   IBluetoothLowEnergyAlpsEvolutionExt& ble = *_ble;

   // messageItem.message is allowed to be NULL e.g. in case of indication/signal message

   if(true == ble.getEnabled())
   {
      return ble.handleIpc2BtsSeamlessPairingErrorCodeReceived(bts2IpcMsgList, bts2AppMsgList, messageItem, message);
   }

   return false;
}

bool AlpsEvolutionConnectionControl::handleIpc2BtsSeamlessPairingAccessRequest(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_SeamlessPairingAccessRequest* message)
{
   FW_IF_NULL_PTR_RETURN_FALSE(message);
   FW_IF_NULL_PTR_RETURN_FALSE(_ble);
   IBluetoothLowEnergyAlpsEvolutionExt& ble = *_ble;

   // messageItem.message is allowed to be NULL e.g. in case of indication/signal message

   if(true == ble.getEnabled())
   {
      return ble.handleIpc2BtsSeamlessPairingAccessRequest(bts2IpcMsgList, bts2AppMsgList, messageItem, message);
   }

   return false;
}

bool AlpsEvolutionConnectionControl::handleIpc2BtsSeamlessPairingPairingDataUpdate(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_SeamlessPairingPairingDataUpdate* message)
{
   FW_IF_NULL_PTR_RETURN_FALSE(message);
   FW_IF_NULL_PTR_RETURN_FALSE(_ble);
   IBluetoothLowEnergyAlpsEvolutionExt& ble = *_ble;

   // messageItem.message is allowed to be NULL e.g. in case of indication/signal message

   if(true == ble.getEnabled())
   {
      return ble.handleIpc2BtsSeamlessPairingPairingDataUpdate(bts2IpcMsgList, bts2AppMsgList, messageItem, message);
   }

   return false;
}

} //alpsevolutionext
} //genivi
} //btstackif
