/**
 * @file IWblWiFiSetupStatusCallback.h
 *
 * @par SW-Component
 * State machine for WBL WiFi setup status
 *
 * @brief Callback interface definition for WBL WiFi setup status state machine.
 *
 * @copyright (C) 2017 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details This file defines the callback interface to be used by specific WBL WiFi setup status state machine.
 *          Callback interface must be provided by generic WBL WiFi setup status state machine.
 */

#ifndef _I_WBL_WIFI_SETUP_STATUS_CALLBACK_H_
#define _I_WBL_WIFI_SETUP_STATUS_CALLBACK_H_

#include "BtStackInternalTypes.h"

namespace btstackif {

// class forward declarations
class Bts2Ipc_BaseMessage;
class Bts2App_BaseMessage;

/**
 * Interface definition.
 */
class IWblWiFiSetupStatusCallback
{
public:
   /**
    * Destructor.
    */
   virtual ~IWblWiFiSetupStatusCallback() {}

   /**
    * WiFi setup entry added.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] id: identifier
    */
   virtual void wifiSetupEntryAdded(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSWblWiFiSetupId id) = 0;

   /**
    * WiFi setup entry removed.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] id: identifier
    */
   virtual void wifiSetupEntryRemoved(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSWblWiFiSetupId id) = 0;

   /**
    * WiFi setup entry available.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] id: identifier
    */
   virtual void wifiSetupEntryAvailable(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSWblWiFiSetupId id) = 0;

   /**
    * WiFi setup entry unavailable.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] id: identifier
    */
   virtual void wifiSetupEntryUnavailable(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSWblWiFiSetupId id) = 0;

   /**
    * Get WiFi setup list result.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] result: result
    */
   virtual void getWifiSetupListResult(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSIpcCommonErrorCode result) = 0;

   /**
    * Update WiFi mode.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] result: result
    * @param[in] id: identifier
    * @param[in] mode: mode
    */
   virtual void updateWifiMode(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSIpcCommonErrorCode result, IN const BTSWblWiFiSetupId id, IN const BTSWblWifiMode mode) = 0;

   /**
    * Update WiFi power state.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] result: result
    * @param[in] id: identifier
    * @param[in] powerState: power state
    */
   virtual void updateWifiPowerState(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSIpcCommonErrorCode result, IN const BTSWblWiFiSetupId id, IN const BTSWblWifiPowerState powerState) = 0;

   /**
    * Update WiFi frequency.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] result: result
    * @param[in] id: identifier
    * @param[in] frequency: frequency
    */
   virtual void updateWifiFrequency(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSIpcCommonErrorCode result, IN const BTSWblWiFiSetupId id, IN const BTSWblWifiFrequency frequency) = 0;

   /**
    * Update access point config type.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] result: result
    * @param[in] id: identifier
    * @param[in] type: type
    */
   virtual void updateAccessPointConfigType(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSIpcCommonErrorCode result, IN const BTSWblWiFiSetupId id, IN const BTSWblAccessPointConfigType type) = 0;

   /**
    * Update supported restrictions.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] result: result
    * @param[in] id: identifier
    * @param[in] supportedRestrictions: supported restrictions
    */
   virtual void updateSupportedRestrictions(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSIpcCommonErrorCode result, IN const BTSWblWiFiSetupId id, IN const BTSWblSupportedRestrictionList& supportedRestrictions) = 0;

   /**
    * Update WiFi healthiness indicator.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] result: result
    * @param[in] id: identifier
    * @param[in] connected: connected flag
    * @param[in] healthIndicator: healthiness indicator
    */
   virtual void updateWifiHealthinessIndicator(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSIpcCommonErrorCode result, IN const BTSWblWiFiSetupId id, IN const bool connected, IN const BTSWblHealthinessIndicator healthIndicator) = 0;
};

} //btstackif

#endif //_I_WBL_WIFI_SETUP_STATUS_CALLBACK_H_
