/**
 * @file WblStartup.cpp
 *
 * @par SW-Component
 * State machine for WBL startup
 *
 * @brief Implementation of generic WBL startup state machine.
 *
 * @copyright (C) 2017 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Source file for implementation of generic WBL startup state machine.
 */

#include "WblStartup.h"
#include "IWblStartupRequest.h"
#include "IBasicControl.h"
#include "IWblStartupObserver.h"
#include "ITimerPool.h"
// #include "Timer.h"
#include "FwAssert.h"
#include "App2Bts_MessageWrapper.h"
#include "Bts2App_MessageWrapper.h"
#include "TraceClasses.h"
#include "FwTrace.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_BTS_CONTROL
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/WblStartup.cpp.trc.h"
#endif
#endif

namespace btstackif {

WblStartup::WblStartup() :
_requestIf(0),
_controlIf(0),
_timerPoolIf(0),
_availability(BTS_DBUS_SERVICE_NOT_AVAILABLE),
_initializedDone(false),
_observerList(),
_startupData()
{
}

WblStartup::~WblStartup()
{
   _requestIf = 0;
   _controlIf = 0;
   _timerPoolIf = 0;
}

void WblStartup::reset(void)
{
   StateMachine::reset();
   // keep _requestIf
   // keep _controlIf
   // keep _timerPoolIf
   // keep _availability
   // keep _initializedDone
   // keep _observerList
   _startupData.reset();
}

void WblStartup::forceInitialState(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList)
{
   (void)(bts2AppMsgList);

   createWblServiceAvailabilityStatusMsg(bts2AppMsgList, 0, 0, true, _availability);

   // reset control data
   reset();
}

void WblStartup::setInstance(IN IWblStartupRequest* instance)
{
   _requestIf = instance;

   FW_IF_NULL_PTR_RETURN(_requestIf);

   _requestIf->setCallback(this);
}

void WblStartup::setControlIf(IN IBasicControl* control)
{
   _controlIf = control;

   FW_IF_NULL_PTR_RETURN(_requestIf);

   _requestIf->setControlIf(_controlIf);
}

void WblStartup::setTimerPoolIf(IN ITimerPool* timerPool)
{
   _timerPoolIf = timerPool;

   FW_NORMAL_ASSERT(0 != _timerPoolIf);
}

void WblStartup::sendInitializedStatus(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN BtStackIfCallback* user, IN const BTSSessionHandle handle, IN const BTSCommonEnumClass statusCode) const
{
   (void)(statusCode);

   // send current status
   createFbWblInitializedStatusMsg(bts2AppMsgList, user, handle, true, BTS_DBUS_SERVICE_AVAILABLE); // indicate that BtStackIf part is running
}

void WblStartup::sendInitializedStatus(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const App2Bts_TriggerFbWblInitialized& request, IN const BTSCommonEnumClass statusCode) const
{
   sendInitializedStatus(bts2AppMsgList, request.getUser(), request.getSessionHandle(), statusCode);
}

void WblStartup::sendServiceAvailabilityStatus(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN BtStackIfCallback* user, IN const BTSSessionHandle handle, IN const BTSCommonEnumClass statusCode) const
{
   (void)(statusCode);

   // send current status
   createWblServiceAvailabilityStatusMsg(bts2AppMsgList, user, handle, false, _availability);
}

void WblStartup::sendServiceAvailabilityStatus(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const App2Bts_GetWblServiceAvailability& request, IN const BTSCommonEnumClass statusCode) const
{
   sendServiceAvailabilityStatus(bts2AppMsgList, request.getUser(), request.getSessionHandle(), statusCode);
}

bool WblStartup::getServiceAvailability(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const App2Bts_GetWblServiceAvailability& request)
{
   (void)(bts2IpcMsgList);

   // send current status
   createWblServiceAvailabilityStatusMsg(bts2AppMsgList, request.getUser(), request.getSessionHandle(), false, _availability);

   return false;
}

bool WblStartup::processTriggerInitializedCallback(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const App2Bts_TriggerFbWblInitialized& request)
{
   (void)(bts2IpcMsgList);

   // check if initialization is already done
   if(true == _initializedDone)
   {
      // send current status
      createFbWblInitializedStatusMsg(bts2AppMsgList, request.getUser(), request.getSessionHandle(), true, BTS_DBUS_SERVICE_AVAILABLE); // indicate that BtStackIf part is running
   }
   else
   {
      // wait for end of initialization
   }

   return false;
}

void WblStartup::processTriggerInitializedCallback(void)
{
   FW_IF_NULL_PTR_RETURN(_controlIf);

   // Note: This function might be called from another thread context.
   // => generate internal application message
   _controlIf->sendInternalApp2BtsMessage(ptrNew_App2Bts_TriggerFbWblInitialized());
}

bool WblStartup::isServiceAvailable(void) const
{
   return (BTS_DBUS_SERVICE_AVAILABLE == _availability);
}

bool WblStartup::isStartupOngoing(void) const
{
   return (BTS_DBUS_SERVICE_WAITING == _availability);
}

void WblStartup::registerObserver(IN IWblStartupObserver* observer)
{
   FW_IF_NULL_PTR_RETURN(observer);

   _observerList.insert(observer);
}

void WblStartup::informAboutInitialDataStatus(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const WblStartupData::InitialDataPendingBit pendingBit, IN const bool enable)
{
   if(WblStartupData::INITIAL_DATA_LAST <= pendingBit)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   const unsigned int oldPendingData(_startupData.pending.getData());

   if(true == enable)
   {
      _startupData.pending.setBit(pendingBit);
   }
   else
   {
      _startupData.pending.resetBit(pendingBit);
   }

   // handle only different status
   if(oldPendingData == _startupData.pending.getData())
   {
      return;
   }

   BTSDbusServiceAvailability availability(BTS_DBUS_SERVICE_NOT_AVAILABLE);
   if(_startupData.pending.getData() == _startupData.pending.getDataMask())
   {
      availability = BTS_DBUS_SERVICE_AVAILABLE;
   }

   // inform observers
   for(::std::set< IWblStartupObserver* >::const_iterator it = _observerList.begin(); it != _observerList.end(); ++it)
   {
      if(0 != *it)
      {
         (*it)->initialDataAvailabilityChanged(bts2IpcMsgList, bts2AppMsgList, messageItem, availability);
      }
   }

   // send current status
   if(BTS_DBUS_SERVICE_AVAILABLE == availability)
   {
      createWblServiceAvailabilityStatusMsg(bts2AppMsgList, 0, 0, true, _availability);
   }
}

void WblStartup::updateServiceAvailability(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSDbusServiceAvailability availability)
{
   // handle only different status
   if(availability == _availability)
   {
      return;
   }

   const BTSDbusServiceAvailability oldAvailability(_availability);
   _availability = availability;

   ETG_TRACE_USR3((" updateServiceAvailability: oldAvailability=%d newAvailability=%d", oldAvailability, _availability));

   if(BTS_DBUS_SERVICE_WAITING == _availability)
   {
      ETG_TRACE_USR1((" ### WBL startup started ###"));

      // send current status
      if(false == _initializedDone)
      {
         createFbWblInitializedStatusMsg(bts2AppMsgList, 0, 0, true, BTS_DBUS_SERVICE_AVAILABLE); // indicate that BtStackIf part is running
         _initializedDone = true;
      }
   }
   else if(BTS_DBUS_SERVICE_AVAILABLE == _availability)
   {
      if(BTS_DBUS_SERVICE_WAITING == oldAvailability)
      {
         ETG_TRACE_USR1((" ### WBL startup finished (success) ###"));
      }
      else if(BTS_DBUS_SERVICE_NOT_AVAILABLE == oldAvailability)
      {
         ETG_TRACE_USR1((" ### WBL startup finished (success after timeout) ###"));
      }
      else
      {
         FW_NORMAL_ASSERT_ALWAYS();
      }
   }
   else if(BTS_DBUS_SERVICE_NOT_AVAILABLE == _availability)
   {
      if(BTS_DBUS_SERVICE_WAITING == oldAvailability)
      {
         ETG_TRACE_USR1((" ### WBL startup finished (failed) ###"));
      }
      else if(BTS_DBUS_SERVICE_AVAILABLE == oldAvailability)
      {
         ETG_TRACE_USR1((" ### WBL crashed !? ###"));
      }
      else
      {
         FW_NORMAL_ASSERT_ALWAYS();
      }
   }
   else
   {
      FW_NORMAL_ASSERT_ALWAYS();
   }

   // send current status
   if(BTS_DBUS_SERVICE_AVAILABLE != _availability)
   {
      createWblServiceAvailabilityStatusMsg(bts2AppMsgList, 0, 0, true, _availability);
   }

   informObservers(bts2IpcMsgList, bts2AppMsgList, messageItem, _availability);
}

void WblStartup::createFbWblInitializedStatusMsg(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN BtStackIfCallback* user, IN const BTSSessionHandle handle, IN const bool sendStatusToAll, IN const BTSDbusServiceAvailability availability) const
{
   Bts2App_FbWblInitialized* msg = ptrNew_Bts2App_FbWblInitialized();
   if(0 != msg)
   {
      if((true == sendStatusToAll) || (0 == user))
      {
         msg->setUser(0); // send status message to all
         msg->setSessionHandle(0); // send status message to all
      }
      else
      {
         msg->setUser(user);
         msg->setSessionHandle(handle);
      }
      if(BTS_DBUS_SERVICE_AVAILABLE == availability)
      {
         msg->setRequestResult(BTS_REQ_SUCCESS);
      }
      else
      {
         msg->setRequestResult(BTS_REQ_FAILED);
      }

      bts2AppMsgList.push_back(msg);
   }
}

void WblStartup::createWblServiceAvailabilityStatusMsg(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN BtStackIfCallback* user, IN const BTSSessionHandle handle, IN const bool sendStatusToAll, IN const BTSDbusServiceAvailability availability) const
{
   Bts2App_WblServiceAvailability* msg = ptrNew_Bts2App_WblServiceAvailability();
   if(0 != msg)
   {
      if((true == sendStatusToAll) || (0 == user))
      {
         msg->setUser(0); // send status message to all
         msg->setSessionHandle(0); // send status message to all
      }
      else
      {
         msg->setUser(user);
         msg->setSessionHandle(handle);
      }
      msg->setServiceAvailable((BTS_DBUS_SERVICE_AVAILABLE == availability));

      bts2AppMsgList.push_back(msg);
   }
}

void WblStartup::informObservers(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSDbusServiceAvailability availability) const
{
   for(::std::set< IWblStartupObserver* >::const_iterator it = _observerList.begin(); it != _observerList.end(); ++it)
   {
      if(0 != *it)
      {
         (*it)->serviceAvailabilityChanged(bts2IpcMsgList, bts2AppMsgList, messageItem, availability);
      }
   }
}

} //btstackif
