/**
 * @file WblLastMode.cpp
 *
 * @par SW-Component
 * State machine for WBL last mode
 *
 * @brief Implementation of generic WBL last mode state machine.
 *
 * @copyright (C) 2017 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Source file for implementation of generic WBL last mode state machine.
 */

#include "WblLastMode.h"
#include "IWblLastModeRequest.h"
#include "IBasicControl.h"
#include "ITimerPool.h"
// #include "Timer.h"
#include "IWblStartup.h"
#include "FwAssert.h"
#include "App2Bts_MessageWrapper.h"
#include "Bts2App_MessageWrapper.h"
#include "TraceClasses.h"
#include "FwTrace.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_BTS_CONTROL
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/WblLastMode.cpp.trc.h"
#endif
#endif

namespace btstackif {

WblLastMode::WblLastMode() :
_requestIf(0),
_controlIf(0),
_timerPoolIf(0),
_wblStartup(0),
_accessPointList(),
_lastUpdateList(),
_initialUpdateDone(false),
_maxRetry(1),
_nmbRetry(0),
_timer(),
_timeoutRepeatGet(1000),
_getPending(false),
_initialGetDone(false)
{
}

WblLastMode::~WblLastMode()
{
   _requestIf = 0;
   _controlIf = 0;
   _timerPoolIf = 0;
   _wblStartup = 0;
}

void WblLastMode::reset(void)
{
   StateMachine::reset();
   // keep _requestIf
   // keep _controlIf
   // keep _timerPoolIf
   // keep _wblStartup
   _accessPointList.clear();
   _lastUpdateList.clear();
   _initialUpdateDone = false;
   _nmbRetry = 0;
   // stop and release timer
   releaseTimer(_timer);
   _getPending = false;
   _initialGetDone = false;
}

void WblLastMode::forceInitialState(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList)
{
   // update current access point list as empty
   _accessPointList.clear();
   createWblLastIntendedModeStatusMsg(bts2AppMsgList, 0, 0, true, _accessPointList);

   // reset control data
   reset();
}

void WblLastMode::setInstance(IN IWblLastModeRequest* instance)
{
   _requestIf = instance;

   FW_IF_NULL_PTR_RETURN(_requestIf);

   _requestIf->setCallback(this);
}

void WblLastMode::setControlIf(IN IBasicControl* control)
{
   _controlIf = control;

   FW_IF_NULL_PTR_RETURN(_requestIf);

   _requestIf->setControlIf(_controlIf);
}

void WblLastMode::setTimerPoolIf(IN ITimerPool* timerPool)
{
   _timerPoolIf = timerPool;

   FW_NORMAL_ASSERT(0 != _timerPoolIf);
}

void WblLastMode::setWblStartupIf(IN IWblStartup* wblStartup)
{
   _wblStartup = wblStartup;

   FW_NORMAL_ASSERT(0 != _wblStartup);
}

void WblLastMode::sendStatus(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN BtStackIfCallback* user, IN const BTSSessionHandle handle, IN const BTSCommonEnumClass statusCode) const
{
   (void)(statusCode);

   // send current status
   createWblLastIntendedModeStatusMsg(bts2AppMsgList, user, handle, false, _accessPointList);
}

void WblLastMode::sendStatus(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const App2Bts_GetWblLastIntendedMode& request, IN const BTSCommonEnumClass statusCode) const
{
   sendStatus(bts2AppMsgList, request.getUser(), request.getSessionHandle(), statusCode);
}

bool WblLastMode::getLastMode(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const App2Bts_GetWblLastIntendedMode& request)
{
   (void)(bts2IpcMsgList);

   if(true == _getPending)
   {
      // update will be sent after get is finished
   }
   else
   {
      // send current status
      createWblLastIntendedModeStatusMsg(bts2AppMsgList, request.getUser(), request.getSessionHandle(), false, _accessPointList);
   }

   return false;
}

IWblStartupObserver* WblLastMode::getStartupObserver(void)
{
   return this;
}

void WblLastMode::updateLastMode(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSIpcCommonErrorCode result, IN const BTSWblLastIntendedModeList& accessPointStatusList)
{
   ETG_TRACE_USR2((" updateLastMode: accessPointStatusList.size()=%u", accessPointStatusList.size()));

   if(BTS_IPC_SUCCESS != result)
   {
      if(_maxRetry > _nmbRetry)
      {
         _nmbRetry++;

         // start timer
         startTimer(_timer, _timeoutRepeatGet);
      }
      else
      {
         // stop timer
         stopTimer(_timer);

         // no retries left, failed answer for get request

         _accessPointList.clear();
         _getPending = false;
         if(true == _initialGetDone)
         {
            createWblLastIntendedModeStatusMsgWrapper(bts2AppMsgList, 0, 0, true, _accessPointList);
         }
         else
         {
            // mark initial get as done
            handleChangedInitialGetDone(bts2IpcMsgList, bts2AppMsgList, messageItem, true);
         }
      }
   }
   else
   {
      // stop timer
      stopTimer(_timer);

      // either property update or successful answer for get request

      _accessPointList = accessPointStatusList;
      _getPending = false;
      if(true == _initialGetDone)
      {
         createWblLastIntendedModeStatusMsgWrapper(bts2AppMsgList, 0, 0, true, _accessPointList);
      }
      else
      {
         // mark initial get as done
         handleChangedInitialGetDone(bts2IpcMsgList, bts2AppMsgList, messageItem, true);
      }
   }
}

void WblLastMode::handleExtendedTimeout(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSTimerId timerId)
{
   (void)(messageItem);

   ETG_TRACE_USR2((" handleExtendedTimeout"));

   FW_IF_NULL_PTR_RETURN(_requestIf);

   // check timer id
   if(false == _timer.compare(timerId))
   {
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   _requestIf->getLastMode(bts2IpcMsgList, bts2AppMsgList);
}

void WblLastMode::serviceAvailabilityChanged(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSDbusServiceAvailability availability)
{
   FW_IF_NULL_PTR_RETURN(_requestIf);

   if(BTS_DBUS_SERVICE_AVAILABLE == availability)
   {
      stopTimer(_timer);
      _nmbRetry = 0;

      // send get
      _requestIf->getLastMode(bts2IpcMsgList, bts2AppMsgList);
      _getPending = true;

      // mark initial get as not done
      handleChangedInitialGetDone(bts2IpcMsgList, bts2AppMsgList, messageItem, false);
   }
   else if(BTS_DBUS_SERVICE_NOT_AVAILABLE == availability)
   {
      stopTimer(_timer);

      // reset status
      _accessPointList.clear();
      _getPending = false;
      if(true == _initialGetDone)
      {
         createWblLastIntendedModeStatusMsgWrapper(bts2AppMsgList, 0, 0, true, _accessPointList);
      }
      else
      {
         // mark initial get as done
         handleChangedInitialGetDone(bts2IpcMsgList, bts2AppMsgList, messageItem, true);
      }
   }
   else if(BTS_DBUS_SERVICE_WAITING == availability)
   {
      // ignore
   }
   else
   {
      FW_NORMAL_ASSERT_ALWAYS();
   }
}

void WblLastMode::initialDataAvailabilityChanged(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSDbusServiceAvailability availability)
{
   (void)(bts2IpcMsgList);
   (void)(messageItem);

   if(BTS_DBUS_SERVICE_AVAILABLE == availability)
   {
      // all initial data is available => update now
      createWblLastIntendedModeStatusMsgWrapper(bts2AppMsgList, 0, 0, true, _accessPointList);
   }
   else if(BTS_DBUS_SERVICE_NOT_AVAILABLE == availability)
   {
      // ignore
   }
   else if(BTS_DBUS_SERVICE_WAITING == availability)
   {
      // ignore
   }
   else
   {
      FW_NORMAL_ASSERT_ALWAYS();
   }
}

void WblLastMode::handleChangedInitialGetDone(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const bool newInitialGetDone)
{
   FW_IF_NULL_PTR_RETURN(_wblStartup);

   // handle only different status
   if(_initialGetDone == newInitialGetDone)
   {
      return;
   }

   // remember new status
   _initialGetDone = newInitialGetDone;

   // inform startup handler
   _wblStartup->informAboutInitialDataStatus(bts2IpcMsgList, bts2AppMsgList, messageItem, WblStartupData::LAST_MODE_PENDING, _initialGetDone);
}

void WblLastMode::createWblLastIntendedModeStatusMsgWrapper(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN BtStackIfCallback* user, IN const BTSSessionHandle handle, IN const bool sendStatusToAll, IN const BTSWblLastIntendedModeList& accessPointStatusList)
{
   if((true == _initialUpdateDone) && (accessPointStatusList == _lastUpdateList))
   {
      return;
   }

   _initialUpdateDone = true;

   _lastUpdateList = accessPointStatusList;

   createWblLastIntendedModeStatusMsg(bts2AppMsgList, user, handle, sendStatusToAll, _lastUpdateList);
}

void WblLastMode::createWblLastIntendedModeStatusMsg(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN BtStackIfCallback* user, IN const BTSSessionHandle handle, IN const bool sendStatusToAll, IN const BTSWblLastIntendedModeList& accessPointStatusList) const
{
   Bts2App_WblLastIntendedMode* msg = ptrNew_Bts2App_WblLastIntendedMode();
   if(0 != msg)
   {
      if((true == sendStatusToAll) || (0 == user))
      {
         msg->setUser(0); // send status message to all
         msg->setSessionHandle(0); // send status message to all
      }
      else
      {
         msg->setUser(user);
         msg->setSessionHandle(handle);
      }
      msg->setWblLastIntendedModeList(accessPointStatusList);

      bts2AppMsgList.push_back(msg);
   }
}

void WblLastMode::startTimer(IN ExtendedTimerEntry& timer, IN const BTSTimeValue timeout)
{
   ETG_TRACE_USR3((" startTimer: timeout=%u", timeout));

   FW_IF_NULL_PTR_RETURN(_timerPoolIf);
   FW_IF_NULL_PTR_RETURN(_controlIf);

   timer.setTimerPool(_timerPoolIf);
   timer.start(_controlIf, this, timeout);
}

void WblLastMode::stopTimer(IN ExtendedTimerEntry& timer) const
{
   ETG_TRACE_USR3((" stopTimer"));

   timer.stop();

   // do not release timer
}

void WblLastMode::releaseTimer(IN ExtendedTimerEntry& timer) const
{
   ETG_TRACE_USR3((" releaseTimer"));

   timer.release();
}

} //btstackif
