/**
 * @file IWblLastMode.h
 *
 * @par SW-Component
 * State machine for WBL last mode
 *
 * @brief Interface definition for generic WBL last mode state machine.
 *
 * @copyright (C) 2017 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details This file defines the generic interface for WBL last mode state machine.
 */

#ifndef _I_WBL_LAST_MODE_H_
#define _I_WBL_LAST_MODE_H_

#include "BtStackInternalTypes.h"

namespace btstackif {

// class forward declarations
class IWblLastModeRequest;
class IBasicControl;
class ITimerPool;
class IWblStartup;
class App2Bts_GetWblLastIntendedMode;
class Bts2Ipc_BaseMessage;
class Bts2App_BaseMessage;
class IWblStartupObserver;

/**
 * Interface definition.
 */
class IWblLastMode
{
public:
   /**
    * Destructor.
    */
   virtual ~IWblLastMode() {}

   /**
    * Set specific instance (implementation).
    *
    * @param[in] instance: instance (implementation)
    */
   virtual void setInstance(IN IWblLastModeRequest* instance) = 0;

   /**
    * Set basic control interface.
    *
    * @param[in] control: basic control interface
    */
   virtual void setControlIf(IN IBasicControl* control) = 0;

   /**
    * Set timer pool interface.
    *
    * @param[in] timerPool: timer pool interface
    */
   virtual void setTimerPoolIf(IN ITimerPool* timerPool) = 0;

   /**
    * Set WBL startup interface.
    *
    * @param[in] wblStartup: WBL startup interface
    */
   virtual void setWblStartupIf(IN IWblStartup* wblStartup) = 0;

   /**
    * Send current status for given control data.
    * <BR><B>This method shall be used to reject application requests or handle doubled application requests.</B>
    *
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[in] user: user
    * @param[in] handle: handle
    * @param[in] statusCode: status code
    */
   virtual void sendStatus(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN BtStackIfCallback* user, IN const BTSSessionHandle handle, IN const BTSCommonEnumClass statusCode) const = 0;

   /**
    * Send current status for given control data.
    * <BR><B>This method shall be used to reject application requests or handle doubled application requests.</B>
    *
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[in] request: request
    * @param[in] statusCode: status code
    */
   virtual void sendStatus(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const App2Bts_GetWblLastIntendedMode& request, IN const BTSCommonEnumClass statusCode) const = 0;

   /**
    * Get last mode.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[in] request: request
    *
    * @return = true: request processing was started (request shall not be deleted),
    * @return = false: no further processing necessary (request can be deleted)
    */
   virtual bool getLastMode(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const App2Bts_GetWblLastIntendedMode& request) = 0;

   /**
    * Get startup observer.
    *
    * @return = startup observer
    */
   virtual IWblStartupObserver* getStartupObserver(void) = 0;
};

} //btstackif

#endif //_I_WBL_LAST_MODE_H_
