/**
 * @file TestMode.cpp
 *
 * @par SW-Component
 * State machine for test mode
 *
 * @brief Implementation of generic test mode state machine.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Source file for implementation of generic test mode state machine.
 */

#include "TestMode.h"
#include "ITestModeRequest.h"
#include "IBasicControl.h"
#include "FwAssert.h"
#include "FwStringUtils.h"
#include "FwBluetoothStringUtils.h"
#include "App2Bts_MessageWrapper.h"
#include "Bts2App_MessageWrapper.h"

namespace btstackif {

TestMode::TestMode() :
_requestIf(0),
_controlIf(0),
_requestItem(),
_active(false),
_address()
{
}

TestMode::~TestMode()
{
   _requestIf = 0;
   _controlIf = 0;
}

void TestMode::reset(void)
{
   StateMachine::reset();
   // keep _requestIf
   // keep _controlIf
   _requestItem.reset();
   _active = false;
   _address.clear();

   FW_IF_NULL_PTR_RETURN(_requestIf);
   _requestIf->reset();
}

void TestMode::forceInitialState(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList)
{
   // check current action
   switch(_requestItem.item.opCode)
   {
      case App2BtsOC_Last:
         // no request is pending
         if(true == _active)
         {
            // test mode is active => update status as false
            createTestModeStatusMsg(bts2AppMsgList, 0, 0, _address, false);
         }
         else
         {
            // test mode is inactive => nothing to do
         }
         break;
      case App2BtsOC_StartTestMode:
         // switching to test mode ongoing => update status as false and result as failed
         createTestModeStatusMsg(bts2AppMsgList, 0, 0, _address, false);
         createStartTestModeResultMsg(bts2AppMsgList, _requestItem.user, _requestItem.handle, BTS_REQ_FAILED);
         break;
      case App2BtsOC_StopTestMode:
         // leaving test mode ongoing => update status as false and result as success
         createTestModeStatusMsg(bts2AppMsgList, 0, 0, _address, false);
         createStopTestModeResultMsg(bts2AppMsgList, _requestItem.user, _requestItem.handle, BTS_REQ_SUCCESS);
         break;
      case App2BtsOC_RequestTestModeLinkQuality:
         // getting link quality ongoing => update result as failed
         createLinkQualityResultMsg(bts2AppMsgList, _requestItem.user, _requestItem.handle, BTS_REQ_FAILED, defaultRssiValue, defaultLinkQualityValue);
         break;
      default:
         FW_NORMAL_ASSERT_ALWAYS();
         break;
   }

   // reset control data
   reset();
}

void TestMode::setInstance(IN ITestModeRequest* instance)
{
   _requestIf = instance;

   FW_IF_NULL_PTR_RETURN(_requestIf);

   _requestIf->setCallback(this);
}

void TestMode::setControlIf(IN IBasicControl* control)
{
   _controlIf = control;

   FW_NORMAL_ASSERT(0 != _controlIf);

   FW_IF_NULL_PTR_RETURN(_requestIf);

   _requestIf->setControlIf(_controlIf);
}

IStateMachine* TestMode::getSmEntryInterface(void)
{
   return this;
}

void TestMode::sendStatus(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN BtStackIfCallback* user, IN const BTSSessionHandle handle, IN const BTSCommonEnumClass statusCode) const
{
   (void)(statusCode);

   // send current status for local control data
   createTestModeStatusMsg(bts2AppMsgList, user, handle, _address, _active);
}

void TestMode::sendStatusAndStartResult(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const App2Bts_StartTestMode& request, IN const bool sendStatusToAll, IN const BTSCommonEnumClass resultCode, IN const BTSCommonEnumClass statusCode) const
{
   (void)(statusCode);

   // collect data for sending status and result
   BtStackIfCallback* sendUser;
   BTSSessionHandle sendHandle;
   BTSBDAddress sendAddress(request.getBDAddress());
   ::fw::convertString2LowerCase(sendAddress);
   bool sendActive;
   BTSRequestResult sendResult;
   const BTSRequestResult inputResult = (BTSRequestResult)resultCode;

   if((true == sendStatusToAll) || (0 == request.getUser()))
   {
      sendUser = 0;
      sendHandle = 0;
   }
   else
   {
      sendUser = request.getUser();
      sendHandle = request.getSessionHandle();
   }

   if(false == isValidStartRequest(request))
   {
      sendActive = false;
      sendResult = BTS_REQ_INVALID_PARAM;
   }
   else if(true == _active)
   {
      if(sendAddress == _address)
      {
         // test mode is already active for given address
         sendActive = true;
         sendResult = BTS_REQ_SUCCESS;
      }
      else
      {
         // test mode is active but for another address
         sendActive = false;
         sendResult = BTS_REQ_FAILED;
      }
   }
   else
   {
      // test mode is inactive
      sendActive = false;
      sendResult = BTS_REQ_FAILED;
   }

   // check given result
   if((BTS_REQ_LAST > inputResult) && (BTS_REQ_SUCCESS != inputResult) && (BTS_REQ_SUCCESS != sendResult))
   {
      // use given result
      sendResult = inputResult;
   }

   // send status and start result for given control data
   createTestModeStatusMsg(bts2AppMsgList, sendUser, sendHandle, sendAddress, sendActive);
   createStartTestModeResultMsg(bts2AppMsgList, request.getUser(), request.getSessionHandle(), sendResult);
}

void TestMode::sendStatusAndStopResult(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const App2Bts_StopTestMode& request, IN const bool sendStatusToAll, IN const BTSCommonEnumClass resultCode, IN const BTSCommonEnumClass statusCode) const
{
   (void)(statusCode);

   // collect data for sending status and result
   BtStackIfCallback* sendUser;
   BTSSessionHandle sendHandle;
   const BTSBDAddress& sendAddress = _address;
   bool sendActive;
   BTSRequestResult sendResult;
   const BTSRequestResult inputResult = (BTSRequestResult)resultCode;

   if((true == sendStatusToAll) || (0 == request.getUser()))
   {
      sendUser = 0;
      sendHandle = 0;
   }
   else
   {
      sendUser = request.getUser();
      sendHandle = request.getSessionHandle();
   }

   if(false == isValidStopRequest(request))
   {
      sendActive = _active;
      sendResult = BTS_REQ_INVALID_PARAM;
   }
   else if(true == _active)
   {
      // test mode is active
      sendActive = _active;
      sendResult = BTS_REQ_FAILED;
   }
   else
   {
      // test mode is inactive
      sendActive = _active;
      sendResult = BTS_REQ_SUCCESS;
   }

   // check given result
   if((BTS_REQ_LAST > inputResult) && (BTS_REQ_SUCCESS != inputResult) && (BTS_REQ_SUCCESS != sendResult))
   {
      // use given result
      sendResult = inputResult;
   }

   // send status and start result for given control data
   createTestModeStatusMsg(bts2AppMsgList, sendUser, sendHandle, sendAddress, sendActive);
   createStopTestModeResultMsg(bts2AppMsgList, request.getUser(), request.getSessionHandle(), sendResult);
}

void TestMode::sendLinkQualityResult(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const App2Bts_RequestTestModeLinkQuality& request, IN const BTSCommonEnumClass resultCode) const
{
   // collect data for sending result
   BTSRequestResult sendResult;
   const BTSRequestResult inputResult = (BTSRequestResult)resultCode;

   if(false == isValidLinkQualityRequest(request))
   {
      sendResult = BTS_REQ_INVALID_PARAM;
   }
   else
   {
      // in all other cases return failed
      sendResult = BTS_REQ_FAILED;
   }

   // check given result
   if((BTS_REQ_LAST > inputResult) && (BTS_REQ_SUCCESS != inputResult))
   {
      // use given result
      sendResult = inputResult;
   }

   // send result for given control data
   createLinkQualityResultMsg(bts2AppMsgList, request.getUser(), request.getSessionHandle(), sendResult, defaultRssiValue, defaultLinkQualityValue);
}

bool TestMode::isValidStartRequest(IN const App2Bts_StartTestMode& request) const
{
   return ::fw::isValidBdAddress(request.getBDAddress());
}

bool TestMode::isValidStopRequest(IN const App2Bts_StopTestMode& request) const
{
   // no parameter to be checked
   (void)(request);

   return true;
}

bool TestMode::isValidLinkQualityRequest(IN const App2Bts_RequestTestModeLinkQuality& request) const
{
   // no parameter to be checked
   (void)(request);

   return true;
}

bool TestMode::start(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const App2Bts_StartTestMode& request)
{
   FW_IF_NULL_PTR_RETURN_FALSE(_requestIf);

   BTSBDAddress workingAddress(request.getBDAddress());
   ::fw::convertString2LowerCase(workingAddress);

   if(false == isValidStartRequest(request))
   {
      createTestModeStatusMsg(bts2AppMsgList, request.getUser(), request.getSessionHandle(), workingAddress, false);
      createStartTestModeResultMsg(bts2AppMsgList, request.getUser(), request.getSessionHandle(), BTS_REQ_INVALID_PARAM);

      return false;
   }

   if(true == _active)
   {
      if(workingAddress == _address)
      {
         // test mode is already active for given address
         createTestModeStatusMsg(bts2AppMsgList, request.getUser(), request.getSessionHandle(), workingAddress, true);
         createStartTestModeResultMsg(bts2AppMsgList, request.getUser(), request.getSessionHandle(), BTS_REQ_SUCCESS);
      }
      else
      {
         // test mode is active but for another address
         createTestModeStatusMsg(bts2AppMsgList, request.getUser(), request.getSessionHandle(), workingAddress, false);
         createStartTestModeResultMsg(bts2AppMsgList, request.getUser(), request.getSessionHandle(), BTS_REQ_FAILED);
      }

      return false;
   }

   // store data and process request
   request.getCompareItem(_requestItem.item);
   _requestItem.user = request.getUser();
   _requestItem.handle = request.getSessionHandle();
   _address = workingAddress;
   _requestIf->start(bts2IpcMsgList, bts2AppMsgList, _address);

   return true;
}

bool TestMode::stop(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const App2Bts_StopTestMode& request)
{
   FW_IF_NULL_PTR_RETURN_FALSE(_requestIf);

   if(false == isValidStopRequest(request))
   {
      createTestModeStatusMsg(bts2AppMsgList, request.getUser(), request.getSessionHandle(), _address, _active);
      createStopTestModeResultMsg(bts2AppMsgList, request.getUser(), request.getSessionHandle(), BTS_REQ_INVALID_PARAM);

      return false;
   }

   if(false == _active)
   {
      // test mode is already stopped
      createTestModeStatusMsg(bts2AppMsgList, request.getUser(), request.getSessionHandle(), _address, _active);
      createStopTestModeResultMsg(bts2AppMsgList, request.getUser(), request.getSessionHandle(), BTS_REQ_SUCCESS);

      return false;
   }

   // store data and process request
   request.getCompareItem(_requestItem.item);
   _requestItem.user = request.getUser();
   _requestItem.handle = request.getSessionHandle();
   _requestIf->stop(bts2IpcMsgList, bts2AppMsgList, _address);

   return true;
}

bool TestMode::getLinkQuality(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const App2Bts_RequestTestModeLinkQuality& request)
{
   FW_IF_NULL_PTR_RETURN_FALSE(_requestIf);

   if(false == isValidLinkQualityRequest(request))
   {
      createLinkQualityResultMsg(bts2AppMsgList, request.getUser(), request.getSessionHandle(), BTS_REQ_INVALID_PARAM, defaultRssiValue, defaultLinkQualityValue);

      return false;
   }

   if(false == _active)
   {
      // test mode is inactive => get link quality makes no sense
      createLinkQualityResultMsg(bts2AppMsgList, request.getUser(), request.getSessionHandle(), BTS_REQ_FAILED, defaultRssiValue, defaultLinkQualityValue);

      return false;
   }

   // store data and process request
   request.getCompareItem(_requestItem.item);
   _requestItem.user = request.getUser();
   _requestItem.handle = request.getSessionHandle();
   _requestIf->getLinkQuality(bts2IpcMsgList, bts2AppMsgList);

   return true;
}

bool TestMode::isActive(void) const
{
   return _active;
}

bool TestMode::isActive(IN const BTSBDAddress& address) const
{
   BTSBDAddress workingAddress(address);
   ::fw::convertString2LowerCase(workingAddress);

   if((_address == workingAddress) && (true == _active))
   {
      return true;
   }
   else
   {
      return false;
   }
}

const BTSBDAddress& TestMode::getAddress(void) const
{
   return _address;
}

void TestMode::handleSetHciModeStatus(IN const bool hciMode)
{
   FW_IF_NULL_PTR_RETURN(_requestIf);

   _requestIf->handleSetHciModeStatus(hciMode);
}

void TestMode::handleSetHciModeResult(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const bool hciMode, IN const bool success)
{
   FW_IF_NULL_PTR_RETURN(_requestIf);

   // forward only if start test mode is ongoing
   if(App2BtsOC_StartTestMode == _requestItem.item.opCode)
   {
      _requestIf->handleSetHciModeResult(bts2IpcMsgList, bts2AppMsgList, messageItem, hciMode, success);
   }
}

void TestMode::startResult(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const bool success)
{
   (void)(bts2IpcMsgList);

   _active = success;

   BTSRequestResult sendResult(BTS_REQ_FAILED);

   if(true == _active)
   {
      sendResult = BTS_REQ_SUCCESS;
   }

   createTestModeStatusMsg(bts2AppMsgList, 0, 0, _address, _active);
   createStartTestModeResultMsg(bts2AppMsgList, _requestItem.user, _requestItem.handle, sendResult);

   // action is finished
   handleActionFinished(messageItem);

   // reset control data because action is finished
   _requestItem.reset();
   if(false == _active)
   {
      _address.clear();
   }
}

void TestMode::stopResult(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const bool success)
{
   (void)(bts2IpcMsgList);

   _active = (false == success);

   BTSRequestResult sendResult(BTS_REQ_FAILED);

   if(false == _active)
   {
      sendResult = BTS_REQ_SUCCESS;
   }

   createTestModeStatusMsg(bts2AppMsgList, 0, 0, _address, _active);
   createStopTestModeResultMsg(bts2AppMsgList, _requestItem.user, _requestItem.handle, sendResult);

   // action is finished
   handleActionFinished(messageItem);

   // reset control data because action is finished
   _requestItem.reset();
   if(false == _active)
   {
      _address.clear();
   }
}

void TestMode::status(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const bool active)
{
   (void)(bts2IpcMsgList);
   (void)(messageItem);

   _active = active;

   createTestModeStatusMsg(bts2AppMsgList, 0, 0, _address, _active);

   // reset control data
   if(false == _active)
   {
      _address.clear();
   }
}

void TestMode::getLinkQualityResult(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const bool success, IN const bool rssiSupported, IN const BTSRssi rssi, IN const bool linkQualitySupported, IN const BTSLinkQuality linkQuality)
{
   (void)(bts2IpcMsgList);

   BTSRequestResult sendResult(BTS_REQ_FAILED);
   BTSRssi sendRssi(defaultRssiValue);
   BTSLinkQuality sendLinkQuality(defaultLinkQualityValue);

   if(true == success)
   {
      sendResult = BTS_REQ_SUCCESS;
      if(true == rssiSupported)
      {
         sendRssi = rssi;
      }
      if(true == linkQualitySupported)
      {
         sendLinkQuality = linkQuality;
      }
   }

   createLinkQualityResultMsg(bts2AppMsgList, _requestItem.user, _requestItem.handle, sendResult, sendRssi, sendLinkQuality);

   // action is finished
   handleActionFinished(messageItem);

   // reset control data because action is finished
   _requestItem.reset();
}

App2Bts_BaseMessage* TestMode::getApp2BtsWorkingMessage(void)
{
   FW_IF_NULL_PTR_RETURN_NULL(_controlIf);

   return _controlIf->findApp2BtsWorkingMessageWrapper(_requestItem.item);
}

void TestMode::handleActionFinished(OUT BTSHandleIpc2BtsMessageItem& messageItem)
{
   messageItem.deleteMessage = true;
   if(0 == messageItem.message)
   {
      messageItem.message = getApp2BtsWorkingMessage();
   }
   FW_NORMAL_ASSERT(0 != messageItem.message);
}

void TestMode::createTestModeStatusMsg(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN BtStackIfCallback* user, IN const BTSSessionHandle handle, IN const BTSBDAddress& address, IN const bool active) const
{
   Bts2App_TestModeStatus* msg = ptrNew_Bts2App_TestModeStatus();
   if(0 != msg)
   {
      if(0 == user)
      {
         msg->setUser(0); // send status message to all
         msg->setSessionHandle(0); // send status message to all
      }
      else
      {
         msg->setUser(user);
         msg->setSessionHandle(handle);
      }
      msg->setBDAddress(address); // can be empty
      if(true == active)
      {
         msg->setConnectionStatus(BTS_CONN_CONNECTED);
      }
      else
      {
         msg->setConnectionStatus(BTS_CONN_DISCONNECTED);
      }

      bts2AppMsgList.push_back(msg);
   }
}

void TestMode::createStartTestModeResultMsg(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN BtStackIfCallback* user, IN const BTSSessionHandle handle, IN const BTSRequestResult result) const
{
   if(0 != user)
   {
      Bts2App_StartTestModeResult* msg = ptrNew_Bts2App_StartTestModeResult();
      if(0 != msg)
      {
         msg->setUser(user);
         msg->setSessionHandle(handle);
         msg->setRequestResult(result);

         bts2AppMsgList.push_back(msg);
      }
   }
}

void TestMode::createStopTestModeResultMsg(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN BtStackIfCallback* user, IN const BTSSessionHandle handle, IN const BTSRequestResult result) const
{
   if(0 != user)
   {
      Bts2App_StopTestModeResult* msg = ptrNew_Bts2App_StopTestModeResult();
      if(0 != msg)
      {
         msg->setUser(user);
         msg->setSessionHandle(handle);
         msg->setRequestResult(result);

         bts2AppMsgList.push_back(msg);
      }
   }
}

void TestMode::createLinkQualityResultMsg(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN BtStackIfCallback* user, IN const BTSSessionHandle handle, IN const BTSRequestResult result, IN const BTSRssi rssi, IN const BTSLinkQuality linkQuality) const
{
   if(0 != user)
   {
      Bts2App_TestModeLinkQualityResult* msg = ptrNew_Bts2App_TestModeLinkQualityResult();
      if(0 != msg)
      {
         msg->setUser(user);
         msg->setSessionHandle(handle);
         msg->setRequestResult(result);
         msg->setRssi(rssi);
         msg->setLinkQuality(linkQuality);

         bts2AppMsgList.push_back(msg);
      }
   }
}

} //btstackif
