/**
 * @file ITestModeRequest.h
 *
 * @par SW-Component
 * State machine for test mode
 *
 * @brief Request interface definition for test mode state machine.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details This file defines the request interface to be implemented by specific test mode state machine.
 */

#ifndef _I_TEST_MODE_REQUEST_H_
#define _I_TEST_MODE_REQUEST_H_

#include "BtStackInternalTypes.h"

namespace btstackif {

// class forward declarations
class ITestModeCallback;
class IBasicControl;
class Bts2Ipc_BaseMessage;
class Bts2App_BaseMessage;

/**
 * Interface definition.
 */
class ITestModeRequest
{
public:
   /**
    * Destructor.
    */
   virtual ~ITestModeRequest() {}

   /**
    * Reset internal data.
    */
   virtual void reset(void) = 0;

   /**
    * Set callback instance (implementation).
    *
    * @param[in] callback: callback instance (implementation)
    */
   virtual void setCallback(IN ITestModeCallback* callback) = 0;

   /**
    * Set basic control interface.
    *
    * @param[in] control: basic control interface
    */
   virtual void setControlIf(IN IBasicControl* control) = 0;

   /**
    * Start test mode.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[in] address: device address
    */
   virtual void start(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address) = 0;

   /**
    * Stop test mode.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[in] address: device address
    */
   virtual void stop(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address) = 0;

   /**
    * Get link quality.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    */
   virtual void getLinkQuality(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList) = 0;

   /**
    * Handle set HCI mode status. This interface shall be used if HCI is already active before starting test mode activities.
    *
    * @param[in] hciMode: HCI mode flag
    */
   virtual void handleSetHciModeStatus(IN const bool hciMode) = 0;

   /**
    * Handle set HCI mode result.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] hciMode: HCI mode flag
    * @param[in] success: success flag
    */
   virtual void handleSetHciModeResult(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const bool hciMode, IN const bool success) = 0;
};

} //btstackif

#endif //_I_TEST_MODE_REQUEST_H_
