/**
 * @file ITestMode.h
 *
 * @par SW-Component
 * State machine for test mode
 *
 * @brief Interface definition for generic test mode state machine.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details This file defines the generic interface for test mode state machine.
 */

#ifndef _I_TEST_MODE_H_
#define _I_TEST_MODE_H_

#include "BtStackInternalTypes.h"

namespace btstackif {

// class forward declarations
class ITestModeRequest;
class IBasicControl;
class IStateMachine;
class App2Bts_StartTestMode;
class App2Bts_StopTestMode;
class App2Bts_RequestTestModeLinkQuality;
class Bts2Ipc_BaseMessage;
class Bts2App_BaseMessage;

/**
 * Interface definition.
 */
class ITestMode
{
public:
   /**
    * Destructor.
    */
   virtual ~ITestMode() {}

   /**
    * Set specific instance (implementation).
    *
    * @param[in] instance: instance (implementation)
    */
   virtual void setInstance(IN ITestModeRequest* instance) = 0;

   /**
    * Set basic control interface.
    *
    * @param[in] control: basic control interface
    */
   virtual void setControlIf(IN IBasicControl* control) = 0;

   /**
    * Get SM entry interface.
    *
    * @return = SM entry interface
    */
   virtual IStateMachine* getSmEntryInterface(void) = 0;

   /**
    * Send current status for local control data.
    * <BR><B>This method shall be used to reject application requests or handle doubled application requests.</B>
    *
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[in] user: user
    * @param[in] handle: handle
    * @param[in] statusCode: status code
    */
   virtual void sendStatus(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN BtStackIfCallback* user, IN const BTSSessionHandle handle, IN const BTSCommonEnumClass statusCode) const = 0;

   /**
    * Send status and start result for given control data.
    * <BR><B>This method shall be used to reject application requests or handle doubled application requests.</B>
    *
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[in] request: request
    * @param[in] sendStatusToAll: flag for sending status to all or not
    * @param[in] resultCode: result code
    * @param[in] statusCode: status code
    */
   virtual void sendStatusAndStartResult(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const App2Bts_StartTestMode& request, IN const bool sendStatusToAll, IN const BTSCommonEnumClass resultCode, IN const BTSCommonEnumClass statusCode) const = 0;

   /**
    * Send status and stop result for given control data.
    * <BR><B>This method shall be used to reject application requests or handle doubled application requests.</B>
    *
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[in] request: request
    * @param[in] sendStatusToAll: flag for sending status to all or not
    * @param[in] resultCode: result code
    * @param[in] statusCode: status code
    */
   virtual void sendStatusAndStopResult(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const App2Bts_StopTestMode& request, IN const bool sendStatusToAll, IN const BTSCommonEnumClass resultCode, IN const BTSCommonEnumClass statusCode) const = 0;

   /**
    * Send link quality result.
    * <BR><B>This method shall be used to reject application requests or handle doubled application requests.</B>
    *
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[in] request: request
    * @param[in] resultCode: result code
    */
   virtual void sendLinkQualityResult(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const App2Bts_RequestTestModeLinkQuality& request, IN const BTSCommonEnumClass resultCode) const = 0;

   /**
    * Check if given request is valid.
    *
    * @param[in] request: request
    *
    * @return = true: valid request,
    * @return = false: invalid request
    */
   virtual bool isValidStartRequest(IN const App2Bts_StartTestMode& request) const = 0;

   /**
    * Check if given request is valid.
    *
    * @param[in] request: request
    *
    * @return = true: valid request,
    * @return = false: invalid request
    */
   virtual bool isValidStopRequest(IN const App2Bts_StopTestMode& request) const = 0;

   /**
    * Check if given request is valid.
    *
    * @param[in] request: request
    *
    * @return = true: valid request,
    * @return = false: invalid request
    */
   virtual bool isValidLinkQualityRequest(IN const App2Bts_RequestTestModeLinkQuality& request) const = 0;

   /**
    * Start test mode.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[in] request: request
    *
    * @return = true: request processing was started (request shall not be deleted),
    * @return = false: no further processing necessary (request can be deleted)
    */
   virtual bool start(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const App2Bts_StartTestMode& request) = 0;

   /**
    * Stop test mode.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[in] request: request
    *
    * @return = true: request processing was started (request shall not be deleted),
    * @return = false: no further processing necessary (request can be deleted)
    */
   virtual bool stop(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const App2Bts_StopTestMode& request) = 0;

   /**
    * Get link quality.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[in] request: request
    *
    * @return = true: request processing was started (request shall not be deleted),
    * @return = false: no further processing necessary (request can be deleted)
    */
   virtual bool getLinkQuality(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const App2Bts_RequestTestModeLinkQuality& request) = 0;

   /**
    * Checks if test mode is active.
    *
    * @return = true: test mode is active,
    * @return = false: test mode is inactive
    */
   virtual bool isActive(void) const = 0;

   /**
    * Checks if test mode is active for the given device address.
    *
    * @param[in] address: device address
    *
    * @return = true: test mode is active,
    * @return = false: test mode is inactive
    */
   virtual bool isActive(IN const BTSBDAddress& address) const = 0;

   /**
    * Returns the device address used for test mode.
    *
    * @return = device address
    */
   virtual const BTSBDAddress& getAddress(void) const = 0;

   /**
    * Handle set HCI mode status. This interface shall be used if HCI is already active before starting test mode activities.
    *
    * @param[in] hciMode: HCI mode flag
    */
   virtual void handleSetHciModeStatus(IN const bool hciMode) = 0;

   /**
    * Handle set HCI mode result.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] hciMode: HCI mode flag
    * @param[in] success: success flag
    */
   virtual void handleSetHciModeResult(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const bool hciMode, IN const bool success) = 0;
};

} //btstackif

#endif //_I_TEST_MODE_H_
