/**
 * @file SwitchBluetooth.cpp
 *
 * @par SW-Component
 * State machine for switch Bluetooth on/off
 *
 * @brief Implementation of generic switch Bluetooth on/off state machine.
 *
 * @copyright (C) 2018 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Source file for implementation of generic switch Bluetooth on/off state machine.
 */

#include "SwitchBluetooth.h"
#include "ISwitchBluetoothRequest.h"
#include "IBasicControl.h"
#include "IDeviceManager.h"
#include "IConfiguration.h"
#include "ITestMode.h"
#include "ILocalAdapterModes.h"
#include "ITimerPool.h"
#include "FwErrmemPrint.h"
#include "App2Bts_MessageWrapper.h"
#include "Bts2App_MessageWrapper.h"
#include "FwBoschRegistryWrapper.h"
#include "TraceClasses.h"
#include "FwTrace.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_BTS_CONTROL
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/SwitchBluetooth.cpp.trc.h"
#endif
#endif

namespace btstackif {

SwitchBluetooth::SwitchBluetooth() :
_requestIf(0),
_controlIf(0),
_deviceManagerIf(0),
_configurationIf(0),
_configurationMaster(0),
_testModeIf(0),
_localAdapterModesIf(0),
_timerPoolIf(0),
_requestItem(),
_newBtMode(BTS_BT_MODE_OFF),
_currentBtMode(BTS_BT_MODE_OFF),
_offReason(BTS_BT_OFF_REASON_STARTUP),
_newAdMode(BTS_ADAPTER_MODE_APP),
_currentAdMode(BTS_ADAPTER_MODE_APP),
_localAddress(),
_switchActive(false),
_stackPowered(false),
_stackHciMode(false),
_setActive(false),
_timerUpdate(),
_globalConfigActive(false),
_configRequestOngoing(false),
_configRequestDone(false),
_nmbRetry(0),
_maxRetry(1),
_app2BtsSwitchBtQueue(),
_regPath("/dev/registry/LOCAL_MACHINE/SOFTWARE/BLAUPUNKT/VERSIONS"),
_chipIdKey("Bluetooth_HW_Version"),
_buildIdKey("Bluetooth_HW_LoaderVersion"),
_fwVersionKey("Bluetooth_SW_Version"),
_vendorId(BTS_HCI_CHIP_VENDOR_UNKNOWN),
_moduleId(BTS_BT_MODULE_UNKNOWN)
{
   _app2BtsSwitchBtQueue.setWarningSize(128);
}

SwitchBluetooth::~SwitchBluetooth()
{
   _requestIf = 0;
   _controlIf = 0;
   _deviceManagerIf = 0;
   _configurationIf = 0;
   _configurationMaster = 0;
   _testModeIf = 0;
   _localAdapterModesIf = 0;
   _timerPoolIf = 0;
   _app2BtsSwitchBtQueue.empty();
}

void SwitchBluetooth::reset(void)
{
   StateMachine::reset();
   // keep _requestIf
   // keep _controlIf
   // keep _deviceManagerIf
   // keep _configurationIf
   // keep _configurationMaster
   // keep _testModeIf
   // keep _localAdapterModesIf
   // keep _timerPoolIf
   _requestItem.reset();
   _newBtMode = BTS_BT_MODE_OFF;
   _currentBtMode = BTS_BT_MODE_OFF;
   if((BTS_BT_OFF_REASON_STACK_NOT_RUNNING != _offReason) && (BTS_BT_OFF_REASON_DBUS_ERROR != _offReason))
   {
      _offReason = BTS_BT_OFF_REASON_STARTUP;
   }
   _newAdMode = BTS_ADAPTER_MODE_APP;
   _currentAdMode = BTS_ADAPTER_MODE_APP;
   // keep _localAddress
   _switchActive = false;
   _stackPowered = false;
   _stackHciMode = false;
   _setActive = false;
   stopTimer(_timerUpdate);
   _globalConfigActive = false;
   _configRequestOngoing = false;
   _configRequestDone = false;
   _nmbRetry = 0;
   rejectStoredRequests();
   // keep _vendorId
   // keep _moduleId

   FW_ERRMEM_IF_NULL_PTR_RETURN(_requestIf);
   _requestIf->reset();
}

void SwitchBluetooth::forceInitialState(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList)
{
   const bool updateStatus(BTS_BT_MODE_ON == getCurrentBtMode());

   // reset data before updating status/result
   _currentBtMode = BTS_BT_MODE_OFF;
   if((BTS_BT_OFF_REASON_STACK_NOT_RUNNING != _offReason) && (BTS_BT_OFF_REASON_DBUS_ERROR != _offReason))
   {
      _offReason = BTS_BT_OFF_REASON_STARTUP;
   }
   _currentAdMode = BTS_ADAPTER_MODE_APP;

   // check current state/action
   if(App2BtsOC_SwitchBluetoothOnOff == _requestItem.item.opCode)
   {
      // request ongoing => update status and result
      createStatusMsg(bts2AppMsgList, 0, 0, true);
      createResultMsg(bts2AppMsgList, _requestItem.user, _requestItem.handle, ((BTS_BT_MODE_ON == getNewBtMode()) ? BTS_REQ_FAILED : BTS_REQ_SUCCESS));
   }
   else
   {
      // no request ongoing
      if(true == updateStatus)
      {
         // BT is on => update status
         createStatusMsg(bts2AppMsgList, 0, 0, true);
      }
   }

   // reset control data
   reset();
}

void SwitchBluetooth::setInstance(IN ISwitchBluetoothRequest* instance)
{
   _requestIf = instance;

   FW_ERRMEM_IF_NULL_PTR_RETURN(_requestIf);

   _requestIf->setCallback(this);
}

void SwitchBluetooth::setControlIf(IN IBasicControl* control)
{
   _controlIf = control;

   FW_ERRMEM_ASSERT(0 != _controlIf);

   FW_ERRMEM_IF_NULL_PTR_RETURN(_requestIf);

   _requestIf->setControlIf(_controlIf);
}

void SwitchBluetooth::setDeviceManagerIf(IN IDeviceManager* deviceManagerIf)
{
   _deviceManagerIf = deviceManagerIf;

   FW_ERRMEM_ASSERT(0 != _deviceManagerIf);
}

void SwitchBluetooth::setConfigurationIf(IN IConfiguration* configurationIf)
{
   _configurationIf = configurationIf;

   FW_ERRMEM_ASSERT(0 != _configurationIf);
}

IConfigurationClient* SwitchBluetooth::getConfigurationClientHandler(void)
{
   return this;
}

void SwitchBluetooth::setTestModeIf(IN ITestMode* testModeIf)
{
   _testModeIf = testModeIf;

   FW_ERRMEM_ASSERT(0 != _testModeIf);
}

void SwitchBluetooth::setLocalAdapterModesIf(IN ILocalAdapterModes* localAdapterModes)
{
   _localAdapterModesIf = localAdapterModes;

   FW_ERRMEM_ASSERT(0 != _localAdapterModesIf);
}

void SwitchBluetooth::setTimerPoolIf(IN ITimerPool* timerPool)
{
   _timerPoolIf = timerPool;

   FW_ERRMEM_ASSERT(0 != _timerPoolIf);
}

IStateMachine* SwitchBluetooth::getSmEntryInterface(void)
{
   return this;
}

void SwitchBluetooth::sendStatus(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const App2Bts_GetBluetoothOnOffStatus& request, IN const BTSCommonEnumClass statusCode) const
{
   (void)(statusCode);

   if(false == isValidGetRequest(request))
   {
      FW_ERRMEM_ASSERT_ALWAYS();
      return;
   }

   createStatusMsg(bts2AppMsgList, request.getUser(), request.getSessionHandle(), false);
}

void SwitchBluetooth::sendStatusAndResult(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const App2Bts_SwitchBluetoothOnOff& request, IN const bool sendStatusToAll, IN const BTSCommonEnumClass resultCode, IN const BTSCommonEnumClass statusCode) const
{
   (void)(statusCode);

   // collect data for sending status and result
   BTSRequestResult sendResult;
   const BTSRequestResult inputResult = (BTSRequestResult)resultCode;

   if(false == isValidSetRequest(request))
   {
      FW_ERRMEM_ASSERT_ALWAYS();

      sendResult = BTS_REQ_INVALID_PARAM;
   }
   else if(BTS_BT_MODE_ON == request.getBluetoothMode())
   {
      sendResult = ((BTS_BT_MODE_ON == getCurrentBtMode()) ? BTS_REQ_SUCCESS : BTS_REQ_FAILED);
   }
   else
   {
      sendResult = ((BTS_BT_MODE_ON == getCurrentBtMode()) ? BTS_REQ_FAILED : BTS_REQ_SUCCESS);
   }

   // check given result
   if((BTS_REQ_LAST > inputResult) && (BTS_REQ_SUCCESS != inputResult) && (BTS_REQ_SUCCESS != sendResult))
   {
      // use given result
      sendResult = inputResult;
   }

   // update status
   createStatusMsg(bts2AppMsgList, request.getUser(), request.getSessionHandle(), sendStatusToAll);
   // update result
   createResultMsg(bts2AppMsgList, request.getUser(), request.getSessionHandle(), sendResult);
}

bool SwitchBluetooth::isValidGetRequest(IN const App2Bts_GetBluetoothOnOffStatus& request) const
{
   (void)(request);
   return true;
}

bool SwitchBluetooth::isValidSetRequest(IN const App2Bts_SwitchBluetoothOnOff& request) const
{
   (void)(request);
   return true;
}

bool SwitchBluetooth::switchBtOnOff(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const App2Bts_SwitchBluetoothOnOff& request)
{
   ETG_TRACE_USR2((" switchBtOnOff"));

   FW_ERRMEM_IF_NULL_PTR_RETURN_FALSE(_requestIf);
   FW_ERRMEM_IF_NULL_PTR_RETURN_FALSE(_deviceManagerIf);
   FW_ERRMEM_IF_NULL_PTR_RETURN_FALSE(_configurationIf);

   if(false == isValidSetRequest(request))
   {
      FW_ERRMEM_ASSERT_ALWAYS();

      createStatusMsg(bts2AppMsgList, request.getUser(), request.getSessionHandle(), false);
      createResultMsg(bts2AppMsgList, request.getUser(), request.getSessionHandle(), BTS_REQ_INVALID_PARAM);
      return false;
   }

   // check current adapter mode
   if(BTS_ADAPTER_MODE_HCI == getCurrentAdMode())
   {
      // do not sent any request to stack; just answer
      if(BTS_BT_MODE_ON == request.getBluetoothMode())
      {
         setCurrentBtMode(BTS_BT_MODE_ON);
         setBtOffReason(BTS_BT_OFF_REASON_NOT_VALID);
      }
      else
      {
         setCurrentBtMode(BTS_BT_MODE_OFF);
         setBtOffReason(BTS_BT_OFF_REASON_APP_REQUESTED);
      }

      createStatusMsg(bts2AppMsgList, request.getUser(), request.getSessionHandle(), false);
      createResultMsg(bts2AppMsgList, request.getUser(), request.getSessionHandle(), BTS_REQ_SUCCESS);
      return false;
   }

   // check current Bluetooth on/off status (ignore adapter mode)
   if((getCurrentBtMode() == request.getBluetoothMode()) /*&&
      (getCurrentAdMode() == request.getAdapterMode())*/)
   {
      // send answer directly because requested mode is already set
      createStatusMsg(bts2AppMsgList, request.getUser(), request.getSessionHandle(), false);
      createResultMsg(bts2AppMsgList, request.getUser(), request.getSessionHandle(), BTS_REQ_SUCCESS);
      return false;
   }

   // check for connected devices
   if(true == _deviceManagerIf->isAnyDeviceConnected()) // TODO: [low]: what about ongoing connect?
   {
      // reject request because switching BT on/off is not allowed while any BT device is connected
      // send answer directly because request was rejected
      createStatusMsg(bts2AppMsgList, request.getUser(), request.getSessionHandle(), false);
      createResultMsg(bts2AppMsgList, request.getUser(), request.getSessionHandle(), BTS_REQ_FAILED);
      return false;
   }

   // different mode requested
   // store data and process request
   request.getCompareItem(_requestItem.item);
   _requestItem.user = request.getUser();
   _requestItem.handle = request.getSessionHandle();
   resetRetry();

   // mark switch BT on/off as ongoing
   setSwitchActive(true);

   if(BTS_BT_MODE_ON == request.getBluetoothMode())
   {
      setNewBtMode(BTS_BT_MODE_ON);
      setNewAdMode(getCurrentAdMode()); // HCI mode switch is only done internally
      // start configuration sequence
      _configurationIf->startSequence(bts2IpcMsgList, bts2AppMsgList);
   }
   else // related to Evolution: always ON or OFF
   {
      setNewBtMode(BTS_BT_MODE_OFF);
      setNewAdMode(getCurrentAdMode()); // HCI mode switch is only done internally
      // send request
      sendPoweredMode(bts2IpcMsgList, bts2AppMsgList, false);
   }

   return true;
}

bool SwitchBluetooth::getAppBtMode(void) const
{
   return ((BTS_BT_MODE_ON == _currentBtMode) && (BTS_ADAPTER_MODE_APP == _currentAdMode));
}

bool SwitchBluetooth::getSwitchActive(void) const
{
   return _switchActive;
}

BTSAdapterMode SwitchBluetooth::getCurrentAdMode(void) const
{
   return _currentAdMode;
}

BTSBluetoothMode SwitchBluetooth::getNewBtMode(void) const
{
   return _newBtMode;
}

const BTSBDAddress& SwitchBluetooth::getLocalAddress(void) const
{
   return _localAddress;
}

void SwitchBluetooth::updateCurrentStatus(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList) const
{
   createStatusMsg(bts2AppMsgList, 0, 0, true);
}

void SwitchBluetooth::setCurrentBtMode(IN const BTSBluetoothMode mode)
{
   _currentBtMode = mode;
}

void SwitchBluetooth::setBtOffReason(IN const BTSBluetoothOffReason reason)
{
   _offReason = reason;
}

void SwitchBluetooth::sendGetHwVersionRequest(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList)
{
   FW_ERRMEM_IF_NULL_PTR_RETURN(_requestIf);

   _requestIf->getHwVersion(bts2IpcMsgList, bts2AppMsgList);
}

void SwitchBluetooth::readRegistryData(void)
{
   ::fw::BoschRegistryWrapper registry;
   ::std::string chipId;
   ::std::string buildId;
   ::std::string fwVersion;

   registry.open(_regPath);

   if(true == registry.read(chipId, _chipIdKey))
   {
      ETG_TRACE_USR3((" readRegistryData(): chipId=%s", chipId.c_str()));
   }
   else
   {
      ETG_TRACE_USR3((" readRegistryData(): reading chipId failed"));
   }

   if(true == registry.read(buildId, _buildIdKey))
   {
      ETG_TRACE_USR3((" readRegistryData(): buildId=%s", buildId.c_str()));
   }
   else
   {
      ETG_TRACE_USR3((" readRegistryData(): reading buildId failed"));
   }

   if(true == registry.read(fwVersion, _fwVersionKey))
   {
      ETG_TRACE_USR3((" readRegistryData(): fwVersion=%s", fwVersion.c_str()));
   }
   else
   {
      ETG_TRACE_USR3((" readRegistryData(): reading fwVersion failed"));
   }

   registry.close();
}

void SwitchBluetooth::pushToSwitchBtQueue(IN App2Bts_BaseMessage* message, bool withLock /*= true*/)
{
   FW_ERRMEM_IF_NULL_PTR_RETURN(message);

   _app2BtsSwitchBtQueue.push(message, withLock);
}

BTSHciChipVendorId SwitchBluetooth::getHciChipVendorId(void) const
{
   return _vendorId;
}

BTSBtModuleId SwitchBluetooth::getBtModuleId(void) const
{
   return _moduleId;
}

void SwitchBluetooth::setStackLocalBDAddress(IN const BTSBDAddress& address)
{
   _localAddress = address;
}

void SwitchBluetooth::setStackPoweredMode(IN const bool enable)
{
   _stackPowered = enable;
}

void SwitchBluetooth::setStackHciMode(IN const bool enable)
{
   FW_ERRMEM_IF_NULL_PTR_RETURN(_testModeIf);

   _stackHciMode = enable;

   if(true == _stackHciMode)
   {
      setCurrentAdMode(BTS_ADAPTER_MODE_HCI);
   }
   else
   {
      setCurrentAdMode(BTS_ADAPTER_MODE_APP);
   }

   _testModeIf->handleSetHciModeStatus(enable);
}

void SwitchBluetooth::reportVersionInfo(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem,
         IN const ::std::string& chipId, IN const ::std::string& buildId, IN const ::std::string& fwVersion, IN const BTSHciChipVendorId vendorId, IN const BTSBtModuleId moduleId)
{
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);
   (void)(messageItem);

   writeRegistryData(chipId, buildId, fwVersion);

   _vendorId = vendorId;
   _moduleId = moduleId;
}

void SwitchBluetooth::updateHciMode(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const bool hciMode)
{
   (void)(bts2IpcMsgList);
   (void)(messageItem);

   if(true == hciMode)
   {
      setCurrentAdMode(BTS_ADAPTER_MODE_HCI);
   }
   else
   {
      setCurrentAdMode(BTS_ADAPTER_MODE_APP);
   }

   createStatusMsg(bts2AppMsgList, 0, 0, true);
}

void SwitchBluetooth::updatePowered(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const bool powered, IN const BTSIpcCommonErrorCode errorCode, IN const bool force /*= false*/)
{
   (void)(force);

   FW_ERRMEM_IF_NULL_PTR_RETURN(_configurationIf);
   FW_ERRMEM_IF_NULL_PTR_RETURN(_configurationMaster);

   // check if request is ongoing
   if(true == getRequestActive())
   {
      // request was sent before

      // reset flag and stop timer
      setRequestActive(false);

      // remember current mode
      const BTSBluetoothMode oldBtMode(getCurrentBtMode());
      bool continueWithConfiguration(false);

      if(BTS_IPC_SUCCESS == errorCode)
      {
         if(BTS_BT_MODE_ON == getNewBtMode())
         {
            // on was requested
            if(true == powered)
            {
               setCurrentBtMode(BTS_BT_MODE_ON);
               setCurrentAdMode(getNewAdMode());

               continueWithConfiguration = true;
            }
            else
            {
               setCurrentBtMode(BTS_BT_MODE_OFF);
               // keep current adapter mode
            }
         }
         else
         {
            // off was requested
            if(false == powered)
            {
               setCurrentBtMode(BTS_BT_MODE_OFF);
               setCurrentAdMode(getNewAdMode());
            }
            else
            {
               setCurrentBtMode(BTS_BT_MODE_ON);
               // keep current adapter mode
            }
         }

         // store new value
         setStackPoweredMode(powered);
      }
      else
      {
         // keep current values
      }

      if(true == continueWithConfiguration)
      {
         // successfully switched on
         setBtOffReason(BTS_BT_OFF_REASON_NOT_VALID);
         // mark as not ongoing
         _configRequestOngoing = false;
         // mark as finished
         _configRequestDone = true;
         // handle updated BT status
         createStatusMsg(bts2AppMsgList, 0, 0, true);

#ifdef ENABLE_NEW_CONFIGURATION_HANDLING
         // inform configuration master to execute next step
         (void)_configurationMaster->doNextConfigurationStep(bts2IpcMsgList, bts2AppMsgList, messageItem);
#endif
      }
      else
      {
         // switching on/off failed or successfully switched off
         // check error code, if UPDATE_TIMEOUT then do 1 retry
         bool doRetry(false);
         if(BTS_IPC_UPDATE_TIMEOUT == errorCode)
         {
            doRetry = checkRetry();
         }

         if(true == doRetry)
         {
            // send request
            sendPoweredMode(bts2IpcMsgList, bts2AppMsgList, (BTS_BT_MODE_ON == getNewBtMode()));
         }
         else
         {
            // sequence finished
            if(BTS_BT_MODE_OFF == getCurrentBtMode())
            {
               setBtOffReason(BTS_BT_OFF_REASON_APP_REQUESTED);
            }
            else
            {
               setBtOffReason(BTS_BT_OFF_REASON_NOT_VALID);
            }
            // remember configuration request ongoing flag
            const bool tmpConfigRequestOngoing(_configRequestOngoing);
            // mark as not ongoing
            _configRequestOngoing = false;
            // mark as finished
            _configRequestDone = true;
            // handle updated BT status
            createStatusMsg(bts2AppMsgList, 0, 0, true);

            if(true == tmpConfigRequestOngoing)
            {
               // set error flag
               _configurationIf->setError(true);

#ifdef ENABLE_NEW_CONFIGURATION_HANDLING
               // inform configuration master to execute next step
               (void)_configurationMaster->doNextConfigurationStep(bts2IpcMsgList, bts2AppMsgList, messageItem);
#endif
            }
            else
            {
               handleEndOfSwitchSequence(bts2IpcMsgList, bts2AppMsgList, messageItem, false);
            }
         }
      }

      // in case of Bluetooth was switched off and discoverable / pairable / connectable mode was enabled before switching off Bluetooth stack will automatically disable discoverable / pairable / connectable mode
      if(BTS_IPC_SUCCESS == errorCode)
      {
         if((oldBtMode != getCurrentBtMode()) && (BTS_BT_MODE_OFF == getCurrentBtMode()))
         {
            forceLocalModesUpdate(bts2IpcMsgList, bts2AppMsgList, messageItem);
         }
      }
   }
   else
   {
      // handle spontaneous update (it could be possible that it takes more than 3s to switch BT on or off)
      if(BTS_IPC_SUCCESS == errorCode)
      {
         const BTSBluetoothMode oldBtMode(getCurrentBtMode());

         // set current BT mode as well as BT off reason
         if(true == powered)
         {
            setCurrentBtMode(BTS_BT_MODE_ON);
            setBtOffReason(BTS_BT_OFF_REASON_NOT_VALID);
         }
         else
         {
            setCurrentBtMode(BTS_BT_MODE_OFF);
            setBtOffReason(BTS_BT_OFF_REASON_APP_REQUESTED);
         }

         // keep current adapter mode

         // store new value
         setStackPoweredMode(powered);

         // update status
         createStatusMsg(bts2AppMsgList, 0, 0, true);

         // in case of Bluetooth was switched off and discoverable / pairable / connectable mode was enabled before switching off Evolution stack will automatically disable discoverable / pairable / connectable mode
         if((oldBtMode != getCurrentBtMode()) && (BTS_BT_MODE_OFF == getCurrentBtMode()))
         {
            forceLocalModesUpdate(bts2IpcMsgList, bts2AppMsgList, messageItem);
         }
      }
   }
}

void SwitchBluetooth::handleExtendedTimeout(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSTimerId timerId)
{
   (void)(messageItem);

   FW_ERRMEM_IF_NULL_PTR_RETURN(_requestIf);

   // find related timer
   if(true == _timerUpdate.compare(timerId))
   {
      _requestIf->sendVirtualPoweredUpdate(bts2IpcMsgList, bts2AppMsgList, false, BTS_IPC_UPDATE_TIMEOUT);
   }
}

void SwitchBluetooth::setConfigurationMasterIf(IN IConfigurationMaster* master)
{
   _configurationMaster = master;

   FW_ERRMEM_ASSERT(0 != _configurationMaster);
}

void SwitchBluetooth::startGlobalConfiguration(void)
{
   _globalConfigActive = true;
   _configRequestOngoing = false;
   _configRequestDone = false; // because this configuration shall be done during each configuration sequence
}

void SwitchBluetooth::stopGlobalConfiguration(void)
{
   _globalConfigActive = false;
   _configRequestOngoing = false;
}

bool SwitchBluetooth::isSingleConfigurationOngoing(void) const
{
   if(false == _globalConfigActive)
   {
      return false;
   }

   // check if call to switch BT is ongoing
   return _configRequestOngoing;
}

bool SwitchBluetooth::setSingleConfiguration(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const bool beforeBtOn, IN const bool errorOccurred)
{
   (void)(errorOccurred);

   FW_ERRMEM_IF_NULL_PTR_RETURN_FALSE(_requestIf);
   FW_ERRMEM_IF_NULL_PTR_RETURN_FALSE(_configurationIf);

   if(false == _globalConfigActive)
   {
      return false;
   }

   // ETG_TRACE_USR3((" SwitchBluetooth::setSingleConfiguration(): beforeBtOn=%d _configRequestOngoing=%d errorOccurred=%d _configRequestDone=%d", beforeBtOn, _configRequestOngoing, errorOccurred, _configRequestDone));

   if(true == _configRequestOngoing)
   {
      // configuration request is ongoing => wait for result
      return true;
   }

   /*
    * (true == beforeBtOn):
    *    - switch BT on
    *    - update only BT status
    *    - _configRequestOngoing indicates ongoing switch; will be reset after switch is done
    *    - ignore errorOccurred flag, continue with sequence
    * (else):
    *    - update switch BT result
    *    - _configRequestOngoing is not modified because no further communication with Bluetooth stack necessary
    *    - ignore errorOccurred flag, continue with sequence
    */

   if(true == beforeBtOn)
   {
      if(false == _configRequestDone)
      {
         // switch BT => send request
         sendPoweredMode(bts2IpcMsgList, bts2AppMsgList, true);

         // mark as ongoing
         _configRequestOngoing = true;

         return true;
      }

      return false;
   }
   else
   {
      // send switch result
      handleEndOfSwitchSequence(bts2IpcMsgList, bts2AppMsgList, messageItem, true);

      return false;
   }
}

void SwitchBluetooth::handleEndOfSwitchSequence(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const bool switchOn)
{
   (void)(bts2IpcMsgList);

   FW_ERRMEM_IF_NULL_PTR_RETURN(_controlIf);

   // stop timers
   stopTimer(_timerUpdate);

   // BT switch on and configuration sequence is finished => update result
   BTSRequestResult result(BTS_REQ_FAILED);

   if(true == switchOn)
   {
      result = ((BTS_BT_MODE_ON == getCurrentBtMode()) ? BTS_REQ_SUCCESS : BTS_REQ_FAILED);
   }
   else
   {
      result = ((BTS_BT_MODE_ON == getCurrentBtMode()) ? BTS_REQ_FAILED : BTS_REQ_SUCCESS);
   }

   createResultMsg(bts2AppMsgList, _requestItem.user, _requestItem.handle, result);

   // user request is ongoing => action is finished
   handleActionFinished(messageItem);

   // reset control data because action is finished
   _requestItem.reset();

   // mark switch BT on/off as inactive
   setSwitchActive(false);

   // process queued messages (push again to App2Bts main queue)
   for(MessageQueue< App2Bts_BaseMessage >::It it = _app2BtsSwitchBtQueue.getBegin(); it != _app2BtsSwitchBtQueue.getEnd(); ++it)
   {
      _controlIf->sendInternalApp2BtsMessage(*it);
   }
   _app2BtsSwitchBtQueue.clear(false); // single worker thread

   // TODO: [low]: inform observers
}

void SwitchBluetooth::handleActionFinished(OUT BTSHandleIpc2BtsMessageItem& messageItem)
{
   messageItem.item = _requestItem.item;
   messageItem.item.opCode = App2BtsOC_None; // to force default handling for end of request sequence
   messageItem.deleteMessage = true;
   findApp2BtsWorkingMessage(messageItem);
}

void SwitchBluetooth::findApp2BtsWorkingMessage(OUT BTSHandleIpc2BtsMessageItem& messageItem)
{
   if(0 == messageItem.message)
   {
      messageItem.message = getApp2BtsWorkingMessage();
   }
   FW_ERRMEM_ASSERT(0 != messageItem.message);
}

App2Bts_BaseMessage* SwitchBluetooth::getApp2BtsWorkingMessage(void)
{
   FW_ERRMEM_IF_NULL_PTR_RETURN_NULL(_controlIf);

   return _controlIf->findApp2BtsWorkingMessageWrapper(_requestItem.item);
}

void SwitchBluetooth::rejectStoredRequests(void)
{
   FW_ERRMEM_IF_NULL_PTR_RETURN(_controlIf);

   // reject all requests stored in startup queue
   ::std::vector< App2Bts_BaseMessage* > msgList;
   msgList.reserve(_app2BtsSwitchBtQueue.getSize());

   for(MessageQueue< App2Bts_BaseMessage >::It it = _app2BtsSwitchBtQueue.getBegin(); it != _app2BtsSwitchBtQueue.getEnd(); ++it)
   {
      msgList.push_back(*it);
   }

   _controlIf->sendDirectAnswerForApp2BtsMessages(msgList, (BTSCommonEnumClass)BTS_REQ_DBUS_ERROR, BTS_COMMON_ENUM_CLASS_DEFAULT_VALUE);

   _app2BtsSwitchBtQueue.empty(false); // single worker thread
}

void SwitchBluetooth::writeRegistryData(IN const ::std::string& chipId, IN const ::std::string& buildId, IN const ::std::string& fwVersion)
{
   ::fw::BoschRegistryWrapper registry;

   registry.open(_regPath);

   if(0 < chipId.length())
   {
      registry.write(_chipIdKey, chipId);
   }

   if(0 < buildId.length())
   {
      registry.write(_buildIdKey, buildId);
   }

   if(0 < fwVersion.length())
   {
      registry.write(_fwVersionKey, fwVersion);
   }

   registry.close();
}

void SwitchBluetooth::sendPoweredMode(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const bool enable)
{
   FW_ERRMEM_IF_NULL_PTR_RETURN(_requestIf);

   if(_stackPowered != enable)
   {
      _requestIf->setPowered(bts2IpcMsgList, bts2AppMsgList, enable);
   }
   else
   {
      // we will send a virtual Ipc2Bts message to follow normal sequence
      _requestIf->sendVirtualPoweredUpdate(bts2IpcMsgList, bts2AppMsgList, enable, BTS_IPC_SUCCESS);
   }

   setRequestActive(true);
}

void SwitchBluetooth::setRequestActive(IN const bool enable)
{
   _setActive = enable;

   // start/stop timer
   if(true == enable)
   {
      startTimer(_timerUpdate, TIMEOUT_UPDATE_ADAPTER_POWERED);
   }
   else
   {
      stopTimer(_timerUpdate);
   }
}

bool SwitchBluetooth::getRequestActive(void) const
{
   return _setActive;
}

bool SwitchBluetooth::checkRetry(void)
{
   if(_maxRetry > _nmbRetry)
   {
      _nmbRetry++;
      return true;
   }
   else
   {
      return false;
   }
}

void SwitchBluetooth::resetRetry(void)
{
   _nmbRetry = 0;
}

BTSBluetoothMode SwitchBluetooth::getCurrentBtMode(void) const
{
   return _currentBtMode;
}

void SwitchBluetooth::setCurrentAdMode(IN const BTSAdapterMode mode)
{
   _currentAdMode = mode;
}

void SwitchBluetooth::setNewBtMode(IN const BTSBluetoothMode mode)
{
   _newBtMode = mode;
}

void SwitchBluetooth::setNewAdMode(IN const BTSAdapterMode mode)
{
   _newAdMode = mode;
}

BTSAdapterMode SwitchBluetooth::getNewAdMode(void) const
{
   return _newAdMode;
}

void SwitchBluetooth::setSwitchActive(IN const bool enable)
{
   _switchActive = enable;
}

void SwitchBluetooth::forceLocalModesUpdate(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem)
{
   FW_ERRMEM_IF_NULL_PTR_RETURN(_localAdapterModesIf);

   _localAdapterModesIf->forceLocalModesUpdate(bts2IpcMsgList, bts2AppMsgList, messageItem);
}

void SwitchBluetooth::startTimer(IN ExtendedTimerEntry& timer, IN const BTSTimeValue timeout)
{
   ETG_TRACE_USR3((" startTimer: timeout=%u", timeout));

   FW_ERRMEM_IF_NULL_PTR_RETURN(_timerPoolIf);
   FW_ERRMEM_IF_NULL_PTR_RETURN(_controlIf);

   timer.setTimerPool(_timerPoolIf);
   timer.start(_controlIf, this, timeout);
}

void SwitchBluetooth::stopTimer(IN ExtendedTimerEntry& timer) const
{
   ETG_TRACE_USR3((" stopTimer"));

   timer.stop();

   // do not release timer
}

void SwitchBluetooth::releaseTimer(IN ExtendedTimerEntry& timer) const
{
   ETG_TRACE_USR3((" releaseTimer"));

   timer.release();
}

bool SwitchBluetooth::isTimerActive(IN const ExtendedTimerEntry& timer) const
{
   return timer.isActive();
}

void SwitchBluetooth::createStatusMsg(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN BtStackIfCallback* user, IN const BTSSessionHandle handle, IN const bool sendStatusToAll) const
{
   Bts2App_CurrentBluetoothOnOffStatus* msg = ptrNew_Bts2App_CurrentBluetoothOnOffStatus();
   if(0 != msg)
   {
      if((true == sendStatusToAll) || (0 == user))
      {
         msg->setUser(0); // send status message to all
         msg->setSessionHandle(0); // send status message to all
      }
      else
      {
         msg->setUser(user);
         msg->setSessionHandle(handle);
      }
      msg->setBluetoothMode(_currentBtMode);
      msg->setBluetoothOffReason(_offReason);
      msg->setAdapterMode(_currentAdMode);
      msg->setBDAddress(_localAddress); // can be empty

      bts2AppMsgList.push_back(msg);
   }
}

void SwitchBluetooth::createResultMsg(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN BtStackIfCallback* user, IN const BTSSessionHandle handle, IN const BTSRequestResult result) const
{
   if(0 != user)
   {
      Bts2App_SwitchBluetoothOnOffResult* msg = ptrNew_Bts2App_SwitchBluetoothOnOffResult();
      if(0 != msg)
      {
         msg->setUser(user);
         msg->setSessionHandle(handle);
         msg->setRequestResult(result);

         bts2AppMsgList.push_back(msg);
      }
   }
}

} //btstackif
