/**
 * @file ISwitchBluetoothCallback.h
 *
 * @par SW-Component
 * State machine for switch Bluetooth on/off
 *
 * @brief Callback interface definition for switch Bluetooth on/off state machine.
 *
 * @copyright (C) 2018 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details This file defines the callback interface to be used by specific switch Bluetooth on/off state machine.
 *          Callback interface must be provided by generic switch Bluetooth on/off state machine.
 */

#ifndef _I_SWITCH_BLUETOOTH_CALLBACK_H_
#define _I_SWITCH_BLUETOOTH_CALLBACK_H_

#include "BtStackInternalTypes.h"

namespace btstackif {

// class forward declarations
class Bts2Ipc_BaseMessage;
class Bts2App_BaseMessage;

/**
 * Interface definition.
 */
class ISwitchBluetoothCallback
{
public:
   /**
    * Destructor.
    */
   virtual ~ISwitchBluetoothCallback() {}

   /**
    * Set current stack value for local BD address.
    *
    * @param[in] address: address
    */
   virtual void setStackLocalBDAddress(IN const BTSBDAddress& address) = 0;

   /**
    * Set current stack value for powered mode.
    *
    * @param[in] enable: powered mode
    */
   virtual void setStackPoweredMode(IN const bool enable) = 0;

   /**
    * Set current stack value for HCI mode.
    *
    * @param[in] enable: HCI mode
    */
   virtual void setStackHciMode(IN const bool enable) = 0;

   /**
    * Report version information.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] chipId: chip ID
    * @param[in] buildId: build ID
    * @param[in] fwVersion: FW version
    * @param[in] vendorId: vendor ID
    * @param[in] moduleId: module ID
    */
   virtual void reportVersionInfo(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem,
            IN const ::std::string& chipId, IN const ::std::string& buildId, IN const ::std::string& fwVersion, IN const BTSHciChipVendorId vendorId, IN const BTSBtModuleId moduleId) = 0;

   /**
    * Update HCI mode.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] hciMode: HCI mode
    */
   virtual void updateHciMode(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const bool hciMode) = 0;

   /**
    * Update powered mode.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] powered: powered mode
    * @param[in] errorCode: error code
    * @param[in] force: force flag
    */
   virtual void updatePowered(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const bool powered, IN const BTSIpcCommonErrorCode errorCode, IN const bool force = false) = 0;
};

} //btstackif

#endif //_I_SWITCH_BLUETOOTH_CALLBACK_H_
