/**
 * @file ISwitchBluetooth.h
 *
 * @par SW-Component
 * State machine for switch Bluetooth on/off
 *
 * @brief Interface definition for generic switch Bluetooth on/off state machine.
 *
 * @copyright (C) 2017 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details This file defines the generic interface for switch Bluetooth on/off state machine.
 */

#ifndef _I_SWITCH_BLUETOOTH_H_
#define _I_SWITCH_BLUETOOTH_H_

#include "BtStackInternalTypes.h"

namespace btstackif {

// class forward declarations
class ISwitchBluetoothRequest;
class IBasicControl;
class IDeviceManager;
class IConfiguration;
class IConfigurationClient;
class ITestMode;
class ILocalAdapterModes;
class ITimerPool;
class IStateMachine;
class Bts2Ipc_BaseMessage;
class Bts2App_BaseMessage;
class App2Bts_SwitchBluetoothOnOff;
class App2Bts_GetBluetoothOnOffStatus;

/**
 * Interface definition.
 */
class ISwitchBluetooth
{
public:
   /**
    * Destructor.
    */
   virtual ~ISwitchBluetooth() {}

   /**
    * Set specific instance (implementation).
    *
    * @param[in] instance: instance (implementation)
    */
   virtual void setInstance(IN ISwitchBluetoothRequest* instance) = 0;

   /**
    * Set basic control interface.
    *
    * @param[in] control: basic control interface
    */
   virtual void setControlIf(IN IBasicControl* control) = 0;

   /**
    * Set device manager interface.
    *
    * @param[in] deviceManagerIf: device manager interface
    */
   virtual void setDeviceManagerIf(IN IDeviceManager* deviceManagerIf) = 0;

   /**
    * Set configuration interface.
    *
    * @param[in] configurationIf: configuration interface
    */
   virtual void setConfigurationIf(IN IConfiguration* configurationIf) = 0;

   /**
    * Get configuration client handler.
    *
    * @return = configuration client handler
    */
   virtual IConfigurationClient* getConfigurationClientHandler(void) = 0;

   /**
    * Set test mode interface.
    *
    * @param[in] testModeIf: test mode interface
    */
   virtual void setTestModeIf(IN ITestMode* testModeIf) = 0;

   /**
    * Set local adapter modes interface.
    *
    * @param[in] localAdapterModes: local adapter modes interface
    */
   virtual void setLocalAdapterModesIf(IN ILocalAdapterModes* localAdapterModes) = 0;

   /**
    * Set timer pool interface.
    *
    * @param[in] timerPool: timer pool interface
    */
   virtual void setTimerPoolIf(IN ITimerPool* timerPool) = 0;

   /**
    * Get SM entry interface.
    *
    * @return = SM entry interface
    */
   virtual IStateMachine* getSmEntryInterface(void) = 0;

   /**
    * Send current status for given control data.
    * <BR><B>This method shall be used to reject application requests or handle doubled application requests.</B>
    *
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[in] request: request
    * @param[in] statusCode: status code
    */
   virtual void sendStatus(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const App2Bts_GetBluetoothOnOffStatus& request, IN const BTSCommonEnumClass statusCode) const = 0;

   /**
    * Send status and result for given control data.
    * <BR><B>This method shall be used to reject application requests or handle doubled application requests.</B>
    *
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[in] request: request
    * @param[in] sendStatusToAll: flag for sending status to all or not
    * @param[in] resultCode: result code
    * @param[in] statusCode: status code
    */
   virtual void sendStatusAndResult(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const App2Bts_SwitchBluetoothOnOff& request, IN const bool sendStatusToAll, IN const BTSCommonEnumClass resultCode, IN const BTSCommonEnumClass statusCode) const = 0;

   /**
    * Check if given request is valid.
    *
    * @param[in] request: request
    *
    * @return = true: valid request,
    * @return = false: invalid request
    */
   virtual bool isValidGetRequest(IN const App2Bts_GetBluetoothOnOffStatus& request) const = 0;

   /**
    * Check if given request is valid.
    *
    * @param[in] request: request
    *
    * @return = true: valid request,
    * @return = false: invalid request
    */
   virtual bool isValidSetRequest(IN const App2Bts_SwitchBluetoothOnOff& request) const = 0;

   /**
    * Switch BT on/off.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[in] request: request
    *
    * @return = true: request processing was started (request shall not be deleted),
    * @return = false: no further processing necessary (request can be deleted)
    */
   virtual bool switchBtOnOff(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const App2Bts_SwitchBluetoothOnOff& request) = 0;

   /**
    * Get application Bluetooth mode (mode from application point of view => is BT on from user point of view).
    *
    * @return = application Bluetooth mode
    */
   virtual bool getAppBtMode(void) const = 0;

   /**
    * Get switch active flag (flag indicating that switching BT on or off is ongoing).
    *
    * @return = switch active flag
    */
   virtual bool getSwitchActive(void) const = 0;

   /**
    * Get current adapter mode.
    *
    * @return = adapter mode
    */
   virtual BTSAdapterMode getCurrentAdMode(void) const = 0;

   /**
    * Get new BT mode (requested mode).
    *
    * @return = BT mode
    */
   virtual BTSBluetoothMode getNewBtMode(void) const = 0;

   /**
    * Get local BT address.
    *
    * @return = local BT address
    */
   virtual const BTSBDAddress& getLocalAddress(void) const = 0;

   /**
    * Update current status.
    *
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    */
   virtual void updateCurrentStatus(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList) const = 0;

   /**
    * Set current BT mode.
    *
    * @param[in] mode: BT mode
    */
   virtual void setCurrentBtMode(IN const BTSBluetoothMode mode) = 0;

   /**
    * Set BT off reason.
    *
    * @param[in] reason: BT off reason
    */
   virtual void setBtOffReason(IN const BTSBluetoothOffReason reason) = 0;

   /**
    * Send request to get HW version information.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    */
   virtual void sendGetHwVersionRequest(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList) = 0;

   /**
    * Read registry data.
    */
   virtual void readRegistryData(void) = 0;

   /**
    * Push message to switch BT queue.
    *
    * @param[in] message: message
    * @param[in] withLock: flag for locking
    */
   virtual void pushToSwitchBtQueue(IN App2Bts_BaseMessage* message, bool withLock = true) = 0;

   /**
    * Get HCI chip vendor ID.
    *
    * @return = HCI chip vendor ID
    */
   virtual BTSHciChipVendorId getHciChipVendorId(void) const = 0;

   /**
    * Get BT module ID.
    *
    * @return = BT module ID
    */
   virtual BTSBtModuleId getBtModuleId(void) const = 0;
};

} //btstackif

#endif //_I_SWITCH_BLUETOOTH_H_
