/**
 * @file Startup.h
 *
 * @par SW-Component
 * State machine for startup
 *
 * @brief Implementation of generic startup state machine.
 *
 * @copyright (C) 2018 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Header file for implementation of generic startup state machine.
 */

#ifndef _STARTUP_H_
#define _STARTUP_H_

#include "StateMachine.h"
#include "IStartup.h"
#include "IStartupCallback.h"
#include "IExtendedTimeoutHandler.h"
#include "BtsMessageQueue.h"
#include "ExtendedTimerEntry.h"

namespace btstackif {

/**
 * Startup class.
 */
class Startup : public StateMachine, public IStartup, public IStartupCallback, public IExtendedTimeoutHandler
{
public:
   /**
    * Default constructor.
    */
   Startup();

   /**
    * Destructor.
    */
   virtual ~Startup();

   virtual void reset(void);

   virtual void forceInitialState(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList);

   virtual void setInstance(IN IStartupRequest* instance);

   virtual void setControlIf(IN IBasicControl* control);

   virtual void setConfigurationIf(IN IConfiguration* configurationIf);

   virtual void setSwitchBluetoothIf(IN ISwitchBluetooth* switchBluetooth);

   virtual void setLocalAdapterModesIf(IN ILocalAdapterModes* localAdapterModes);

   virtual void setTimerPoolIf(IN ITimerPool* timerPool);

   virtual IStateMachine* getSmEntryInterface(void);

   virtual void handleTriggerInitializedCallback(void);

   virtual bool handleFbConnectionInitialized(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList);

   virtual BTSDbusServiceAvailability getServiceAvailability(void) const;

   virtual bool getStartupOngoing(void) const;

   virtual void pushToStartupQueue(IN App2Bts_BaseMessage* message, bool withLock = true);

   virtual void registerSmEntry(IN IStateMachine* entry);

   virtual void startupSequenceStarted(IN const BTSDbusServiceAvailability availability);

   virtual void indicateAvailability(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSDbusServiceAvailability availability, IN const BTSBluetoothOffReason reason = BTS_BT_OFF_REASON_NOT_VALID);

   virtual bool getStartupOngoingFlag(void) const;

   virtual void handleExtendedTimeout(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSTimerId timerId);

private:
   IStartupRequest* _requestIf; /**< request interface */
   IBasicControl* _controlIf; /**< basic control interface */
   IConfiguration* _configurationIf; /**< configuration interface */
   ISwitchBluetooth* _switchBluetoothIf; /**< switch Bluetooth interface */
   ILocalAdapterModes* _localAdapterModesIf; /**< local adapter modes interface */
   ITimerPool* _timerPoolIf; /**< timer pool interface */
   bool _startupOngoing; /**< flag for ongoing startup */
   BTSDbusServiceAvailability _serviceAvailability; /**< service availability */
   BTSRequestResult _initStatus; /**< initialization result/status */
   MessageQueue< App2Bts_BaseMessage > _app2BtsStartupQueue; /**< startup queue */
   ::std::vector< IStateMachine* > _smList; /**< SM list */
   ExtendedTimerEntry _startupTimer; /**< startup timer */
   const BTSTimeValue _startupTimeout; /**< startup timeout */
   unsigned int _startupCounter; /**< startup counter */

   void updateInitializedStatus(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList);

   void setInitializedStatus(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSRequestResult status, IN const bool update = true);

   void processStartupQueue(void);

   void rejectStoredRequests(void);

   void handleStackReset(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem);

   void startTimer(IN ExtendedTimerEntry& timer, IN const BTSTimeValue timeout);

   void stopTimer(IN ExtendedTimerEntry& timer) const;

   void releaseTimer(IN ExtendedTimerEntry& timer) const;

   bool isTimerActive(IN const ExtendedTimerEntry& timer) const;
};

} //btstackif

#endif //_STARTUP_H_
