/**
 * @file ServiceSearch.cpp
 *
 * @par SW-Component
 * State machine for service search
 *
 * @brief Implementation of generic service search state machine.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Source file for implementation of generic service search state machine.
 */

#include "ServiceSearch.h"
#include "IServiceSearchRequest.h"
#include "IBasicControl.h"
#include "ITimerPool.h"
#include "IConfiguration.h"
#include "IDeviceManager.h"
#include "IProtocolManager.h"
#include "Timer.h"
#include "FwErrmemPrint.h"
#include "FwStringUtils.h"
#include "FwBluetoothStringUtils.h"
#include "App2Bts_MessageWrapper.h"
#include "Bts2App_MessageWrapper.h"
#include "IServiceSearchObserver.h"
#include "TraceClasses.h"
#include "FwTrace.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_BTS_CONTROL
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/ServiceSearch.cpp.trc.h"
#endif
#endif

namespace btstackif {

ServiceSearch::ServiceSearch() :
_requestIf(0),
_controlIf(0),
_timerPoolIf(0),
_configurationIf(0),
_deviceManagerIf(0),
_protocolManagerIf(0),
_serviceSearchList(),
_observerList(),
_simulateSppSupport(false),
_simulatePanSupport(false),
_simulateSppUuid("db23ce977ba04df2b841ad9c1dff3559"),
_testTriggerCancelServiceSearch(false),
_ignoreServiceSearchData(false),
_ignoreSppUuids(false),
_timeoutRetry(1000),
_maxRetry(30)
{
}

ServiceSearch::~ServiceSearch()
{
   _requestIf = 0;
   _controlIf = 0;
   _timerPoolIf = 0;
   _configurationIf = 0;
   _deviceManagerIf = 0;
   _protocolManagerIf = 0;
}

void ServiceSearch::reset(void)
{
   StateMachine::reset();
   // keep _requestIf
   // keep _controlIf
   // keep _timerPoolIf
   // keep _configurationIf
   // keep _deviceManagerIf
   // keep _protocolManagerIf
   // stop and release all timer
   for(::std::map< BTSBDAddress, ServiceSearchData >::iterator it = _serviceSearchList.begin(); it != _serviceSearchList.end(); ++it)
   {
      releaseTimer(it->second.retryTimer);
      releaseTimer(it->second.pageTimeoutTimer);
   }
   _serviceSearchList.clear();
   // keep _observerList
   _simulateSppSupport = false;
   _simulatePanSupport = false;
   _testTriggerCancelServiceSearch = false;
   _ignoreServiceSearchData = false;
   _ignoreSppUuids = false;

   FW_ERRMEM_IF_NULL_PTR_RETURN(_requestIf);
   _requestIf->reset();
}

void ServiceSearch::forceInitialState(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList)
{
   // check current state/action
   for(::std::map< BTSBDAddress, ServiceSearchData >::const_iterator it = _serviceSearchList.begin(); it != _serviceSearchList.end(); ++it)
   {
      const BTSBDAddress& workingAddress = it->first;
      const ServiceSearchData& entry = it->second;

      // check for ongoing service search
      if(true == entry.ongoing)
      {
         // send result
         sendStatusAndResult(bts2AppMsgList, workingAddress, entry.type, entry.requestItem.user, entry.requestItem.handle, false, (BTSCommonEnumClass)BTS_REQ_FAILED, BTS_COMMON_ENUM_CLASS_DEFAULT_VALUE);
      }
   }

   // reset control data
   reset();
}

void ServiceSearch::setInstance(IN IServiceSearchRequest* instance)
{
   _requestIf = instance;

   FW_ERRMEM_IF_NULL_PTR_RETURN(_requestIf);

   _requestIf->setCallback(this);
}

void ServiceSearch::setControlIf(IN IBasicControl* control)
{
   _controlIf = control;

   FW_ERRMEM_ASSERT(0 != _controlIf);

   FW_ERRMEM_IF_NULL_PTR_RETURN(_requestIf);

   _requestIf->setControlIf(_controlIf);
}

void ServiceSearch::setTimerPoolIf(IN ITimerPool* timerPool)
{
   _timerPoolIf = timerPool;

   FW_ERRMEM_ASSERT(0 != _timerPoolIf);
}

void ServiceSearch::setConfigurationIf(IN IConfiguration* configurationIf)
{
   _configurationIf = configurationIf;

   FW_ERRMEM_ASSERT(0 != _configurationIf);
}

void ServiceSearch::setDeviceManagerIf(IN IDeviceManager* deviceManagerIf)
{
   _deviceManagerIf = deviceManagerIf;

   FW_ERRMEM_ASSERT(0 != _deviceManagerIf);
}

void ServiceSearch::setProtocolManagerIf(IN IProtocolManager* protocolManagerIf)
{
   _protocolManagerIf = protocolManagerIf;

   FW_ERRMEM_ASSERT(0 != _protocolManagerIf);
}

IDeviceObserver* ServiceSearch::getDeviceObserver(void)
{
   return this;
}

IStateMachine* ServiceSearch::getSmEntryInterface(void)
{
   return this;
}

void ServiceSearch::sendStatusAndResult(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const App2Bts_StartRemoteServiceSearch& request, IN const bool sendStatusToAll, IN const BTSCommonEnumClass resultCode, IN const BTSCommonEnumClass statusCode) const
{
   sendStatusAndResult(bts2AppMsgList, request.getBDAddress(), request.getSearchType(), request.getUser(), request.getSessionHandle(), sendStatusToAll, resultCode, statusCode, request.getInternalMessageFlag());
}

void ServiceSearch::sendStatusAndResult(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address, IN const BTSSearchType searchType, IN BtStackIfCallback* user, IN const BTSSessionHandle handle, IN const bool sendStatusToAll, IN const BTSCommonEnumClass resultCode, IN const BTSCommonEnumClass statusCode, IN const bool internalRequest /*= false*/) const
{
   (void)(statusCode);
   (void)(internalRequest);

   BTSRequestResult resultServices = BTS_REQ_SEARCH_NOT_STARTED;
   BTSRequestResult resultSpp = BTS_REQ_SEARCH_NOT_STARTED;
   BTSRequestResult resultDid = BTS_REQ_SEARCH_NOT_STARTED;

   if(false == isValidStartRequest(address, searchType))
   {
      FW_ERRMEM_ASSERT_ALWAYS();

      if(BTS_SEARCH_ALL == searchType)
      {
         resultServices = BTS_REQ_INVALID_PARAM;
      }

      if((BTS_SEARCH_ALL == searchType) || (BTS_SEARCH_SPP == searchType))
      {
         resultSpp = BTS_REQ_INVALID_PARAM;
      }

      if((BTS_SEARCH_ALL == searchType) || (BTS_SEARCH_DID == searchType))
      {
         resultDid = BTS_REQ_INVALID_PARAM;
      }

      // use given address
      createSupportedServicesUpdate(bts2AppMsgList, address, resultServices, 0, user, handle);
      createSppCapabilitiesUpdate(bts2AppMsgList, address, resultSpp, 0, user, handle);
      createServiceRecordUpdate(bts2AppMsgList, address, resultDid, 0, user, handle);
      createServiceSearchResult(bts2AppMsgList, address, resultServices, resultSpp, resultDid, user, handle);
      return;
   }

   BTSRequestResult result((BTSRequestResult)resultCode);
   if(BTS_REQ_LAST <= result)
   {
      FW_ERRMEM_ASSERT_ALWAYS();
      result = BTS_REQ_FAILED;
   }

   BTSBDAddress workingAddress(address);
   ::fw::convertString2LowerCase(workingAddress);
   ::std::map< BTSBDAddress, ServiceSearchData >::const_iterator it = getEntry(workingAddress);
   resultServices = result;
   resultSpp = result;
   resultDid = result;
   const ServiceSearchData* entry = 0;
   BtStackIfCallback* sendUser = user;
   BTSSessionHandle sendHandle = handle;

   if(_serviceSearchList.end() != it)
   {
      entry = &(it->second);
   }

   if(true == sendStatusToAll)
   {
      sendUser = 0;
      sendHandle = 0;
   }

   if(BTS_SEARCH_ALL == searchType)
   {
      if((0 != entry) && (true == entry->sdpRecordAvailable.getBit(ServiceSearchData::SERVICES_RECORD_AVAILABLE)))
      {
         resultServices = BTS_REQ_SUCCESS;
         createSupportedServicesUpdate(bts2AppMsgList, workingAddress, resultServices, entry, sendUser, sendHandle);
      }
      else
      {
         if(BTS_REQ_SUCCESS == resultServices)
         {
            resultServices = BTS_REQ_SEARCH_NOT_STARTED;
         }
         createSupportedServicesUpdate(bts2AppMsgList, workingAddress, resultServices, 0, sendUser, sendHandle);
      }
   }
   else if(BTS_SEARCH_PBAP == searchType)
   {
      if((0 != entry) && (true == entry->sdpRecordAvailable.getBit(ServiceSearchData::SERVICES_RECORD_AVAILABLE)) && (true == entry->sdpRecordAvailable.getBit(ServiceSearchData::PBAP_RECORD_AVAILABLE)))
      {
         resultServices = BTS_REQ_SUCCESS;
         createSupportedServicesUpdate(bts2AppMsgList, workingAddress, resultServices, entry, sendUser, sendHandle);
      }
   }
   else if(BTS_SEARCH_MAP == searchType)
   {
      if((0 != entry) && (true == entry->sdpRecordAvailable.getBit(ServiceSearchData::SERVICES_RECORD_AVAILABLE)) && (true == entry->sdpRecordAvailable.getBit(ServiceSearchData::MAP_RECORD_AVAILABLE)))
      {
         resultServices = BTS_REQ_SUCCESS;
         createSupportedServicesUpdate(bts2AppMsgList, workingAddress, resultServices, entry, sendUser, sendHandle);
      }
   }
   else if(BTS_SEARCH_PAN == searchType)
   {
      if((0 != entry) && (true == entry->sdpRecordAvailable.getBit(ServiceSearchData::SERVICES_RECORD_AVAILABLE)) && (true == entry->sdpRecordAvailable.getBit(ServiceSearchData::PAN_RECORD_AVAILABLE)))
      {
         resultServices = BTS_REQ_SUCCESS;
         createSupportedServicesUpdate(bts2AppMsgList, workingAddress, resultServices, entry, sendUser, sendHandle);
      }
   }

   if((BTS_SEARCH_ALL == searchType) || (BTS_SEARCH_SPP == searchType) || (BTS_SEARCH_PAN == searchType))
   {
      // check for automatic stack search after pairing
      if((0 != entry) && (true == entry->automaticSdpSearch))
      {
         if(true == entry->sdpRecordAvailable.getBit(ServiceSearchData::SPP_RECORD_AVAILABLE))
         {
            resultSpp = BTS_REQ_SUCCESS;
            createSppCapabilitiesUpdate(bts2AppMsgList, workingAddress, resultSpp, entry, sendUser, sendHandle);
         }
         else
         {
            // set result to not started and do not update SPP capabilities
            resultSpp = BTS_REQ_SEARCH_NOT_STARTED;
         }
      }
      else
      {
         // default handling
         if((0 != entry) && (true == entry->sdpRecordAvailable.getBit(ServiceSearchData::SPP_RECORD_AVAILABLE)))
         {
            resultSpp = BTS_REQ_SUCCESS;
            createSppCapabilitiesUpdate(bts2AppMsgList, workingAddress, resultSpp, entry, sendUser, sendHandle);
         }
         else
         {
            if(BTS_REQ_SUCCESS == resultSpp)
            {
               resultSpp = BTS_REQ_SEARCH_NOT_STARTED;
            }
            if(BTS_SEARCH_SPP == searchType)
            {
               resultSpp = BTS_REQ_FAILED;
            }
            createSppCapabilitiesUpdate(bts2AppMsgList, workingAddress, resultSpp, 0, sendUser, sendHandle);
         }
      }
   }

   if((BTS_SEARCH_ALL == searchType) || (BTS_SEARCH_DID == searchType))
   {
      if((0 != entry) && (true == entry->sdpRecordAvailable.getBit(ServiceSearchData::DID_RECORD_AVAILABLE)))
      {
         resultDid = BTS_REQ_SUCCESS;
         createServiceRecordUpdate(bts2AppMsgList, workingAddress, resultDid, entry, sendUser, sendHandle);
      }
      else
      {
         if(BTS_REQ_SUCCESS == resultDid)
         {
            resultDid = BTS_REQ_SEARCH_NOT_STARTED;
         }
         if(BTS_SEARCH_DID == searchType)
         {
            resultDid = BTS_REQ_FAILED;
         }
         createServiceRecordUpdate(bts2AppMsgList, workingAddress, resultDid, 0, sendUser, sendHandle);
      }
   }

   // no result for BTS_SEARCH_PBAP
   // no result for BTS_SEARCH_MAP
   // no result for BTS_SEARCH_PAN

   // correct results
   if(BTS_SEARCH_SPP == searchType)
   {
      resultServices = BTS_REQ_SEARCH_NOT_STARTED;
      resultDid = BTS_REQ_SEARCH_NOT_STARTED;
   }
   else if(BTS_SEARCH_DID == searchType)
   {
      resultServices = BTS_REQ_SEARCH_NOT_STARTED;
      resultSpp = BTS_REQ_SEARCH_NOT_STARTED;
   }

   if((BTS_SEARCH_ALL == searchType) || (BTS_SEARCH_SPP == searchType) || (BTS_SEARCH_DID == searchType))
   {
      createServiceSearchResult(bts2AppMsgList, workingAddress, resultServices, resultSpp, resultDid, user, handle);
   }
}

bool ServiceSearch::isValidStartRequest(IN const App2Bts_StartRemoteServiceSearch& request) const
{
   return isValidStartRequest(request.getBDAddress(), request.getSearchType());
}

bool ServiceSearch::isValidStartRequest(IN const BTSBDAddress& address, IN const BTSSearchType searchType) const
{
   return ((true == ::fw::isValidBdAddress(address)) && (BTS_SEARCH_LAST > searchType));
}

bool ServiceSearch::isValidCancelRequest(IN const App2Bts_CancelRemoteServiceSearch& request) const
{
   return ((true == ::fw::isValidBdAddress(request.getBDAddress())) && (BTS_SEARCH_LAST > request.getSearchType()));
}

bool ServiceSearch::startSearch(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const App2Bts_StartRemoteServiceSearch& request)
{
   ETG_TRACE_USR3((" startSearch: address=%s", request.getBDAddress().c_str()));

   FW_ERRMEM_IF_NULL_PTR_RETURN_FALSE(_requestIf);
   FW_ERRMEM_IF_NULL_PTR_RETURN_FALSE(_deviceManagerIf);

   BTSRequestResult resultServices = BTS_REQ_SEARCH_NOT_STARTED;
   BTSRequestResult resultSpp = BTS_REQ_SEARCH_NOT_STARTED;
   BTSRequestResult resultDid = BTS_REQ_SEARCH_NOT_STARTED;

   if(false == isValidStartRequest(request))
   {
      FW_ERRMEM_ASSERT_ALWAYS();

      if(BTS_SEARCH_ALL == request.getSearchType())
      {
         resultServices = BTS_REQ_INVALID_PARAM;
      }

      if((BTS_SEARCH_ALL == request.getSearchType()) || (BTS_SEARCH_SPP == request.getSearchType()))
      {
         resultSpp = BTS_REQ_INVALID_PARAM;
      }

      if((BTS_SEARCH_ALL == request.getSearchType()) || (BTS_SEARCH_DID == request.getSearchType()))
      {
         resultDid = BTS_REQ_INVALID_PARAM;
      }

      // use given address
      createSupportedServicesUpdate(bts2AppMsgList, request.getBDAddress(), resultServices, 0, request.getUser(), request.getSessionHandle());
      createSppCapabilitiesUpdate(bts2AppMsgList, request.getBDAddress(), resultSpp, 0, request.getUser(), request.getSessionHandle());
      createServiceRecordUpdate(bts2AppMsgList, request.getBDAddress(), resultDid, 0, request.getUser(), request.getSessionHandle());
      createServiceSearchResult(bts2AppMsgList, request.getBDAddress(), resultServices, resultSpp, resultDid, request.getUser(), request.getSessionHandle());
      return false;
   }

   BTSBDAddress workingAddress(request.getBDAddress());
   ::fw::convertString2LowerCase(workingAddress);

   const bool rejectIfNotConnected(request.getRejectIfNotConnected());
   const bool anyProfileConnected(getAnyProfileConnected(workingAddress));

   if((true == rejectIfNotConnected) && (false == anyProfileConnected))
   {
      ETG_TRACE_USR2((" startSearch: address=%s: request rejected", request.getBDAddress().c_str()));

      // update status and result
      sendStatusAndResult(bts2AppMsgList, workingAddress, request.getSearchType(), request.getUser(), request.getSessionHandle(), false, (BTSCommonEnumClass)BTS_REQ_SUCCESS, BTS_COMMON_ENUM_CLASS_DEFAULT_VALUE);
      return false;
   }

   ServiceSearchData& entry = checkServiceSearchList(workingAddress);

   // store data and process request
   // keep stored data
   if(BTS_SEARCH_DID == request.getSearchType())
   {
      entry.didDataAvailable.setData(0);
   }
   entry.sdpRecordAvailable.resetBit(convertSearchType2SearchDoneBit(request.getSearchType()));
   entry.pendingGetData.setData(getPendingGetDataBits(request.getSearchType(), false));
   entry.automaticSdpSearch = false;
   entry.ongoing = true;
   entry.resultPending = true;
   entry.dataPending = true;
   entry.cancelRequested = false;
   entry.cancelPending = false;
   entry.type = request.getSearchType();
   entry.internal = request.getInternalMessageFlag();
   request.getCompareItem(entry.requestItem.item);
   entry.requestItem.user = request.getUser();
   entry.requestItem.handle = request.getSessionHandle();

   // stop page timeout timer
   stopTimer(entry.pageTimeoutTimer);
   entry.pageTimeoutHappened = false;
   entry.pageTimeout = 0;

   // service search is normally triggered while device is already connected => therefore page timeout handling not necessary, keep page timeout value of 0

   // remember device connection status
   entry.deviceConnected = _deviceManagerIf->isDeviceConnected(workingAddress, true);

   if(true == _deviceManagerIf->isDeviceAvailable(workingAddress))
   {
      _requestIf->startSearch(bts2IpcMsgList, bts2AppMsgList, workingAddress, request.getSearchType());

      startPageTimeoutTimer(entry);
   }
   else
   {
      // trigger create device
      entry.createPending = true;
      _deviceManagerIf->createDevice(bts2IpcMsgList, bts2AppMsgList, workingAddress);
   }

   // stop any retry
   entry.retryNmb = 0;
   stopTimer(entry.retryTimer);

   return true;
}

bool ServiceSearch::startSearch(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address, IN const BTSSearchType searchType, IN const BTSTimeValue pageTimeout)
{
   // this function is called only internally

   ETG_TRACE_USR3((" startSearch: address=%s", address.c_str()));

   FW_ERRMEM_IF_NULL_PTR_RETURN_FALSE(_requestIf);
   FW_ERRMEM_IF_NULL_PTR_RETURN_FALSE(_deviceManagerIf);

   // check for valid address and searchType not necessary

   BTSBDAddress workingAddress(address);
   ::fw::convertString2LowerCase(workingAddress);

   ServiceSearchData& entry = checkServiceSearchList(workingAddress);

   // store data and process request
   // keep stored data
   if(BTS_SEARCH_DID == searchType)
   {
      entry.didDataAvailable.setData(0);
   }
   entry.sdpRecordAvailable.resetBit(convertSearchType2SearchDoneBit(searchType));
   entry.pendingGetData.setData(getPendingGetDataBits(searchType, false));
   entry.automaticSdpSearch = false;
   entry.ongoing = true;
   entry.resultPending = true;
   entry.dataPending = true;
   entry.cancelRequested = false;
   entry.cancelPending = false;
   entry.type = searchType;
   entry.internal = true;
   // entry.requestItem.item
   entry.requestItem.user = 0;
   entry.requestItem.handle = 0;

   // stop page timeout timer
   stopTimer(entry.pageTimeoutTimer);
   entry.pageTimeoutHappened = false;

   entry.pageTimeout = pageTimeout; // take over given page timeout value

   // remember device connection status
   entry.deviceConnected = _deviceManagerIf->isDeviceConnected(workingAddress, true);

   if(true == _deviceManagerIf->isDeviceAvailable(workingAddress))
   {
      _requestIf->startSearch(bts2IpcMsgList, bts2AppMsgList, workingAddress, searchType);

      startPageTimeoutTimer(entry);
   }
   else
   {
      // trigger create device
      entry.createPending = true;
      _deviceManagerIf->createDevice(bts2IpcMsgList, bts2AppMsgList, workingAddress);
   }

   // stop any retry
   entry.retryNmb = 0;
   stopTimer(entry.retryTimer);

   return true;
}

void ServiceSearch::triggerSdpSearchRequest(IN const BTSBDAddress& address, IN const BTSSearchType searchType, IN const bool highPrio /*= true*/)
{
   FW_ERRMEM_IF_NULL_PTR_RETURN(_controlIf);

   // valid for all
   if(false == isValidStartRequest(address, searchType))
   {
      FW_ERRMEM_ASSERT_ALWAYS();
      return;
   }

   App2Bts_StartRemoteServiceSearch* msg = ptrNew_App2Bts_StartRemoteServiceSearch();
   if(0 != msg)
   {
      // keep user as NULL
      msg->setBDAddress(address);
      msg->setSearchType(searchType);
      msg->setRejectIfNotConnected(true); // internal service search to update related information shall not be started in case of no profile connection
      // keep session handle as 0
      msg->setInternalMessageFlag(true);
   }

   _controlIf->sendInternalApp2BtsMessage(msg, highPrio);
}

bool ServiceSearch::cancelSearch(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const App2Bts_CancelRemoteServiceSearch& request)
{
   ETG_TRACE_USR3((" cancelSearch: address=%s", request.getBDAddress().c_str()));

   FW_ERRMEM_IF_NULL_PTR_RETURN_FALSE(_requestIf);
   FW_ERRMEM_IF_NULL_PTR_RETURN_FALSE(_deviceManagerIf);

   if(false == isValidCancelRequest(request))
   {
      FW_ERRMEM_ASSERT_ALWAYS();

      // nothing else to do; wait for end of ongoing search
      return false;
   }

   BTSBDAddress workingAddress(request.getBDAddress());
   ::fw::convertString2LowerCase(workingAddress);

   // it could be possible that the device is not in list (valid scenario)
   ::std::map< BTSBDAddress, ServiceSearchData >::iterator it = getEntry(workingAddress);
   if(_serviceSearchList.end() == it)
   {
      return false;
   }

   ServiceSearchData& entry = it->second;

   // check for matching search type
   if(entry.type != request.getSearchType())
   {
      return false;
   }

   // check if device creation is ongoing
   if(true == entry.createPending)
   {
      entry.cancelRequested = true;
   }

   // check for device available: it could be possible that the device is no longer available or device creation was started before (valid scenario)
   if(false == _deviceManagerIf->isDeviceAvailable(workingAddress))
   {
      return false;
   }

   // search ongoing?
   if(false == entry.ongoing)
   {
      return false;
   }

   // cancel already requested?
   if(true == entry.cancelRequested)
   {
      return false;
   }

   // check if result is pending or retry was started (timer active)
   if(true == isTimerActive(entry.retryTimer))
   {
      // no result pending but retry timer active => same handling as failed service search result
      // timer is stopped below
      // send virtual failed service search result message to force end of sequence
      _requestIf->sendVirtualFailedResult(bts2IpcMsgList, bts2AppMsgList, workingAddress);
   }
   else if(true == entry.resultPending)
   {
      // result pending => cancel can be sent
      _requestIf->cancelSearch(bts2IpcMsgList, bts2AppMsgList, workingAddress);
      // set marker for sent cancel request
      entry.cancelPending = true;
   }
   else
   {
      // success case and waiting for data updates => cancel has no effect
   }

   entry.cancelRequested = true;

   // stop page timeout timer
   stopTimer(entry.pageTimeoutTimer);
   entry.pageTimeoutHappened = false;
   entry.pageTimeout = 0;

   // stop any retry
   entry.retryNmb = 0;
   stopTimer(entry.retryTimer);

   // no need to store the request => return false
   return false;
}

bool ServiceSearch::cancelSearch(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address, IN const BTSSearchType searchType)
{
   // this function is called only internally

   ETG_TRACE_USR3((" cancelSearch: address=%s", address.c_str()));

   FW_ERRMEM_IF_NULL_PTR_RETURN_FALSE(_requestIf);
   FW_ERRMEM_IF_NULL_PTR_RETURN_FALSE(_deviceManagerIf);

   // check for valid address and searchType not necessary

   BTSBDAddress workingAddress(address);
   ::fw::convertString2LowerCase(workingAddress);

   // it could be possible that the device is not in list (valid scenario)
   ::std::map< BTSBDAddress, ServiceSearchData >::iterator it = getEntry(workingAddress);
   if(_serviceSearchList.end() == it)
   {
      return false;
   }

   ServiceSearchData& entry = it->second;

   // check for matching search type
   if(entry.type != searchType)
   {
      return false;
   }

   // check if device creation is ongoing
   if(true == entry.createPending)
   {
      entry.cancelRequested = true;
   }

   // check for device available: it could be possible that the device is no longer available or device creation was started before (valid scenario)
   if(false == _deviceManagerIf->isDeviceAvailable(workingAddress))
   {
      return false;
   }

   // search ongoing?
   if(false == entry.ongoing)
   {
      return false;
   }

   // cancel already requested?
   if(true == entry.cancelRequested)
   {
      return false;
   }

   // check if result is pending or retry was started (timer active)
   if(true == isTimerActive(entry.retryTimer))
   {
      // no result pending but retry timer active => same handling as failed service search result
      // timer is stopped below
      // send virtual failed service search result message to force end of sequence
      _requestIf->sendVirtualFailedResult(bts2IpcMsgList, bts2AppMsgList, workingAddress);
   }
   else if(true == entry.resultPending)
   {
      // result pending => cancel can be sent
      _requestIf->cancelSearch(bts2IpcMsgList, bts2AppMsgList, workingAddress);
      // set marker for sent cancel request
      entry.cancelPending = true;
   }
   else
   {
      // success case and waiting for data updates => cancel has no effect
   }

   entry.cancelRequested = true;

   // stop page timeout timer
   stopTimer(entry.pageTimeoutTimer);
   entry.pageTimeoutHappened = false;
   entry.pageTimeout = 0;

   // stop any retry
   entry.retryNmb = 0;
   stopTimer(entry.retryTimer);

   // return true to indicate that processing was started properly
   return true;
}

void ServiceSearch::cancelSearch(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address)
{
   for(::std::map< BTSBDAddress, ServiceSearchData >::const_iterator it = _serviceSearchList.begin(); it != _serviceSearchList.end(); ++it)
   {
      BTSBDAddress workingAddress(address);
      ::fw::convertString2LowerCase(workingAddress);

      if(workingAddress == it->first)
      {
         const ServiceSearchData& entry = it->second;

         (void)cancelSearch(bts2IpcMsgList, bts2AppMsgList, workingAddress, entry.type);
      }
   }
}

bool ServiceSearch::isAnySearchOngoing(void) const
{
   for(::std::map< BTSBDAddress, ServiceSearchData >::const_iterator it = _serviceSearchList.begin(); it != _serviceSearchList.end(); ++it)
   {
      if((true == it->second.ongoing) || (true == it->second.stackSdpSearchActive))
      {
         return true;
      }
   }

   return false;
}

bool ServiceSearch::isSearchOngoing(IN const BTSBDAddress& address) const
{
   ::std::map< BTSBDAddress, ServiceSearchData >::const_iterator it = getEntry(address);

   if(_serviceSearchList.end() != it)
   {
      return it->second.ongoing;
   }

   return false;
}

bool ServiceSearch::isUuidAvailable(IN const BTSBDAddress& address, IN const BTSUuid& uuid) const
{
   ::std::map< BTSBDAddress, ServiceSearchData >::const_iterator it = getEntry(address);

   if(_serviceSearchList.end() == it)
   {
      return false;
   }

   BTSUuid copyUuid(uuid);
   ::fw::convertString2LowerCase(copyUuid);
   const ServiceSearchData& entry = it->second;

   for(size_t i = 0; i < entry.sspCapabilities.size(); i++)
   {
      // each entry has only 1 UUID
      if(0 < entry.sspCapabilities[i].uuidList.size())
      {
         if(copyUuid == entry.sspCapabilities[i].uuidList[0])
         {
            return true;
         }
      }
   }

   return false;
}

bool ServiceSearch::isSdpSearchDone(IN const BTSBDAddress& address, IN const BTSSearchType searchType)
{
   // valid for all
   if(BTS_SEARCH_LAST <= searchType)
   {
      FW_ERRMEM_ASSERT_ALWAYS();
      return false;
   }

   ::std::map< BTSBDAddress, ServiceSearchData >::const_iterator it = getEntry(address);

   if(_serviceSearchList.end() == it)
   {
      return false;
   }

   const ServiceSearchData& entry = it->second;

   return entry.sdpRecordAvailable.getBit(convertSearchType2SearchDoneBit(searchType));
}

bool ServiceSearch::isSdpRecordAvailable(IN const BTSBDAddress& address, IN const BTSSearchType searchType)
{
   // valid for all
   if(BTS_SEARCH_LAST <= searchType)
   {
      FW_ERRMEM_ASSERT_ALWAYS();
      return false;
   }

   ::std::map< BTSBDAddress, ServiceSearchData >::const_iterator it = getEntry(address);

   if(_serviceSearchList.end() == it)
   {
      return false;
   }

   const ServiceSearchData& entry = it->second;

   return entry.sdpRecordAvailable.getBit(convertSearchType2RecordAvailableBit(searchType));
}

bool ServiceSearch::isDeviceCreationOngoing(void) const
{
   for(::std::map< BTSBDAddress, ServiceSearchData >::const_iterator it = _serviceSearchList.begin(); it != _serviceSearchList.end(); ++it)
   {
      if(true == it->second.createPending)
      {
         return true;
      }
   }

   return false;
}

bool ServiceSearch::getMasInstanceId(OUT BTSMasInstanceId& instanceId, const BTSBDAddress& address, IN const BTSMasInstanceName& instanceName) const
{
   FW_ERRMEM_IF_NULL_PTR_RETURN_FALSE(_configurationIf);

   ::std::map< BTSBDAddress, ServiceSearchData >::const_iterator it = getEntry(address);

   if(_serviceSearchList.end() == it)
   {
      ETG_TRACE_USR3((" getMasInstanceId: address=%12s instanceId=%u instanceName=%s (no matching device)", address.c_str(), instanceId, instanceName.c_str()));
      return false;
   }

   const ServiceSearchData& entry = it->second;

   if(false == instanceName.empty())
   {
      // try to find matching instance
      for(size_t i = 0; i < entry.mapRecords.size(); i++)
      {
         if(entry.mapRecords[i].name == instanceName)
         {
            instanceId = entry.mapRecords[i].instanceId;
            ETG_TRACE_USR3((" getMasInstanceId: address=%12s instanceId=%u instanceName=%s", address.c_str(), instanceId, instanceName.c_str()));
            return true;
         }
      }
   }

   // either instance name is empty or no matching instance was found => use default instance
   BTSMasInstanceId instance = 0;
   unsigned int prio;
   unsigned int lastPrio = 0;

   for(size_t i = 0; i < entry.mapRecords.size(); i++)
   {
      // set to first available instance (to have a valid instance); assuming that instance supports at least SMS if there is only 1 instance available
      if(0 == i)
      {
         instance = entry.mapRecords[i].instanceId;
      }

      // default instance shall not support EMAIL because this one shall be explicitly selected by user; then we will never reach this block here
      // default instance shall support SMS
      // default instance shall support MMS if MMS is configured as enabled
      prio = 0;
      if(false == entry.mapRecords[i].messageTypes.getBit(BTS_MAP_MSG_TYPE_EMAIL))
      {
         // no EMAIL
         prio += 1;
      }

      if((true == entry.mapRecords[i].messageTypes.getBit(BTS_MAP_MSG_TYPE_SMS_GSM)) || (true == entry.mapRecords[i].messageTypes.getBit(BTS_MAP_MSG_TYPE_SMS_CDMA)))
      {
         // SMS
         prio += 4;
      }

      if((true == entry.mapRecords[i].messageTypes.getBit(BTS_MAP_MSG_TYPE_MMS)) && (true == _configurationIf->getConfiguration().mapMmsEnabled))
      {
         // MMS supported and enabled
         prio += 2;
      }
      else if((false == entry.mapRecords[i].messageTypes.getBit(BTS_MAP_MSG_TYPE_MMS)) && (false == _configurationIf->getConfiguration().mapMmsEnabled))
      {
         // MMS not supported and not enabled
         prio += 2;
      }

      if(prio > lastPrio)
      {
         instance = entry.mapRecords[i].instanceId;
         lastPrio = prio;
      }
   }

   if(0 < entry.mapRecords.size())
   {
      instanceId = instance;
      ETG_TRACE_USR3((" getMasInstanceId: address=%12s instanceId=%u instanceName=%s", address.c_str(), instanceId, instanceName.c_str()));
      return true;
   }

   ETG_TRACE_USR3((" getMasInstanceId: address=%12s instanceId=%u instanceName=%s (no matching instance)", address.c_str(), instanceId, instanceName.c_str()));
   return false;
}

bool ServiceSearch::getMasInstanceName(OUT BTSMasInstanceName& instanceName, const BTSBDAddress& address, IN const BTSMasInstanceId instanceId) const
{
   ::std::map< BTSBDAddress, ServiceSearchData >::const_iterator it = getEntry(address);

   if(_serviceSearchList.end() == it)
   {
      ETG_TRACE_USR3((" getMasInstanceName: address=%12s instanceId=%u instanceName=%s (no matching device)", address.c_str(), instanceId, instanceName.c_str()));
      return false;
   }

   const ServiceSearchData& entry = it->second;

   // try to find matching instance
   for(size_t i = 0; i < entry.mapRecords.size(); i++)
   {
      if(entry.mapRecords[i].instanceId == instanceId)
      {
         instanceName = entry.mapRecords[i].name;
         ETG_TRACE_USR3((" getMasInstanceName: address=%12s instanceId=%u instanceName=%s", address.c_str(), instanceId, instanceName.c_str()));
         return true;
      }
   }

   ETG_TRACE_USR3((" getMasInstanceName: address=%12s instanceId=%u instanceName=%s (no matching instance)", address.c_str(), instanceId, instanceName.c_str()));
   return false;
}

void ServiceSearch::requestServiceVersion(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address, IN const BTSProtocolId protocol)
{
   FW_ERRMEM_IF_NULL_PTR_RETURN(_requestIf);

   if((BTS_PROTO_HFP != protocol) && (BTS_PROTO_AVP != protocol))
   {
      FW_ERRMEM_ASSERT_ALWAYS();
      return;
   }

   if(false == ::fw::isValidBdAddress(address))
   {
      FW_ERRMEM_ASSERT_ALWAYS();
      return;
   }

   BTSBDAddress workingAddress(address);
   ::fw::convertString2LowerCase(workingAddress);

   // remember that version was requested from outside, limit to AVP
   if(BTS_PROTO_AVP == protocol)
   {
      ServiceSearchData& entry = checkServiceSearchList(workingAddress);
      entry.avrcpVersionRequested = true;
   }

   _requestIf->getServiceVersion(bts2IpcMsgList, bts2AppMsgList, workingAddress, protocol);
}

void ServiceSearch::setSppSupportSimulation(IN const bool enable)
{
   _simulateSppSupport = enable;
}

void ServiceSearch::setPanSupportSimulation(IN const bool enable)
{
   _simulatePanSupport = enable;
}

void ServiceSearch::setTestTriggerCancelServiceSearch(IN const bool enable)
{
   _testTriggerCancelServiceSearch = enable;
}

void ServiceSearch::setIgnoreServiceSearchData(IN const bool enable)
{
   _ignoreServiceSearchData = enable;
}

void ServiceSearch::setIgnoreSppUuids(IN const bool enable)
{
   _ignoreSppUuids = enable;
}

void ServiceSearch::registerObserver(IN IServiceSearchObserver* observer)
{
   FW_ERRMEM_IF_NULL_PTR_RETURN(observer);

   _observerList.insert(observer);
}

void ServiceSearch::handleTestCommandGetDeviceVendorId(void)
{
   FW_ERRMEM_IF_NULL_PTR_RETURN(_requestIf);
   FW_ERRMEM_IF_NULL_PTR_RETURN(_controlIf);

   ::std::map< BTSBDAddress, ServiceSearchData >::const_iterator it = _serviceSearchList.begin();

   if(_serviceSearchList.end() == it)
   {
      return;
   }

   ::std::vector< Bts2Ipc_BaseMessage* > bts2IpcMsgList;
   ::std::vector< Bts2App_BaseMessage* > bts2AppMsgList;

   _requestIf->getDeviceVendorId(bts2IpcMsgList, bts2AppMsgList, it->first);

   for(size_t i = 0; i < bts2IpcMsgList.size(); i++)
   {
      _controlIf->sendInternalBts2IpcMessage(bts2IpcMsgList[i]);
   }
}

void ServiceSearch::handleTestCommandGetAllSearchResults(void)
{
   FW_ERRMEM_IF_NULL_PTR_RETURN(_requestIf);
   FW_ERRMEM_IF_NULL_PTR_RETURN(_controlIf);

   ::std::map< BTSBDAddress, ServiceSearchData >::const_iterator it = _serviceSearchList.begin();

   if(_serviceSearchList.end() == it)
   {
      return;
   }

   ::std::vector< Bts2Ipc_BaseMessage* > bts2IpcMsgList;
   bts2IpcMsgList.reserve(10);
   ::std::vector< Bts2App_BaseMessage* > bts2AppMsgList;

   _requestIf->getDeviceVendorId(bts2IpcMsgList, bts2AppMsgList, it->first);
   _requestIf->getDeviceVendorIdSource(bts2IpcMsgList, bts2AppMsgList, it->first);
   _requestIf->getDeviceProductId(bts2IpcMsgList, bts2AppMsgList, it->first);
   _requestIf->getDeviceVersion(bts2IpcMsgList, bts2AppMsgList, it->first);
   _requestIf->getServiceVersion(bts2IpcMsgList, bts2AppMsgList, it->first, BTS_PROTO_HFP);
   _requestIf->getServiceVersion(bts2IpcMsgList, bts2AppMsgList, it->first, BTS_PROTO_AVP);
   _requestIf->getDeviceUuids(bts2IpcMsgList, bts2AppMsgList, it->first);
   _requestIf->getPbapInfo(bts2IpcMsgList, bts2AppMsgList, it->first);
   _requestIf->getMapInfo(bts2IpcMsgList, bts2AppMsgList, it->first);

   for(size_t i = 0; i < bts2IpcMsgList.size(); i++)
   {
      _controlIf->sendInternalBts2IpcMessage(bts2IpcMsgList[i]);
   }
}

void ServiceSearch::setRetryMaxCounter(IN const unsigned int max)
{
   _maxRetry = max;
}

void ServiceSearch::startTestSdpSearchAfterPairing(IN const BTSBDAddress& address)
{
   ::std::vector< Bts2Ipc_BaseMessage* > bts2IpcMsgList;
   ::std::vector< Bts2App_BaseMessage* > bts2AppMsgList;
   BTSHandleIpc2BtsMessageItem messageItem;

   (void)startSdpSearchAfterPairing(bts2IpcMsgList, bts2AppMsgList, messageItem, address, true);
}

void ServiceSearch::serviceSearchResult(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, OUT bool& connectFailed, IN const BTSBDAddress& address, IN const BTSIpcCommonErrorCode result)
{
   FW_ERRMEM_IF_NULL_PTR_RETURN(_deviceManagerIf);

   ETG_TRACE_USR3((" serviceSearchResult: address=%s", address.c_str()));

   // device must be in list
   ::std::map< BTSBDAddress, ServiceSearchData >::iterator it = getEntry(address);
   if(_serviceSearchList.end() == it)
   {
      FW_ERRMEM_ASSERT_ALWAYS();
      return;
   }

   ServiceSearchData& entry = it->second;

   entry.resultPending = false;

   //===================================================================================================================
   // debug section start
   if(true == _testTriggerCancelServiceSearch)
   {
      _testTriggerCancelServiceSearch = false;

      if(0 != _controlIf)
      {
         App2Bts_CancelRemoteServiceSearch* msg = ptrNew_App2Bts_CancelRemoteServiceSearch();
         if(0 != msg)
         {
            // keep user as NULL
            msg->setBDAddress(address);
            msg->setSearchType(entry.type);
            // keep session handle as 0
            msg->setInternalMessageFlag(true);

            _controlIf->sendInternalApp2BtsMessage(msg, true);
         }
      }
   }
   // debug section end
   //===================================================================================================================

   // check error code
   if(BTS_IPC_SUCCESS == result)
   {
      // search request was successful
      // trigger a get request to retrieve the data
      triggerGetRequests(bts2IpcMsgList, bts2AppMsgList, address, entry.type, false, entry);

      // wait for data update --- entry.dataPending
      checkForSearchCompleted(bts2IpcMsgList, bts2AppMsgList, messageItem, address, entry);
   }
   else
   {
      if((false == entry.cancelRequested) && (true == retryStarted(result, entry, _timeoutRetry)))
      {
         // retry was started
         // search sequence is ongoing

         // TODO: [low]: timer has to be stopped somehow during BT off, new pairing => observer?
      }
      else
      {
         entry.dataPending = false;

         // check for pending cancel result
         if(true == entry.cancelPending)
         {
            // wait for result of cancel request
            // search sequence is ongoing
         }
         else
         {
            if(false == entry.ongoing)
            {
               // search sequence is already finished
               FW_ERRMEM_ASSERT_ALWAYS();
               return;
            }

            // search sequence is finished
            handleEndOfFailedSearch(bts2IpcMsgList, bts2AppMsgList, messageItem, &connectFailed, entry, address);
         }
      }
   }
}

void ServiceSearch::cancelServiceSearchResult(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const BTSIpcCommonErrorCode result)
{
   ETG_TRACE_USR3((" cancelServiceSearchResult: address=%s", address.c_str()));

   // device must be in list
   ::std::map< BTSBDAddress, ServiceSearchData >::iterator it = getEntry(address);
   if(_serviceSearchList.end() == it)
   {
      FW_ERRMEM_ASSERT_ALWAYS();
      return;
   }

   ServiceSearchData& entry = it->second;

   entry.cancelPending = false;

   bool endOfSequence(false);

   // check for pending search result
   if(true == entry.resultPending)
   {
      // search result did not return until now => wait for search result
      // search sequence is ongoing

      // use case 2: search success, cancel failed, cancel result returned first
      // use case 4: search failed, cancel success, cancel result returned first
      // use case 6: search failed, cancel failed, cancel result returned first
      // use case 8: search success, cancel success, cancel result returned first
   }
   else
   {
      // search result already returned

      // check error code
      if(BTS_IPC_SUCCESS == result)
      {
         if(true == entry.dataPending)
         {
            // get requests active
            // search sequence is ongoing

            // use case 7: search success, cancel success, search result returned first
         }
         else
         {
            // nothing ongoing

            // use case 3: search failed, cancel success, search result returned first => handle end of sequence
            endOfSequence = true;
         }
      }
      else
      {
         if(true == entry.dataPending)
         {
            // get requests active
            // search sequence is ongoing

            // use case 1: search success, cancel failed, search result returned first (assumption: get requests active, cancel was sent too late, cancel result returned before last get request returns)
         }
         else
         {
            // nothing ongoing

            // use case 5: search failed, cancel failed, search result returned first (assumption: search failed e.g. page timeout, cancel was sent too late) => handle end of sequence
            endOfSequence = true;
         }
      }
   }

   // handle end of sequence?
   if(true == endOfSequence)
   {
      if(false == entry.ongoing)
      {
         // search sequence is already finished
         FW_ERRMEM_ASSERT_ALWAYS();
         return;
      }

      // search sequence is finished
      handleEndOfFailedSearch(bts2IpcMsgList, bts2AppMsgList, messageItem, 0, entry, address);
   }
}

void ServiceSearch::setSdpSearchActiveAfterPairing(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const bool enable, IN const bool success)
{
   // Bluetooth automatically starts SDP search after successful pairing
   ETG_TRACE_USR3((" setSdpSearchActiveAfterPairing: address=%s", address.c_str()));

   ServiceSearchData& entry = checkServiceSearchList(address);

   entry.stackSdpSearchActive = enable;

   if(true == enable)
   {
      // search was started => similar to startSearch() call but without any result message

      // keep stored data
      // set flags
      entry.didDataAvailable.setData(0);
      resetSearchDoneBits(entry.sdpRecordAvailable);
      entry.pendingGetData.setData(getPendingGetDataBits(BTS_SEARCH_ALL, true));
      entry.automaticSdpSearch = true;
      entry.ongoing = true;
      entry.resultPending = false;
      entry.dataPending = true;
      entry.cancelRequested = false;
      entry.cancelPending = false;
      entry.type = BTS_SEARCH_ALL;
      entry.internal = true;
      entry.requestItem.reset();

      // stop page timeout timer
      stopTimer(entry.pageTimeoutTimer);
      entry.pageTimeoutHappened = false;
      entry.pageTimeout = 0;

      // stop any retry
      entry.retryNmb = 0;
      stopTimer(entry.retryTimer);
   }
   else
   {
      // search was completed => similar to serviceSearchResult()
      if(true == success)
      {
         // search request was successful
         // trigger a get request to retrieve the data
         triggerGetRequests(bts2IpcMsgList, bts2AppMsgList, address, BTS_SEARCH_ALL, true, entry);

         // wait for data update --- entry.dataPending
         checkForSearchCompleted(bts2IpcMsgList, bts2AppMsgList, messageItem, address, entry);
      }
      else
      {
         // search sequence is finished
         entry.ongoing = false;
         entry.dataPending = false;

         // no update status and result

         // search sequence is finished; keep entry in every case

         // in case of automatic search we have no user request => fake necessary data
         messageItem.item.opCode = App2BtsOC_StartRemoteServiceSearch;
         messageItem.item.deviceAddress = address;
         messageItem.item.searchType = entry.type;
         messageItem.deleteMessage = true;

         // reset control data because action is finished
         entry.requestItem.reset();

         // reset data
         entry.automaticSdpSearch = false;
         entry.cancelRequested = false;
         entry.cancelPending = false;
         entry.type = BTS_SEARCH_LAST;
         entry.internal = false;

         // stop page timeout timer
         stopTimer(entry.pageTimeoutTimer);
         entry.pageTimeoutHappened = false;
         entry.pageTimeout = 0;
      }
   }
}

bool ServiceSearch::startSdpSearchAfterPairing(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const bool success)
{
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);
   (void)(messageItem);

   FW_ERRMEM_IF_NULL_PTR_RETURN_FALSE(_controlIf);

   ETG_TRACE_USR3((" startSdpSearchAfterPairing: address=%s", address.c_str()));

   // start SDP search only after successful pairing
   if(true != success)
   {
      return false;
   }

   App2Bts_StartRemoteServiceSearch* msg = ptrNew_App2Bts_StartRemoteServiceSearch();
   if(0 != msg)
   {
      // keep user as NULL
      msg->setBDAddress(address);
      msg->setSearchType(BTS_SEARCH_ALL);
      msg->setRejectIfNotConnected(false); // service search after pairing shall be started independent of any profile connection
      // keep session handle as 0
      msg->setInternalMessageFlag(true);

      _controlIf->sendInternalApp2BtsMessage(msg, true);

      return true;
   }

   return false;
}

bool ServiceSearch::startSdpSearchAfterPairing(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address)
{
   ServiceSearchData& entry = checkServiceSearchList(address);

   // continue if:
   // - last pairing was successful
   // - stack internal SDP search is finished

   if((true == entry.lastPairingSuccessful) && (false == entry.stackSdpSearchActive))
   {
      // reset last pairing successful flag
      entry.lastPairingSuccessful = false;

      return startSdpSearchAfterPairing(bts2IpcMsgList, bts2AppMsgList, messageItem, address, true);
   }

   return false;
}

void ServiceSearch::setPairingStatus(IN const BTSBDAddress& address, IN const bool success)
{
   if(false == ::fw::isValidBdAddress(address))
   {
      FW_ERRMEM_ASSERT_ALWAYS();
      return;
   }

   ServiceSearchData& entry = checkServiceSearchList(address);

   entry.lastPairingSuccessful = success;
}

void ServiceSearch::updateDeviceUuids(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSIpcCommonErrorCode result, IN const bool response, IN const BTSBDAddress& address, IN const BTSSupportedServices& supportedServices, IN const BTSUuidList& uuidList)
{
   FW_ERRMEM_IF_NULL_PTR_RETURN(_requestIf);

   ServiceSearchData& entry = checkServiceSearchList(address);
   bool addSimulatedUuid = false;

   // HINT: store data in every success case and non-empty service list (in case of failed automatic stack search the get is successful but the list is empty) => keep existing list
   if((BTS_IPC_SUCCESS == result) && (0 != supportedServices.getData()) && (false == _ignoreServiceSearchData))
   {
      // copy bit field
      entry.supportedServices = supportedServices;

      // copy service information, re-use version information if available
      BTSServiceInfoList list;
      BTSServiceInfo info;

      if(true == supportedServices.getBit(BTS_SUPP_SRV_HFP))
      {
         info.service = BTS_SUPP_SRV_HFP;
         info.version = getProfileVersion(entry.serviceInfoList, BTS_SUPP_SRV_HFP);
         list.push_back(info);
      }
      if(true == supportedServices.getBit(BTS_SUPP_SRV_A2DP))
      {
         info.service = BTS_SUPP_SRV_A2DP;
         info.version = getProfileVersion(entry.serviceInfoList, BTS_SUPP_SRV_A2DP);
         list.push_back(info);
      }
      if(true == supportedServices.getBit(BTS_SUPP_SRV_AVRCP))
      {
         info.service = BTS_SUPP_SRV_AVRCP;
         info.version = getProfileVersion(entry.serviceInfoList, BTS_SUPP_SRV_AVRCP);
         list.push_back(info);
      }
      if(true == supportedServices.getBit(BTS_SUPP_SRV_PBAP))
      {
         info.service = BTS_SUPP_SRV_PBAP;
         info.version = getProfileVersion(entry.serviceInfoList, BTS_SUPP_SRV_PBAP);
         list.push_back(info);
      }
      if(true == supportedServices.getBit(BTS_SUPP_SRV_MAP))
      {
         info.service = BTS_SUPP_SRV_MAP;
         info.version = getProfileVersion(entry.serviceInfoList, BTS_SUPP_SRV_MAP);
         list.push_back(info);
      }
      if(true == supportedServices.getBit(BTS_SUPP_SRV_DUN))
      {
         info.service = BTS_SUPP_SRV_DUN;
         info.version = getProfileVersion(entry.serviceInfoList, BTS_SUPP_SRV_DUN);
         list.push_back(info);
      }
      if(true == supportedServices.getBit(BTS_SUPP_SRV_PAN))
      {
         info.service = BTS_SUPP_SRV_PAN;
         info.version = getProfileVersion(entry.serviceInfoList, BTS_SUPP_SRV_PAN);
         list.push_back(info);
      }
      else
      {
         if(true == _simulatePanSupport)
         {
            ETG_TRACE_USR1((" ### simulate PAN support ###"));
            info.service = BTS_SUPP_SRV_PAN;
            info.version = getProfileVersion(entry.serviceInfoList, BTS_SUPP_SRV_PAN);
            list.push_back(info);
         }
      }
      if(true == supportedServices.getBit(BTS_SUPP_SRV_SPP))
      {
         info.service = BTS_SUPP_SRV_SPP;
         info.version = getProfileVersion(entry.serviceInfoList, BTS_SUPP_SRV_SPP);
         list.push_back(info);
      }
      else
      {
         if(true == _simulateSppSupport)
         {
            info.service = BTS_SUPP_SRV_SPP;
            info.version = getProfileVersion(entry.serviceInfoList, BTS_SUPP_SRV_SPP);
            list.push_back(info);
         }
      }
      if(true == supportedServices.getBit(BTS_SUPP_SRV_PNP))
      {
         info.service = BTS_SUPP_SRV_PNP;
         info.version = getProfileVersion(entry.serviceInfoList, BTS_SUPP_SRV_PNP);
         list.push_back(info);
      }
      if(true == supportedServices.getBit(BTS_SUPP_SRV_DID))
      {
         info.service = BTS_SUPP_SRV_DID;
         info.version = getProfileVersion(entry.serviceInfoList, BTS_SUPP_SRV_DID);
         list.push_back(info);
      }

      entry.serviceInfoList = list;

      if(true == _simulateSppSupport)
      {
         addSimulatedUuid = true;
         for(size_t i = 0; i < uuidList.size(); i++)
         {
            if(uuidList[i] == _simulateSppUuid)
            {
               addSimulatedUuid = false;
               break;
            }
         }
      }

      entry.sspCapabilities.clear();
      if(true == addSimulatedUuid)
      {
         entry.sspCapabilities.reserve(1 + uuidList.size());
      }
      else
      {
         entry.sspCapabilities.reserve(uuidList.size());
      }

      BTSSppCapability capability;
      for(size_t i = 0; i < uuidList.size(); i++)
      {
         if(false == isUuidInList(uuidList[i], entry.sspCapabilities))
         {
            capability.uuidList.push_back(uuidList[i]);
            if(false == _ignoreSppUuids)
            {
               entry.sspCapabilities.push_back(capability);
            }
            capability.uuidList.clear();
         }
      }

      if(true == addSimulatedUuid)
      {
         capability.uuidList.push_back(_simulateSppUuid);
         entry.sspCapabilities.push_back(capability);
         capability.uuidList.clear();
      }
   }

   // HINT: set record available in success case and search type is ALL or SPP or PAN
   if((BTS_IPC_SUCCESS == result) &&
      ((BTS_SEARCH_ALL == entry.type) || (BTS_SEARCH_SPP == entry.type) || (BTS_SEARCH_PAN == entry.type)))
   {
      entry.sdpRecordAvailable.setBit(ServiceSearchData::SERVICES_SEARCH_DONE);
      if((0 != supportedServices.getData()) && (false == _ignoreServiceSearchData))
      {
         // HINT: set record available only in case of non-empty service list (in case of failed automatic stack search the get is successful but the list is empty) => keep existing list/setting
         entry.sdpRecordAvailable.setBit(ServiceSearchData::SERVICES_RECORD_AVAILABLE);
      }

      // if SPP services are reported then set availability to true
      if((0 < uuidList.size()) || (true == addSimulatedUuid))
      {
         entry.sdpRecordAvailable.setBit(ServiceSearchData::SPP_SEARCH_DONE);
         entry.sdpRecordAvailable.setBit(ServiceSearchData::SPP_RECORD_AVAILABLE);
      }
      else if(false == entry.automaticSdpSearch)
      {
         entry.sdpRecordAvailable.setBit(ServiceSearchData::SPP_SEARCH_DONE);
         entry.sdpRecordAvailable.setBit(ServiceSearchData::SPP_RECORD_AVAILABLE);
      }

      // PAN
      entry.sdpRecordAvailable.setBit(ServiceSearchData::PAN_SEARCH_DONE);
      if((true == supportedServices.getBit(BTS_SUPP_SRV_PAN)) || (true == _simulatePanSupport))
      {
         entry.sdpRecordAvailable.setBit(ServiceSearchData::PAN_RECORD_AVAILABLE);
      }
   }

   // HINT: set search done in response case and search type is ALL or SPP or PAN
   if((true == response) &&
      ((BTS_SEARCH_ALL == entry.type) || (BTS_SEARCH_SPP == entry.type) || (BTS_SEARCH_PAN == entry.type)))
   {
      entry.sdpRecordAvailable.setBit(ServiceSearchData::SERVICES_SEARCH_DONE);
      if(false == entry.automaticSdpSearch)
      {
         entry.sdpRecordAvailable.setBit(ServiceSearchData::SPP_SEARCH_DONE);
      }
      entry.sdpRecordAvailable.setBit(ServiceSearchData::PAN_SEARCH_DONE);
   }

   // HINT: check for get version in response case and search type is ALL or SPP (nothing in case of PAN)
   if((true == response) &&
      ((BTS_SEARCH_ALL == entry.type) || (BTS_SEARCH_SPP == entry.type)))
   {
      entry.protocolVersionAvailable.setData(entry.protocolVersionAvailable.getDataMask());

      if(true == entry.supportedServices.getBit(BTS_SUPP_SRV_HFP))
      {
         entry.protocolVersionAvailable.resetBit(BTS_SUPP_SRV_HFP); // we have to request version for HFP
         entry.pendingGetData.setBit(ServiceSearchData::GET_HFP_VERSION);
         _requestIf->getServiceVersion(bts2IpcMsgList, bts2AppMsgList, address, BTS_PROTO_HFP);
      }

      if(true == entry.supportedServices.getBit(BTS_SUPP_SRV_AVRCP))
      {
         entry.protocolVersionAvailable.resetBit(BTS_SUPP_SRV_AVRCP); // we have to request version for AVRCP; in case of A2DP only do nothing
         entry.pendingGetData.setBit(ServiceSearchData::GET_AVRCP_VERSION);
         _requestIf->getServiceVersion(bts2IpcMsgList, bts2AppMsgList, address, BTS_PROTO_AVP);
      }
   }

   // HINT: reset pending get requests information in response case
   if(true == response)
   {
      entry.pendingGetData.resetBit(ServiceSearchData::GET_UUIDS);
   }

   // check for intermediate update of UUIDs
   if((true == entry.stackSdpSearchActive) && (false == response))
   {
      if(true == entry.sdpRecordAvailable.getBit(ServiceSearchData::SERVICES_RECORD_AVAILABLE))
      {
         createSupportedServicesUpdate(bts2AppMsgList, address, BTS_REQ_SUCCESS, entry, 0, 0);
      }

      if(true == entry.sdpRecordAvailable.getBit(ServiceSearchData::SPP_RECORD_AVAILABLE))
      {
         createSppCapabilitiesUpdate(bts2AppMsgList, address, BTS_REQ_SUCCESS, entry, 0, 0);
      }

      return;
   }

   // check if search sequence is completed
   checkForSearchCompleted(bts2IpcMsgList, bts2AppMsgList, messageItem, address, entry);
}

void ServiceSearch::updateServiceVersion(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSIpcCommonErrorCode result, IN const bool response, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSProfileVersion version)
{
   (void)(bts2IpcMsgList);

   ServiceSearchData& entry = checkServiceSearchList(address);

   if(BTS_PROTO_HFP == protocol)
   {
      entry.protocolVersionAvailable.setBit(BTS_SUPP_SRV_HFP);

      // HINT: store data in every success case
      if(BTS_IPC_SUCCESS == result)
      {
         if(emptyProtocolVersion == version)
         {
            // keep stored value
         }
         else
         {
            setProfileVersion(entry.serviceInfoList, BTS_SUPP_SRV_HFP, version);
         }
      }

      // HINT: reset pending get requests information in response case
      if(true == response)
      {
         entry.pendingGetData.resetBit(ServiceSearchData::GET_HFP_VERSION);
      }
   }
   else if(BTS_PROTO_AVP == protocol)
   {
      entry.protocolVersionAvailable.setBit(BTS_SUPP_SRV_AVRCP);

      // HINT: store data in every success case
      if(BTS_IPC_SUCCESS == result)
      {
         if(emptyProtocolVersion == version)
         {
            // keep stored value
         }
         else
         {
            setProfileVersion(entry.serviceInfoList, BTS_SUPP_SRV_AVRCP, version);
         }
      }

      // correct stored profile version in case of response because version will only be requested in case of AVRCP was indicated as supported profile or AVRCP got connected
      if(true == response)
      {
         const BTSProfileVersion tmpVersion(getProfileVersion(entry.serviceInfoList, BTS_SUPP_SRV_AVRCP));
         if((emptyProtocolVersion == tmpVersion) || (defaultProtocolVersion == tmpVersion))
         {
            const BTSProfileVersion avrcpMinimumVersion(0x0100);
            setProfileVersion(entry.serviceInfoList, BTS_SUPP_SRV_AVRCP, avrcpMinimumVersion);
         }

         // ensure that AVRCP bit is set in supported services
         entry.supportedServices.setBit(BTS_SUPP_SRV_AVRCP);
      }

      // check whether version was requested from outside, in case of version was requested we have to check the response, limit to AVP
      if(true == response)
      {
         if(true == entry.avrcpVersionRequested)
         {
            entry.avrcpVersionRequested = false;

            ServiceSearchData partialEntry;
            partialEntry.supportedServices.setBit(BTS_SUPP_SRV_A2DP);
            partialEntry.supportedServices.setBit(BTS_SUPP_SRV_AVRCP);
            BTSServiceInfo info;
            info.service = BTS_SUPP_SRV_A2DP;
            info.version = getProfileVersion(partialEntry.serviceInfoList, BTS_SUPP_SRV_A2DP); // A2DP version is not available, take from dummy list
            partialEntry.serviceInfoList.push_back(info);
            info.service = BTS_SUPP_SRV_AVRCP;
            info.version = getProfileVersion(entry.serviceInfoList, BTS_SUPP_SRV_AVRCP); // take from internal list, version was corrected before (if necessary)
            partialEntry.serviceInfoList.push_back(info);

            createSupportedServicesUpdate(bts2AppMsgList, address, BTS_REQ_PARTIAL_UPDATE, partialEntry, 0, 0);
         }
      }

      // HINT: reset pending get requests information in response case
      if(true == response)
      {
         entry.pendingGetData.resetBit(ServiceSearchData::GET_AVRCP_VERSION);
      }
   }
   else
   {
      FW_ERRMEM_ASSERT_ALWAYS();
      return;
   }

   // check if search sequence is completed
   checkForSearchCompleted(bts2IpcMsgList, bts2AppMsgList, messageItem, address, entry);
}

void ServiceSearch::updateDeviceVendorId(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSIpcCommonErrorCode result, IN const bool response, IN const BTSBDAddress& address, IN const BTSVendorId vendorId)
{
   (void)(bts2IpcMsgList);

   ServiceSearchData& entry = checkServiceSearchList(address);

   entry.didDataAvailable.setBit(ServiceSearchData::VENDOR_ID);

   // HINT: store data in every success case
   // HINT: set record available in every success case
   if(BTS_IPC_SUCCESS == result)
   {
      if(false == _ignoreServiceSearchData)
      {
         entry.serviceRecord.vendorID = vendorId;
      }
      if(entry.didDataAvailable.getData() == entry.didDataAvailable.getDataMask())
      {
         entry.sdpRecordAvailable.setBit(ServiceSearchData::DID_SEARCH_DONE);
         entry.sdpRecordAvailable.setBit(ServiceSearchData::DID_RECORD_AVAILABLE);
         checkDidServiceRecord(entry);
      }
   }

   // HINT: set search done in response case
   // HINT: reset pending get requests information in response case
   if(true == response)
   {
      if(entry.didDataAvailable.getData() == entry.didDataAvailable.getDataMask())
      {
         entry.sdpRecordAvailable.setBit(ServiceSearchData::DID_SEARCH_DONE);
      }
      entry.pendingGetData.resetBit(ServiceSearchData::GET_VENDOR_ID);
   }

   // check if search sequence is completed
   checkForSearchCompleted(bts2IpcMsgList, bts2AppMsgList, messageItem, address, entry);
}

void ServiceSearch::updateDeviceVendorIdSource(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSIpcCommonErrorCode result, IN const bool response, IN const BTSBDAddress& address, IN const BTSVendorIdSource vendorIdSource)
{
   (void)(bts2IpcMsgList);

   ServiceSearchData& entry = checkServiceSearchList(address);

   entry.didDataAvailable.setBit(ServiceSearchData::VENDOR_ID_SOURCE);

   // HINT: store data in every success case
   // HINT: set record available in every success case
   if(BTS_IPC_SUCCESS == result)
   {
      if(false == _ignoreServiceSearchData)
      {
         entry.serviceRecord.vendorIDSource = vendorIdSource;
      }
      if(entry.didDataAvailable.getData() == entry.didDataAvailable.getDataMask())
      {
         entry.sdpRecordAvailable.setBit(ServiceSearchData::DID_SEARCH_DONE);
         entry.sdpRecordAvailable.setBit(ServiceSearchData::DID_RECORD_AVAILABLE);
         checkDidServiceRecord(entry);
      }
   }

   // HINT: set search done in response case
   // HINT: reset pending get requests information in response case
   if(true == response)
   {
      if(entry.didDataAvailable.getData() == entry.didDataAvailable.getDataMask())
      {
         entry.sdpRecordAvailable.setBit(ServiceSearchData::DID_SEARCH_DONE);
      }
      entry.pendingGetData.resetBit(ServiceSearchData::GET_VENDOR_ID_SOURCE);
   }

   // check if search sequence is completed
   checkForSearchCompleted(bts2IpcMsgList, bts2AppMsgList, messageItem, address, entry);
}

void ServiceSearch::updateDeviceProductId(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSIpcCommonErrorCode result, IN const bool response, IN const BTSBDAddress& address, IN const BTSProductId productId)
{
   (void)(bts2IpcMsgList);

   ServiceSearchData& entry = checkServiceSearchList(address);

   entry.didDataAvailable.setBit(ServiceSearchData::PRODUCT_ID);

   // HINT: store data in every success case
   // HINT: set record available in every success case
   if(BTS_IPC_SUCCESS == result)
   {
      if(false == _ignoreServiceSearchData)
      {
         entry.serviceRecord.productID = productId;
      }
      if(entry.didDataAvailable.getData() == entry.didDataAvailable.getDataMask())
      {
         entry.sdpRecordAvailable.setBit(ServiceSearchData::DID_SEARCH_DONE);
         entry.sdpRecordAvailable.setBit(ServiceSearchData::DID_RECORD_AVAILABLE);
         checkDidServiceRecord(entry);
      }
   }

   // HINT: set search done in response case
   // HINT: reset pending get requests information in response case
   if(true == response)
   {
      if(entry.didDataAvailable.getData() == entry.didDataAvailable.getDataMask())
      {
         entry.sdpRecordAvailable.setBit(ServiceSearchData::DID_SEARCH_DONE);
      }
      entry.pendingGetData.resetBit(ServiceSearchData::GET_PRODUCT_ID);
   }

   // check if search sequence is completed
   checkForSearchCompleted(bts2IpcMsgList, bts2AppMsgList, messageItem, address, entry);
}

void ServiceSearch::updateDeviceVersion(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSIpcCommonErrorCode result, IN const bool response, IN const BTSBDAddress& address, IN const BTSVersion version)
{
   (void)(bts2IpcMsgList);

   ServiceSearchData& entry = checkServiceSearchList(address);

   entry.didDataAvailable.setBit(ServiceSearchData::VERSION);

   // HINT: store data in every success case
   // HINT: set record available in every success case
   if(BTS_IPC_SUCCESS == result)
   {
      if(false == _ignoreServiceSearchData)
      {
         entry.serviceRecord.version = version;
      }
      if(entry.didDataAvailable.getData() == entry.didDataAvailable.getDataMask())
      {
         entry.sdpRecordAvailable.setBit(ServiceSearchData::DID_SEARCH_DONE);
         entry.sdpRecordAvailable.setBit(ServiceSearchData::DID_RECORD_AVAILABLE);
         checkDidServiceRecord(entry);
      }
   }

   // HINT: set search done in response case
   // HINT: reset pending get requests information in response case
   if(true == response)
   {
      if(entry.didDataAvailable.getData() == entry.didDataAvailable.getDataMask())
      {
         entry.sdpRecordAvailable.setBit(ServiceSearchData::DID_SEARCH_DONE);
      }
      entry.pendingGetData.resetBit(ServiceSearchData::GET_VERSION);
   }

   // check if search sequence is completed
   checkForSearchCompleted(bts2IpcMsgList, bts2AppMsgList, messageItem, address, entry);
}

void ServiceSearch::updatePbapInfo(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSIpcCommonErrorCode result, IN const bool response, IN const BTSBDAddress& address, IN const BTSPbapSdpRecord& record, IN const bool valid)
{
   (void)(bts2IpcMsgList);

   ServiceSearchData& entry = checkServiceSearchList(address);

   // HINT: store data in every success case
   // HINT: set record available in every success case
   if(BTS_IPC_SUCCESS == result)
   {
      if((true == valid) && (false == _ignoreServiceSearchData))
      {
         entry.pbapRecord = record;
         entry.sdpRecordAvailable.setBit(ServiceSearchData::PBAP_RECORD_AVAILABLE);
         setProfileVersion(entry.serviceInfoList, BTS_SUPP_SRV_PBAP, entry.pbapRecord.version);
      }
      entry.sdpRecordAvailable.setBit(ServiceSearchData::PBAP_SEARCH_DONE);
   }

   // HINT: set search done in response case
   // HINT: reset pending get requests information in response case
   if(true == response)
   {
      entry.sdpRecordAvailable.setBit(ServiceSearchData::PBAP_SEARCH_DONE);
      entry.pendingGetData.resetBit(ServiceSearchData::GET_PBAP_INFO);
   }

   // check if search sequence is completed
   checkForSearchCompleted(bts2IpcMsgList, bts2AppMsgList, messageItem, address, entry);
}

void ServiceSearch::updateMapInfo(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSIpcCommonErrorCode result, IN const bool response, IN const BTSBDAddress& address, IN const BTSMapSdpRecordList& recordList, IN const bool valid)
{
   (void)(bts2IpcMsgList);

   ServiceSearchData& entry = checkServiceSearchList(address);

   // HINT: store data in every success case
   // HINT: set record available in every success case
   if(BTS_IPC_SUCCESS == result)
   {
      if((true == valid) && (false == _ignoreServiceSearchData))
      {
         entry.mapRecords = recordList;
         entry.sdpRecordAvailable.setBit(ServiceSearchData::MAP_RECORD_AVAILABLE);
         // assume that all MAS instances have same profile version
         if(0 < entry.mapRecords.size())
         {
            setProfileVersion(entry.serviceInfoList, BTS_SUPP_SRV_MAP, entry.mapRecords[0].version);
         }
      }
      entry.sdpRecordAvailable.setBit(ServiceSearchData::MAP_SEARCH_DONE);
   }

   // HINT: set search done in response case
   // HINT: reset pending get requests information in response case
   if(true == response)
   {
      entry.sdpRecordAvailable.setBit(ServiceSearchData::MAP_SEARCH_DONE);
      entry.pendingGetData.resetBit(ServiceSearchData::GET_MAP_INFO);
   }

   // check if search sequence is completed
   checkForSearchCompleted(bts2IpcMsgList, bts2AppMsgList, messageItem, address, entry);
}

App2Bts_BaseMessage* ServiceSearch::getApp2BtsWorkingMessage(IN const BTSBDAddress& address)
{
   FW_ERRMEM_IF_NULL_PTR_RETURN_NULL(_controlIf);

   ServiceSearchData& entry = checkServiceSearchList(address);

   return _controlIf->findApp2BtsWorkingMessageWrapper(entry.requestItem.item);
}

ServiceSearchData& ServiceSearch::getDataEntry(IN const BTSBDAddress& address)
{
   return checkServiceSearchList(address);
}

void ServiceSearch::deviceAdded(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address)
{
   // information not needed
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);
   (void)(messageItem);
   (void)(address);
}

void ServiceSearch::deviceRemoved(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address)
{
   // check for ongoing service search
   checkForOngoingSearch(bts2IpcMsgList, bts2AppMsgList, messageItem, address);
}

void ServiceSearch::deviceAvailable(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address)
{
   FW_ERRMEM_IF_NULL_PTR_RETURN(_requestIf);

   BTSBDAddress workingAddress(address);
   ::std::map< BTSBDAddress, ServiceSearchData >::iterator it = getEntry(workingAddress);
   if(_serviceSearchList.end() == it)
   {
      // there is no matching entry => ignore
      return;
   }

   ServiceSearchData& entry = it->second;

   if(true == entry.createPending)
   {
      entry.createPending = false;

      // check if cancel was requested
      if(true == entry.cancelRequested)
      {
         // no search request was sent until now
         // no cancel request was sent until now
         // => send answer for service search

         entry.resultPending = false;
         entry.dataPending = false;

         // search sequence is finished
         handleEndOfFailedSearch(bts2IpcMsgList, bts2AppMsgList, messageItem, 0, entry, workingAddress);
      }
      else
      {
         // start search
         _requestIf->startSearch(bts2IpcMsgList, bts2AppMsgList, workingAddress, entry.type);

         startPageTimeoutTimer(entry);
      }
   }
}

void ServiceSearch::deviceUnavailable(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address)
{
   // information not needed
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);
   (void)(messageItem);
   (void)(address);
}

void ServiceSearch::deviceConnectionStatus(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const BTSStatusTransition aclTransition, IN const bool aclConnected, IN const BTSStatusTransition anyProfileTransition, IN const bool anyProfileConnected)
{
   // information not needed
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);
   (void)(messageItem);
   (void)(aclTransition);

   FW_ERRMEM_IF_NULL_PTR_RETURN(_configurationIf);

   // check for first protocol connected
   if((BTS_STATUS_TRANSITION_ENABLED == anyProfileTransition) && (true == anyProfileConnected))
   {
      // workaround for periodic service search: trigger DID and service search internally after first protocol gets connected
      if(true == _configurationIf->getConfiguration().enableServiceSearchAfterFirstProtocol)
      {
         if(false == isStartRequestPending(address, BTS_SEARCH_DID))
         {
            triggerSdpSearchRequest(address, BTS_SEARCH_DID, false);
         }
         if(false == isStartRequestPending(address, BTS_SEARCH_ALL))
         {
            triggerSdpSearchRequest(address, BTS_SEARCH_ALL, false);
         }
      }
   }

   // find matching service search entries, remember device connection status
   for(::std::map< BTSBDAddress, ServiceSearchData >::iterator it = _serviceSearchList.begin(); it != _serviceSearchList.end(); ++it)
   {
      if(address == it->first)
      {
         ServiceSearchData& entry = it->second;

         if((true == aclConnected) || (true == anyProfileConnected))
         {
            entry.deviceConnected = true;

            // in case of device gets connected a page timeout cannot happen
            entry.pageTimeoutHappened = false;
            stopPageTimeoutTimer(entry);
         }
      }
   }
}

void ServiceSearch::deviceCreationFinished(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const BTSRequestResult result)
{
   if(BTS_REQ_SUCCESS == result)
   {
      return;
   }

   BTSBDAddress workingAddress(address);
   ::std::map< BTSBDAddress, ServiceSearchData >::iterator it = getEntry(workingAddress);
   if(_serviceSearchList.end() == it)
   {
      // there is no matching entry => ignore
      return;
   }

   ServiceSearchData& entry = it->second;

   if(true == entry.createPending)
   {
      entry.createPending = false;
      entry.resultPending = false;
      entry.dataPending = false;

      // search sequence is finished
      handleEndOfFailedSearch(bts2IpcMsgList, bts2AppMsgList, messageItem, 0, entry, workingAddress);
   }
}

void ServiceSearch::deviceRemovalFinished(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const BTSRequestResult result)
{
   // information not needed
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);
   (void)(messageItem);
   (void)(address);
   (void)(result);
}

void ServiceSearch::handleExtendedTimeout(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSTimerId timerId)
{
   // timeout after retry

   FW_ERRMEM_IF_NULL_PTR_RETURN(_requestIf);

   // find matching entry via timer id
   ::std::map< BTSBDAddress, ServiceSearchData >::iterator it;
   for(it = _serviceSearchList.begin(); it != _serviceSearchList.end(); ++it)
   {
      const ServiceSearchData& entry = it->second;

      if((true == entry.retryTimer.compare(timerId)) || (true == entry.pageTimeoutTimer.compare(timerId)))
      {
         // matching timer found
         break;
      }
   }

   if(_serviceSearchList.end() == it)
   {
      // no match
      FW_ERRMEM_ASSERT_ALWAYS();
      return;
   }

   const BTSBDAddress& address = it->first;
   ServiceSearchData& entry = it->second;

   ETG_TRACE_USR3((" handleExtendedTimeout: address=%s", address.c_str()));

   if(true == entry.retryTimer.compare(timerId))
   {
      // mark result as pending (all other meta data is still set)
      entry.resultPending = true;

      // find related application message (to have compare item for service search request)
      if(0 == messageItem.message)
      {
         messageItem.message = getWorkingMessage(entry.requestItem.item);
      }
      FW_ERRMEM_ASSERT(0 != messageItem.message);

      // send service search request
      _requestIf->startSearch(bts2IpcMsgList, bts2AppMsgList, address, entry.type);

      startPageTimeoutTimer(entry);
   }
   else if(true == entry.pageTimeoutTimer.compare(timerId))
   {
      // page timeout happened => set marker
      entry.pageTimeoutHappened = true;
   }
}

ServiceSearchData& ServiceSearch::checkServiceSearchList(IN const BTSBDAddress& address)
{
   ::std::map< BTSBDAddress, ServiceSearchData >::iterator it = getEntry(address);

   if(_serviceSearchList.end() == it)
   {
      // add new entry
      ETG_TRACE_USR3((" checkServiceSearchList: _serviceSearchList.size()=%u", (1 + _serviceSearchList.size())));
      return _serviceSearchList[address];
   }
   else
   {
      // continue with existing entry
      return it->second;
   }
}

::std::map< BTSBDAddress, ServiceSearchData >::iterator ServiceSearch::getEntry(IN const BTSBDAddress& address)
{
   return _serviceSearchList.find(address);
}

::std::map< BTSBDAddress, ServiceSearchData >::const_iterator ServiceSearch::getEntry(IN const BTSBDAddress& address) const
{
   return _serviceSearchList.find(address);
}

ServiceSearchData::SdpRecordAvailableBit ServiceSearch::convertSearchType2SearchDoneBit(IN const BTSSearchType searchType) const
{
   ServiceSearchData::SdpRecordAvailableBit bit = ServiceSearchData::SERVICES_SEARCH_DONE;

   switch(searchType)
   {
      case BTS_SEARCH_ALL:
         bit = ServiceSearchData::SERVICES_SEARCH_DONE;
         break;
      case BTS_SEARCH_SPP:
         bit = ServiceSearchData::SPP_SEARCH_DONE;
         break;
      case BTS_SEARCH_DID:
         bit = ServiceSearchData::DID_SEARCH_DONE;
         break;
      case BTS_SEARCH_PBAP:
         bit = ServiceSearchData::PBAP_SEARCH_DONE;
         break;
      case BTS_SEARCH_MAP:
         bit = ServiceSearchData::MAP_SEARCH_DONE;
         break;
      case BTS_SEARCH_PAN:
         bit = ServiceSearchData::PAN_SEARCH_DONE;
         break;
      default:
         FW_ERRMEM_ASSERT_ALWAYS();
         break;
   }

   return bit;
}

ServiceSearchData::SdpRecordAvailableBit ServiceSearch::convertSearchType2RecordAvailableBit(IN const BTSSearchType searchType) const
{
   ServiceSearchData::SdpRecordAvailableBit bit = ServiceSearchData::SERVICES_RECORD_AVAILABLE;

   switch(searchType)
   {
      case BTS_SEARCH_ALL:
         bit = ServiceSearchData::SERVICES_RECORD_AVAILABLE;
         break;
      case BTS_SEARCH_SPP:
         bit = ServiceSearchData::SPP_RECORD_AVAILABLE;
         break;
      case BTS_SEARCH_DID:
         bit = ServiceSearchData::DID_RECORD_AVAILABLE;
         break;
      case BTS_SEARCH_PBAP:
         bit = ServiceSearchData::PBAP_RECORD_AVAILABLE;
         break;
      case BTS_SEARCH_MAP:
         bit = ServiceSearchData::MAP_RECORD_AVAILABLE;
         break;
      case BTS_SEARCH_PAN:
         bit = ServiceSearchData::PAN_RECORD_AVAILABLE;
         break;
      default:
         FW_ERRMEM_ASSERT_ALWAYS();
         break;
   }

   return bit;
}

BTSProfileVersion ServiceSearch::getProfileVersion(IN const BTSServiceInfoList& infoList, IN const BTSSupportedServicesBit service)
{
   for(::std::vector< BTSServiceInfo >::const_iterator it = infoList.begin(); it != infoList.end(); ++it)
   {
      if(it->service == service)
      {
         return it->version;
      }
   }

   return defaultProtocolVersion;
}

void ServiceSearch::setProfileVersion(IN BTSServiceInfoList& infoList, IN const BTSSupportedServicesBit service, IN const BTSProfileVersion version)
{
   for(::std::vector< BTSServiceInfo >::iterator it = infoList.begin(); it != infoList.end(); ++it)
   {
      if(it->service == service)
      {
         it->version = version;
         return;
      }
   }

   BTSServiceInfo info;
   info.service = service;
   info.version = version;
   infoList.push_back(info);
}

void ServiceSearch::checkForSearchCompleted(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN ServiceSearchData& entry)
{
   /*
    * If we trigger a service search we have to check following:
    * - search result
    * - data update
    *
    * This function is not called in case of search result returns with failed result.
    */

   ETG_TRACE_USR3((" checkForSearchCompleted(): ongoing=%d resultPending=%d dataPending=%d pendingGetData=0x%08X", entry.ongoing, entry.resultPending, entry.dataPending, entry.pendingGetData.getData()));

   if(false == entry.ongoing)
   {
      // search sequence is already finished
      return;
   }

   bool pending = false;

   if(true == entry.resultPending)
   {
      // search result is still pending
      pending = true;
   }

   else if(true == entry.dataPending)
   {
      // data update is still pending
      // check received data
      if(0 == entry.pendingGetData.getData())
      {
         entry.dataPending = false;
      }
      else
      {
         pending = true;
      }
   }

   if(true == pending)
   {
      // search sequence is ongoing
   }
   else
   {
      // search sequence is finished
      entry.ongoing = false;

      // update status and result
      sendStatusAndResult(bts2AppMsgList, address, entry.type, entry.requestItem.user, entry.requestItem.handle, true, (BTSCommonEnumClass)BTS_REQ_SUCCESS, BTS_COMMON_ENUM_CLASS_DEFAULT_VALUE);

      const bool tmpAutomaticSdpSearch(entry.automaticSdpSearch);
      const BTSSearchType tmpType(entry.type);

      // search sequence is finished; keep entry in every case

      // check for internal request
      // note: in case of internal triggered search we can have item request or not => check stored item
      if((true == entry.internal) && (App2BtsOC_StartRemoteServiceSearch != entry.requestItem.item.opCode))
      {
         // in case of internal triggered search we have no user request => fake necessary data
         messageItem.item.opCode = App2BtsOC_StartRemoteServiceSearch;
         messageItem.item.deviceAddress = address;
         messageItem.item.searchType = entry.type;
         messageItem.deleteMessage = true;
      }
      else
      {
         handleActionFinished(messageItem, entry.requestItem.item);
      }

      // reset control data because action is finished
      entry.requestItem.reset();

      // reset data
      entry.automaticSdpSearch = false;
      entry.cancelRequested = false;
      entry.cancelPending = false;
      entry.type = BTS_SEARCH_LAST;
      entry.internal = false;
      entry.deviceConnected = false;

      // stop page timeout timer
      stopTimer(entry.pageTimeoutTimer);
      entry.pageTimeoutHappened = false;
      entry.pageTimeout = 0;

      // inform observer (do not check user or internal flag; check only automaticSdpSearch flag)
      if(false == tmpAutomaticSdpSearch)
      {
         informObserver(bts2IpcMsgList, bts2AppMsgList, messageItem, address, tmpType, BTS_REQ_SUCCESS);
      }
   }
}

void ServiceSearch::checkForOngoingSearch(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address)
{
   ETG_TRACE_USR2((" checkForOngoingSearch: address=%s", address.c_str()));

   // device was removed
   // in case of ongoing service search the search sequence failed

   /*
    * in case of removed device all results/indications are filtered in ServiceSearchGenivi SW part: no impact (only normal assert)
    */

   ::std::map< BTSBDAddress, ServiceSearchData >::iterator it = _serviceSearchList.find(address);

   if(_serviceSearchList.end() == it)
   {
      // no matching entry
      return;
   }

   ServiceSearchData& entry(it->second);

   // check for any ongoing service search
   if(true == entry.ongoing)
   {
      // invalidate supported services information
      entry.supportedServices.setData(0);
      entry.serviceInfoList.clear();
      entry.protocolVersionAvailable.setData(0);
      entry.sdpRecordAvailable.resetBit(ServiceSearchData::SERVICES_SEARCH_DONE);
      entry.sdpRecordAvailable.resetBit(ServiceSearchData::SERVICES_RECORD_AVAILABLE);

      // invalidate DID information
      entry.serviceRecord.vendorID = 0;
      entry.serviceRecord.productID = 0;
      entry.serviceRecord.version = 0;
      entry.serviceRecord.vendorIDSource = 0;
      entry.serviceRecord.clientExecutableURL.clear();
      entry.serviceRecord.serviceDescription.clear();
      entry.serviceRecord.clientDocumentationURL.clear();
      entry.didDataAvailable.setData(0);
      entry.sdpRecordAvailable.resetBit(ServiceSearchData::DID_SEARCH_DONE);
      entry.sdpRecordAvailable.resetBit(ServiceSearchData::DID_RECORD_AVAILABLE);

      // invalidate SPP information
      entry.sspCapabilities.clear();
      entry.sdpRecordAvailable.resetBit(ServiceSearchData::SPP_SEARCH_DONE);
      entry.sdpRecordAvailable.resetBit(ServiceSearchData::SPP_RECORD_AVAILABLE);

      // invalidate PBAP information
      entry.pbapRecord.version = 0;
      entry.pbapRecord.repositories.setData(0);
      entry.pbapRecord.features.setData(0);
      entry.sdpRecordAvailable.resetBit(ServiceSearchData::PBAP_SEARCH_DONE);
      entry.sdpRecordAvailable.resetBit(ServiceSearchData::PBAP_RECORD_AVAILABLE);

      // invalidate MAP information
      entry.mapRecords.clear();
      entry.sdpRecordAvailable.resetBit(ServiceSearchData::MAP_SEARCH_DONE);
      entry.sdpRecordAvailable.resetBit(ServiceSearchData::MAP_RECORD_AVAILABLE);

      // invalidate PAN information
      entry.sdpRecordAvailable.resetBit(ServiceSearchData::PAN_SEARCH_DONE);
      entry.sdpRecordAvailable.resetBit(ServiceSearchData::PAN_RECORD_AVAILABLE);

      if(true == entry.stackSdpSearchActive)
      {
         // Bluetooth stack internal search is ongoing => no status update needed
         // in case of automatic search we have no user request => fake necessary data
         messageItem.item.opCode = App2BtsOC_StartRemoteServiceSearch;
         messageItem.item.deviceAddress = address;
         messageItem.item.searchType = entry.type;
         messageItem.deleteMessage = true;
      }
      else
      {
         // local initiated service search ongoing => update status and result (failed)
         handleEndOfFailedSearch(bts2IpcMsgList, bts2AppMsgList, messageItem, 0, entry, address);
      }

      // search sequence is finished
      entry.ongoing = false;
      entry.dataPending = false;
      entry.resultPending = false;
      // reset control data because action is finished
      entry.requestItem.reset();
      // reset data
      entry.pendingGetData.setData(0);
      entry.stackSdpSearchActive = false;
      entry.automaticSdpSearch = false;
      entry.cancelRequested = false;
      entry.cancelPending = false;
      entry.type = BTS_SEARCH_LAST;
      entry.internal = false;
      entry.deviceConnected = false;
      entry.lastPairingSuccessful = false;
      entry.createPending = false;
      // stop page timeout timer
      stopTimer(entry.pageTimeoutTimer);
      entry.pageTimeoutHappened = false;
      entry.pageTimeout = 0;
      // stop any retry
      entry.retryNmb = 0;
      stopTimer(entry.retryTimer);
   }
}

void ServiceSearch::checkDidServiceRecord(IN ServiceSearchData& entry)
{
   // set general data
   entry.serviceRecord.specificationID = ::fw::getDeviceIdSpecificationId();
   entry.serviceRecord.primaryRecord = true;

   // check received data
   if((::fw::getVendorIdSourceBluetoothSIG() == entry.serviceRecord.vendorIDSource) ||
      (::fw::getVendorIdSourceUSBImplementersForum() == entry.serviceRecord.vendorIDSource) ||
      (::fw::getVendorIdSourceReserved() == entry.serviceRecord.vendorIDSource))
   {
      // assume valid data
   }
   else
   {
      // assume invalid data
      entry.serviceRecord.vendorID = 0;
      entry.serviceRecord.productID = 0;
      entry.serviceRecord.version = 0;
      entry.serviceRecord.vendorIDSource = 0;
      entry.didDataAvailable.setData(0);
      entry.sdpRecordAvailable.resetBit(ServiceSearchData::DID_RECORD_AVAILABLE);
   }
}

unsigned int ServiceSearch::getPendingGetDataBits(IN const BTSSearchType searchType, IN const bool sdpAfterPairing) const
{
   ServiceSearchData::PendingGetData bits;

   if(true == sdpAfterPairing)
   {
      bits.setBit(ServiceSearchData::GET_UUIDS);
      bits.setBit(ServiceSearchData::GET_VENDOR_ID);
      bits.setBit(ServiceSearchData::GET_PRODUCT_ID);
      bits.setBit(ServiceSearchData::GET_VENDOR_ID_SOURCE);
      bits.setBit(ServiceSearchData::GET_VERSION);
      bits.setBit(ServiceSearchData::GET_PBAP_INFO);
      bits.setBit(ServiceSearchData::GET_MAP_INFO);
   }
   else
   {
      switch(searchType)
      {
         case BTS_SEARCH_ALL:
            bits.setBit(ServiceSearchData::GET_UUIDS);
            break;
         case BTS_SEARCH_SPP:
            bits.setBit(ServiceSearchData::GET_UUIDS);
            break;
         case BTS_SEARCH_DID:
            bits.setBit(ServiceSearchData::GET_VENDOR_ID);
            bits.setBit(ServiceSearchData::GET_PRODUCT_ID);
            bits.setBit(ServiceSearchData::GET_VENDOR_ID_SOURCE);
            bits.setBit(ServiceSearchData::GET_VERSION);
            break;
         case BTS_SEARCH_PBAP:
            bits.setBit(ServiceSearchData::GET_PBAP_INFO);
            break;
         case BTS_SEARCH_MAP:
            bits.setBit(ServiceSearchData::GET_MAP_INFO);
            break;
         case BTS_SEARCH_PAN:
            bits.setBit(ServiceSearchData::GET_UUIDS);
            break;
         default:
            FW_ERRMEM_ASSERT_ALWAYS();
            break;
      }
   }

   return bits.getData();
}

void ServiceSearch::resetSearchDoneBits(IN ServiceSearchData::SdpRecordAvailable& recordAvailable) const
{
   recordAvailable.resetBit(ServiceSearchData::SERVICES_SEARCH_DONE);
   recordAvailable.resetBit(ServiceSearchData::SPP_SEARCH_DONE);
   recordAvailable.resetBit(ServiceSearchData::DID_SEARCH_DONE);
   recordAvailable.resetBit(ServiceSearchData::PBAP_SEARCH_DONE);
   recordAvailable.resetBit(ServiceSearchData::MAP_SEARCH_DONE);
}

void ServiceSearch::triggerGetRequests(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address, IN const BTSSearchType searchType, IN const bool sdpAfterPairing, IN ServiceSearchData& entry)
{
   FW_ERRMEM_IF_NULL_PTR_RETURN(_requestIf);

   if(true == sdpAfterPairing)
   {
      if(true == entry.pendingGetData.getBit(ServiceSearchData::GET_UUIDS))
      {
         _requestIf->getDeviceUuids(bts2IpcMsgList, bts2AppMsgList, address);
      }
      if((true == entry.pendingGetData.getBit(ServiceSearchData::GET_VENDOR_ID)) ||
         (true == entry.pendingGetData.getBit(ServiceSearchData::GET_PRODUCT_ID)) ||
         (true == entry.pendingGetData.getBit(ServiceSearchData::GET_VENDOR_ID_SOURCE)) ||
         (true == entry.pendingGetData.getBit(ServiceSearchData::GET_VERSION)))
      {
         entry.pendingGetData.setBit(ServiceSearchData::GET_VENDOR_ID);
         entry.pendingGetData.setBit(ServiceSearchData::GET_PRODUCT_ID);
         entry.pendingGetData.setBit(ServiceSearchData::GET_VENDOR_ID_SOURCE);
         entry.pendingGetData.setBit(ServiceSearchData::GET_VERSION);

         _requestIf->getDeviceVendorId(bts2IpcMsgList, bts2AppMsgList, address);
         _requestIf->getDeviceProductId(bts2IpcMsgList, bts2AppMsgList, address);
         _requestIf->getDeviceVendorIdSource(bts2IpcMsgList, bts2AppMsgList, address);
         _requestIf->getDeviceVersion(bts2IpcMsgList, bts2AppMsgList, address);
      }
      if(true == entry.pendingGetData.getBit(ServiceSearchData::GET_PBAP_INFO))
      {
         _requestIf->getPbapInfo(bts2IpcMsgList, bts2AppMsgList, address);
      }
      if(true == entry.pendingGetData.getBit(ServiceSearchData::GET_MAP_INFO))
      {
         _requestIf->getMapInfo(bts2IpcMsgList, bts2AppMsgList, address);
      }
   }
   else
   {
      switch(searchType)
      {
         case BTS_SEARCH_ALL:
            if(true == entry.pendingGetData.getBit(ServiceSearchData::GET_UUIDS))
            {
               _requestIf->getDeviceUuids(bts2IpcMsgList, bts2AppMsgList, address);
            }
            break;
         case BTS_SEARCH_SPP:
            if(true == entry.pendingGetData.getBit(ServiceSearchData::GET_UUIDS))
            {
               _requestIf->getDeviceUuids(bts2IpcMsgList, bts2AppMsgList, address);
            }
            break;
         case BTS_SEARCH_DID:
            if((true == entry.pendingGetData.getBit(ServiceSearchData::GET_VENDOR_ID)) ||
               (true == entry.pendingGetData.getBit(ServiceSearchData::GET_PRODUCT_ID)) ||
               (true == entry.pendingGetData.getBit(ServiceSearchData::GET_VENDOR_ID_SOURCE)) ||
               (true == entry.pendingGetData.getBit(ServiceSearchData::GET_VERSION)))
            {
               entry.pendingGetData.setBit(ServiceSearchData::GET_VENDOR_ID);
               entry.pendingGetData.setBit(ServiceSearchData::GET_PRODUCT_ID);
               entry.pendingGetData.setBit(ServiceSearchData::GET_VENDOR_ID_SOURCE);
               entry.pendingGetData.setBit(ServiceSearchData::GET_VERSION);

               _requestIf->getDeviceVendorId(bts2IpcMsgList, bts2AppMsgList, address);
               _requestIf->getDeviceProductId(bts2IpcMsgList, bts2AppMsgList, address);
               _requestIf->getDeviceVendorIdSource(bts2IpcMsgList, bts2AppMsgList, address);
               _requestIf->getDeviceVersion(bts2IpcMsgList, bts2AppMsgList, address);
            }
            break;
         case BTS_SEARCH_PBAP:
            if(true == entry.pendingGetData.getBit(ServiceSearchData::GET_PBAP_INFO))
            {
               _requestIf->getPbapInfo(bts2IpcMsgList, bts2AppMsgList, address);
            }
            break;
         case BTS_SEARCH_MAP:
            if(true == entry.pendingGetData.getBit(ServiceSearchData::GET_MAP_INFO))
            {
               _requestIf->getMapInfo(bts2IpcMsgList, bts2AppMsgList, address);
            }
            break;
         case BTS_SEARCH_PAN:
            if(true == entry.pendingGetData.getBit(ServiceSearchData::GET_UUIDS))
            {
               _requestIf->getDeviceUuids(bts2IpcMsgList, bts2AppMsgList, address);
            }
            break;
         default:
            FW_ERRMEM_ASSERT_ALWAYS();
            break;
      }
   }
}

void ServiceSearch::createServiceSearchResult(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address, IN const BTSRequestResult resultServices, IN const BTSRequestResult resultSpp, IN const BTSRequestResult resultDid, IN BtStackIfCallback* user, IN const BTSSessionHandle handle) const
{
   if(0 != user)
   {
      Bts2App_RemoteServiceSearchResult* msg = ptrNew_Bts2App_RemoteServiceSearchResult();
      if(0 != msg)
      {
         msg->setUser(user);
         msg->setSessionHandle(handle);

         msg->setBDAddress(address);
         msg->setSupportedServicesResult(resultServices);
         msg->setSppCapabilitiesResult(resultSpp);
         msg->setDeviceIdServiceRecordsResult(resultDid);

         bts2AppMsgList.push_back(msg);
      }
   }
}

void ServiceSearch::createSupportedServicesUpdate(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address, IN const BTSRequestResult result, IN const ServiceSearchData& entry, IN BtStackIfCallback* ptrUser, IN const BTSSessionHandle sessionHandle) const
{
   createSupportedServicesUpdate(bts2AppMsgList, address, result, &entry, ptrUser, sessionHandle);
}

void ServiceSearch::createSupportedServicesUpdate(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address, IN const BTSRequestResult result, IN const ServiceSearchData* entry, IN BtStackIfCallback* ptrUser, IN const BTSSessionHandle sessionHandle) const
{
   Bts2App_RemoteSupportedServices* msg = ptrNew_Bts2App_RemoteSupportedServices();
   if(0 != msg)
   {
      if(0 == ptrUser)
      {
         msg->setUser(0); // to all
         msg->setSessionHandle(0); // to all
      }
      else
      {
         msg->setUser(ptrUser);
         msg->setSessionHandle(sessionHandle);
      }

      msg->setBDAddress(address);
      msg->setRequestResult(result);

      if(0 != entry)
      {
         msg->setSupportedServices(entry->supportedServices);
         msg->setServiceInfoList(entry->serviceInfoList);
      }

      bts2AppMsgList.push_back(msg);
   }
}

void ServiceSearch::createServiceRecordUpdate(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address, IN const BTSRequestResult result, IN const ServiceSearchData& entry, IN BtStackIfCallback* ptrUser, IN const BTSSessionHandle sessionHandle) const
{
   createServiceRecordUpdate(bts2AppMsgList, address, result, &entry, ptrUser, sessionHandle);
}

void ServiceSearch::createServiceRecordUpdate(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address, IN const BTSRequestResult result, IN const ServiceSearchData* entry, IN BtStackIfCallback* ptrUser, IN const BTSSessionHandle sessionHandle) const
{
   Bts2App_RemoteDeviceIdServiceRecords* msg = ptrNew_Bts2App_RemoteDeviceIdServiceRecords();
   if(0 != msg)
   {
      if(0 == ptrUser)
      {
         msg->setUser(0); // to all
         msg->setSessionHandle(0); // to all
      }
      else
      {
         msg->setUser(ptrUser);
         msg->setSessionHandle(sessionHandle);
      }

      msg->setBDAddress(address);
      msg->setRequestResult(result);

      if(0 != entry)
      {
         BTSDeviceIdServiceRecordList& recordList = msg->getDeviceIdServiceRecordListMutable();
         recordList.push_back(entry->serviceRecord);
      }

      bts2AppMsgList.push_back(msg);
   }
}

void ServiceSearch::createSppCapabilitiesUpdate(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address, IN const BTSRequestResult result, IN const ServiceSearchData& entry, IN BtStackIfCallback* ptrUser, IN const BTSSessionHandle sessionHandle) const
{
   createSppCapabilitiesUpdate(bts2AppMsgList, address, result, &entry, ptrUser, sessionHandle);
}

void ServiceSearch::createSppCapabilitiesUpdate(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address, IN const BTSRequestResult result, IN const ServiceSearchData* entry, IN BtStackIfCallback* ptrUser, IN const BTSSessionHandle sessionHandle) const
{
   Bts2App_RemoteSppCapabilities* msg = ptrNew_Bts2App_RemoteSppCapabilities();
   if(0 != msg)
   {
      if(0 == ptrUser)
      {
         msg->setUser(0); // to all
         msg->setSessionHandle(0); // to all
      }
      else
      {
         msg->setUser(ptrUser);
         msg->setSessionHandle(sessionHandle);
      }

      msg->setBDAddress(address);
      msg->setRequestResult(result);

      if(0 != entry)
      {
         msg->setSppCapabilityList(entry->sspCapabilities);
      }

      bts2AppMsgList.push_back(msg);
   }
}

void ServiceSearch::startPageTimeoutTimer(IN ServiceSearchData& entry)
{
   // start only if page timeout is set
   if(0 == entry.pageTimeout)
   {
      return;
   }

   // start only if page timeout marker is not set
   if(true == entry.pageTimeoutHappened)
   {
      return;
   }

   // start only if device is not connected
   if(true == entry.deviceConnected)
   {
      return;
   }

   // given timeout value has to be reduced to cover internal timer inaccuracy
   const BTSTimeValue timerInaccuracy(500);
   if(entry.pageTimeout <= timerInaccuracy)
   {
      // given timeout value is too small
      return;
   }

   entry.pageTimeout = (BTSTimeValue)(entry.pageTimeout - timerInaccuracy);

   ETG_TRACE_USR2((" startPageTimeoutTimer: timeout=%u", entry.pageTimeout));
   startTimer(entry.pageTimeoutTimer, entry.pageTimeout);
}

void ServiceSearch::stopPageTimeoutTimer(IN ServiceSearchData& entry)
{
   if(true == isTimerActive(entry.pageTimeoutTimer))
   {
      ETG_TRACE_USR2((" stopPageTimeoutTimer: pageTimeoutHappened=%d", entry.pageTimeoutHappened));
      stopTimer(entry.pageTimeoutTimer);
   }
}

bool ServiceSearch::pageTimeoutHappened(IN ServiceSearchData& entry) const
{
   if(true == entry.deviceConnected)
   {
      return false;
   }
   else if(false == entry.pageTimeoutHappened)
   {
      return false;
   }
   else
   {
      ETG_TRACE_USR2((" pageTimeoutHappened"));
      return true;
   }
}

BTSRequestResult ServiceSearch::getSearchResult(IN const BTSRequestResult result, IN const bool pageTimeoutOccurred) const
{
   BTSRequestResult returnResult(result);

   if(true == pageTimeoutOccurred)
   {
      returnResult = BTS_REQ_CONNECT_ACL_FAILED;
   }

   return returnResult;
}

void ServiceSearch::startTimer(IN ExtendedTimerEntry& timer, IN const BTSTimeValue timeout)
{
   ETG_TRACE_USR3((" startTimer: timeout=%u", timeout));

   FW_ERRMEM_IF_NULL_PTR_RETURN(_timerPoolIf);
   FW_ERRMEM_IF_NULL_PTR_RETURN(_controlIf);

   timer.setTimerPool(_timerPoolIf);
   timer.start(_controlIf, this, timeout);
}

void ServiceSearch::stopTimer(IN ExtendedTimerEntry& timer) const
{
   ETG_TRACE_USR3((" stopTimer"));

   timer.stop();

   // do not release timer
}

void ServiceSearch::releaseTimer(IN ExtendedTimerEntry& timer) const
{
   ETG_TRACE_USR3((" releaseTimer"));

   timer.release();
}

bool ServiceSearch::isTimerActive(IN const ExtendedTimerEntry& timer) const
{
   return timer.isActive();
}

bool ServiceSearch::retryStarted(IN const BTSIpcCommonErrorCode errorCode, IN ServiceSearchData& entry, IN const BTSTimeValue timeout)
{
   if(BTS_IPC_BUSY != errorCode)
   {
      return false;
   }

   if(_maxRetry > entry.retryNmb)
   {
      entry.retryNmb++;
      startTimer(entry.retryTimer, timeout);

      ETG_TRACE_USR1((" retryStarted(): retry %u/%u TO=%ums started",
               entry.retryNmb,
               _maxRetry,
               timeout));

      return true;
   }
   else
   {
      ETG_TRACE_USR1((" retryStarted(): retry limit %u/%u reached",
               entry.retryNmb,
               _maxRetry));

      entry.retryNmb = 0;
      stopTimer(entry.retryTimer);

      return false;
   }
}

void ServiceSearch::handleActionFinished(OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSApp2BtsMessageCompareItem& item)
{
   messageItem.item = item;
   messageItem.deleteMessage = true;
   if(0 == messageItem.message)
   {
      messageItem.message = getWorkingMessage(item);
   }
   FW_ERRMEM_ASSERT(0 != messageItem.message);
}

App2Bts_BaseMessage* ServiceSearch::getWorkingMessage(IN const BTSApp2BtsMessageCompareItem& item)
{
   FW_ERRMEM_IF_NULL_PTR_RETURN_NULL(_controlIf);

   return _controlIf->findApp2BtsWorkingMessageWrapper(item);
}

void ServiceSearch::informObserver(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const BTSSearchType searchType, IN const BTSRequestResult result)
{
   for(::std::set< IServiceSearchObserver* >::const_iterator it = _observerList.begin(); it != _observerList.end(); ++it)
   {
      if(0 != *it)
      {
         (*it)->serviceSearchFinished(bts2IpcMsgList, bts2AppMsgList, messageItem, address, searchType, result);
      }
   }
}

void ServiceSearch::handleEndOfFailedSearch(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, OUT bool* connectFailed, INOUT ServiceSearchData& entry, IN const BTSBDAddress& address)
{
   FW_ERRMEM_IF_NULL_PTR_RETURN(_deviceManagerIf);

   entry.ongoing = false;
   entry.dataPending = false;

   const bool pageTimeoutOccurred(pageTimeoutHappened(entry));
   const BTSRequestResult result(getSearchResult(BTS_REQ_FAILED, pageTimeoutOccurred));

   if(0 != connectFailed)
   {
      // check if this was the first connect to the device
      *connectFailed = pageTimeoutOccurred;
   }

   // update status and result
   sendStatusAndResult(bts2AppMsgList, address, entry.type, entry.requestItem.user, entry.requestItem.handle, false, (BTSCommonEnumClass)result, BTS_COMMON_ENUM_CLASS_DEFAULT_VALUE);

   const BTSSearchType tmpType(entry.type);

   // search sequence is finished; keep entry in every case

   // check for internal request
   // note: in case of internal triggered search we can have item request or not => check stored item
   if((true == entry.internal) && (App2BtsOC_StartRemoteServiceSearch != entry.requestItem.item.opCode))
   {
      // in case of internal triggered search we have no user request => fake necessary data
      messageItem.item.opCode = App2BtsOC_StartRemoteServiceSearch;
      messageItem.item.deviceAddress = address;
      messageItem.item.searchType = entry.type;
      messageItem.deleteMessage = true;
   }
   else
   {
      handleActionFinished(messageItem, entry.requestItem.item);
   }

   // reset control data because action is finished
   entry.requestItem.reset();

   // reset data
   entry.automaticSdpSearch = false;
   entry.cancelRequested = false;
   entry.cancelPending = false;
   entry.type = BTS_SEARCH_LAST;
   entry.internal = false;
   entry.deviceConnected = false;

   // stop page timeout timer
   stopTimer(entry.pageTimeoutTimer);
   entry.pageTimeoutHappened = false;
   entry.pageTimeout = 0;

   // inform observer
   informObserver(bts2IpcMsgList, bts2AppMsgList, messageItem, address, tmpType, result);
}

bool ServiceSearch::isUuidAvailableInSdpRecord(IN const ServiceSearchData& entry, IN const BTSUuid& uuid) const
{
   // given UUID is already in lower case format

   BTSUuid tmpUuid;

   for(size_t i = 0; i < entry.sspCapabilities.size(); i++)
   {
      const BTSUuidList& uuidList(entry.sspCapabilities[i].uuidList);

      for(size_t j = 0; j < uuidList.size(); j++)
      {
         tmpUuid = uuidList[j];
         ::fw::convertString2LowerCase(tmpUuid);
         if(uuid == tmpUuid)
         {
            return true;
         }
      }
   }

   return false;
}

bool ServiceSearch::isStartRequestPending(IN const BTSBDAddress& address, IN const BTSSearchType searchType) const
{
   FW_ERRMEM_IF_NULL_PTR_RETURN_FALSE(_controlIf);

   BTSApp2BtsMessageMasterCompareItem item;
   ::std::vector< BTSApp2BtsMessageMasterCompareItem > itemList;

   item.deviceAddress = address;
   item.searchType = searchType;

   item.compareDeviceAddress = true;
   item.compareSearchType = true;

   item.opCode = App2BtsOC_StartRemoteServiceSearch;
   itemList.push_back(item);

   // check if opcode is in waiting queue
   return _controlIf->isSimilarOpCodeInWaitingQueue(itemList);
}

bool ServiceSearch::getAnyProfileConnected(IN const BTSBDAddress& address) const
{
   FW_ERRMEM_IF_NULL_PTR_RETURN_FALSE(_protocolManagerIf);

   ::std::vector< BTSProtocolId > connectedProtocols;
   connectedProtocols.reserve((unsigned int)(BTS_PROTO_LAST) << 1);

   _protocolManagerIf->getConnectedProtocols(connectedProtocols, address);

   for(::std::vector< BTSProtocolId >::const_iterator it = connectedProtocols.begin(); it != connectedProtocols.end(); ++it)
   {
      if((BTS_PROTO_HFP == *it) ||
         (BTS_PROTO_AVP == *it) ||
         (BTS_PROTO_DUN == *it) ||
         (BTS_PROTO_PAN == *it) ||
         (BTS_PROTO_SPP == *it))
      {
         return true;
      }
   }

   return false;
}

bool ServiceSearch::isUuidInList(IN const BTSUuid& uuid, IN const BTSSppCapabilityList& capabilityList) const
{
   for(::std::vector< BTSSppCapability >::const_iterator it1 = capabilityList.begin(); it1 != capabilityList.end(); ++it1)
   {
      const BTSUuidList& uuidList(it1->uuidList);

      for(::std::vector< BTSUuid >::const_iterator it2 = uuidList.begin(); it2 != uuidList.end(); ++it2)
      {
         if(uuid == *it2)
         {
            return true;
         }
      }
   }

   return false;
}

} //btstackif
