/**
 * @file ServiceSearchData.h
 *
 * @par SW-Component
 * State machine for service search
 *
 * @brief Implementation of service search data.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Header file for implementation of service search data.
 */

#ifndef _SERVICE_SEARCH_DATA_H_
#define _SERVICE_SEARCH_DATA_H_

#include "BtStackInternalTypes.h"
#include "ExtendedTimerEntry.h"

namespace btstackif {

/**
 * Service search data class.
 */
class ServiceSearchData
{
public:
   /**
    * Type definition for DID information available.
    * #DidDataAvailableBit#ignore#
    */
   enum DidDataAvailableBit
   {
      VENDOR_ID = 0, /**< vendor id available */
      PRODUCT_ID, /**< product id available */
      VENDOR_ID_SOURCE, /**< vendor id source available */
      VERSION, /**< version available */
      // -------------------------------------------------------------------------
      DDA_LAST /**< end marker, must be the last one */
   };

   /**
    * Type definition for device data available.
    */
   typedef BTSBitfield<unsigned char, DidDataAvailableBit, DDA_LAST> DidDataAvailable;

   /**
    * Bit definition for finished SDP search and available SDP record.
    * #SdpRecordAvailableBit#ignore#
    */
   enum SdpRecordAvailableBit
   {
      SERVICES_SEARCH_DONE = 0, /**< flag for finished service search */
      SERVICES_RECORD_AVAILABLE, /**< flag for available record */
      DID_SEARCH_DONE, /**< flag for finished service search */
      DID_RECORD_AVAILABLE, /**< flag for available record */
      SPP_SEARCH_DONE, /**< flag for finished service search */
      SPP_RECORD_AVAILABLE, /**< flag for available record */
      PBAP_SEARCH_DONE, /**< flag for finished service search */
      PBAP_RECORD_AVAILABLE, /**< flag for available record */
      MAP_SEARCH_DONE, /**< flag for finished service search */
      MAP_RECORD_AVAILABLE, /**< flag for available record */
      PAN_SEARCH_DONE, /**< flag for finished service search */
      PAN_RECORD_AVAILABLE, /**< flag for available record */
      // -------------------------------------------------------------------------
      SDP_LAST /**< end marker, must be the last one */
   };

   /**
    * Type definition for finished SDP search and available SDP record.
    */
   typedef BTSBitfield<unsigned int, SdpRecordAvailableBit, SDP_LAST> SdpRecordAvailable;

   /**
    * Bit definition for pending get data requests.
    * #PendingGetDataBit#ignore#
    */
   enum PendingGetDataBit
   {
      GET_UUIDS = 0, /**< flag for get uuids (supported services and SPP) */
      GET_VENDOR_ID, /**< flag for get vendor id */
      GET_PRODUCT_ID, /**< flag for get product id */
      GET_VENDOR_ID_SOURCE, /**< flag for get vendor id source */
      GET_VERSION, /**< flag for get version */
      GET_PBAP_INFO, /**< flag for get pbap info */
      GET_MAP_INFO, /**< flag for get map info */
      GET_HFP_VERSION, /**< flag for get hfp version */
      GET_AVRCP_VERSION, /**< flag for get avrcp version */
      // -------------------------------------------------------------------------
      GET_LAST /**< end marker, must be the last one */
   };

   /**
    * Type definition for pending get data requests.
    */
   typedef BTSBitfield<unsigned int, PendingGetDataBit, GET_LAST> PendingGetData;

   /**
    * Default constructor.
    */
   ServiceSearchData();

   /**
    * Destructor.
    */
   virtual ~ServiceSearchData();

   /**
    * Copy constructor.
    *
    * @param[in] ref: reference of message to be copied
    */
   ServiceSearchData(const ServiceSearchData& ref);

   /**
    * Assignment operator.
    *
    * @param[in] ref: reference of message to be assigned
    */
   ServiceSearchData& operator=(const ServiceSearchData& ref);

   /**
    * Equality operator.
    *
    * @param[in] ref: reference of message to be compared
    */
   bool operator==(const ServiceSearchData& ref) const;

   /**
    * Inequality operator.
    *
    * @param[in] ref: reference of message to be compared
    */
   bool operator!=(const ServiceSearchData& ref) const;

   // member with public access
   SdpRecordAvailable sdpRecordAvailable; /**< SDP record availability information */
   PendingGetData pendingGetData; /**< pending get requests information */
   BTSSupportedServices supportedServices; /**< supported services (bit field) */
   BTSServiceInfoList serviceInfoList; /**< service information list including protocol version number */
   BTSSupportedServices protocolVersionAvailable; /**< bit field indicating availability of protocol version */
   BTSDeviceIdServiceRecord serviceRecord; /**< DID service record */
   DidDataAvailable didDataAvailable; /**< DID availability information */
   BTSSppCapabilityList sspCapabilities; /**< SPP capabilities */
   BTSPbapSdpRecord pbapRecord; /**< PBAP SDP record */
   BTSMapSdpRecordList mapRecords; /**< MAP SDP records */
   bool stackSdpSearchActive; /**< flag for SDP search started by Bluetooth stack */
   bool automaticSdpSearch; /**< flag for automatic SDP search (stack SDP search including get for all properties) */
   bool ongoing; /**< flag for ongoing search */
   bool resultPending; /**< flag for pending result */
   bool dataPending; /**< flag for pending data */
   bool cancelRequested; /**< flag for requested cancel */
   bool cancelPending; /**< flag for sent cancel request (pending cancel result) */
   bool createPending; /**< flag for sent create request (pending device available) */
   BTSSearchType type; /**< search type */
   bool internal; /**< flag indicating internal/external request */
   unsigned int retryNmb; /**< current retry number */
   BTSApp2BtsMessageRequestItem requestItem; /**< request item */
   ExtendedTimerEntry retryTimer; /**< retry timer */
   ExtendedTimerEntry pageTimeoutTimer; /**< page timeout timer */
   bool pageTimeoutHappened; /**< flag indicating that page timeout happened (timer expired) */
   BTSTimeValue pageTimeout; /**< page timeout */
   bool deviceConnected; /**< flag indicating that device is/was connected */
   bool lastPairingSuccessful; /**< flag indicating that last pairing with device was successful */
   bool avrcpVersionRequested; /**< flag indicating that AVRCP version was requested from outside */
};

} //btstackif

#endif //_SERVICE_SEARCH_DATA_H_
