/**
 * @file IServiceSearchCallback.h
 *
 * @par SW-Component
 * State machine for service search
 *
 * @brief Callback interface definition for service search state machine.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details This file defines the callback interface to be used by specific service search state machine.
 *          Callback interface must be provided by generic service search state machine.
 */

#ifndef _I_SERVICE_SEARCH_CALLBACK_H_
#define _I_SERVICE_SEARCH_CALLBACK_H_

#include "BtStackInternalTypes.h"

namespace btstackif {

// class forward declarations
class Bts2Ipc_BaseMessage;
class Bts2App_BaseMessage;
class App2Bts_BaseMessage;
class ServiceSearchData;

/**
 * Interface definition.
 */
class IServiceSearchCallback
{
public:
   /**
    * Destructor.
    */
   virtual ~IServiceSearchCallback() {}

   /**
    * Result for service search.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[out] connectFailed: flag for ACL connect failed
    * @param[in] address: device address
    * @param[in] result: result
    */
   virtual void serviceSearchResult(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, OUT bool& connectFailed, IN const BTSBDAddress& address, IN const BTSIpcCommonErrorCode result) = 0;

   /**
    * Result for cancel service search.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] address: device address
    * @param[in] result: result
    */
   virtual void cancelServiceSearchResult(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const BTSIpcCommonErrorCode result) = 0;

   /**
    * Setter for active SDP search after successful pairing (automatic search started by Bluetooth stack).
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] address: device address
    * @param[in] enable: enable flag
    * @param[in] success: success flag
    */
   virtual void setSdpSearchActiveAfterPairing(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const bool enable, IN const bool success) = 0;

   /**
    * Start SDP search after pairing (BtStackIf internal search).
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] address: device address
    * @param[in] success: success flag (result)
    *
    * @return = true: manual SDP search was triggered,
    * @return = false: no SDP search was triggered
    */
   virtual bool startSdpSearchAfterPairing(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const bool success) = 0;

   /**
    * Start SDP search after pairing (BtStackIf internal search).
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] address: device address
    *
    * @return = true: manual SDP search was triggered,
    * @return = false: no SDP search was triggered
    */
   virtual bool startSdpSearchAfterPairing(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address) = 0;

   /**
    * Set current pairing status.
    *
    * @param[in] address: device address
    * @param[in] success: pairing status
    */
   virtual void setPairingStatus(IN const BTSBDAddress& address, IN const bool success) = 0;

   /**
    * Update uuids data.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] result: result (needed for get requests)
    * @param[in] response: response flag
    * @param[in] address: device address
    * @param[in] supportedServices: supported services
    * @param[in] uuidList: list with UUIDs (SPP capabilities)
    */
   virtual void updateDeviceUuids(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSIpcCommonErrorCode result, IN const bool response, IN const BTSBDAddress& address, IN const BTSSupportedServices& supportedServices, IN const BTSUuidList& uuidList) = 0;

   /**
    * Update protocol version.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] result: result (needed for get requests)
    * @param[in] response: response flag
    * @param[in] address: device address
    * @param[in] protocolId: protocol id
    * @param[in] version: protocol version
    */
   virtual void updateServiceVersion(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSIpcCommonErrorCode result, IN const bool response, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSProfileVersion version) = 0;

   /**
    * Update vendor id (DID).
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] result: result (needed for get requests)
    * @param[in] response: response flag
    * @param[in] address: device address
    * @param[in] vendorId: vendor id
    */
   virtual void updateDeviceVendorId(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSIpcCommonErrorCode result, IN const bool response, IN const BTSBDAddress& address, IN const BTSVendorId vendorId) = 0;

   /**
    * Update vendor id source (DID).
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] result: result (needed for get requests)
    * @param[in] response: response flag
    * @param[in] address: device address
    * @param[in] vendorIdSource: vendor id source
    */
   virtual void updateDeviceVendorIdSource(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSIpcCommonErrorCode result, IN const bool response, IN const BTSBDAddress& address, IN const BTSVendorIdSource vendorIdSource) = 0;

   /**
    * Update product id (DID).
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] result: result (needed for get requests)
    * @param[in] response: response flag
    * @param[in] address: device address
    * @param[in] productId: product id
    */
   virtual void updateDeviceProductId(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSIpcCommonErrorCode result, IN const bool response, IN const BTSBDAddress& address, IN const BTSProductId productId) = 0;

   /**
    * Update version (DID).
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] result: result (needed for get requests)
    * @param[in] response: response flag
    * @param[in] address: device address
    * @param[in] version: version
    */
   virtual void updateDeviceVersion(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSIpcCommonErrorCode result, IN const bool response, IN const BTSBDAddress& address, IN const BTSVersion version) = 0;

   /**
    * Update PBAP info.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] result: result (needed for get requests)
    * @param[in] response: response flag
    * @param[in] address: device address
    * @param[in] record: SDP record
    * @param[in] valid: valid flag
    */
   virtual void updatePbapInfo(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSIpcCommonErrorCode result, IN const bool response, IN const BTSBDAddress& address, IN const BTSPbapSdpRecord& record, IN const bool valid) = 0;

   /**
    * Update MAP info.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] result: result (needed for get requests)
    * @param[in] response: response flag
    * @param[in] address: device address
    * @param[in] recordList: SDP record list
    * @param[in] valid: valid flag
    */
   virtual void updateMapInfo(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSIpcCommonErrorCode result, IN const bool response, IN const BTSBDAddress& address, IN const BTSMapSdpRecordList& recordList, IN const bool valid) = 0;

   /**
    * Get App2Bts message.
    *
    * @param[in] address: device address
    *
    * @return = App2Bts message if found else NULL
    */
   virtual App2Bts_BaseMessage* getApp2BtsWorkingMessage(IN const BTSBDAddress& address) = 0;

   /**
    * Get service search data entry.
    *
    * @param[in] address: device address
    *
    * @return = device entry
    */
   virtual ServiceSearchData& getDataEntry(IN const BTSBDAddress& address) = 0;
};

} //btstackif

#endif //_I_SERVICE_SEARCH_CALLBACK_H_
