/**
 * @file IRssi.h
 *
 * @par SW-Component
 * State machine for RSSI
 *
 * @brief Interface definition for generic RSSI state machine.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details This file defines the generic interface for RSSI + link quality state machine.
 */

#ifndef _I_RSSI_H_
#define _I_RSSI_H_

#include "BtStackInternalTypes.h"

namespace btstackif {

// class forward declarations
class IRssiRequest;
class IBasicControl;
class IDeviceManager;
class IStateMachine;
class App2Bts_RequestLinkQuality;
class Bts2Ipc_BaseMessage;
class Bts2App_BaseMessage;

/**
 * Interface definition.
 */
class IRssi
{
public:
   /**
    * Destructor.
    */
   virtual ~IRssi() {}

   /**
    * Set specific instance (implementation).
    *
    * @param[in] instance: instance (implementation)
    */
   virtual void setInstance(IN IRssiRequest* instance) = 0;

   /**
    * Set basic control interface.
    *
    * @param[in] control: basic control interface
    */
   virtual void setControlIf(IN IBasicControl* control) = 0;

   /**
    * Set device manager interface.
    *
    * @param[in] deviceManager: device manager interface
    */
   virtual void setDeviceManagerIf(IN IDeviceManager* deviceManager) = 0;

   /**
    * Get SM entry interface.
    *
    * @return = SM entry interface
    */
   virtual IStateMachine* getSmEntryInterface(void) = 0;

   /**
    * Send result for given control data.
    * <BR><B>This method shall be used to reject application requests or handle doubled application requests.</B>
    *
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[in] request: request
    * @param[in] resultCode: result code
    */
   virtual void sendResult(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const App2Bts_RequestLinkQuality& request, IN const BTSCommonEnumClass resultCode) const = 0;

   /**
    * Check if given request is valid.
    *
    * @param[in] request: request
    *
    * @return = true: valid request,
    * @return = false: invalid request
    */
   virtual bool isValidRequest(IN const App2Bts_RequestLinkQuality& request) const = 0;

   /**
    * Get link quality.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[in] request: request
    *
    * @return = true: request processing was started (request shall not be deleted),
    * @return = false: no further processing necessary (request can be deleted)
    */
   virtual bool getLinkQuality(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const App2Bts_RequestLinkQuality& request) = 0;
};

} //btstackif

#endif //_I_RSSI_H_
