/**
 * @file RestrictedPairingConnecting.cpp
 *
 * @par SW-Component
 * State machine for restricted pairing and connecting
 *
 * @brief Implementation of generic restricted pairing and connecting state machine.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Source file for implementation of generic restricted pairing and connecting state machine.
 */

#include "RestrictedPairingConnecting.h"
#include "FwAssert.h"
#include "FwStringUtils.h"
#include "FwBluetoothStringUtils.h"
#include "App2Bts_MessageWrapper.h"
#include "Bts2App_MessageWrapper.h"

namespace btstackif {

RestrictedPairingConnecting::RestrictedPairingConnecting() :
_restrictedPairingSupported(true), // supported
_restrictedConnectingSupported(true), // supported
_address()
{
}

RestrictedPairingConnecting::~RestrictedPairingConnecting()
{
}

void RestrictedPairingConnecting::reset(void)
{
   StateMachine::reset();
   _address.clear();
}

void RestrictedPairingConnecting::forceInitialState(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList)
{
   // check current restriction
   if(false == _address.empty())
   {
      // restriction active => reset
      _address.clear();
      createStatusMsg(bts2AppMsgList, 0, 0, _address);
   }

   // reset control data
   reset();
}

IStateMachine* RestrictedPairingConnecting::getSmEntryInterface(void)
{
   return this;
}

void RestrictedPairingConnecting::setRestrictionAndSendStatus(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const App2Bts_SetDeviceForRestrictedPairingConnecting& request)
{
   // use setRestriction() because there is no message to be sent to stack or any other action necessary
   ::std::vector< Bts2Ipc_BaseMessage* > bts2IpcMsgList;
   (void)setRestriction(bts2IpcMsgList, bts2AppMsgList, request);
}

bool RestrictedPairingConnecting::isValidRequest(IN const App2Bts_SetDeviceForRestrictedPairingConnecting& request) const
{
   return ::fw::isValidBdAddress(request.getBDAddress(), false, true); // empty address is allowed
}

bool RestrictedPairingConnecting::setRestriction(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const App2Bts_SetDeviceForRestrictedPairingConnecting& request)
{
   (void)(bts2IpcMsgList);

   BTSBDAddress workingAddress(request.getBDAddress());
   ::fw::convertString2LowerCase(workingAddress);

   if(false == isValidRequest(request))
   {
      createStatusMsg(bts2AppMsgList, request.getUser(), request.getSessionHandle(), _address);

      return false;
   }

   // store data
   _address = workingAddress;

   createStatusMsg(bts2AppMsgList, 0, 0, _address);

   return false; // request is finished now
}

bool RestrictedPairingConnecting::isPairingRestricted(OUT BTSBDAddress& address) const
{
   if((true == _restrictedPairingSupported) && (false == _address.empty()))
   {
      address = _address;

      return true;
   }
   else
   {
      return false;
   }
}

bool RestrictedPairingConnecting::isConnectingRestricted(OUT BTSBDAddress& address) const
{
   if((true == _restrictedConnectingSupported) && (false == _address.empty()))
   {
      address = _address;

      return true;
   }
   else
   {
      return false;
   }
}

void RestrictedPairingConnecting::createStatusMsg(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN BtStackIfCallback* user, IN const BTSSessionHandle handle, IN const BTSBDAddress& address) const
{
   Bts2App_CurrentDeviceForRestrictedPairingConnecting* msg = ptrNew_Bts2App_CurrentDeviceForRestrictedPairingConnecting();
   if(0 != msg)
   {
      if(0 == user)
      {
         msg->setUser(0); // send status message to all
         msg->setSessionHandle(0); // send status message to all
      }
      else
      {
         msg->setUser(user);
         msg->setSessionHandle(handle);
      }
      msg->setBDAddress(address); // can be empty

      bts2AppMsgList.push_back(msg);
   }
}

} //btstackif
