/**
 * @file ProtocolDisconnect.cpp
 *
 * @par SW-Component
 * State machine for protocol manager
 *
 * @brief Implementation of generic protocol disconnect state machine.
 *
 * @copyright (C) 2017 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Source file for implementation of generic protocol disconnect state machine.
 */

#include "ProtocolDisconnect.h"
#include "IProtocolManagerRequest.h"
#include "IProtocolSmHelper.h"
#include "FwErrmemPrint.h"
#include "TraceClasses.h"
#include "FwTrace.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_BTS_SM
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/ProtocolDisconnect.cpp.trc.h"
#endif
#endif

namespace btstackif {

ProtocolDisconnect::ProtocolDisconnect() :
_protocolManagerRequest(0),
_protocolSmHelper(0),
_address(),
_protocol(BTS_PROTO_LAST),
_sppInstance(0),
_uuid(),
_masInstanceName(),
_masInstanceId(0),
_pauseBtStreaming(false),
_cancel(false),
_final(false),
_disconnecting(false),
_result(BTS_REQ_LAST),
_status(BTS_REQ_LAST),
_smProcessingActive(false),
_testBlockSendingDisconnect(false)
{
}

ProtocolDisconnect::~ProtocolDisconnect()
{
   _protocolManagerRequest = 0;
   _protocolSmHelper = 0;
}

//===================================================================================================================

void ProtocolDisconnect::setProtocolManagerRequest(IN IProtocolManagerRequest* manager)
{
   _protocolManagerRequest = manager;

   FW_ERRMEM_ASSERT(0 != _protocolManagerRequest);
}

void ProtocolDisconnect::setProtocolSmHelper(IN IProtocolSmHelper* helper)
{
   _protocolSmHelper = helper;

   FW_ERRMEM_ASSERT(0 != _protocolSmHelper);
}

void ProtocolDisconnect::resetStateMachine(void)
{
   ETG_TRACE_USR3((" [SM]: resetSm()"));

   resetSm();
}

void ProtocolDisconnect::sendDisconnectEvent(void)
{
   ETG_TRACE_USR3((" [SM]: sendDisconnectEvent()"));

   FW_ERRMEM_ASSERT(0 == SendEvent(DISCONNECT, 0));
}

void ProtocolDisconnect::sendCancelEvent(void)
{
   ETG_TRACE_USR3((" [SM]: sendCancelEvent()"));

   FW_ERRMEM_ASSERT(0 == SendEvent(CANCEL, 0));
}

void ProtocolDisconnect::sendBusyEvent(void)
{
   ETG_TRACE_USR3((" [SM]: sendBusyEvent()"));

   FW_ERRMEM_ASSERT(0 == SendEvent(BUSY, 0));
}

void ProtocolDisconnect::sendDisconnectingEvent(void)
{
   ETG_TRACE_USR3((" [SM]: sendDisconnectingEvent()"));

   FW_ERRMEM_ASSERT(0 == SendEvent(DISCONNECTING, 0));
}

void ProtocolDisconnect::sendDisconnectResultEvent(const BTSRequestResult result)
{
   ETG_TRACE_USR3((" [SM]: sendDisconnectResultEvent(): result=%d", result));

   char parameters[getMarshalIntSize()];
   FW_ERRMEM_ASSERT(0 == ParameterDISCONNECT_RESULT(parameters, sizeof(parameters), result));
   FW_ERRMEM_ASSERT(0 == SendEvent(DISCONNECT_RESULT, parameters));
}

void ProtocolDisconnect::sendFinalStateEvent(const BTSRequestResult result)
{
   ETG_TRACE_USR3((" [SM]: sendFinalStateEvent(): result=%d", result));

   char parameters[getMarshalIntSize()];
   FW_ERRMEM_ASSERT(0 == ParameterFINAL_STATE(parameters, sizeof(parameters), result));
   FW_ERRMEM_ASSERT(0 == SendEvent(FINAL_STATE, parameters));
}

void ProtocolDisconnect::setConnectionData(IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceName& masInstanceName, IN const BTSMasInstanceId masInstanceId, IN const bool pauseBtStreaming)
{
   ETG_TRACE_USR3((" [SM]: setConnectionData()"));

   _address = address;
   _protocol = protocol;
   _sppInstance = sppInstance;
   _uuid = uuid;
   _masInstanceName = masInstanceName;
   _masInstanceId = masInstanceId;
   _pauseBtStreaming = pauseBtStreaming;
}

bool ProtocolDisconnect::doDisconnectSmProcessing(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN BtStackIfCallback* user /*= 0*/, IN const BTSSessionHandle handle /*= 0*/)
{
   // avoid recursion
   if(true == _smProcessingActive)
   {
      return false;
   }
   else
   {
      _smProcessingActive = true;
      const bool result(doSmProcessing(bts2IpcMsgList, bts2AppMsgList, messageItem, user, handle));
      _smProcessingActive = false;
      return result;
   }
}

void ProtocolDisconnect::setTestBlockSendingDisconnect(IN const bool enable)
{
   _testBlockSendingDisconnect = enable;
}

//===================================================================================================================

int ProtocolDisconnect::entryBusy()
{
   ETG_TRACE_USR3((" [SM]: entryBusy()"));

   FW_ERRMEM_IF_NULL_PTR_RETURN_NULL(_protocolSmHelper);

   /*
    * status: 100%
    *
    * triggered by BUSY event
    * check if retry is allowed
    * if yes start timer
    * if not send DISCONNECT_RESULT event to end the sequence
    */

   if(true == _protocolSmHelper->isRetryAllowed(_address, _protocol, _uuid, _masInstanceId))
   {
      _protocolSmHelper->startRetryTimer(_address, _protocol, _uuid, _masInstanceId);
   }
   else
   {
      sendDisconnectResultEvent(BTS_REQ_FAILED);
   }

   return 0;
}

int ProtocolDisconnect::entryCheckForRetry()
{
   ETG_TRACE_USR3((" [SM]: entryCheckForRetry()"));

   FW_ERRMEM_IF_NULL_PTR_RETURN_NULL(_protocolSmHelper);

   /*
    * status: 100%
    *
    * if status is set to success then disconnect was successful
    * check if retry shall be executed
    * general precondition for retry:
    * - not canceled
    * - not disconnected
    * - max retry not reached (retry allowed)
    * - disconnect timer still active
    *
    * increase retry counter
    * reset internal members
    */

   if(BTS_REQ_SUCCESS == _status)
   {
      _result = BTS_REQ_SUCCESS;
   }

   if((true == _cancel) && (BTS_REQ_SUCCESS != _result))
   {
      _result = BTS_REQ_DISCONNECT_ABORTED;
   }

   bool goToEnd(true);

   if((false == _cancel) &&
      (BTS_REQ_SUCCESS != _result) &&
      (true == _protocolSmHelper->isRetryAllowed(_address, _protocol, _uuid, _masInstanceId)) &&
      (true == _protocolSmHelper->isConnectTimerActive(_address, _protocol, _uuid, _masInstanceId, false)))
   {
      // start retry
      goToEnd = false;

      ETG_TRACE_USR1((" [SM]: entryCheckForRetry(): retry"));
   }

   // exit or retry?
   if(true == goToEnd)
   {
      sendExitNoRetryEvent();
   }
   else
   {
      // reset internal members
      _final = false;
      _disconnecting = false;
      _result = BTS_REQ_LAST;
      _status = BTS_REQ_LAST;

      _protocolSmHelper->increaseRetryCounter(_address, _protocol, _uuid, _masInstanceId);

      sendRetryDisconnectEvent();
   }

   return 0;
}

int ProtocolDisconnect::exitBusy()
{
   ETG_TRACE_USR3((" [SM]: exitBusy()"));

   FW_ERRMEM_IF_NULL_PTR_RETURN_NULL(_protocolSmHelper);

   /*
    * status: 100%
    *
    * triggered by DISCONNECT_RESULT, FINAL_STATE or DISCONNECT event
    * stop timer
    */

   _protocolSmHelper->stopRetryTimer(_address, _protocol, _uuid, _masInstanceId);

   return 0;
}

int ProtocolDisconnect::handleBusyCanceled(const BTSRequestResult result)
{
   ETG_TRACE_USR3((" [SM]: handleBusyCanceled()"));

   /*
    * status: 100%
    *
    * triggered by DISCONNECT_RESULT event
    * store result
    * handle end of sequence
    */

   _result = result;

   handleDisconnectFinished();

   return 0;
}

int ProtocolDisconnect::handleCancel()
{
   ETG_TRACE_USR3((" [SM]: handleCancel()"));

   /*
    * status: 100%
    *
    * triggered by CANCEL event
    * store cancel event
    */

   _cancel = true;

   return 0;
}

int ProtocolDisconnect::handleCancelDuringBusy()
{
   ETG_TRACE_USR3((" [SM]: handleCancelDuringBusy()"));

   FW_ERRMEM_IF_NULL_PTR_RETURN_NULL(_protocolManagerRequest);
   FW_ERRMEM_IF_NULL_PTR_RETURN_NULL(_protocolSmHelper);

   /*
    * status: 100%
    *
    * triggered by CANCEL event
    * store cancel event
    * create virtual failed disconnect result message to end the sequence in context of Ipc2Bts message processing
    * stop timer
    */

   _cancel = true;

   ::std::vector< Bts2App_BaseMessage* > _tempBts2AppList;
   _protocolManagerRequest->sendVirtualFailedDisconnectResult(_bts2IpcList, _tempBts2AppList, _address, _protocol, _sppInstance, _uuid, _masInstanceId, _masInstanceName, BTS_IPC_RETRY_ABORTED);
   FW_ERRMEM_ASSERT(0 == _tempBts2AppList.size());

   _protocolSmHelper->stopRetryTimer(_address, _protocol, _uuid, _masInstanceId);

   return 0;
}

int ProtocolDisconnect::handleDisconnectCompleted(const BTSRequestResult result)
{
   ETG_TRACE_USR3((" [SM]: handleDisconnectCompleted(): result=%d", result));

   /*
    * status: 100%
    *
    * triggered by FINAL_STATE event
    * if called: triggered by final connection state update (connected/disconnected)
    * store result
    * handle end of sequence
    */

   _status = result;

   handleDisconnectFinished();

   return 0;
}

int ProtocolDisconnect::handleDisconnectFinalState(const BTSRequestResult result)
{
   ETG_TRACE_USR3((" [SM]: handleDisconnectFinalState(): result=%d", result));

   /*
    * status: 100%
    *
    * triggered by TRUE (isFinalStateReached) event
    * if called: triggered by disconnect result; final connection state was updated before (stored in _result)
    * ignore result
    * handle end of sequence
    */

   (void)(result);

   handleDisconnectFinished();

   return 0;
}

int ProtocolDisconnect::handleDisconnecting()
{
   ETG_TRACE_USR3((" [SM]: handleDisconnecting()"));

   /*
    * status: 100%
    *
    * triggered by DISCONNECTING event
    * disconnecting state was indicated
    */

   _disconnecting = true;

   return 0;
}

int ProtocolDisconnect::handleDisconnectingCanceled()
{
   ETG_TRACE_USR3((" [SM]: handleDisconnectingCanceled()"));

   /*
    * status: 100%
    *
    * triggered by CANCEL or TRUE (isCanceled) event
    * disconnecting was canceled
    * set result to ABORTED
    * handle end of sequence
    */

   _result = BTS_REQ_DISCONNECT_ABORTED;

   handleDisconnectFinished();

   return 0;
}

int ProtocolDisconnect::handleExitNoRetry()
{
   ETG_TRACE_USR3((" [SM]: handleExitNoRetry()"));

   /*
    * status: 100%
    *
    * triggered by EXIT_NO_RETRY event
    * handle end of sequence
    */

   handleDisconnectFinishedNew();

   return 0;
}

int ProtocolDisconnect::initSm()
{
   ETG_TRACE_USR3((" [SM]: initSm()"));

   /*
    * status: 100%
    *
    * reset member variables
    */

   _address.clear();
   _protocol = BTS_PROTO_LAST;
   _sppInstance = 0;
   _uuid.clear();
   _masInstanceName.clear();
   _masInstanceId = 0;
   _pauseBtStreaming = false;
   _cancel = false;
   _final = false;
   _disconnecting = false;
   _result = BTS_REQ_LAST;
   _status = BTS_REQ_LAST;

   // reset basic data
   resetSmfWrapperData();

   return 0;
}

int ProtocolDisconnect::isCanceled()
{
   ETG_TRACE_USR3((" [SM]: isCanceled()"));

   /*
    * status: 100%
    *
    * check if action was canceled
    */

   if(true == _cancel)
   {
      return 1;
   }
   else
   {
      return 0;
   }
}

int ProtocolDisconnect::isFinalStateReached(const BTSRequestResult result)
{
   ETG_TRACE_USR3((" [SM]: isFinalStateReached(): result=%d", result));

   /*
    * status: 100%
    *
    * check if final state was reached
    * store given result
    */

   _result = result;

   if(true == _final)
   {
      return 1;
   }
   else
   {
      if(false == _disconnecting)
      {
         return 1;
      }
      else
      {
         return 0;
      }
   }
}

int ProtocolDisconnect::sendDisconnect()
{
   ETG_TRACE_USR3((" [SM]: sendDisconnect()"));

   FW_ERRMEM_IF_NULL_PTR_RETURN_NULL(_protocolManagerRequest);
   FW_ERRMEM_IF_NULL_PTR_RETURN_NULL(_protocolSmHelper);

   /*
    * status: 100%
    *
    * triggered by DISCONNECT event
    * send local disconnect request
    * start timer if not active
    */

   if(true == _testBlockSendingDisconnect)
   {
      // reset test flag
      _testBlockSendingDisconnect = false;
   }
   else
   {
      ::std::vector< Bts2App_BaseMessage* > _tempBts2AppList;
      _protocolManagerRequest->disconnect(_bts2IpcList, _tempBts2AppList, _address, _protocol, _sppInstance, _uuid, _masInstanceId, _masInstanceName, _pauseBtStreaming);
      FW_ERRMEM_ASSERT(0 == _tempBts2AppList.size());
   }

   if(false == _protocolSmHelper->isConnectTimerActive(_address, _protocol, _uuid, _masInstanceId, false))
   {
      _protocolSmHelper->startConnectTimer(_address, _protocol, _uuid, _masInstanceId, false);
   }

   return 0;
}

int ProtocolDisconnect::storeFinalState(const BTSRequestResult result)
{
   ETG_TRACE_USR3((" [SM]: storeFinalState(): result=%d", result));

   /*
    * status: 100%
    *
    * triggered by FINAL_STATE event
    * store given result
    * set flag for final state update
    */

   _status = result;
   _final = true;

   return 0;
}

//===================================================================================================================

void ProtocolDisconnect::handleDisconnectFinished(void)
{
   // nothing
}

void ProtocolDisconnect::handleDisconnectFinishedNew(void)
{
   FW_ERRMEM_IF_NULL_PTR_RETURN(_protocolSmHelper);

   /*
    * status: 100%
    *
    * disconnect sequence is finished
    * stop timer
    * if status is set to success then disconnect was successful
    * calculate disconnect reason
    * calculate disconnection status
    * set connection status
    * update connection status
    * update connection result
    */

   _protocolSmHelper->stopConnectTimer(_address, _protocol, _uuid, _masInstanceId, false);

   if(BTS_REQ_SUCCESS == _status)
   {
      _result = BTS_REQ_SUCCESS;
   }

   if((true == _cancel) && (BTS_REQ_SUCCESS != _result))
   {
      _result = BTS_REQ_DISCONNECT_ABORTED;
   }

   if((BTS_REQ_SUCCESS != _result) && (BTS_REQ_DISCONNECT_ABORTED != _result))
   {
      // disconnect failed, also after retries, set to success to avoid device disconnect failed issues
      _result = BTS_REQ_SUCCESS;

      // write error memory entry
      ETG_TRACE_ERRMEM((" #CONN: BtStackIf: disconnect for device=%12s profile=%d uuid=%32s instanceId=%u failed", _address.c_str(), _protocol, _uuid.c_str(), _masInstanceId));
   }

   BTSDisconnectReason reason;

   if(BTS_REQ_SUCCESS == _result)
   {
      // disconnect success => disconnect was triggered from local side (ignore disconnect reason from stack)
      reason = BTS_DISCONNECT_REASON_NORMAL_LOSS_LOCAL;
   }
   else
   {
      // disconnect failed => protocol still connected
      reason = BTS_DISCONNECT_REASON_NOT_VALID;
   }

   const bool disconnected(BTS_REQ_SUCCESS == _result);
   _protocolSmHelper->setConnectionStatus(_address, _protocol, _uuid, _masInstanceId, (false == disconnected), reason);
   _protocolSmHelper->updateConnectionStatus(_bts2AppStatusList, _address, _protocol, _uuid, _masInstanceId, disconnected);
   _protocolSmHelper->updateConnectionResult(_bts2AppStatusList, _address, _protocol, _uuid, _masInstanceId, false, _result);
   _protocolSmHelper->handleChangedProtocolConnectionStatus(_bts2IpcList, _bts2AppStatusList, _messageItem, _address, reason);
}

void ProtocolDisconnect::sendExitNoRetryEvent(void)
{
   ETG_TRACE_USR3((" [SM]: sendExitNoRetryEvent()"));

   FW_ERRMEM_ASSERT(0 == SendEvent(EXIT_NO_RETRY, 0));
}

void ProtocolDisconnect::sendRetryDisconnectEvent(void)
{
   ETG_TRACE_USR3((" [SM]: sendRetryDisconnectEvent()"));

   FW_ERRMEM_ASSERT(0 == SendEvent(RETRY_DISCONNECT, 0));
}

} //btstackif
