/**
 * @file ProtocolConnect.cpp
 *
 * @par SW-Component
 * State machine for protocol manager
 *
 * @brief Implementation of generic protocol connect state machine.
 *
 * @copyright (C) 2017 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Source file for implementation of generic protocol connect state machine.
 */

#include "ProtocolConnect.h"
#include "IProtocolManager.h"
#include "IDeviceManager.h"
#include "IServiceSearch.h"
#include "IProtocolManagerRequest.h"
#include "IProtocolSmHelper.h"
#include "FwErrmemPrint.h"
#include "TraceClasses.h"
#include "FwTrace.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_BTS_SM
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/ProtocolConnect.cpp.trc.h"
#endif
#endif

namespace btstackif {

ProtocolConnect::ProtocolConnect() :
_protocolManager(0),
_deviceManager(0),
_serviceSearch(0),
_protocolManagerRequest(0),
_protocolSmHelper(0),
_address(),
_protocol(BTS_PROTO_LAST),
_sppInstance(0),
_outUuid(),
_inUuid(),
_masInstanceName(),
_masInstanceId(0),
_cancel(false),
_final(false),
_finalDisconnected(false),
_connecting(false),
_disconnecting(false),
_localConnectOngoing(false),
_localConnectSent(false),
_result(BTS_REQ_LAST),
_remoteConnectForwarded(false),
_remoteConnectReceived(false),
_acceptReceived(false),
_rejectReceived(false),
_deviceConnected(false),
_noAclFailed(false),
_restricted(false),
_smProcessingActive(false),
_testDisableSearching(false),
_testDisableSetting(false),
_pageTimeout(0)
{
}

ProtocolConnect::~ProtocolConnect()
{
   _protocolManager = 0;
   _deviceManager = 0;
   _serviceSearch = 0;
   _protocolManagerRequest = 0;
   _protocolSmHelper = 0;
}

//===================================================================================================================

void ProtocolConnect::setProtocolManager(IN IProtocolManager* manager)
{
   _protocolManager = manager;

   FW_ERRMEM_ASSERT(0 != _protocolManager);
}

void ProtocolConnect::setDeviceManager(IN IDeviceManager* manager)
{
   _deviceManager = manager;

   FW_ERRMEM_ASSERT(0 != _deviceManager);
}

void ProtocolConnect::setServiceSearch(IN IServiceSearch* manager)
{
   _serviceSearch = manager;

   FW_ERRMEM_ASSERT(0 != _serviceSearch);
}

void ProtocolConnect::setProtocolManagerRequest(IN IProtocolManagerRequest* manager)
{
   _protocolManagerRequest = manager;

   FW_ERRMEM_ASSERT(0 != _protocolManagerRequest);
}

void ProtocolConnect::setProtocolSmHelper(IN IProtocolSmHelper* helper)
{
   _protocolSmHelper = helper;

   FW_ERRMEM_ASSERT(0 != _protocolSmHelper);
}

void ProtocolConnect::resetStateMachine(void)
{
   ETG_TRACE_USR3((" [SM]: resetStateMachine()"));

   resetSm();
}

void ProtocolConnect::sendStartEvent(void)
{
   ETG_TRACE_USR3((" [SM]: sendStartEvent()"));

   FW_ERRMEM_ASSERT(0 == SendEvent(START, 0));
}

void ProtocolConnect::sendConnectEvent(void)
{
   ETG_TRACE_USR3((" [SM]: sendConnectEvent()"));

   FW_ERRMEM_ASSERT(0 == SendEvent(CONNECT, 0));
}

void ProtocolConnect::sendAcceptEvent(void)
{
   ETG_TRACE_USR3((" [SM]: sendAcceptEvent()"));

   FW_ERRMEM_ASSERT(0 == SendEvent(ACCEPT_CONNECT, 0));
}

void ProtocolConnect::sendRejectEvent(void)
{
   ETG_TRACE_USR3((" [SM]: sendRejectEvent()"));

   FW_ERRMEM_ASSERT(0 == SendEvent(REJECT_CONNECT, 0));
}

void ProtocolConnect::sendCancelEvent(void)
{
   ETG_TRACE_USR3((" [SM]: sendCancelEvent()"));

   FW_ERRMEM_ASSERT(0 == SendEvent(CANCEL, 0));
}

void ProtocolConnect::sendWaitingForServiceSuccessEvent(void)
{
   ETG_TRACE_USR3((" [SM]: sendWaitingForServiceSuccessEvent()"));

   FW_ERRMEM_ASSERT(0 == SendEvent(WAITING_FOR_SERVICE_SUCCESS, 0));
}

void ProtocolConnect::sendWaitingForServiceFailedEvent(void)
{
   ETG_TRACE_USR3((" [SM]: sendWaitingForServiceFailedEvent()"));

   FW_ERRMEM_ASSERT(0 == SendEvent(WAITING_FOR_SERVICE_FAILED, 0));
}

void ProtocolConnect::sendCreateSuccessEvent(void)
{
   ETG_TRACE_USR3((" [SM]: sendCreateSuccessEvent()"));

   FW_ERRMEM_ASSERT(0 == SendEvent(CREATE_SUCCESS, 0));
}

void ProtocolConnect::sendCreateFailedEvent(void)
{
   ETG_TRACE_USR3((" [SM]: sendCreateFailedEvent()"));

   FW_ERRMEM_ASSERT(0 == SendEvent(CREATE_FAILED, 0));
}

void ProtocolConnect::sendSearchSuccessEvent(void)
{
   ETG_TRACE_USR3((" [SM]: sendSearchSuccessEvent()"));

   FW_ERRMEM_ASSERT(0 == SendEvent(SEARCH_SUCCESS, 0));
}

void ProtocolConnect::sendSearchFailedEvent(const BTSRequestResult result)
{
   ETG_TRACE_USR3((" [SM]: sendSearchFailedEvent(): result=%d", result));

   char parameters[getMarshalIntSize()];
   FW_ERRMEM_ASSERT(0 == ParameterSEARCH_FAILED(parameters, sizeof(parameters), result));
   FW_ERRMEM_ASSERT(0 == SendEvent(SEARCH_FAILED, parameters));
}

void ProtocolConnect::sendWaitingForProtocolSuccessEvent(void)
{
   ETG_TRACE_USR3((" [SM]: sendWaitingForProtocolSuccessEvent()"));

   FW_ERRMEM_ASSERT(0 == SendEvent(WAITING_FOR_PROTOCOL_SUCCESS, 0));
}

void ProtocolConnect::sendWaitingForProtocolFailedEvent(void)
{
   ETG_TRACE_USR3((" [SM]: sendWaitingForProtocolFailedEvent()"));

   FW_ERRMEM_ASSERT(0 == SendEvent(WAITING_FOR_PROTOCOL_FAILED, 0));
}

void ProtocolConnect::sendSettingSuccessEvent(void)
{
   ETG_TRACE_USR3((" [SM]: sendSettingSuccessEvent()"));

   FW_ERRMEM_ASSERT(0 == SendEvent(SETTING_SUCCESS, 0));
}

void ProtocolConnect::sendSettingFailedEvent(void)
{
   ETG_TRACE_USR3((" [SM]: sendSettingFailedEvent()"));

   FW_ERRMEM_ASSERT(0 == SendEvent(SETTING_FAILED, 0));
}

void ProtocolConnect::sendBusyEvent(void)
{
   ETG_TRACE_USR3((" [SM]: sendBusyEvent()"));

   FW_ERRMEM_ASSERT(0 == SendEvent(BUSY, 0));
}

void ProtocolConnect::sendRemoteConnectEvent(void)
{
   ETG_TRACE_USR3((" [SM]: sendRemoteConnectEvent()"));

   FW_ERRMEM_ASSERT(0 == SendEvent(CONNECT_EXTERN, 0));
}

void ProtocolConnect::sendConnectingEvent(void)
{
   ETG_TRACE_USR3((" [SM]: sendConnectingEvent()"));

   FW_ERRMEM_ASSERT(0 == SendEvent(CONNECTING, 0));
}

void ProtocolConnect::sendDisconnectingEvent(void)
{
   ETG_TRACE_USR3((" [SM]: sendDisconnectingEvent()"));

   FW_ERRMEM_ASSERT(0 == SendEvent(DISCONNECTING, 0));
}

void ProtocolConnect::sendCanceledEvent(void)
{
   ETG_TRACE_USR3((" [SM]: sendCanceledEvent()"));

   FW_ERRMEM_ASSERT(0 == SendEvent(CANCELED, 0));
}

void ProtocolConnect::sendConnectResultEvent(const BTSRequestResult result)
{
   ETG_TRACE_USR3((" [SM]: sendConnectResultEvent(): result=%d", result));

   char parameters[getMarshalIntSize()];
   FW_ERRMEM_ASSERT(0 == ParameterCONNECT_RESULT(parameters, sizeof(parameters), result));
   FW_ERRMEM_ASSERT(0 == SendEvent(CONNECT_RESULT, parameters));
}

void ProtocolConnect::sendFinalStateEvent(const BTSRequestResult result)
{
   ETG_TRACE_USR3((" [SM]: sendFinalStateEvent(): result=%d", result));

   char parameters[getMarshalIntSize()];
   FW_ERRMEM_ASSERT(0 == ParameterFINAL_STATE(parameters, sizeof(parameters), result));
   FW_ERRMEM_ASSERT(0 == SendEvent(FINAL_STATE, parameters));
}

void ProtocolConnect::sendCancelResultEvent(const BTSRequestResult result)
{
   ETG_TRACE_USR3((" [SM]: sendCancelResultEvent(): result=%d", result));

   char parameters[getMarshalIntSize()];
   FW_ERRMEM_ASSERT(0 == ParameterCANCEL_RESULT(parameters, sizeof(parameters), result));
   FW_ERRMEM_ASSERT(0 == SendEvent(CANCEL_RESULT, parameters));
}

void ProtocolConnect::sendGuardPeriodExpiredEvent(void)
{
   ETG_TRACE_USR3((" [SM]: sendGuardPeriodExpiredEvent()"));

   FW_ERRMEM_ASSERT(0 == SendEvent(GUARD_PERIOD_EXPIRED, 0));
}

void ProtocolConnect::sendCharacterDeviceAvailableEvent(void)
{
   ETG_TRACE_USR3((" [SM]: sendCharacterDeviceAvailableEvent()"));

   FW_ERRMEM_ASSERT(0 == SendEvent(CHARACTER_DEVICE_AVAILABLE, 0));
}

void ProtocolConnect::sendSetPageTimeoutSuccessEvent(void)
{
   ETG_TRACE_USR3((" [SM]: sendSetPageTimeoutSuccessEvent()"));

   FW_ERRMEM_ASSERT(0 == SendEvent(SET_PAGE_TIMEOUT_SUCCESS, 0));
}

void ProtocolConnect::setConnectionData(IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& outgoingUuid, IN const BTSUuid& incomingUuid, IN const BTSMasInstanceName& masInstanceName, IN const BTSMasInstanceId masInstanceId, IN const bool deviceConnected)
{
   ETG_TRACE_USR3((" [SM]: setConnectionData()"));

   _address = address;
   _protocol = protocol;
   _sppInstance = sppInstance;
   _outUuid = outgoingUuid;
   _inUuid = incomingUuid;
   _masInstanceName = masInstanceName;
   _masInstanceId = masInstanceId;
   setDeviceConnectionStatus(deviceConnected);
}

void ProtocolConnect::setPageTimeout(IN const BTSTimeValue pageTimeout)
{
   ETG_TRACE_USR3((" [SM]: setPageTimeout()"));

   _pageTimeout = pageTimeout;
}

void ProtocolConnect::setDeviceConnectionStatus(IN const bool deviceConnected)
{
   ETG_TRACE_USR3((" [SM]: setDeviceConnectionStatus(): deviceConnected=%d", deviceConnected));

   // set only to true
   if(true == deviceConnected)
   {
      _deviceConnected = deviceConnected;
   }
}

void ProtocolConnect::setMasInstanceNameAndId(IN const BTSMasInstanceName& masInstanceName, IN const BTSMasInstanceId masInstanceId)
{
   ETG_TRACE_USR3((" [SM]: setMasInstanceNameAndId()"));

   _masInstanceName = masInstanceName;
   _masInstanceId = masInstanceId;
}

void ProtocolConnect::setRestrictedConnecting(void)
{
   ETG_TRACE_USR3((" [SM]: setRestrictedConnecting()"));

   _restricted = true;
}

bool ProtocolConnect::doConnectSmProcessing(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN BtStackIfCallback* user /*= 0*/, IN const BTSSessionHandle handle /*= 0*/)
{
   // avoid recursion
   if(true == _smProcessingActive)
   {
      return false;
   }
   else
   {
      _smProcessingActive = true;
      const bool result(doSmProcessing(bts2IpcMsgList, bts2AppMsgList, messageItem, user, handle));
      _smProcessingActive = false;
      return result;
   }
}

bool ProtocolConnect::isConnectingState(OUT bool& localConnect) const
{
   localConnect = _localConnectSent;

   return Connecting == GetState();
}

bool ProtocolConnect::isDisconnectedState(void) const
{
   return Disconnected == GetState();
}

void ProtocolConnect::setTestDisableSearching(IN const bool enable)
{
   _testDisableSearching = enable;
}

void ProtocolConnect::setTestDisableSetting(IN const bool enable)
{
   _testDisableSetting = enable;
}

//===================================================================================================================

int ProtocolConnect::checkCreating()
{
   ETG_TRACE_USR3((" [SM]: checkCreating()"));

   FW_ERRMEM_IF_NULL_PTR_RETURN_NULL(_protocolManager);
   FW_ERRMEM_IF_NULL_PTR_RETURN_NULL(_deviceManager);

   /*
    * status: 100%
    *
    * triggered by WAITING_FOR_SERVICE_SUCCESS event
    * indicates local connect but local connect request will be sent during later step in the sequence
    * check is needed for available device / protocol
    */

   // check for available protocol TODO: [low]: timer is needed for protocol available
   if(false ==_protocolManager->isProtocolAvailable(_address, _protocol, _outUuid, _masInstanceName, false))
   {
      // create device is needed
      ::std::vector< Bts2App_BaseMessage* > _tempBts2AppList;
      _deviceManager->createDevice(_bts2IpcList, _tempBts2AppList, _address);
      FW_ERRMEM_ASSERT(0 == _tempBts2AppList.size());
      // no state change
   }
   else
   {
      // device / protocol is already available
      // change to next state
      sendCreateSuccessEvent();
   }

   return 0;
}

int ProtocolConnect::checkSearching()
{
   ETG_TRACE_USR3((" [SM]: checkSearching()"));

   FW_ERRMEM_IF_NULL_PTR_RETURN_NULL(_protocolManagerRequest);
   FW_ERRMEM_IF_NULL_PTR_RETURN_NULL(_serviceSearch);

   /*
    * status: 100%
    *
    * triggered by SET_PAGE_TIMEOUT_SUCCESS event
    * check is needed for service search
    */

   BTSSearchType searchType(BTS_SEARCH_ALL);
   if((false == _testDisableSearching) && (true == _protocolManagerRequest->isServiceSearchNeeded(searchType, _address, _protocol, _sppInstance, _outUuid, _masInstanceId, _masInstanceName)))
   {
      // service search is needed before starting connect
      ::std::vector< Bts2App_BaseMessage* > _tempBts2AppList;
      const bool result(_serviceSearch->startSearch(_bts2IpcList, _tempBts2AppList, _address, searchType, _pageTimeout));
      FW_ERRMEM_ASSERT(0 == _tempBts2AppList.size());
      FW_ERRMEM_ASSERT(true == result);
      // no state change
   }
   else
   {
      // no service search needed
      // change to next state
      sendSearchSuccessEvent();

      // reset test flag
      _testDisableSearching = false;
   }

   return 0;
}

int ProtocolConnect::checkSetting()
{
   ETG_TRACE_USR3((" [SM]: checkSetting()"));

   FW_ERRMEM_IF_NULL_PTR_RETURN_NULL(_protocolManagerRequest);
   FW_ERRMEM_IF_NULL_PTR_RETURN_NULL(_protocolSmHelper);

   /*
    * status: 100%
    *
    * triggered by WAITING_FOR_PROTOCOL_SUCCESS event
    * check is needed for setting UUID
    */

   if((false == _testDisableSetting) && (true == _protocolManagerRequest->isSettingUuidNeeded(_address, _protocol, _sppInstance, _outUuid, _masInstanceId, _masInstanceName)))
   {
      // setting UUID necessary
      ::std::vector< Bts2App_BaseMessage* > _tempBts2AppList;
      _protocolSmHelper->setSppUuid(_bts2IpcList, _tempBts2AppList, _address, _sppInstance, _outUuid, _inUuid);
      FW_ERRMEM_ASSERT(0 == _tempBts2AppList.size());
      // no state change
   }
   else
   {
      // setting UUID not necessary
      // change to next state
      sendSettingSuccessEvent();

      // reset test flag
      _testDisableSetting = false;
   }

   return 0;
}

int ProtocolConnect::checkSettingPageTimeout()
{
   ETG_TRACE_USR3((" [SM]: checkSettingPageTimeout()"));

   FW_ERRMEM_IF_NULL_PTR_RETURN_NULL(_protocolSmHelper);

   /*
    * status: 100%
    *
    * triggered by CREATE_SUCCESS event
    * configure page timeout
    */

   ::std::vector< Bts2App_BaseMessage* > _tempBts2AppList;

   if(true == _protocolSmHelper->configurePageTimeout(_bts2IpcList, _tempBts2AppList, _address, _protocol, _outUuid, _masInstanceId, _pageTimeout))
   {
      // wait for end of configuration
   }
   else
   {
      // continue
      sendSetPageTimeoutSuccessEvent();
   }

   FW_ERRMEM_ASSERT(0 == _tempBts2AppList.size());

   return 0;
}

int ProtocolConnect::checkWaitingForProtocol()
{
   ETG_TRACE_USR3((" [SM]: checkWaitingForProtocol()"));

   FW_ERRMEM_IF_NULL_PTR_RETURN_NULL(_protocolManager);
   FW_ERRMEM_IF_NULL_PTR_RETURN_NULL(_protocolManagerRequest);

   /*
    * status: 100%
    *
    * triggered by SEARCH_SUCCESS event
    * check is needed for available protocol
    */

   if(false ==_protocolManager->isProtocolAvailable(_address, _protocol, _outUuid, _masInstanceName, true))
   {
      // waiting is needed
      ::std::vector< Bts2Ipc_BaseMessage* > _tempBts2IpcList;
      ::std::vector< Bts2App_BaseMessage* > _tempBts2AppList;
      _protocolManagerRequest->waitForProtocolAvailable(_tempBts2IpcList, _tempBts2AppList, _address, _protocol, _sppInstance, _outUuid, _masInstanceId, _masInstanceName);
      FW_ERRMEM_ASSERT(0 == _tempBts2IpcList.size());
      FW_ERRMEM_ASSERT(0 == _tempBts2AppList.size());
      // no state change
   }
   else
   {
      // no waiting necessary
      // change to next state
      sendWaitingForProtocolSuccessEvent();
   }

   return 0;
}

int ProtocolConnect::checkWaitingForService()
{
   ETG_TRACE_USR3((" [SM]: checkWaitingForService()"));

   FW_ERRMEM_IF_NULL_PTR_RETURN_NULL(_protocolManagerRequest);

   /*
    * status: 100%
    *
    * triggered by CONNECT event
    * indicates local connect but local connect request will be sent during later step in sequence
    * check is needed for available service
    */

   if(false == _protocolManagerRequest->isServiceAvailable(_address, _protocol, _sppInstance, _outUuid, _masInstanceId, _masInstanceName))
   {
      // waiting is needed
      ::std::vector< Bts2Ipc_BaseMessage* > _tempBts2IpcList;
      ::std::vector< Bts2App_BaseMessage* > _tempBts2AppList;
      _protocolManagerRequest->waitForServiceAvailable(_tempBts2IpcList, _tempBts2AppList, _address, _protocol, _sppInstance, _outUuid, _masInstanceId, _masInstanceName);
      FW_ERRMEM_ASSERT(0 == _tempBts2IpcList.size());
      FW_ERRMEM_ASSERT(0 == _tempBts2AppList.size());
      // no state change
   }
   else
   {
      // service is already available
      // change to next state
      sendWaitingForServiceSuccessEvent();
   }

   return 0;
}

int ProtocolConnect::entryBusy()
{
   ETG_TRACE_USR3((" [SM]: entryBusy()"));

   FW_ERRMEM_IF_NULL_PTR_RETURN_NULL(_protocolSmHelper);

   /*
    * status: 100%
    *
    * triggered by BUSY event
    * check if retry is allowed
    * if yes start timer
    * if not send CONNECT_RESULT event to end the sequence
    */

   if(true == _protocolSmHelper->isRetryAllowed(_address, _protocol, _outUuid, _masInstanceId))
   {
      _protocolSmHelper->startRetryTimer(_address, _protocol, _outUuid, _masInstanceId);
   }
   else
   {
      sendConnectResultEvent(BTS_REQ_CONNECT_PROTOCOL_FAILED);
   }

   return 0;
}

int ProtocolConnect::entryCheckForCharacterDevice()
{
   ETG_TRACE_USR3((" [SM]: entryCheckForCharacterDevice()"));

   FW_ERRMEM_IF_NULL_PTR_RETURN_NULL(_protocolSmHelper);

   /*
    * status: 100%
    *
    * check whether character device was already reported by udev monitor
    */

   if(BTS_REQ_SUCCESS != _result)
   {
      // connect failed or aborted => continue
      sendCharacterDeviceAvailableEvent();
   }
   else if(true == _protocolSmHelper->wasCharacterDeviceReported(_address, _protocol, _outUuid, _masInstanceId))
   {
      // character device is available => continue
      sendCharacterDeviceAvailableEvent();
   }
   else
   {
      // character device is not available => start timer
      _protocolSmHelper->startWaitForCharacterDeviceTimer(_address, _protocol, _outUuid, _masInstanceId);
   }

   return 0;
}

int ProtocolConnect::entryCheckForRetry()
{
   ETG_TRACE_USR3((" [SM]: entryCheckForRetry()"));

   FW_ERRMEM_IF_NULL_PTR_RETURN_NULL(_protocolSmHelper);

   /*
    * status: 100%
    *
    * check if retry shall be executed
    * general precondition for retry:
    * - local connect was sent
    * - not restricted
    * - not canceled
    * - not connected
    * - max retry not reached (retry allowed)
    *
    * precondition for retry (1):
    * - disconnect reason different than BTS_DISCONNECT_REASON_MISSING_LINK_KEY
    * - ACL was connected during the connect sequence
    * - only for first protocol (=> to cover connect failed due to REMOTE_DISCONNECT_ACL)
    *
    * precondition for retry (2):
    * - disconnect reason equals BTS_DISCONNECT_REASON_CONFLICT_DIFFERENT_DEVICE
    * - limit to HFP, AVP, SPP, PAN
    *
    * precondition for retry (3):
    * - disconnect reason equals BTS_DISCONNECT_REASON_CONFLICT_SAME_DEVICE
    * - ACL was connected during the connect sequence
    * - limit to HFP, AVP, SPP, PAN
    *
    * precondition for retry (4):
    * - disconnect reason different than BTS_DISCONNECT_REASON_MISSING_LINK_KEY
    * - ACL was connected during the connect sequence
    * - connect timer active
    * - limit to HFP, AVP, SPP, PAN
    *
    * stop timer
    * reset internal members
    */

   // check for disconnect reason BTS_DISCONNECT_REASON_MISSING_LINK_KEY
   const BTSDisconnectReason stackReason(_protocolSmHelper->getDisconnectReason(_address, _protocol, _outUuid, _masInstanceId));

   bool goToEnd(true);

   if((true == _localConnectSent) && (false == _restricted) && (false == _cancel) && (BTS_REQ_SUCCESS != _result) && (true == _protocolSmHelper->isRetryAllowed(_address, _protocol, _outUuid, _masInstanceId)))
   {
      if((BTS_DISCONNECT_REASON_MISSING_LINK_KEY != stackReason) && (true == _deviceConnected) && (false == _protocolSmHelper->isAnyProtocolConnected(_address)))
      {
         // start retry
         goToEnd = false;

         ETG_TRACE_USR1((" [SM]: entryCheckForRetry(): retry (1)"));
      }
      else if((BTS_DISCONNECT_REASON_CONFLICT_DIFFERENT_DEVICE == stackReason) && (true == isRetryProtocol(_protocol)))
      {
         // start retry
         goToEnd = false;

         ETG_TRACE_USR1((" [SM]: entryCheckForRetry(): retry (2)"));
      }
      else if((BTS_DISCONNECT_REASON_CONFLICT_SAME_DEVICE == stackReason) && (true == _deviceConnected) && (true == isRetryProtocol(_protocol)))
      {
         // start retry
         goToEnd = false;

         ETG_TRACE_USR1((" [SM]: entryCheckForRetry(): retry (3)"));
      }
      else if((BTS_DISCONNECT_REASON_MISSING_LINK_KEY != stackReason) && (true == _deviceConnected) && (true == _protocolSmHelper->isConnectTimerActive(_address, _protocol, _outUuid, _masInstanceId, true)) && (true == isRetryProtocol(_protocol)))
      {
         // start retry
         goToEnd = false;

         ETG_TRACE_USR1((" [SM]: entryCheckForRetry(): retry (4)"));
      }
   }

   _protocolSmHelper->stopConnectTimer(_address, _protocol, _outUuid, _masInstanceId, true);

   // exit or retry?
   if(true == goToEnd)
   {
      sendExitNoRetryEvent();
   }
   else
   {
      // reset internal members
      _final = false;
      _finalDisconnected = false;
      _connecting = false;
      _disconnecting = false;
      _localConnectOngoing = false;
      _localConnectSent = false;
      _result = BTS_REQ_LAST;
      _remoteConnectForwarded = false;
      _remoteConnectReceived = false;
      _acceptReceived = false;
      _rejectReceived = false;
      // do not reset --- _deviceConnected = false;
      _noAclFailed = false;
      _restricted = false;

      sendRetryConnectEvent();
   }

   return 0;
}

int ProtocolConnect::entryDisconnected()
{
   ETG_TRACE_USR3((" [SM]: entryDisconnected()"));

   FW_ERRMEM_IF_NULL_PTR_RETURN_NULL(_protocolManagerRequest);

   /*
    * status: 100%
    *
    * call prepare
    */

   _protocolManagerRequest->prepareProtocolConnect(_bts2IpcList, _bts2AppStatusList, _address, _protocol, _sppInstance, _outUuid, _masInstanceId, _masInstanceName);

   return 0;
}

int ProtocolConnect::exitBusy()
{
   ETG_TRACE_USR3((" [SM]: exitBusy()"));

   FW_ERRMEM_IF_NULL_PTR_RETURN_NULL(_protocolSmHelper);

   /*
    * status: 100%
    *
    * triggered by CONNECT_RESULT, CONNECT_EXTERN or CONNECT event
    * stop timer
    */

   _protocolSmHelper->stopRetryTimer(_address, _protocol, _outUuid, _masInstanceId);

   return 0;
}

int ProtocolConnect::handleBusyCanceled(const BTSRequestResult result)
{
   ETG_TRACE_USR3((" [SM]: handleBusyCanceled()"));

   /*
    * status: 100%
    *
    * triggered by CONNECT_RESULT event
    * store result
    * handle end of sequence
    */

   _result = result;

   handleConnectFinished();

   return 0;
}

int ProtocolConnect::exitConnecting()
{
   ETG_TRACE_USR3((" [SM]: exitConnecting()"));

   /*
    * status: 100%
    *
    * exit connecting state
    * reset flag for local connect
    */

   _localConnectOngoing = false;

   return 0;
}

int ProtocolConnect::handleCancel()
{
   ETG_TRACE_USR3((" [SM]: handleCancel()"));

   /*
    * status: 100%
    *
    * triggered by CANCEL event
    * store cancel event
    */

   _cancel = true;

   return 0;
}

int ProtocolConnect::handleCancelDuringBusy()
{
   ETG_TRACE_USR3((" [SM]: handleCancelDuringBusy()"));

   FW_ERRMEM_IF_NULL_PTR_RETURN_NULL(_protocolManagerRequest);
   FW_ERRMEM_IF_NULL_PTR_RETURN_NULL(_protocolSmHelper);

   /*
    * status: 100%
    *
    * triggered by CANCEL event
    * store cancel event
    * create virtual failed connect result message to end the sequence in context of Ipc2Bts message processing
    * stop timer
    */

   _cancel = true;

   ::std::vector< Bts2App_BaseMessage* > _tempBts2AppList;
   _protocolManagerRequest->sendVirtualFailedConnectResult(_bts2IpcList, _tempBts2AppList, _address, _protocol, _sppInstance, _outUuid, _masInstanceId, _masInstanceName, BTS_IPC_RETRY_ABORTED);
   FW_ERRMEM_ASSERT(0 == _tempBts2AppList.size());

   _protocolSmHelper->stopRetryTimer(_address, _protocol, _outUuid, _masInstanceId);

   return 0;
}

int ProtocolConnect::handleCancelDuringConnecting()
{
   ETG_TRACE_USR3((" [SM]: handleCancelDuringConnecting()"));

   /*
    * status: 100%
    *
    * triggered by CANCEL event during connecting state
    * store cancel event
    * check whether cancel shall be continued:
    * - do not continue if remote connection request was rejected
    */

   _cancel = true;

   bool continueCancel(true);

   if(true == _rejectReceived)
   {
      continueCancel = false;
   }

   if(true == continueCancel)
   {
      sendContinueCancelEvent();
   }

   return 0;
}

int ProtocolConnect::handleCancelDuringSearching()
{
   ETG_TRACE_USR3((" [SM]: handleCancelDuringSearching()"));

   FW_ERRMEM_IF_NULL_PTR_RETURN_NULL(_protocolManagerRequest);
   FW_ERRMEM_IF_NULL_PTR_RETURN_NULL(_serviceSearch);

   /*
    * status: 100%
    *
    * triggered by CANCEL event during searching state
    * store cancel event
    * try to cancel the service search sequence
    */

   _cancel = true;

   BTSSearchType searchType(BTS_SEARCH_ALL);
   if(true == _protocolManagerRequest->isServiceSearchNeeded(searchType, _address, _protocol, _sppInstance, _outUuid, _masInstanceId, _masInstanceName))
   {
      // cancel service search
      ::std::vector< Bts2App_BaseMessage* > _tempBts2AppList;
      const bool result(_serviceSearch->cancelSearch(_bts2IpcList, _tempBts2AppList, _address, searchType));
      FW_ERRMEM_ASSERT(0 == _tempBts2AppList.size());
      FW_ERRMEM_ASSERT(true == result);
      // no state change
   }
   else
   {
      // there is no service search for other protocols
      FW_ERRMEM_ASSERT_ALWAYS();
      // no state change
   }

   return 0;
}

int ProtocolConnect::handleCharacterDeviceAvailable()
{
   ETG_TRACE_USR3((" [SM]: handleCharacterDeviceAvailable()"));

   FW_ERRMEM_IF_NULL_PTR_RETURN_NULL(_protocolSmHelper);

   /*
    * status: 100%
    *
    * stop timer
    * handle end of sequence
    */

   _protocolSmHelper->stopWaitForCharacterDeviceTimer(_address, _protocol, _outUuid, _masInstanceId);

   handleConnectFinishedNew();

   return 0;
}

int ProtocolConnect::handleConnectCompleted(const BTSRequestResult result)
{
   ETG_TRACE_USR3((" [SM]: handleConnectCompleted(): result=%d", result));

   /*
    * status: 100%
    *
    * triggered by FINAL_STATE event
    * if called: triggered by final connection state update (connected/disconnected)
    * store result
    * handle end of sequence
    */

   _result = result;

   handleConnectFinished();

   return 0;
}

int ProtocolConnect::handleConnectFinalState(const BTSRequestResult result)
{
   ETG_TRACE_USR3((" [SM]: handleConnectFinalState(): result=%d", result));

   /*
    * status: 100%
    *
    * triggered by TRUE (isFinalStateReached) event
    * if called: triggered by connect result; final connection state was updated before (stored in _result)
    * ignore result
    * handle end of sequence
    */

   (void)(result);

   handleConnectFinished();

   return 0;
}

int ProtocolConnect::handleConnecting()
{
   ETG_TRACE_USR3((" [SM]: handleConnecting()"));

   /*
    * status: 100%
    *
    * triggered by CONNECTING event
    * connecting state was indicated, store information
    */

   _connecting = true;

   return 0;
}

int ProtocolConnect::handleConnectingCanceled()
{
   ETG_TRACE_USR3((" [SM]: handleConnectingCanceled()"));

   /*
    * status: 100%
    *
    * triggered by TRUE (isCanceled) or CANCEL event
    * connecting was canceled
    * set result to ABORTED
    * handle end of sequence
    */

   _result = BTS_REQ_CONNECT_ABORTED;

   handleConnectFinished();

   return 0;
}

int ProtocolConnect::handleContinueCancel()
{
   ETG_TRACE_USR3((" [SM]: handleContinueCancel()"));

   FW_ERRMEM_IF_NULL_PTR_RETURN_NULL(_protocolManagerRequest);

   /*
    * status: 100%
    *
    * triggered by CONTINUE_CANCEL event during connecting state
    * store cancel event
    * reset disconnecting flag because a disconnecting indication will follow after sending cancel request
    * reset final state disconnected flag because a disconnected state update will follow after sending cancel request
    * try to cancel the connect sequence
    */

   _cancel = true;
   _disconnecting = false;
   _finalDisconnected = false;

   ::std::vector< Bts2App_BaseMessage* > _tempBts2AppList;
   _protocolManagerRequest->cancel(_bts2IpcList, _tempBts2AppList, _address, _protocol, _sppInstance, _outUuid, _masInstanceId, _masInstanceName);
   FW_ERRMEM_ASSERT(0 == _tempBts2AppList.size());

   return 0;
}

int ProtocolConnect::handleCreateCanceled()
{
   ETG_TRACE_USR3((" [SM]: handleCreateCanceled()"));

   /*
    * status: 100%
    *
    * triggered by TRUE (isCanceled) event
    * creating was canceled
    * set result to ABORTED
    * handle end of sequence
    */

   _result = BTS_REQ_CONNECT_ABORTED;

   handleConnectFinished();

   return 0;
}

int ProtocolConnect::handleCreateFailed()
{
   ETG_TRACE_USR3((" [SM]: handleCreateFailed()"));

   /*
    * status: 100%
    *
    * triggered by CREATE_FAILED event
    * creating failed
    * set result to PROTOCOL_FAILED (ACL_FAILED makes no sense because we did not try any connect)
    * handle end of sequence
    */

   _result = BTS_REQ_CONNECT_PROTOCOL_FAILED;
   _noAclFailed = true;

   handleConnectFinished();

   return 0;
}

int ProtocolConnect::handleDisconnectedReached(const BTSRequestResult result)
{
   ETG_TRACE_USR3((" [SM]: handleDisconnectedReached()"));

   /*
    * status: 100%
    *
    * triggered by CANCEL_RESULT (TRUE path) event
    * ignore result
    */

   (void)(result);

   return 0;
}

int ProtocolConnect::handleDisconnecting()
{
   ETG_TRACE_USR3((" [SM]: handleDisconnecting()"));

   /*
    * status: 100%
    *
    * triggered by DISCONNECTING event
    * disconnecting state was indicated, store information
    * reset final state disconnected flag because a disconnected state update will follow after disconnecting
    */

   _disconnecting = true;
   _finalDisconnected = false;

   return 0;
}

int ProtocolConnect::handleExitNoRetry()
{
   ETG_TRACE_USR3((" [SM]: handleExitNoRetry()"));

   /*
    * status: 100%
    *
    * triggered by EXIT_NO_RETRY event
    */

   return 0;
}

int ProtocolConnect::handleFinalStateDuringGuardPeriod(const BTSRequestResult result)
{
   ETG_TRACE_USR3((" [SM]: handleFinalStateDuringGuardPeriod(): result=%d", result));

   /*
    * status: 100%
    *
    * store given result
    * function is called after reception of cancel result, therefore this state update must be the final one, and therefore trigger change to final SM state
    */

   _result = result;

   sendGuardPeriodExpiredEvent();

   return 0;
}

int ProtocolConnect::handleFinalStateDuringWaitForCharacterDevice(const BTSRequestResult result)
{
   ETG_TRACE_USR3((" [SM]: handleFinalStateDuringWaitForCharacterDevice(): result=%d", result));

   FW_ERRMEM_IF_NULL_PTR_RETURN_NULL(_protocolSmHelper);

   /*
    * status: 100%
    *
    * store given result
    * function is called while waiting for availability of character device
    * stop timer
    * handle end of sequence
    */

   _result = result;

   _protocolSmHelper->stopWaitForCharacterDeviceTimer(_address, _protocol, _outUuid, _masInstanceId);

   handleConnectFinishedNew();

   return 0;
}

int ProtocolConnect::handleGuardPeriodExpired()
{
   ETG_TRACE_USR3((" [SM]: handleGuardPeriodExpired()"));

   /*
    * status: 100%
    *
    * triggered by GUARD_PERIOD_EXPIRED event
    */

   return 0;
}

int ProtocolConnect::handleRemoteConnect()
{
   ETG_TRACE_USR3((" [SM]: handleRemoteConnect()"));

   FW_ERRMEM_IF_NULL_PTR_RETURN_NULL(_protocolSmHelper);

   /*
    * status: 100%
    *
    * triggered by CONNECT_EXTERN event
    * triggered during remote connect (AUTHORIZE request)
    * check for restricted connecting
    * forward request to application
    * set flag for received remote connect request
    */

   bool reject(false);

   if(false == _remoteConnectForwarded)
   {
      if(true == _restricted)
      {
         // reject directly
         reject = true;
      }
      else
      {
         _protocolSmHelper->createRemoteConnectRequest(_bts2AppStatusList, _address, _protocol, _outUuid, _masInstanceId);
      }
      _remoteConnectForwarded = true;
   }

   _remoteConnectReceived = true;

   // check for reject
   if(true == reject)
   {
      sendRejectEvent();
   }

   return 0;
}

int ProtocolConnect::handleRemoteConnectDuringCanceled()
{
   ETG_TRACE_USR3((" [SM]: handleRemoteConnectDuringCanceled()"));

   FW_ERRMEM_IF_NULL_PTR_RETURN_NULL(_protocolManagerRequest);

   /*
    * status: 100%
    *
    * triggered by CONNECT_EXTERN event
    * triggered during remote connect while connect was canceled
    * reject the external connection request
    */

   ::std::vector< Bts2App_BaseMessage* > _tempBts2AppList;
   _protocolManagerRequest->accept(_bts2IpcList, _tempBts2AppList, _address, _protocol, _sppInstance, _outUuid, _masInstanceId, _masInstanceName, false);
   FW_ERRMEM_ASSERT(0 == _tempBts2AppList.size());

   return 0;
}

int ProtocolConnect::handleRemoteConnectDuringConnecting()
{
   ETG_TRACE_USR3((" [SM]: handleRemoteConnectDuringConnecting()"));

   FW_ERRMEM_IF_NULL_PTR_RETURN_NULL(_protocolManagerRequest);

   /*
    * status: 100%
    *
    * triggered by CONNECT_EXTERN event
    * triggered during remote connect while connect ongoing
    * a) local connect ongoing: reject the external connection request
    * b) remote connect ongoing: accept the external connection request (normal case: CONNECTING was indicated before CONNECT_EXTERN request)
    *
    * related to b) because of SM flow it might be possible that this function is triggered by delayed remote connection request, check received accept/reject
    */

   if((true == _localConnectOngoing) || (true == _rejectReceived))
   {
      ::std::vector< Bts2App_BaseMessage* > _tempBts2AppList;
      _protocolManagerRequest->accept(_bts2IpcList, _tempBts2AppList, _address, _protocol, _sppInstance, _outUuid, _masInstanceId, _masInstanceName, false);
      FW_ERRMEM_ASSERT(0 == _tempBts2AppList.size());
   }
   else
   {
      ::std::vector< Bts2App_BaseMessage* > _tempBts2AppList;
      _protocolManagerRequest->accept(_bts2IpcList, _tempBts2AppList, _address, _protocol, _sppInstance, _outUuid, _masInstanceId, _masInstanceName, true);
      FW_ERRMEM_ASSERT(0 == _tempBts2AppList.size());
   }

   return 0;
}

int ProtocolConnect::handleRemoteConnectDuringSearching()
{
   ETG_TRACE_USR3((" [SM]: handleRemoteConnectDuringSearching()"));

   FW_ERRMEM_IF_NULL_PTR_RETURN_NULL(_protocolManagerRequest);

   /*
    * status: 100%
    *
    * triggered by CONNECT_EXTERN event
    * triggered during remote connect while searching
    * reject the external connection request to avoid conflicting local and remote connect scenario
    */

   ::std::vector< Bts2App_BaseMessage* > _tempBts2AppList;
   _protocolManagerRequest->accept(_bts2IpcList, _tempBts2AppList, _address, _protocol, _sppInstance, _outUuid, _masInstanceId, _masInstanceName, false);
   FW_ERRMEM_ASSERT(0 == _tempBts2AppList.size());

   return 0;
}

int ProtocolConnect::handleRemoteConnectDuringSetting()
{
   ETG_TRACE_USR3((" [SM]: handleRemoteConnectDuringSetting()"));

   FW_ERRMEM_IF_NULL_PTR_RETURN_NULL(_protocolManagerRequest);

   /*
    * status: 100%
    *
    * triggered by CONNECT_EXTERN event
    * triggered during remote connect while setting UUID ongoing
    * reject the external connection request to avoid conflicting local and remote connect scenario
    */

   ::std::vector< Bts2App_BaseMessage* > _tempBts2AppList;
   _protocolManagerRequest->accept(_bts2IpcList, _tempBts2AppList, _address, _protocol, _sppInstance, _outUuid, _masInstanceId, _masInstanceName, false);
   FW_ERRMEM_ASSERT(0 == _tempBts2AppList.size());

   return 0;
}

int ProtocolConnect::handleRemoteConnectDuringWaiting()
{
   ETG_TRACE_USR3((" [SM]: handleRemoteConnectDuringWaiting()"));

   FW_ERRMEM_IF_NULL_PTR_RETURN_NULL(_protocolManagerRequest);

   /*
    * status: 100%
    *
    * triggered by CONNECT_EXTERN event (during waiting)
    * triggered during overlapping remote and local connect situation
    * a) connect result indicated successful: reject the external connection request
    * b) connect result indicated failed: reject the external connection request to avoid conflicting state updates (for local and remote connect)
    */

   ::std::vector< Bts2App_BaseMessage* > _tempBts2AppList;
   _protocolManagerRequest->accept(_bts2IpcList, _tempBts2AppList, _address, _protocol, _sppInstance, _outUuid, _masInstanceId, _masInstanceName, false);
   FW_ERRMEM_ASSERT(0 == _tempBts2AppList.size());

   return 0;
}

int ProtocolConnect::handleRemoteConnectDuringWaitingForProtocol()
{
   ETG_TRACE_USR3((" [SM]: handleRemoteConnectDuringWaitingForProtocol()"));

   FW_ERRMEM_IF_NULL_PTR_RETURN_NULL(_protocolManagerRequest);

   /*
    * status: 100%
    *
    * triggered by CONNECT_EXTERN event
    * triggered during remote connect while waiting for protocol
    * reject the external connection request to avoid conflicting local and remote connect scenario
    */

   ::std::vector< Bts2App_BaseMessage* > _tempBts2AppList;
   _protocolManagerRequest->accept(_bts2IpcList, _tempBts2AppList, _address, _protocol, _sppInstance, _outUuid, _masInstanceId, _masInstanceName, false);
   FW_ERRMEM_ASSERT(0 == _tempBts2AppList.size());

   return 0;
}

int ProtocolConnect::handleRemoteConnecting()
{
   ETG_TRACE_USR3((" [SM]: handleRemoteConnecting()"));

   FW_ERRMEM_IF_NULL_PTR_RETURN_NULL(_protocolSmHelper);

   /*
    * status: 100%
    *
    * triggered by CONNECTING event
    * triggered during remote connect (state update to CONNECTING)
    * connecting state was indicated, store information
    * forward request to application
    */

   if(false == _remoteConnectForwarded)
   {
      _protocolSmHelper->createRemoteConnectRequest(_bts2AppStatusList, _address, _protocol, _outUuid, _masInstanceId);
      _remoteConnectForwarded = true;
   }

  _connecting = true;

   return 0;
}

int ProtocolConnect::handleSearchCanceled()
{
   ETG_TRACE_USR3((" [SM]: handleSearchCanceled()"));

   /*
    * status: 100%
    *
    * triggered by TRUE (isCanceled) event
    * searching was canceled
    * set result to ABORTED
    * handle end of sequence
    */

   _result = BTS_REQ_CONNECT_ABORTED;

   handleConnectFinished();

   return 0;
}

int ProtocolConnect::handleSearchFailed(const BTSRequestResult result)
{
   ETG_TRACE_USR3((" [SM]: handleSearchFailed(): result=%d", result));

   /*
    * status: 100%
    *
    * triggered by SEARCH_FAILED event
    * searching failed: result can be PROTOCOL_NOT_SUPPORTED, FAILED, CONNECT_ACL_FAILED or SUCCESS
    * set result to given value
    * handle end of sequence
    */

   _result = result;

   if(BTS_REQ_FAILED == result)
   {
      // it seems that device could be connected but service search failed, therefore ACL_FAILED makes no sense
      _noAclFailed = true;
   }

   handleConnectFinished();

   return 0;
}

int ProtocolConnect::handleSetPageTimeoutCanceled()
{
   ETG_TRACE_USR3((" [SM]: handleSetPageTimeoutCanceled()"));

   /*
    * status: 100%
    *
    * triggered by TRUE (isCanceled) event
    * set page timeout was canceled
    * set result to ABORTED
    * handle end of sequence
    */

   _result = BTS_REQ_CONNECT_ABORTED;

   handleConnectFinished();

   return 0;
}

int ProtocolConnect::handleSettingCanceled()
{
   ETG_TRACE_USR3((" [SM]: handleSettingCanceled()"));

   /*
    * status: 100%
    *
    * triggered by TRUE (isCanceled) event
    * setting was canceled
    * set result to ABORTED
    * handle end of sequence
    */

   _result = BTS_REQ_CONNECT_ABORTED;

   handleConnectFinished();

   return 0;
}

int ProtocolConnect::handleSettingFailed()
{
   ETG_TRACE_USR3((" [SM]: handleSettingFailed()"));

   /*
    * status: 100%
    *
    * triggered by SETTING_FAILED event
    * setting failed
    * set result to PROTOCOL_FAILED (ACL_FAILED makes no sense because we did not try any connect)
    * handle end of sequence
    */

   _result = BTS_REQ_CONNECT_PROTOCOL_FAILED;
   _noAclFailed = true;

   handleConnectFinished();

   return 0;
}

int ProtocolConnect::handleWaitingForProtocolCanceled()
{
   ETG_TRACE_USR3((" [SM]: handleWaitingForProtocolCanceled()"));

   /*
    * status: 100%
    *
    * triggered by TRUE (isCanceled) event
    * waiting for protocol was canceled
    * set result to ABORTED
    * handle end of sequence
    */

   _result = BTS_REQ_CONNECT_ABORTED;

   handleConnectFinished();

   return 0;
}

int ProtocolConnect::handleWaitingForProtocolFailed()
{
   ETG_TRACE_USR3((" [SM]: handleWaitingForProtocolFailed()"));

   /*
    * status: 100%
    *
    * triggered by WAITING_FOR_PROTOCOL_FAILED event
    * waiting for protocol failed
    * set result to PROTOCOL_FAILED (ACL_FAILED makes no sense because we did not try any connect)
    * handle end of sequence
    */

   _result = BTS_REQ_CONNECT_PROTOCOL_FAILED;
   _noAclFailed = true;

   handleConnectFinished();

   return 0;
}

int ProtocolConnect::handleWaitingForServiceCanceled()
{
   ETG_TRACE_USR3((" [SM]: handleWaitingForServiceCanceled()"));

   /*
    * status: 100%
    *
    * triggered by TRUE (isCanceled) event
    * waiting for service was canceled
    * set result to ABORTED
    * handle end of sequence
    */

   _result = BTS_REQ_CONNECT_ABORTED;

   handleConnectFinished();

   return 0;
}

int ProtocolConnect::handleWaitingForServiceFailed()
{
   ETG_TRACE_USR3((" [SM]: handleWaitingForServiceFailed()"));

   /*
    * status: 100%
    *
    * triggered by WAITING_FOR_SERVICE_FAILED event
    * waiting for service failed
    * set result to PROTOCOL_FAILED (ACL_FAILED makes no sense because we did not try any connect)
    * handle end of sequence
    */

   _result = BTS_REQ_CONNECT_PROTOCOL_FAILED;
   _noAclFailed = true;

   handleConnectFinished();

   return 0;
}

int ProtocolConnect::initSm()
{
   ETG_TRACE_USR3((" [SM]: initSm()"));

   /*
    * status: 100%
    *
    * reset member variables
    */

   _address.clear();
   _protocol = BTS_PROTO_LAST;
   _sppInstance = 0;
   _outUuid.clear();
   _inUuid.clear();
   _masInstanceName.clear();
   _masInstanceId = 0;
   _cancel = false;
   _final = false;
   _finalDisconnected = false;
   _connecting = false;
   _disconnecting = false;
   _localConnectOngoing = false;
   _localConnectSent = false;
   _result = BTS_REQ_LAST;
   _remoteConnectForwarded = false;
   _remoteConnectReceived = false;
   _acceptReceived = false;
   _rejectReceived = false;
   _deviceConnected = false;
   _noAclFailed = false;
   _restricted = false;
   _pageTimeout = 0;

   // reset basic data
   resetSmfWrapperData();

   return 0;
}

int ProtocolConnect::isCanceled()
{
   ETG_TRACE_USR3((" [SM]: isCanceled()"));

   /*
    * status: 100%
    *
    * check if action was canceled
    */

   if(true == _cancel)
   {
      return 1;
   }
   else
   {
      return 0;
   }
}

int ProtocolConnect::isFinalStateDisconnectedReached(const BTSRequestResult result)
{
   ETG_TRACE_USR3((" [SM]: isFinalStateDisconnectedReached(): result=%d", result));

   /*
    * status: 100%
    *
    * check if final state disconnected was reached
    * ignore given result
    */

   (void)(result);

   if(true == _finalDisconnected)
   {
      return 1;
   }
   else
   {
      if(false == _disconnecting)
      {
         return 1;
      }
      else
      {
         return 0;
      }
   }
}

int ProtocolConnect::isFinalStateReached(const BTSRequestResult result)
{
   ETG_TRACE_USR3((" [SM]: isFinalStateReached(): result=%d", result));

   /*
    * status: 100%
    *
    * check if final state was reached
    * store given result
    */

   _result = result;

   if(true == _final)
   {
      return 1;
   }
   else
   {
      if(false == _connecting)
      {
         return 1;
      }
      else
      {
         return 0;
      }
   }
}

int ProtocolConnect::sendAccept()
{
   ETG_TRACE_USR3((" [SM]: sendAccept()"));

   FW_ERRMEM_IF_NULL_PTR_RETURN_NULL(_protocolManagerRequest);

   /*
    * status: 100%
    *
    * action was triggered by accept of application
    * send accept to stack if authorize request was received from stack before
    * no change required w.r.t. originator
    */

   if(true == _remoteConnectReceived)
   {
      ::std::vector< Bts2App_BaseMessage* > _tempBts2AppList;
      _protocolManagerRequest->accept(_bts2IpcList, _tempBts2AppList, _address, _protocol, _sppInstance, _outUuid, _masInstanceId, _masInstanceName, true);
      FW_ERRMEM_ASSERT(0 == _tempBts2AppList.size());
   }

   _acceptReceived = true;

   return 0;
}

int ProtocolConnect::sendConnect()
{
   ETG_TRACE_USR3((" [SM]: sendConnect()"));

   FW_ERRMEM_IF_NULL_PTR_RETURN_NULL(_protocolManagerRequest);
   FW_ERRMEM_IF_NULL_PTR_RETURN_NULL(_protocolSmHelper);

   /*
    * status: 100%
    *
    * send local connect request
    * start timer
    * set flag for local connect ongoing
    * set flag for local connect sent
    * update originator to local => use flag _localConnectSent
    */

   ::std::vector< Bts2App_BaseMessage* > _tempBts2AppList;
   _protocolManagerRequest->connect(_bts2IpcList, _tempBts2AppList, _address, _protocol, _sppInstance, _outUuid, _masInstanceId, _masInstanceName);
   FW_ERRMEM_ASSERT(0 == _tempBts2AppList.size());

   _protocolSmHelper->startConnectTimer(_address, _protocol, _outUuid, _masInstanceId, true);

   _localConnectOngoing = true;
   _localConnectSent = true;

   return 0;
}

int ProtocolConnect::sendDirectAccept()
{
   ETG_TRACE_USR3((" [SM]: sendDirectAccept()"));

   FW_ERRMEM_IF_NULL_PTR_RETURN_NULL(_protocolManagerRequest);

   /*
    * status: 100%
    *
    * send accept directly because action was triggered by authorize request
    * update originator to remote => use flag _localConnectSent
    */

   ::std::vector< Bts2App_BaseMessage* > _tempBts2AppList;
   _protocolManagerRequest->accept(_bts2IpcList, _tempBts2AppList, _address, _protocol, _sppInstance, _outUuid, _masInstanceId, _masInstanceName, true);
   FW_ERRMEM_ASSERT(0 == _tempBts2AppList.size());

   return 0;
}

int ProtocolConnect::sendReject()
{
   ETG_TRACE_USR3((" [SM]: sendReject()"));

   FW_ERRMEM_IF_NULL_PTR_RETURN_NULL(_protocolManagerRequest);

   /*
    * status: 100%
    *
    * action was triggered by reject of application
    * send reject to stack if authorize request was received from stack before
    * no change required w.r.t. originator
    */

   if(true == _remoteConnectReceived)
   {
      ::std::vector< Bts2App_BaseMessage* > _tempBts2AppList;
      _protocolManagerRequest->accept(_bts2IpcList, _tempBts2AppList, _address, _protocol, _sppInstance, _outUuid, _masInstanceId, _masInstanceName, false);
      FW_ERRMEM_ASSERT(0 == _tempBts2AppList.size());
   }

   _rejectReceived = true;

   return 0;
}

int ProtocolConnect::storeFinalState(const BTSRequestResult result)
{
   ETG_TRACE_USR3((" [SM]: storeFinalState(): result=%d", result));

   /*
    * status: 100%
    *
    * store given result
    * set flag for final state update
    * set flag for final state disconnected update if result indicates disconnected state or disconnecting was received before
    */

   _result = result;
   _final = true;

   if((BTS_REQ_SUCCESS != _result) || (true == _disconnecting))
   {
      _finalDisconnected = true;
   }

   return 0;
}

//===================================================================================================================

void ProtocolConnect::handleConnectFinished(void)
{
   // nothing
}

void ProtocolConnect::handleConnectFinishedNew(void)
{
   FW_ERRMEM_IF_NULL_PTR_RETURN(_protocolSmHelper);

   /*
    * status: 100%
    *
    * connect sequence is finished
    * calculate disconnect reason
    * calculate connection status
    * set connection status
    * update connection status
    * update connection result
    */

   if((true == _cancel) && (BTS_REQ_SUCCESS != _result))
   {
      _result = BTS_REQ_CONNECT_ABORTED;
   }

   BTSDisconnectReason reason;

   if(BTS_REQ_SUCCESS == _result)
   {
      reason = _protocolSmHelper->convertConnectResult2DisconnectReason(_result);
   }
   else if(BTS_REQ_CONNECT_ABORTED == _result)
   {
      reason = _protocolSmHelper->convertConnectResult2DisconnectReason(_result);
   }
   else if(BTS_REQ_CONNECT_ACL_FAILED == _result)
   {
      reason = BTS_DISCONNECT_REASON_OUT_OF_RANGE;
   }
   else
   {
      // check for disconnect reason BTS_DISCONNECT_REASON_MISSING_LINK_KEY + BTS_DISCONNECT_REASON_OUT_OF_RANGE
      const BTSDisconnectReason stackReason(_protocolSmHelper->getDisconnectReason(_address, _protocol, _outUuid, _masInstanceId));

      if(BTS_DISCONNECT_REASON_MISSING_LINK_KEY == stackReason)
      {
         _result = BTS_REQ_CONNECT_PROTOCOL_FAILED;
         reason = stackReason;
      }
      else if(BTS_DISCONNECT_REASON_OUT_OF_RANGE == stackReason)
      {
         _result = BTS_REQ_CONNECT_ACL_FAILED;
         reason = stackReason;
      }
      else if(false == _deviceConnected)
      {
         if(true == _noAclFailed)
         {
            // we never send a connect, therefore ACL_FAILED cannot be set as result/reason
            _result = BTS_REQ_CONNECT_PROTOCOL_FAILED;
            reason = _protocolSmHelper->convertConnectResult2DisconnectReason(_result);
         }
         else
         {
            // check stack reason
            if(BTS_DISCONNECT_REASON_PROTO_CONNECT_FAILED == stackReason)
            {
               // other reason than out of range
               _result = BTS_REQ_CONNECT_PROTOCOL_FAILED;
               reason = stackReason;
            }
            else
            {
               // device (ACL) was never connected during the protocol connect sequence
               _result = BTS_REQ_CONNECT_ACL_FAILED;
               reason = BTS_DISCONNECT_REASON_OUT_OF_RANGE;
            }
         }
      }
      else
      {
         // all other failed use cases
         if((BTS_REQ_PROTOCOL_NOT_SUPPORTED == _result) ||
            (BTS_REQ_CONNECT_PROTOCOL_FAILED == _result) ||
            (BTS_REQ_CONNECT_ACL_FAILED == _result) ||
            (BTS_REQ_CONNECT_TIMEOUT == _result) ||
            (BTS_REQ_CONNECT_REJECTED == _result))
         {
            // OK
         }
         else
         {
            // NOK
            _result = BTS_REQ_CONNECT_PROTOCOL_FAILED;
         }

         reason = _protocolSmHelper->convertConnectResult2DisconnectReason(_result);
      }
   }

   const bool connected(BTS_REQ_SUCCESS == _result);
   _protocolSmHelper->setConnectionStatus(_address, _protocol, _outUuid, _masInstanceId, connected, reason);
   _protocolSmHelper->handleChangedProtocolConnectionStatus(_bts2IpcList, _bts2AppStatusList, _messageItem, _address, reason);

   // check access rights for created RFCOMM device
   _protocolSmHelper->checkAccessRightsForCharacterDevice(_address, _protocol, _outUuid, _masInstanceId);

   // do not send any status update or result during restricted connecting
   if(false == _restricted)
   {
      _protocolSmHelper->updateConnectionStatus(_bts2AppStatusList, _address, _protocol, _outUuid, _masInstanceId, connected);
      _protocolSmHelper->updateConnectionResult(_bts2AppStatusList, _address, _protocol, _outUuid, _masInstanceId, true, _result);
   }
}

void ProtocolConnect::sendExitNoRetryEvent(void)
{
   ETG_TRACE_USR3((" [SM]: sendExitNoRetryEvent()"));

   FW_ERRMEM_ASSERT(0 == SendEvent(EXIT_NO_RETRY, 0));
}

void ProtocolConnect::sendRetryConnectEvent(void)
{
   ETG_TRACE_USR3((" [SM]: sendRetryConnectEvent()"));

   FW_ERRMEM_ASSERT(0 == SendEvent(RETRY_CONNECT, 0));
}

void ProtocolConnect::sendContinueCancelEvent(void)
{
   ETG_TRACE_USR3((" [SM]: sendContinueCancelEvent()"));

   FW_ERRMEM_ASSERT(0 == SendEvent(CONTINUE_CANCEL, 0));
}

bool ProtocolConnect::isRetryProtocol(IN const BTSProtocolId protocol) const
{
   if((BTS_PROTO_HFP == protocol) ||
      (BTS_PROTO_AVP == protocol) ||
      (BTS_PROTO_SPP == protocol) ||
      (BTS_PROTO_PAN == protocol))
   {
      return true;
   }
   else
   {
      return false;
   }
}

} //btstackif
