/**
 * @file ProtocolSmEntry.h
 *
 * @par SW-Component
 * State machine for protocol manager
 *
 * @brief Implementation of protocol state machine entry.
 *
 * @copyright (C) 2017 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Header file for implementation of protocol state machine entry.
 */

#ifndef _PROTOCOL_SM_ENTRY_H_
#define _PROTOCOL_SM_ENTRY_H_

#include "BtStackInternalTypes.h"

namespace btstackif {

// class forward declarations
class IProtocolConnect;
class IProtocolDisconnect;
class IConfigurePageTimeout;

/**
 * Protocol state machine entry class.
 */
class ProtocolSmEntry
{
public:
   /**
    * Default constructor.
    */
   ProtocolSmEntry();

   /**
    * Destructor.
    */
   virtual ~ProtocolSmEntry();

   /**
    * Copy constructor.
    *
    * @param[in] ref: reference of message to be copied
    */
   ProtocolSmEntry(const ProtocolSmEntry& ref);

   /**
    * Assignment operator.
    *
    * @param[in] ref: reference of message to be assigned
    */
   ProtocolSmEntry& operator=(const ProtocolSmEntry& ref);

   /**
    * Equality operator.
    *
    * @param[in] ref: reference of message to be compared
    */
   bool operator==(const ProtocolSmEntry& ref) const;

   /**
    * Inequality operator.
    *
    * @param[in] ref: reference of message to be compared
    */
   bool operator!=(const ProtocolSmEntry& ref) const;

   /**
    * Returns in use flag.
    *
    * @return = SM
    */
   inline bool getInUse(void) const { return _inUse; }

   /**
    * Set in use flag.
    *
    * @param[in] enable: in use flag
    */
   inline void setInUse(IN const bool enable) { _inUse = enable; }

   /**
    * Returns SM.
    *
    * @return = SM
    */
   inline IProtocolConnect* getConnectSm(void) { return _smConnect; }

   /**
    * Returns SM.
    *
    * @return = SM
    */
   inline const IProtocolConnect* getConnectSm(void) const { return _smConnect; }

   /**
    * Returns SM.
    *
    * @return = SM
    */
   inline IProtocolDisconnect* getDisconnectSm(void) { return _smDisconnect; }

   /**
    * Returns SM.
    *
    * @return = SM
    */
   inline const IProtocolDisconnect* getDisconnectSm(void) const { return _smDisconnect; }

   /**
    * Set SM.
    *
    * @param[in] sm: SM
    */
   inline void setConnectSm(IN IProtocolConnect* sm) { _smConnect = sm; }

   /**
    * Set SM.
    *
    * @param[in] sm: SM
    */
   inline void setDisconnectSm(IN IProtocolDisconnect* sm) { _smDisconnect = sm; }

   /**
    * Destroy SM.
    */
   void destroySm(void);

   /**
    * Clear SM.
    */
   void clearSm(void);

   /**
    * Checks if any SM is assigned.
    *
    * @return = true/false
    */
   bool anySmAssigned(void) const;

   /**
    * Reset SM.
    */
   void resetSm(void);

   /**
    * Checks if SM is assigned.
    *
    * @param[in] connect: connect/disconnect flag
    *
    * @return = true/false
    */
   bool isSmAssigned(IN const bool connect) const;

   /**
    * Returns SM.
    *
    * @return = SM
    */
   inline IConfigurePageTimeout* getConfigureSm(void) { return _smConfigure; }

   /**
    * Returns SM.
    *
    * @return = SM
    */
   inline const IConfigurePageTimeout* getConfigureSm(void) const { return _smConfigure; }

   /**
    * Set SM.
    *
    * @param[in] sm: SM
    */
   inline void setConfigureSm(IN IConfigurePageTimeout* sm) { _smConfigure = sm; }

private:
   bool _inUse; /**< indicator for used/not used */
   IProtocolConnect* _smConnect; /**< SM for connect */
   IProtocolDisconnect* _smDisconnect; /**< SM for disconnect */
   IConfigurePageTimeout* _smConfigure; /**< SM for configure page timeout */
};

} //btstackif

#endif //_PROTOCOL_SM_ENTRY_H_
